//                                               -*- C++ -*-
/**
 *  @file  LogUniform.cxx
 *  @brief The LogUniform distribution
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2011-04-11 12:32:27 +0200 (lun. 11 avril 2011) $
 *  Id:      $Id: LogUniform.cxx 1866 2011-04-11 10:32:27Z schueller $
 */
#include <cmath>
#include "LogUniform.hxx"
#include "RandomGenerator.hxx"
#include "PersistentObjectFactory.hxx"
#include "Exception.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {

      typedef Base::Stat::RandomGenerator              RandomGenerator;
      typedef Base::Common::NotYetImplementedException NotYetImplementedException;

      CLASSNAMEINIT(LogUniform);

      static Base::Common::Factory<LogUniform> RegisteredFactory("LogUniform");

      /* Default constructor */
      LogUniform::LogUniform()
        : NonEllipticalDistribution("LogUniform"),
          aLog_(-1.0),
          bLog_(1.0),
          a_(exp(-1.0)),
          b_(exp(1.0))
      {
        setDimension( 1 );
        computeRange();
      }

      /* Parameters constructor */
      LogUniform::LogUniform(const NumericalScalar aLog,
                             const NumericalScalar bLog)
        : NonEllipticalDistribution("LogUniform"),
          aLog_(aLog),
          bLog_(bLog),
          a_(exp(aLog_)),
          b_(exp(bLog_))
      {
        if (bLog <= aLog) throw InvalidArgumentException(HERE) << "Error the lower bound aLog of a LogUniform distribution must be lesser than its upper bound bLog, here aLog=" << aLog << " bLog=" << bLog;
        setDimension( 1 );
        computeRange();
      }

      /* Comparison operator */
      Bool LogUniform::operator ==(const LogUniform & other) const
      {
        if (this == &other) return true;
        return (aLog_ == other.aLog_) && (bLog_ == other.bLog_);
      }

      /* String converter */
      String LogUniform::__repr__() const {
        OSS oss;
        oss << "class=" << LogUniform::GetClassName()
            << " name=" << getName()
            << " dimension=" << getDimension()
            << " aLog=" << aLog_
            << " bLog=" << bLog_
            << " a=" << a_
            << " b=" << b_;
        return oss;
      }

      String LogUniform::__str__(const String & offset) const
      {
        OSS oss;
        oss << offset << getClassName() << "(aLog = " << aLog_ << ", bLog = " << bLog_ << ")";
        return oss;
      }

      /* Virtual constructor */
      LogUniform * LogUniform::clone() const
      {
        return new LogUniform(*this);
      }

      /* Compute the numerical range of the distribution given the parameters values */
      void LogUniform::computeRange()
      {
        setRange(Interval(a_, b_));
      }


      /* Get one realization of the distribution */
      LogUniform::NumericalPoint LogUniform::getRealization() const
      {
        return NumericalPoint(1, exp(aLog_ + (bLog_ - aLog_) * RandomGenerator::Generate()));
      }


      /* Get the DDF of the distribution */
      LogUniform::NumericalPoint LogUniform::computeDDF(const NumericalPoint & point) const
      {
        const NumericalScalar x(point[0]);
        if ((x < a_) || (x > b_)) return NumericalPoint(1, 0.0);
        return NumericalPoint(1, -1.0 / (x * x * (bLog_ - aLog_)));
      }


      /* Get the PDF of the distribution */
      NumericalScalar LogUniform::computePDF(const NumericalPoint & point) const
      {
        const NumericalScalar x(point[0]);
        if ((x <= a_) || (x > b_)) return 0.0;
        return 1.0 / (x * (bLog_ - aLog_));
      }


      /* Get the CDF of the distribution */
      NumericalScalar LogUniform::computeCDF(const NumericalPoint & point,
                                             const Bool tail) const
      {
        const NumericalScalar x(point[0]);
        if (x <= a_) return (tail ? 1.0 : 0.0);
        if (x > b_)  return (tail ? 0.0 : 1.0);
        if (tail) return (bLog_ - log(x)) / (bLog_ - aLog_);
        return (log(x) - aLog_) / (bLog_ - aLog_);
      }

      /* Get the characteristic function of the distribution, i.e. phi(u) = E(exp(I*u*X)) */
      NumericalComplex LogUniform::computeCharacteristicFunction(const NumericalScalar x,
                                                                 const Bool logScale) const
      {
        throw NotYetImplementedException(HERE);
      }

      /* Get the PDFGradient of the distribution */
      LogUniform::NumericalPoint LogUniform::computePDFGradient(const NumericalPoint & point) const
      {
        const NumericalScalar x(point[0]);
        if ((x <= a_) || (x > b_)) return 0.0;
        NumericalPoint pdfGradient(2, 0.0);
        const NumericalScalar value(computePDF(point)/ (bLog_ - aLog_));
        pdfGradient[0] = value;
        pdfGradient[1] = -value;
        return pdfGradient;
      }

      /* Get the CDFGradient of the distribution */
      LogUniform::NumericalPoint LogUniform::computeCDFGradient(const NumericalPoint & point) const
      {
        const NumericalScalar x(point[0]);
        if ((x <= a_) || (x > b_)) return 0.0;
        NumericalPoint cdfGradient(2, 0.0);
        const NumericalScalar denominator(pow(bLog_ - aLog_, 2));
        const NumericalScalar logX(log(x));
        cdfGradient[0] = (logX - bLog_) / denominator;
        cdfGradient[1] = (aLog_ - logX) / denominator;
        return cdfGradient;
      }

      /* Get the quantile of the distribution */
      NumericalScalar LogUniform::computeScalarQuantile(const NumericalScalar prob,
                                                        const Bool tail,
                                                        const NumericalScalar precision) const
      {
        if (tail) return exp(bLog_ - prob * (bLog_ - aLog_));
        return exp(aLog_ + prob * (bLog_ - aLog_));
      }

      /* Compute the mean of the distribution */
      void LogUniform::computeMean() const
      {
        mean_ = NumericalPoint(1, (b_ - a_) / (bLog_ - aLog_));
        isAlreadyComputedMean_ = true;
      }

      /* Get the standard deviation of the distribution */
      LogUniform::NumericalPoint LogUniform::getStandardDeviation() const /* throw(NotDefinedException) */
      {
        return NumericalPoint(1, getCovariance()(0, 0));
      }

      /* Get the skewness of the distribution */
      LogUniform::NumericalPoint LogUniform::getSkewness() const /* throw(NotDefinedException) */
      {
        return DistributionImplementation::getSkewness();
      }

      /* Get the kurtosis of the distribution */
      LogUniform::NumericalPoint LogUniform::getKurtosis() const /* throw(NotDefinedException) */
      {
        return DistributionImplementation::getKurtosis();
      }

      /* Compute the covariance of the distribution */
      void LogUniform::computeCovariance() const
      {
        covariance_ = CovarianceMatrix(1);
        const NumericalScalar abLog(bLog_ - aLog_);
        covariance_(0, 0) = 0.5 * (b_ - a_) * (b_ * (abLog - 2.0) + a_ * (abLog + 2.0)) / pow(abLog, 2);
        isAlreadyComputedCovariance_ = true;
      }

      /* Get the moments of the standardized distribution */
      LogUniform::NumericalPoint LogUniform::getStandardMoment(const UnsignedLong n) const
      {
        return NumericalPoint(1, sinh(n) / (2.0 * n));
      }

      /* Parameters value and description accessor */
      LogUniform::NumericalPointWithDescriptionCollection LogUniform::getParametersCollection() const
      {
        NumericalPointWithDescriptionCollection parameters(1);
        NumericalPointWithDescription point(2);
        Description description(point.getDimension());
        point[0] = aLog_;
        point[1] = bLog_;
        description[0] = "aLog";
        description[1] = "bLog";
        point.setDescription(description);
        point.setName(getDescription()[0]);
        parameters[0] = point;
        return parameters;
      }

      void LogUniform::setParametersCollection(const NumericalPointCollection & parametersCollection)
      {
        *this = LogUniform(parametersCollection[0][0], parametersCollection[0][1]);
      }



      /* A accessor */
      void LogUniform::setALog(const NumericalScalar aLog)
      {
        if (aLog != aLog_)
          {
            aLog_ = aLog;
            a_ = exp(aLog_);
            isAlreadyComputedMean_ = false;
            isAlreadyComputedCovariance_ = false;
            computeRange();
          }
      }

      NumericalScalar LogUniform::getALog() const
      {
        return aLog_;
      }


      /* B accessor */
      void LogUniform::setBLog(const NumericalScalar bLog)
      {
        if (bLog != bLog_)
          {
            bLog_ = bLog;
            b_ = exp(bLog);
            isAlreadyComputedMean_ = false;
            isAlreadyComputedCovariance_ = false;
            computeRange();
          }
      }

      NumericalScalar LogUniform::getBLog() const
      {
        return bLog_;
      }

      /* Method save() stores the object through the StorageManager */
      void LogUniform::save(StorageManager::Advocate & adv) const
      {
        NonEllipticalDistribution::save(adv);
        adv.saveAttribute( "aLog_", aLog_ );
        adv.saveAttribute( "bLog_", bLog_ );
        adv.saveAttribute( "a_", a_ );
        adv.saveAttribute( "b_", b_ );
      }

      /* Method load() reloads the object from the StorageManager */
      void LogUniform::load(StorageManager::Advocate & adv)
      {
        NonEllipticalDistribution::load(adv);
        adv.loadAttribute( "aLog_", aLog_ );
        adv.loadAttribute( "bLog_", bLog_ );
        adv.loadAttribute( "a_", a_ );
        adv.loadAttribute( "b_", b_ );
        computeRange();
      }

    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
