//                                               -*- C++ -*-
/**
 *  @file  MultinomialFactory.cxx
 *  @brief Factory for Multinomial distribution
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2011-05-24 19:30:41 +0200 (Tue, 24 May 2011) $
 *  Id:      $Id: MultinomialFactory.cxx 1910 2011-05-24 17:30:41Z schueller $
 */
#include <cmath>
#include "MultinomialFactory.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {

      /* Default constructor */
      MultinomialFactory::MultinomialFactory():
        DistributionImplementationFactory()
      {
        // Nothing to do
      }

      /* Virtual constructor */
      MultinomialFactory * MultinomialFactory::clone() const
      {
        return new MultinomialFactory(*this);
      }


      /* Here is the interface that all derived class must implement */

      Multinomial * MultinomialFactory::build(const NumericalSample & sample) const /* throw(InvalidArgumentException, InternalException) */
      {
        if (sample.getSize() == 0) throw InvalidArgumentException(HERE) << "Error: cannot build a Multinomial distribution from an empty sample";
        // Check if each component of the sample is an integer, compute the mean and extract the maximum value
        UnsignedLong size(sample.getSize());
        UnsignedLong dimension(sample.getDimension());
        NumericalPoint p(dimension, 0.0);
        NumericalScalar max(sample[0][0]);
        for (UnsignedLong i = 0; i < size; i++)
          {
            for (UnsignedLong j = 0; j < dimension; j++)
              {
                NumericalScalar x(sample[i][j]);
                if ((x != trunc(x)) || (x < 0.0)) throw InternalException(HERE) << "Error: can build a Multinomial distribution only from a sample with positive integer components, here sample[" << i << "][" << j << "]=" << x;
                if (x > max) max = x;
                p[j] += x;
              }
          }
        UnsignedLong n((UnsignedLong)max);
        p *= 1.0 / (max * size);
        return Multinomial(n, p).clone();
      }

      Multinomial * MultinomialFactory::build() const /* throw(InvalidArgumentException, InternalException) */
      {
        return Multinomial().clone();
      }


    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
