//                                               -*- C++ -*-
/**
 *  @file  ZipfMandelbrot.cxx
 *  @brief The ZipfMandelbrot distribution
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-07-12 15:45:44 +0200 (lun. 12 juil. 2010) $
 *  Id:      $Id: ZipfMandelbrot.cxx 1581 2010-07-12 13:45:44Z dutka $
 */
#include <cmath>
#include "ZipfMandelbrot.hxx"
#include "DistFunc.hxx"
#include "RandomGenerator.hxx"
#include "PersistentObjectFactory.hxx"
#include "Exception.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {

      typedef Base::Stat::RandomGenerator              RandomGenerator;
      typedef Base::Common::NotYetImplementedException NotYetImplementedException;

      CLASSNAMEINIT(ZipfMandelbrot);

      static Base::Common::Factory<ZipfMandelbrot> RegisteredFactory("ZipfMandelbrot");

      /* Default constructor */
      ZipfMandelbrot::ZipfMandelbrot()
        : DiscreteDistribution("ZipfMandelbrot"),
          n_(1),
          q_(0.0),
          s_(1.0),
          isAlreadyComputedHarmonicNumbers_(false),
          harmonicNumbers_(0)
      {
        // We set the dimension of the ZipfMandelbrot distribution
        setDimension( 1 );
        computeRange();
      }

      /* Parameters constructor */
      ZipfMandelbrot::ZipfMandelbrot(const UnsignedLong n,
                                     const NumericalScalar q,
                                     const NumericalScalar s )
        : DiscreteDistribution("ZipfMandelbrot"),
          n_(n),
          q_(q),
          s_(s),
          isAlreadyComputedHarmonicNumbers_(false),
          harmonicNumbers_(NumericalScalarCollection(0))
      {
        // We set the dimension of the ZipfMandelbrot distribution
        setDimension( 1 );
        computeRange();
      }

      /* Comparison operator */
      Bool ZipfMandelbrot::operator ==(const ZipfMandelbrot & other) const
      {
        if (this == &other) return true;
        return (n_ == other.n_) && (q_ == other.q_) && (s_ == other.s_);
      }

      /* String converter */
      String ZipfMandelbrot::__repr__() const
      {
        OSS oss;
        oss << "class=" << ZipfMandelbrot::GetClassName()
            << " name=" << getName()
            << " dimension=" << getDimension()
            << " n=" << n_
            << " q=" << q_
            << " s=" << s_;
        return oss;
      }

      String ZipfMandelbrot::__str__(const String & offset) const
      {
        OSS oss;
        oss << offset << getClassName() << "(n = " << n_ << ", q = " << q_ << ", s = " << s_ << ")";
        return oss;
      }

      /* Virtual constructor */
      ZipfMandelbrot * ZipfMandelbrot::clone() const
      {
        return new ZipfMandelbrot(*this);
      }

      /* Get one realization of the distribution */
      ZipfMandelbrot::NumericalPoint ZipfMandelbrot::getRealization() const
      {
        const NumericalScalar uniformRealization(1.0 - RandomGenerator::Generate());

        if (!isAlreadyComputedHarmonicNumbers_) computeHarmonicNumbers();

        NumericalScalarCollection::iterator it(lower_bound(harmonicNumbers_.begin(),
                                                           harmonicNumbers_.end(),
                                                           uniformRealization * getHarmonicNumbers(n_))
                                               );
        return NumericalPoint(1, it - harmonicNumbers_.begin() + 1);
      }


      /* Get the PDF of the distribution */
      NumericalScalar ZipfMandelbrot::computePDF(const NumericalPoint & point) const
      {
        const NumericalScalar k(point[0]);

        if ((k < 1 - DiscreteDistribution::SupportEpsilon) || (fabs(k - round(k)) > DiscreteDistribution::SupportEpsilon) || (k > n_ + DiscreteDistribution::SupportEpsilon)) return 0.0;
        return 1.0 / (pow(round(k) + q_, s_) * getHarmonicNumbers(n_) );
      }


      /* Get the CDF of the distribution */
      NumericalScalar ZipfMandelbrot::computeCDF(const NumericalPoint & point,
                                                 const Bool tail) const
      {
        const NumericalScalar k(point[0]);

        if (k < 1 - DiscreteDistribution::SupportEpsilon) return (tail ? 1.0 : 0.0);
        if (k > n_ + DiscreteDistribution::SupportEpsilon) return (tail ? 0.0 : 1.0);

        NumericalScalar value (getHarmonicNumbers(round(k)) / getHarmonicNumbers(n_));

        if (tail) return 1.0 - value;
        return value;

      }

      /* Q accessor */
      NumericalScalar ZipfMandelbrot::getQ() const
      {
        return q_;
      }

      void ZipfMandelbrot::setQ(const NumericalScalar q)
      /* throw(InvalidArgumentException) */
      {
        if (q < 0) throw InvalidArgumentException(HERE) << "q must be >= 0";
	if (q != q_)
	  {
	    q_ = q;
	    isAlreadyComputedMean_ = false;
	    isAlreadyComputedCovariance_ = false;
	    isAlreadyCreatedGeneratingFunction_ = false;
	  }
      }

      /* S accessor */
      NumericalScalar ZipfMandelbrot::getS() const
      {
        return s_;
      }

      void ZipfMandelbrot::setS(const NumericalScalar s)
      /* throw(InvalidArgumentException) */
      {
        if (s <= 0) throw InvalidArgumentException(HERE) << "s must be > 0";
	if (s != s_)
	  {
	    s_ = s;
	    isAlreadyComputedMean_ = false;
	    isAlreadyComputedCovariance_ = false;
	    isAlreadyCreatedGeneratingFunction_ = false;
	  }
      }

      /* N accessor */
      UnsignedLong ZipfMandelbrot::getN() const
      {
        return n_;
      }

      void ZipfMandelbrot::setN(const UnsignedLong n)
      /* throw(InvalidArgumentException) */
      {
        if (n == 0) throw InvalidArgumentException(HERE) << "N must be > 0";
	if (n != n_)
	  {
	    n_ = n;
	    isAlreadyComputedMean_ = false;
	    isAlreadyComputedCovariance_ = false;
	    isAlreadyCreatedGeneratingFunction_ = false;
            computeRange();
          }
      }


      /* Compute the numerical range of the distribution given the parameters values */
      void ZipfMandelbrot::computeRange()
      {
        const NumericalPoint lowerBound(1, 1.0);
        const NumericalPoint upperBound(1, n_);
        const Interval::BoolCollection finiteLowerBound(1, true);
        const Interval::BoolCollection finiteUpperBound(1, true);
        setRange(Interval(lowerBound, upperBound, finiteLowerBound, finiteUpperBound));
      }

      /* Get the support of a discrete distribution that intersect a given interval */
      ZipfMandelbrot::NumericalSample ZipfMandelbrot::getSupport(const Interval & interval) const
      {
        if (interval.getDimension() != getDimension()) throw InvalidArgumentException(HERE) << "Error: the given interval has a dimension that does not match the distribution dimension.";
        const UnsignedLong kMin(static_cast< UnsignedLong > (std::max(ceil(interval.getLowerBound()[0]), 1.0)));
        const UnsignedLong kMax(static_cast< UnsignedLong > (std::min(floor(interval.getUpperBound()[0]), NumericalScalar(n_))));
        NumericalSample result(0, 1);
        for (UnsignedLong k = kMin; k <= kMax; ++k) result.add(NumericalPoint(1, k));
        return result;
      }

      /* Parameters value and description accessor */
      ZipfMandelbrot::NumericalPointWithDescriptionCollection ZipfMandelbrot::getParametersCollection() const
      {
        NumericalPointWithDescriptionCollection parameters(1);
        NumericalPointWithDescription point(3);
        Description description(point.getDimension());
        point[0] = n_;
        point[1] = q_;
        point[2] = s_;
        description[0] = "n";
        description[1] = "q";
        description[2] = "s";
        point.setDescription(description);
        point.setName(getDescription()[0]);
        parameters[0] = point;
        return parameters;
      }

      void ZipfMandelbrot::setParametersCollection(const NumericalPointCollection & parametersCollection)
      {
        *this = ZipfMandelbrot(static_cast< UnsignedLong >(round(parametersCollection[0][0])), parametersCollection[0][1], parametersCollection[0][2]);
      }


      /* Method save() stores the object through the StorageManager */
      void ZipfMandelbrot::save(StorageManager::Advocate & adv) const
      {
        DiscreteDistribution::save(adv);
        adv.saveAttribute( "n_", n_ );
        adv.saveAttribute( "q_", q_ );
        adv.saveAttribute( "s_", s_ );
      }

      /* Method load() reloads the object from the StorageManager */
      void ZipfMandelbrot::load(StorageManager::Advocate & adv)
      {
        DiscreteDistribution::load(adv);
        adv.loadAttribute( "n_", n_ );
        adv.loadAttribute( "q_", q_ );
        adv.loadAttribute( "s_", s_ );
        isAlreadyComputedHarmonicNumbers_ = false;
        harmonicNumbers_ = NumericalScalarCollection(0);
        computeRange();
      }


      /* Method getHarmonicNumbers returns the k-th harmonic number for the current distribution */
      NumericalScalar ZipfMandelbrot::getHarmonicNumbers(UnsignedLong const k ) const
      {
        if (! isAlreadyComputedHarmonicNumbers_)computeHarmonicNumbers();

        if (k < 1) throw InvalidArgumentException(HERE) << "k must be >= 1" ;
        if (k > n_) throw InvalidArgumentException(HERE) << "k must be <= N";

        return harmonicNumbers_[k - 1];
      }

      /* Method  computeHarmonicNumbers computes and stores the  of the k harmonic numbers for
         k = 1..n
         harmonicNumbers_[i] = \sum_{l=1}^i 1./( (i+q)**s )
      */
      void ZipfMandelbrot::computeHarmonicNumbers() const
      {
        harmonicNumbers_ = NumericalScalarCollection(n_);
        harmonicNumbers_[0] = pow(1.0 + q_, -s_);
        for (UnsignedLong i = 2; i <= n_; ++i)
          {
            const NumericalScalar hiqs(pow(i + q_, -s_));
            harmonicNumbers_[i - 1] = harmonicNumbers_[i - 2] + hiqs;
          }
        isAlreadyComputedHarmonicNumbers_ = true;
      }


    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
