//                                               -*- C++ -*-
/**
 *  @file  t_IsoProbabilisticTransformation_EllipticalCopula.cxx
 *  @brief The test file of class IsoProbabilisticTransformation for standard methods
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2011-05-24 19:30:41 +0200 (Tue, 24 May 2011) $
 *  Id:      $Id: t_IsoProbabilisticTransformation_EllipticalCopula.cxx 1910 2011-05-24 17:30:41Z schueller $
 */
#include <iostream>
#include <sstream>
#include <cmath>
#include <iomanip>
#include "OT.hxx"
#include "OTtestcode.hxx"
#include "OStream.hxx"
#include "Normal.hxx"
#include "Uniform.hxx"
#include "Gamma.hxx"
#include "NormalCopula.hxx"
#include "NumericalPoint.hxx"
#include "NumericalSample.hxx"
#include "CorrelationMatrix.hxx"
#include "IdentityMatrix.hxx"
#include "Distribution.hxx"
#include "CenteredFiniteDifferenceGradient.hxx"
#include "CenteredFiniteDifferenceHessian.hxx"
#include "ComposedDistribution.hxx"
#include "IndependentCopula.hxx"

using namespace OT;
using namespace OT::Test;
using namespace OT::Base::Common;
using namespace OT::Base::Type;
using namespace OT::Base::Stat;
using namespace OT::Base::Diff;
using namespace OT::Uncertainty::Model;
using namespace OT::Uncertainty::Distribution;

typedef Distribution::IsoProbabilisticTransformation IsoProbabilisticTransformation;
typedef Distribution::InverseIsoProbabilisticTransformation InverseIsoProbabilisticTransformation;

Matrix clean(Matrix in)
{
  UnsignedLong rowDim(in.getNbRows());
  UnsignedLong colDim(in.getNbColumns());
  for(UnsignedLong i = 0; i < rowDim; i++)
    for(UnsignedLong j = 0; j < colDim; j++)
      {
        in(i, j) = 1.e-6*round(1.e6*in(i, j));
        if (fabs(in(i, j)) < 1.e-6) in(i, j) = 0.0;
      }
  return in;
}

SymmetricTensor clean(SymmetricTensor in)
{
  UnsignedLong rowDim(in.getNbRows());
  UnsignedLong colDim(in.getNbColumns());
  UnsignedLong sheetDim(in.getNbSheets());
  for(UnsignedLong i = 0; i < rowDim; i++)
    for(UnsignedLong j = 0; j < colDim; j++)
      for (UnsignedLong k = 0; k < sheetDim; k++)
        {
          in(i, j, k) = 1.e-4 * round(1.e4*in(i, j, k));
          if (fabs(in(i, j, k)) < 1.e-4) in(i, j, k) = 0.0;
        }
  return in;
}

int main(int argc, char *argv[])
{
  TESTPREAMBLE;
  OStream fullprint(std::cout);
  setRandomGenerator();

  try {
    // Create a collection of distribution
    ComposedDistribution::DistributionCollection aCollection;

    aCollection.add(Uniform(-1.0, 2.0));
    aCollection.add(Gamma(2.0, 2.0, 0.0));

    UnsignedLong dim(aCollection.getSize());

    // Create a copula
    CorrelationMatrix RCopula(dim);
    for (UnsignedLong i = 0; i < dim; i++)
      {
        for (UnsignedLong j = 0; j < i; j++)
          {
            RCopula(i, j) = (i + j + 1.0) / (2.0 * dim);
          }
      }
    ComposedDistribution distribution(aCollection, NormalCopula(RCopula));
    // Test for sampling
    UnsignedLong size = 10000;
    NumericalSample sample = distribution.getNumericalSample( size );
    fullprint << "sample first=" << sample[0] << " last=" << sample[size - 1] << std::endl;
    // Should be close to [0.5, 1, 0.544439]
    fullprint << "sample mean=" << sample.computeMean() << std::endl;
    // Should be close to [0.75, 0, 0, 0, 0.5, 0, 0, 0, 0.739866] for the independent copula
    fullprint << "sample covariance=" << sample.computeCovariance() << std::endl;

    IsoProbabilisticTransformation transform(distribution.getIsoProbabilisticTransformation());
    fullprint << "isoprobabilistic transformation=" << transform << std::endl;
    NumericalSample transformedSample(transform(sample));
    fullprint << "transformed sample first=" << transformedSample[0] << " last=" << transformedSample[size - 1] << std::endl;
    fullprint << "transformed sample mean=" << transformedSample.computeMean() << std::endl;
    fullprint << "transformed sample covariance=" << transformedSample.computeCovariance() << std::endl;

    // Test for evaluation
    InverseIsoProbabilisticTransformation inverseTransform(distribution.getInverseIsoProbabilisticTransformation());
    fullprint << "inverse isoprobabilistic transformation=" << inverseTransform << std::endl;
    NumericalSample transformedBackSample(inverseTransform(transformedSample));
    fullprint << "transformed back sample first=" << transformedBackSample[0] << " last=" << transformedBackSample[size - 1] << std::endl;
    fullprint << "transformed back sample mean=" << transformedBackSample.computeMean() << std::endl;
    fullprint << "transformed back sample covariance=" << transformedBackSample.computeCovariance() << std::endl;
    NumericalPoint point(dim, 0.75);
    fullprint << "point=" << point << std::endl;
    NumericalPoint transformedPoint(transform(point));
    fullprint << "transform value at point        =" << transformedPoint << std::endl;
    fullprint << "transform gradient at point     =" << clean(transform.gradient(point)) << std::endl;
    fullprint << "transform gradient at point (FD)=" << clean(CenteredFiniteDifferenceGradient(1.0e-5, transform.getEvaluationImplementation()).gradient(point)) << std::endl;
    fullprint << "transform hessian at point      =" << std::setprecision(4) << clean(transform.hessian(point)) << std::endl;
    fullprint << "transform hessian at point (FD) =" << std::setprecision(4) << clean(CenteredFiniteDifferenceHessian(1.0e-4, transform.getEvaluationImplementation()).hessian(point)) << std::endl;
    fullprint << "inverse transform value at transformed point        =" << inverseTransform(transformedPoint) << std::endl;
    fullprint << "inverse transform gradient at transformed point     =" << clean(inverseTransform.gradient(transformedPoint)) << std::endl;
    fullprint << "inverse transform gradient at transformed point (FD)=" << clean(CenteredFiniteDifferenceGradient(1.0e-5, inverseTransform.getEvaluationImplementation()).gradient(transformedPoint)) << std::endl;
    fullprint << "inverse transform hessian at transformed point      =" << std::setprecision(4) << clean(inverseTransform.hessian(transformedPoint)) << std::endl;
    fullprint << "inverse transform hessian at transformed point (FD) =" << std::setprecision(4) << clean(CenteredFiniteDifferenceHessian(1.0e-4, inverseTransform.getEvaluationImplementation()).hessian(transformedPoint)) << std::endl;

    // Test for parameters
    fullprint << "parameters gradient at point=" << transform.parametersGradient(point) << std::endl;
    {
      // Validation using finite difference
      NumericalScalar eps(1.0e-5);
      NumericalScalar factor(1.0 / (2.0 * eps));
      Matrix gradient(5, 2);
      NumericalPoint dTdp;
      {
        // dT/dp0
        ComposedDistribution::DistributionCollection coll(aCollection);
        coll[0] = Uniform(-1.0 + eps, 2.0);
        IsoProbabilisticTransformation left(ComposedDistribution(coll, NormalCopula(RCopula)).getIsoProbabilisticTransformation());
        coll[0] = Uniform(-1.0 - eps, 2.0);
        IsoProbabilisticTransformation right(ComposedDistribution(coll, NormalCopula(RCopula)).getIsoProbabilisticTransformation());
        dTdp = (left(point) - right(point)) * factor;
        gradient(0, 0) = dTdp[0];
        gradient(0, 1) = dTdp[1];
      }
      {
        // dT/dp1
        ComposedDistribution::DistributionCollection coll(aCollection);
        coll[0] = Uniform(-1.0, 2.0 + eps);
        IsoProbabilisticTransformation left(ComposedDistribution(coll, NormalCopula(RCopula)).getIsoProbabilisticTransformation());
        coll[0] = Uniform(-1.0, 2.0 - eps);
        IsoProbabilisticTransformation right(ComposedDistribution(coll, NormalCopula(RCopula)).getIsoProbabilisticTransformation());
        dTdp = (left(point) - right(point)) * factor;
        gradient(1, 0) = dTdp[0];
        gradient(1, 1) = dTdp[1];
      }
      {
        // dT/dp2
        ComposedDistribution::DistributionCollection coll(aCollection);
        coll[1] = Gamma(2.0 + eps, 2.0, 0.0);
        IsoProbabilisticTransformation left(ComposedDistribution(coll, NormalCopula(RCopula)).getIsoProbabilisticTransformation());
        coll[1] = Gamma(2.0 - eps, 2.0, 0.0);
        IsoProbabilisticTransformation right(ComposedDistribution(coll, NormalCopula(RCopula)).getIsoProbabilisticTransformation());
        dTdp = (left(point) - right(point)) * factor;
        gradient(2, 0) = dTdp[0];
        gradient(2, 1) = dTdp[1];
      }
      {
        // dT/dp3
        ComposedDistribution::DistributionCollection coll(aCollection);
        coll[1] = Gamma(2.0, 2.0 + eps, 0.0);
        IsoProbabilisticTransformation left(ComposedDistribution(coll, NormalCopula(RCopula)).getIsoProbabilisticTransformation());
        coll[1] = Gamma(2.0, 2.0 - eps, 0.0);
        IsoProbabilisticTransformation right(ComposedDistribution(coll, NormalCopula(RCopula)).getIsoProbabilisticTransformation());
        dTdp = (left(point) - right(point)) * factor;
        gradient(3, 0) = dTdp[0];
        gradient(3, 1) = dTdp[1];
      }
      {
        // dT/dp4
        ComposedDistribution::DistributionCollection coll(aCollection);
        coll[1] = Gamma(2.0, 2.0, 0.0 + eps);
        IsoProbabilisticTransformation left(ComposedDistribution(coll, NormalCopula(RCopula)).getIsoProbabilisticTransformation());
        coll[1] = Gamma(2.0, 2.0, 0.0 - eps);
        IsoProbabilisticTransformation right(ComposedDistribution(coll, NormalCopula(RCopula)).getIsoProbabilisticTransformation());
        dTdp = (left(point) - right(point)) * factor;
        gradient(4, 0) = dTdp[0];
        gradient(4, 1) = dTdp[1];
      }
      fullprint << "parameters gradient (FD)    =" << gradient << std::endl;
    }
    // Test for parameters
    fullprint << "(inverse) parameters gradient at point=" << inverseTransform.parametersGradient(point) << std::endl;
    {
      // Validation using finite difference
      NumericalScalar eps(1.0e-5);
      NumericalScalar factor(1.0 / (2.0 * eps));
      Matrix gradient(5, 2);
      NumericalPoint dTdp;
      {
        // dT/dp0
        ComposedDistribution::DistributionCollection coll(aCollection);
        coll[0] = Uniform(-1.0 + eps, 2.0);
        IsoProbabilisticTransformation left(ComposedDistribution(coll, NormalCopula(RCopula)).getInverseIsoProbabilisticTransformation());
        coll[0] = Uniform(-1.0 - eps, 2.0);
        IsoProbabilisticTransformation right(ComposedDistribution(coll, NormalCopula(RCopula)).getInverseIsoProbabilisticTransformation());
        dTdp = (left(point) - right(point)) * factor;
        gradient(0, 0) = dTdp[0];
        gradient(0, 1) = dTdp[1];
      }
      {
        // dT/dp1
        ComposedDistribution::DistributionCollection coll(aCollection);
        coll[0] = Uniform(-1.0, 2.0 + eps);
        IsoProbabilisticTransformation left(ComposedDistribution(coll, NormalCopula(RCopula)).getInverseIsoProbabilisticTransformation());
        coll[0] = Uniform(-1.0, 2.0 - eps);
        IsoProbabilisticTransformation right(ComposedDistribution(coll, NormalCopula(RCopula)).getInverseIsoProbabilisticTransformation());
        dTdp = (left(point) - right(point)) * factor;
        gradient(1, 0) = dTdp[0];
        gradient(1, 1) = dTdp[1];
      }
      {
        // dT/dp2
        ComposedDistribution::DistributionCollection coll(aCollection);
        coll[1] = Gamma(2.0 + eps, 2.0, 0.0);
        IsoProbabilisticTransformation left(ComposedDistribution(coll, NormalCopula(RCopula)).getInverseIsoProbabilisticTransformation());
        coll[1] = Gamma(2.0 - eps, 2.0, 0.0);
        IsoProbabilisticTransformation right(ComposedDistribution(coll, NormalCopula(RCopula)).getInverseIsoProbabilisticTransformation());
        dTdp = (left(point) - right(point)) * factor;
        gradient(2, 0) = dTdp[0];
        gradient(2, 1) = dTdp[1];
      }
      {
        // dT/dp3
        ComposedDistribution::DistributionCollection coll(aCollection);
        coll[1] = Gamma(2.0, 2.0 + eps, 0.0);
        IsoProbabilisticTransformation left(ComposedDistribution(coll, NormalCopula(RCopula)).getInverseIsoProbabilisticTransformation());
        coll[1] = Gamma(2.0, 2.0 - eps, 0.0);
        IsoProbabilisticTransformation right(ComposedDistribution(coll, NormalCopula(RCopula)).getInverseIsoProbabilisticTransformation());
        dTdp = (left(point) - right(point)) * factor;
        gradient(3, 0) = dTdp[0];
        gradient(3, 1) = dTdp[1];
      }
      {
        // dT/dp4
        ComposedDistribution::DistributionCollection coll(aCollection);
        coll[1] = Gamma(2.0, 2.0, 0.0 + eps);
        IsoProbabilisticTransformation left(ComposedDistribution(coll, NormalCopula(RCopula)).getInverseIsoProbabilisticTransformation());
        coll[1] = Gamma(2.0, 2.0, 0.0 - eps);
        IsoProbabilisticTransformation right(ComposedDistribution(coll, NormalCopula(RCopula)).getInverseIsoProbabilisticTransformation());
        dTdp = (left(point) - right(point)) * factor;
        gradient(4, 0) = dTdp[0];
        gradient(4, 1) = dTdp[1];
      }
      fullprint << "(inverse) parameters gradient (FD)    =" << gradient << std::endl;
    }
  }
  catch (TestFailed & ex) {
    std::cerr << ex << std::endl;
    return ExitCode::Error;
  }


  return ExitCode::Success;
}
