//                                               -*- C++ -*-
/**
 *  @file  UniVariatePolynomialImplementation.hxx
 *  @brief This is a 1D polynomial
 *
 *  (C) Copyright 2005-2012 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-05-21 17:44:02 +0200 (Wed, 21 May 2008) $
 *  Id:      $Id: Object.hxx 818 2008-05-21 15:44:02Z dutka $
 */
#ifndef OPENTURNS_UNIVARIATEPOLYNOMIALIMPLEMENTATION_HXX
#define OPENTURNS_UNIVARIATEPOLYNOMIALIMPLEMENTATION_HXX

#include "OTprivate.hxx"
#include "NumericalPoint.hxx"
#include "Graph.hxx"
#include "Matrix.hxx"
#include "Pointer.hxx"
#include "PersistentObject.hxx"

BEGIN_NAMESPACE_OPENTURNS



/**
 * @class UniVariatePolynomialImplementation
 *
 * This is a 1D polynomial
 */

class UniVariatePolynomialImplementation
  : public PersistentObject
{
  CLASSNAME;
public:

  typedef Pointer<UniVariatePolynomialImplementation> Implementation;
  typedef Matrix::NumericalComplexCollection          NumericalComplexCollection;
  typedef NumericalPoint                              Coefficients;

  /** Default constructor */
  UniVariatePolynomialImplementation();

  /** Constructor from coefficients */
  UniVariatePolynomialImplementation(const Coefficients & coefficients);

  /** Virtual constructor */
  virtual UniVariatePolynomialImplementation * clone() const;

  /** String converter */
  virtual String __repr__() const;
  virtual String __str__(const String & offset = "",
                         const String & variableName = "X") const;

  /** UniVariatePolynomialImplementation are evaluated as functors */
  NumericalScalar operator() (const NumericalScalar x) const;
  NumericalComplex operator() (const NumericalComplex z) const;

  /** UniVariatePolynomialImplementation derivative */
  NumericalScalar derivative(const NumericalScalar x) const;

  /** Compute the derivative of the polynomial */
  UniVariatePolynomialImplementation derivate() const;

  /** Multiply the polynomial P by a NumericalScalar */
  UniVariatePolynomialImplementation operator * (const NumericalScalar scalar) const;

  /** Multiply the polynomial P by a polynomial Q */
  UniVariatePolynomialImplementation operator * (const UniVariatePolynomialImplementation & uniVariatePolynomial) const;

  /** Multiply the polynomial by (x to the power degree) */
  UniVariatePolynomialImplementation incrementDegree (const UnsignedLong degree = 1) const;

  /** Sum of two polynomials of any degree */
  UniVariatePolynomialImplementation operator + (const UniVariatePolynomialImplementation & uniVariatePolynomial) const;

  /** Substraction of two polynomials of any degree */
  UniVariatePolynomialImplementation operator - (const UniVariatePolynomialImplementation & uniVariatePolynomial) const;

  /** Coefficients accessor */
  void setCoefficients(const Coefficients & coefficients);
  Coefficients getCoefficients() const;

  /** Get the degree of the polynomial */
  UnsignedLong getDegree() const;

  /** Root of the polynomial of degree n as the eigenvalues of the associated matrix */
  NumericalComplexCollection getRoots() const;

  /** Method to draw the graph of the polynomial between given bounds */
  Graph draw(const NumericalScalar xMin,
             const NumericalScalar xMax,
             const UnsignedLong pointNumber) const;

  /** Method save() stores the object through the StorageManager */
  void save(Advocate & adv) const;

  /** Method load() reloads the object from the StorageManager */
  void load(Advocate & adv);

protected:

  /** Remove null leading coefficients */
  void compactCoefficients();

  /** The table of polynomial coefficients in ascending order: P(X) = C0 + C1 * X + ... + Cn * X^n */
  Coefficients coefficients_;

private:

} ; /* Class UniVariatePolynomialImplementation */

END_NAMESPACE_OPENTURNS

#endif /* OPENTURNS_UNIVARIATEPOLYNOMIALIMPLEMENTATION_HXX */
