//                                               -*- C++ -*-
/**
 *  @file  GramSchmidtAlgorithm.hxx
 *  @brief Implement the modified Gram Schmidt algorithm to compute the coefficients of
 *         tthe 3 terms recurrence relation of an orthonormal polynomial family
 *
 *  (C) Copyright 2005-2012 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-05-21 17:44:02 +0200 (Wed, 21 May 2008) $
 *  Id:      $Id: Object.hxx 818 2008-05-21 15:44:02Z dutka $
 */
#ifndef OPENTURNS_GRAMSCHMIDTALGORITHM_HXX
#define OPENTURNS_GRAMSCHMIDTALGORITHM_HXX

#include "OrthonormalizationAlgorithmImplementation.hxx"
#include "OrthogonalUniVariatePolynomialFamily.hxx"
#include "UniVariatePolynomial.hxx"
#include "Collection.hxx"

BEGIN_NAMESPACE_OPENTURNS



/**
 * @class GramSchmidtAlgorithm
 *
 * OrthogonalUniVariatePolynomialStandardDistribution polynomial factory
 */

class GramSchmidtAlgorithm
  : public OrthonormalizationAlgorithmImplementation
{
  CLASSNAME;
public:

  typedef Collection<NumericalScalar> NumericalScalarCollection;
  typedef Collection<Coefficients>    CoefficientsCollection;

  /** Default constructor */
  GramSchmidtAlgorithm();

  /** Parameter constructor */
  GramSchmidtAlgorithm(const Distribution & measure);

  /** Parameter constructor */
  GramSchmidtAlgorithm(const Distribution & measure,
                       const OrthogonalUniVariatePolynomialFamily & referenceFamily);

  /** Virtual constructor */
  GramSchmidtAlgorithm * clone() const;

  /** Calculate the coefficients of recurrence a0, a1, a2 such that
      Pn+1(x) = (a0 * x + a1) * Pn(x) + a2 * Pn-1(x) */
  Coefficients getRecurrenceCoefficients(const UnsignedLong n) const;

  /** Reference univariate orthogonal polynomial family accessor */
  void setReferenceFamily(const OrthogonalUniVariatePolynomialFamily & family);
  OrthogonalUniVariatePolynomialFamily getReferenceFamily() const;

  /** String converter */
  String __repr__() const;

  /** Method save() stores the object through the StorageManager */
  virtual void save(Advocate & adv) const;

  /** Method load() reloads the object from the StorageManager */
  virtual void load(Advocate & adv);

private:

  /** Return the order-th raw moment of the underlying measure */
  NumericalScalar getStandardMoment(const UnsignedLong order) const;

  /** Build the coefficients of the kth orthonormal polynomial */
  UniVariatePolynomial buildPolynomial(const UnsignedLong k) const;

  /** Compute the dot product between two general polynomials according to the measure */
  NumericalScalar dotProduct(const UniVariatePolynomial & p1,
                             const UniVariatePolynomial & p2) const;

  /** Cache to store the raw moments */
  mutable NumericalScalarCollection standardMoments_;

  /** Cache to store the coefficients of the orthonormal polynomials */
  mutable CoefficientsCollection coefficientsCache_;

  /** Starting family of polynomials */
  OrthogonalUniVariatePolynomialFamily referenceFamily_;

  /** Flag to tell if we use the canonical basis */
  Bool useCanonicalBasis_;
} ; /* class GramSchmidtAlgorithm */


END_NAMESPACE_OPENTURNS

#endif /* OPENTURNS_GRAMSCHMIDTALGORITHM_HXX */
