//                                               -*- C++ -*-
/**
 *  @file  BetaFactory.cxx
 *  @brief Factory for Beta distribution
 *
 *  (C) Copyright 2005-2012 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2012-02-17 19:35:43 +0100 (Fri, 17 Feb 2012) $
 *  Id:      $Id: BetaFactory.cxx 2392 2012-02-17 18:35:43Z schueller $
 */
#include "BetaFactory.hxx"

BEGIN_NAMESPACE_OPENTURNS



/* Default constructor */
BetaFactory::BetaFactory():
  DistributionImplementationFactory()
{
  // Nothing to do
}

/* Virtual constructor */
BetaFactory * BetaFactory::clone() const
{
  return new BetaFactory(*this);
}


/* Here is the interface that all derived class must implement */

Beta * BetaFactory::build(const NumericalSample & sample) const
{
  const UnsignedLong size(sample.getSize());
  if (size == 0) throw InvalidArgumentException(HERE) << "Error: cannot build a Beta distribution from an empty sample";
  if (sample.getDimension() != 1) throw InvalidArgumentException(HERE) << "Error: can build a Beta distribution only from a sample of dimension 1, here dimension=" << sample.getDimension();
  const NumericalScalar xMin(sample.getMin()[0]);
  const NumericalScalar a(xMin - fabs(xMin) / (2.0 + size));
  const NumericalScalar xMax(sample.getMax()[0]);
  const NumericalScalar b(xMax + fabs(xMax) / (2.0 + size));
  if (a >= b) throw InternalException(HERE) << "Error: can build a Beta distribution only if a < b, here a=" << a << " and b=" << b;
  const NumericalScalar mean(sample.computeMean()[0]);
  const NumericalScalar sigma(sample.computeStandardDeviationPerComponent()[0]);
  const NumericalScalar t((b - mean) * (mean - a) / (sigma * sigma) - 1.0);
  const NumericalScalar r(t * (mean - a) / (b - a));
  return Beta(r, t, a, b).clone();
}

Beta * BetaFactory::build(const NumericalPointCollection & parameters) const
{
  try {
    Beta distribution;
    distribution.setParametersCollection(parameters);
    return distribution.clone();
  }
  catch (InvalidArgumentException & ex)
    {
      throw InternalException(HERE) << "Error: cannot build a Beta distribution from the given parameters";
    }
}

Beta * BetaFactory::build() const
{
  return Beta().clone();
}


END_NAMESPACE_OPENTURNS
