//                                               -*- C++ -*-
/**
 *  @file  LogUniformFactory.cxx
 *  @brief Factory for LogUniform distribution
 *
 *  (C) Copyright 2005-2012 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2011-05-24 19:30:41 +0200 (mar. 24 mai 2011) $
 *  Id:      $Id: LogUniformFactory.cxx 1910 2011-05-24 17:30:41Z schueller $
 */
#include "LogUniformFactory.hxx"

BEGIN_NAMESPACE_OPENTURNS



/* Default constructor */
LogUniformFactory::LogUniformFactory():
  DistributionImplementationFactory()
{
  // Nothing to do
}

/* Virtual constructor */
LogUniformFactory * LogUniformFactory::clone() const
{
  return new LogUniformFactory(*this);
}


/* Here is the interface that all derived class must implement */

LogUniform * LogUniformFactory::build(const NumericalSample & sample) const
{
  const NumericalScalar size(sample.getSize());
  if (size == 0) throw InvalidArgumentException(HERE) << "Error: cannot build a LogUniform distribution from an empty sample";
  if (sample.getDimension() != 1) throw InvalidArgumentException(HERE) << "Error: can build a LogUniform distribution only from a sample of dimension 1, here dimension=" << sample.getDimension();
  const NumericalScalar xMin(sample.getMin()[0]);
  const NumericalScalar a(xMin - fabs(xMin) / (2.0 + size));
  if (a <= 0.0) throw InvalidArgumentException(HERE) << "Error: cannot build a LogUniform distribution from a sample that contains non positive values.";
  const NumericalScalar aLog(log(a));
  const NumericalScalar xMax(sample.getMax()[0]);
  const NumericalScalar b(xMax + fabs(xMax) / (2.0 + size));
  const NumericalScalar bLog(log(b));
  if (aLog >= bLog) throw InternalException(HERE) << "Error: can build a LogUniform distribution only if aLog < bLog, here aLog=" << aLog << " and bLog=" << bLog;
  return LogUniform(aLog, bLog).clone();
}

LogUniform * LogUniformFactory::build(const NumericalPointCollection & parameters) const
{
  try {
    LogUniform distribution;
    distribution.setParametersCollection(parameters);
    return distribution.clone();
  }
  catch (InvalidArgumentException & ex)
    {
      throw InternalException(HERE) << "Error: cannot build a LogUniform distribution from the given parameters";
    }
}

LogUniform * LogUniformFactory::build() const
{
  return LogUniform().clone();
}


END_NAMESPACE_OPENTURNS
