//                                               -*- C++ -*-
/**
 *  @file  LogisticFactory.cxx
 *  @brief Factory for Logistic distribution
 *
 *  (C) Copyright 2005-2012 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2012-02-17 19:35:43 +0100 (Fri, 17 Feb 2012) $
 *  Id:      $Id: LogisticFactory.cxx 2392 2012-02-17 18:35:43Z schueller $
 */
#include "LogisticFactory.hxx"
#include "SpecFunc.hxx"

BEGIN_NAMESPACE_OPENTURNS




/* Default constructor */
LogisticFactory::LogisticFactory():
  DistributionImplementationFactory()
{
  // Nothing to do
}

/* Virtual constructor */
LogisticFactory * LogisticFactory::clone() const
{
  return new LogisticFactory(*this);
}


/* Here is the interface that all derived class must implement */

Logistic * LogisticFactory::build(const NumericalSample & sample) const
{
  if (sample.getSize() == 0) throw InvalidArgumentException(HERE) << "Error: cannot build a Logistic distribution from an empty sample";
  if (sample.getDimension() != 1) throw InvalidArgumentException(HERE) << "Error: can build a Logistic distribution only from a sample of dimension 1, here dimension=" << sample.getDimension();
  NumericalScalar alpha(sample.computeMean()[0]);
  NumericalScalar beta(sample.computeStandardDeviationPerComponent()[0] * SpecFunc::SQRT3_PI);
  if (beta <= 0.0) throw InternalException(HERE) << "Error: can build a Logistic distribution only if beta > 0.0, here beta=" << beta;
  return Logistic(alpha, beta).clone();
}

Logistic * LogisticFactory::build(const NumericalPointCollection & parameters) const
{
  try {
    Logistic distribution;
    distribution.setParametersCollection(parameters);
    return distribution.clone();
  }
  catch (InvalidArgumentException & ex)
    {
      throw InternalException(HERE) << "Error: cannot build a Logistic distribution from the given parameters";
    }
}

Logistic * LogisticFactory::build() const
{
  return Logistic().clone();
}


END_NAMESPACE_OPENTURNS
