//                                               -*- C++ -*-
/**
 * @file  EventDomainImplementation.cxx
 * @brief The class that implements an event random vector (composite) built
 *        upon a boolean function which is described by a domain appartenance.
 *        The antecedent of the event must be a "classical" composite random
 *        vector in the context of OpenTURNS.
 *
 *  (C) Copyright 2005-2012 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 * \author $LastChangedBy: schueller $
 * \date   $LastChangedDate: 2012-02-17 19:35:43 +0100 (Fri, 17 Feb 2012) $
 */

#include "EventDomainImplementation.hxx"
#include "PersistentObjectFactory.hxx"

BEGIN_NAMESPACE_OPENTURNS



CLASSNAMEINIT(EventDomainImplementation);

static Factory<EventDomainImplementation> RegisteredFactory("EventDomainImplementation");
/* Default constructor */
EventDomainImplementation::EventDomainImplementation()
  : CompositeRandomVector()
{
  // Nothing to do
}

/* Constructor from RandomVector */
EventDomainImplementation::EventDomainImplementation(const RandomVectorImplementation & antecedent,
                                                     const Domain & domain,
                                                     const String & name)
  : CompositeRandomVector(name),
    domain_(domain)
{
  // Event can only be constructed from composite random vectors
  if (!antecedent.isComposite())
    throw InvalidArgumentException(HERE) << "Event can only be constructed from composite random vectors. The random vector ("
                                         << antecedent << ") passed as first argument of event '"
                                         << name << "' has incorrect type";
  function_ = antecedent.getFunction();
  p_antecedent_ = antecedent.getAntecedent();
  setName(antecedent.getName());
}

EventDomainImplementation * EventDomainImplementation::clone() const
{
  return new EventDomainImplementation(*this);
}

/* String converter */
String EventDomainImplementation::__repr__() const {
  OSS oss;
  oss << "class=" << EventDomainImplementation::GetClassName()
      << " antecedent=" << CompositeRandomVector::__repr__()
      << " domain=" << domain_;
  return oss;
}

/* Dimension accessor */
UnsignedLong EventDomainImplementation::getDimension() const
{
  return domain_.getDimension();
}

/* Domain accessor */
Domain EventDomainImplementation::getDomain() const
{
  return domain_;
}


/* Realization accessor */
NumericalPoint EventDomainImplementation::getRealization() const
{
  return NumericalPoint(1, domain_.contains(CompositeRandomVector::getRealization()));
}

/* Numerical sample accessor */
NumericalSample EventDomainImplementation::getNumericalSample(UnsignedLong size) const
{
  // We don't build the return sample element by element because it doesn't
  // use the potential distribution of the computation. As the returned
  // sample can be huge, we use it twice in place
  // First, it stores a sample of its antecedent
  NumericalSample returnSample(CompositeRandomVector::getNumericalSample(size));
  // Then, we loop over the sample and substitute realizations of the eventRandomVectorImplementation
  // in place of the realizations of the antecedent
  for (UnsignedLong i = 0; i < size; ++i)
    {
      returnSample[i] = NumericalPoint(1, domain_.contains(returnSample[i]));
    }
  returnSample.setName("EventDomainImplementation sample");
  return returnSample;
}

/* Method save() stores the object through the StorageManager */
void EventDomainImplementation::save(Advocate & adv) const
{
  CompositeRandomVector::save(adv);
  adv.saveAttribute( "domain_", domain_ );
}

/* Method load() reloads the object from the StorageManager */
void EventDomainImplementation::load(Advocate & adv)
{
  CompositeRandomVector::load(adv);
  adv.loadAttribute( "domain_", domain_ );
}

END_NAMESPACE_OPENTURNS
