//                                               -*- C++ -*-
/**
 *  @file  t_Study_save.cxx
 *  @brief The test file of class Study for standard methods
 *
 *  (C) Copyright 2005-2012 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2012-02-17 19:35:43 +0100 (Fri, 17 Feb 2012) $
 *  Id:      $Id: t_Study_save.cxx 2392 2012-02-17 18:35:43Z schueller $
 */
#include <iostream>
#include <sstream>
#include "OT.hxx"
#include "OTtestcode.hxx"
#include "OStream.hxx"
#include "Study.hxx"
#include "XMLStorageManager.hxx"
#include "NumericalPoint.hxx"
#include "NumericalSample.hxx"
#include "NumericalPointWithDescription.hxx"
#include "Matrix.hxx"
#include "LinearModelFactory.hxx"
#include "NearestPointAlgorithm.hxx"
#include "Simulation.hxx"
#include "Beta.hxx"
#include "ComposedDistribution.hxx"
#include "Exponential.hxx"
#include "Gamma.hxx"
#include "Geometric.hxx"
#include "IndependentCopula.hxx"
#include "KernelSmoothing.hxx"
#include "Logistic.hxx"
#include "LogNormal.hxx"
#include "Normal.hxx"
#include "NormalCopula.hxx"
#include "Uniform.hxx"
#include "NumericalMathFunction.hxx"
#include "RandomVector.hxx"
#include "QuadraticCumul.hxx"
#include "FORM.hxx"
#include "SORM.hxx"
#include "Greater.hxx"
#include "UserDefined.hxx"

using namespace OT;
using namespace OT::Test;


int main(int argc, char *argv[])
{
  TESTPREAMBLE;
  OStream fullprint(std::cout);

  const char fileName[] = "myXMLFile";

  try {

    // Create a Study Object
    Study myStudy;
    myStudy.setStorageManager(XMLStorageManager(fileName));


    // Add a PersistentObject to the Study (here a NumericalPoint)
    NumericalPoint p1(3, 0.);
    p1[0] = 10.;
    p1[1] = 11.;
    p1[2] = 12.;
    myStudy.add("Good", p1);

    // Add another PersistentObject to the Study (here a NumericalSample)
    NumericalSample s1(3, 2);
    NumericalPoint p2(2, 0.); p2[0] = 100.; p2[1] = 200.; s1[0] = p2; p2.setName("One");
    NumericalPoint p3(2, 0.); p3[0] = 101.; p3[1] = 201.; s1[1] = p3; p3.setName("Two");
    NumericalPoint p4(2, 0.); p4[0] = 102.; p4[1] = 202.; s1[2] = p4; p4.setName("Three");
    myStudy.add("mySample", s1);

    // Add a point with a description
    NumericalPointWithDescription pDesc(p1);
    Description desc = pDesc.getDescription();
    desc[0] = "x";
    desc[1] = "y";
    desc[2] = "z";
    pDesc.setDescription(desc);
    myStudy.add("pDesc", pDesc);

    // Add a matrix
    Matrix m(2,3);
    m(0,0) = 0; m(0,1) = 1; m(0,2) = 2;
    m(1,0) = 3; m(1,1) = 4; m(1,2) = 5;
    myStudy.add("m", m);

    // Create a linear model
    NumericalSample pred(4,1);
    pred[0] = NumericalPoint(1,0.);
    pred[1] = NumericalPoint(1,1.);
    pred[2] = NumericalPoint(1,2.);
    pred[3] = NumericalPoint(1,3.);
    NumericalSample mes(pred);
    mes.translate(NumericalPoint(1, 2.0));
    LinearModel lm = LinearModelFactory().build(pred,mes);
    myStudy.add("lm", lm);

    // Create a NumericalPoint that we will try to reinstaciate after reloading
    NumericalPoint point(2, 1000.);
    point.setName("point");
    myStudy.add("point", point);

    // Create a NearestPointAlgorithm::Result
    NearestPointAlgorithm::Result nearestPointAlgorithmResult(NumericalPoint(4, 1.0), 15, 1e-10, 2e-10, 3e-10, 4e-10);
    nearestPointAlgorithmResult.setName("nearestPointAlgorithmResult");
    myStudy.add("nearestPointAlgorithmResult", nearestPointAlgorithmResult);

    // Create a Simulation::Result
    SimulationResult simulationResult(Event(), 0.5, 0.01, 150, 4);
    simulationResult.setName("simulationResult");
    myStudy.add("simulationResult", simulationResult);

    // Create a Beta distribution
    Beta beta(3.0, 5.0, -1.0, 4.0);
    myStudy.add("beta", beta);

    // Create an Exponential distribution
    Exponential exponential(3.0, -2.0);
    myStudy.add("exponential", exponential);

    // Create a Gamma distribution
    Gamma gamma(1.5, 3.0, -2.0);
    myStudy.add("gamma", gamma);

    // Create a Geometric distribution
    Geometric geometric(0.15);
    myStudy.add("geometric", geometric);

    // Create a IndependentCopula distribution
    IndependentCopula independentCopula(5);
    myStudy.add("independentCopula", independentCopula);

    // Create a KernelSmoothing algorithm */
    KernelSmoothing kernelSmoothing;
    kernelSmoothing.build(independentCopula.getNumericalSample(20));
    myStudy.add("kernelSmoothing", kernelSmoothing);

    // Create a Logistic distribution
    Logistic logistic(0.5, 1.5);
    myStudy.add("logistic", logistic);

    // Create a LogNormal distribution
    LogNormal logNormal(0.5, 1.5, -1.0);
    myStudy.add("logNormal", logNormal);

    // Create a NormalCopula distribution
    CorrelationMatrix R(3);
    R(0, 1) = 0.5;
    R(1, 2) = 0.5;
    NormalCopula normalCopula(R);
    myStudy.add("normalCopula", normalCopula);

    // Create a Uniform distribution
    Uniform uniform(-1.5, 2.0);
    myStudy.add("uniform", uniform);

    // Create a ComposedDistribution
    ComposedDistribution::DistributionCollection collection;
    collection.add(beta);
    collection.add(gamma);
    collection.add(logistic);
    ComposedDistribution composed(collection, normalCopula);
    myStudy.add("composed", composed);

    // Create a UserDefined distribution
    Collection<UserDefinedPair> pairCollection(3, UserDefinedPair(NumericalPoint(1), 0.0));
    NumericalPoint pairPoint(1);
    pairPoint[0] = 1.0;
    pairCollection[0] = UserDefinedPair(pairPoint,0.3);
    pairPoint[0] = 2.0;
    pairCollection[1] = UserDefinedPair(pairPoint,0.2);
    pairPoint[0] = 3.0;
    pairCollection[2] = UserDefinedPair(pairPoint,0.5);
    UserDefined userDefined(pairCollection);
    myStudy.add("userDefined", userDefined);

    // Create an analytical NumericalMathFunction
    Description input(3);
    input[0] = "a";
    input[1] = "b";
    input[2] = "c";
    Description output(3);
    output[0] = "squaresum";
    output[1] = "prod";
    output[2] = "complex";
    Description formulas(output.getSize());
    formulas[0] = "a+b+c";
    formulas[1] = "a-b*c";
    formulas[2] = "(a+2*b^2+3*c^3)/6";
    NumericalMathFunction analytical(input, output, formulas);
    analytical.setName("analytical");
    myStudy.add("analytical", analytical);

    // Create a QuadraticCumul algorithm
    RandomVector antecedent(IndependentCopula(analytical.getInputDimension()));
    antecedent.setName("antecedent");
    RandomVector composite(analytical, antecedent);
    composite.setName("composite");
    QuadraticCumul quadraticCumul(composite);
    quadraticCumul.setName("quadraticCumul");
    quadraticCumul.getMeanFirstOrder();
    quadraticCumul.getMeanSecondOrder();
    quadraticCumul.getCovariance();
    myStudy.add("quadraticCumul", quadraticCumul);

    // Create a FORMResult
    Description input2(2);
    input2[0] = "x";
    input2[1] = "y";
    Description output2(1);
    output2[0] = "d";
    Description formula2(1);
    formula2[0] = "y^2-x";
    NumericalMathFunction model(input2, output2, formula2);
    model.setName("sum");
    RandomVector input3(Normal(2));
    input3.setName("input");
    RandomVector output3(model, input3);
    output3.setName("output");
    Event event(output3, Greater(), 1.0);
    event.setName("failureEvent");
    NumericalPoint designPoint(2, 0.0);
    designPoint[0] = 1.0;
    FORMResult formResult(NumericalPoint(2, 1.0), event, false);
    formResult.setName("formResult");
    formResult.getImportanceFactors();
    formResult.getImportanceFactors(true);
    formResult.getEventProbabilitySensitivity();
    myStudy.add("formResult", formResult);

    // Create a SORMResult
    SORMResult sormResult(NumericalPoint(2, 1.0), event, false);
    sormResult.setName("sormResult");
    sormResult.getEventProbabilityBreitung();
    sormResult.getEventProbabilityHohenBichler();
    sormResult.getEventProbabilityTvedt();
    sormResult.getGeneralisedReliabilityIndexBreitung();
    sormResult.getGeneralisedReliabilityIndexHohenBichler();
    sormResult.getGeneralisedReliabilityIndexTvedt();
    myStudy.add("sormResult", sormResult);

    fullprint << "Study = " << myStudy << std::endl;

    fullprint << "Save the study..." << std::endl;
    myStudy.save();

  }
  catch (TestFailed & ex) {
    std::cerr << ex << std::endl;
    return ExitCode::Error;
  }




  return ExitCode::Success;
}
