//                                               -*- C++ -*-
/**
 *  @file  t_WhittleFactory_std.cxx
 *  @brief The test file of class WhittleFactory
 *
 *  (C) Copyright 2005-2012 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2012-01-04 12:26:21 +0100 (mer. 04 janv. 2012) $
 *  Id:      $Id: t_WhittleFactory_std.cxx 2302 2012-01-04 11:26:21Z schueller $
 */
#include <iostream>
#include "OT.hxx"
#include "OTtestcode.hxx"

using namespace OT;
using namespace OT::Test;


int main(int argc, char *argv[])
{
  TESTPREAMBLE;
  OStream fullprint(std::cout);

  try
    {

      /* ARMA(p, q) */
      const UnsignedLong p(1);
      const UnsignedLong q(1);

      /* ARMACoefficients intializing */
      NumericalPoint arCoefficients(p, 0.50);
      NumericalPoint maCoefficients(q, 0.20);

      /* ARMA creation */
      ARMA myARMA(ARMACoefficients(arCoefficients), ARMACoefficients(maCoefficients), WhiteNoise(Normal(0.0, 0.05)));
      myARMA.setTimeGrid(RegularGrid(0.0, 0.1, 32));
      fullprint << "myARMA process = " << myARMA << std::endl;

      /* Create a realization */
      TimeSeries timeSeries(myARMA.getRealization());

      /* Create a sample */
      ProcessSample sample(myARMA.getSample(100));

      /* First, build an ARMA based on a given order using the WhittleFactory */
      {
        WhittleFactory factory(p, q);
        factory.setVerbose(false);
        fullprint << "factory=" << factory << std::endl;
        NumericalPoint informationCriteria;
        Process result(factory.build(timeSeries, informationCriteria));
        fullprint << "Estimated ARMA=" << result << std::endl;
        fullprint << "Information criteria=" << informationCriteria << std::endl;
        Process result2(factory.build(sample, informationCriteria));
        fullprint << "Estimated ARMA=" << result2 << std::endl;
        fullprint << "Information criteria=" << informationCriteria << std::endl;
      }

      /* Second, build the best ARMA based on a given range of order using the WhittleFactory */
      Indices pIndices(p + 2);
      pIndices.fill();
      Indices qIndices(q + 2);
      qIndices.fill();
      WhittleFactory factory(pIndices, qIndices);
      fullprint << "factory=" << factory << std::endl;
      NumericalPoint informationCriteria;
      Process result(factory.build(timeSeries, informationCriteria));
      fullprint << "Estimated ARMA=" << result << std::endl;
      fullprint << "Information criteria=" << informationCriteria << std::endl;
      fullprint << "History=" << factory.getHistory()  << std::endl;
      Process result2(factory.build(sample, informationCriteria));
      fullprint << "Estimated ARMA=" << result2 << std::endl;
      fullprint << "Information criteria=" << informationCriteria << std::endl;
      fullprint << "History=" << factory.getHistory()  << std::endl;

      /* Test save / load */
      Study myStudy;
      myStudy.setStorageManager(XMLStorageManager("testStudy.ot"));
      myStudy.add("factory", factory);
      myStudy.save();

      Study myStudy2;
      myStudy2.setStorageManager(XMLStorageManager("testStudy.ot"));
      myStudy2.load();
      WhittleFactory factory2;
      myStudy2.fillObject("factory", factory2);
      fullprint << "loaded factory = " << factory2 << std::endl;

    }
  catch (TestFailed & ex) {
    std::cerr << ex << std::endl;
    return ExitCode::Error;
  }


  return ExitCode::Success;
}
