///////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2012 DreamWorks Animation LLC
//
// All rights reserved. This software is distributed under the
// Mozilla Public License 2.0 ( http://www.mozilla.org/MPL/2.0/ )
//
// Redistributions of source code must retain the above copyright
// and license notice and the following restrictions and disclaimer.
//
// *     Neither the name of DreamWorks Animation nor the names of
// its contributors may be used to endorse or promote products derived
// from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
// IN NO EVENT SHALL THE COPYRIGHT HOLDERS' AND CONTRIBUTORS' AGGREGATE
// LIABILITY FOR ALL CLAIMS REGARDLESS OF THEIR BASIS EXCEED US$250.00.
//
///////////////////////////////////////////////////////////////////////////
//
/// @file math/Frustum.h
/// @author Ken Museth

#ifndef OPENVDB_MATH_FRUSTUM_HAS_BEEN_INCLUDED
#define OPENVDB_MATH_FRUSTUM_HAS_BEEN_INCLUDED

#include <cassert>
#include <string>
#include <iostream>// for cout
#include <iosfwd>
#include <sstream>
#include <algorithm>//for min/max
#include "Maps.h"
#include "Math.h" // for isExactlyEqual()
#include "Vec3.h"
#include "Plane.h"
#include "Ray.h"


namespace openvdb {
OPENVDB_USE_VERSION_NAMESPACE
namespace OPENVDB_VERSION_NAME {
namespace math {

// This class assumes the normal of the front and back faces are
// equal but opposite! The four other faces to not share normals
// Front/back is in the z-direction
// Left/right is in the x-direction
// up/down is in the y-direction

/// @brief Frustum GeClass
template<typename Real>
class Frustum
{
public:
    typedef Vec3<Real>  Point;
    typedef Vec3<Real>  Vector;
    typedef Plane<Real> PlaneType;

private:
    Plane<Real> mFace[6];// 6 faces of frustum

public:
    /// Generates an uninitialized (i.e in-valid) Frustum
OPENVDB_DEPRECATED Frustum() {}

    /// Simple copy constructor
OPENVDB_DEPRECATED Frustum(const Frustum& other)
    {
        mFace[0]=other.mFace[0];
        mFace[1]=other.mFace[1];
        mFace[2]=other.mFace[2];
        mFace[3]=other.mFace[3];
        mFace[4]=other.mFace[4];
        mFace[5]=other.mFace[5];
    }

    /// Constructs a Frustum from the indexToWorld transfrom in a Grid.
    /// GridType (or TransformType) is assumed to have a
    /// indexToWorld(Vector) method! A complicating aspect is that
    /// linear- and frustrum-transforms employ different handedness!
OPENVDB_DEPRECATED Frustum(const Transform& t) { this->init(t); }

    /// Constructor takes the six planes of the Frustum. Note the
    /// ordering: front,back,left,right,up,down
OPENVDB_DEPRECATED Frustum(const PlaneType &front, const PlaneType &back,
            const PlaneType &left,  const PlaneType &right,
            const PlaneType &up,    const PlaneType &down)
    {
        mFace[0]=front;
        mFace[1]=back;
        mFace[2]=left;
        mFace[3]=right;
        mFace[4]=up;
        mFace[5]=down;
    }

    /// @brief Contructor from a Camera Frustum
    ///
    /// @param position is the tip of the frustum (i.e. the camera's position).
    /// @param direction is a unit vector pointing from the center toward the near plane.
    /// @param up,left two non-unit vectors describing the direction and extent of
    ///     the frustum's intersection on the near plane.  Together,
    ///     direction, up and left should form an orthogonal basis for the frustum.
    /// @param zNear,zFar the distance from position along direction to the
    ///     near and far planes of the frustum.
OPENVDB_DEPRECATED Frustum(const Point  &position, //origin = camera position
            const Vector &direction,//z-axis = unit-vector from origin to center
            const Vector &up, //x-axis (=center - upper edge of near plane)
            const Vector &left,//y-axis = center - left edge of near plane
            Real zNear, Real zFar)//two plane offsets along z-axis
    {
        initFromCamera(position, direction, up, left, zNear, zFar);
    }

    /// @brief Constructor for a Frustum from a NonlinearFrustumMap
OPENVDB_DEPRECATED Frustum(const NonlinearFrustumMap& frustum) {

        // an index-space bounding box defines the region transfromed into the frustum
        Vector min = frustum.getBBox().min();
        Vector max = frustum.getBBox().max();
        // compute the four corners of the front face
        Vector ll = frustum.applyMap(min);
        Vector lr = frustum.applyMap(Vec3d(max.x(),min.y(),min.z()));
        Vector ul = frustum.applyMap(Vec3d(min.x(),max.y(),min.z()));
        Vector ur = frustum.applyMap(Vec3d(max.x(),max.y(),min.z()));
        Vector center_near = 0.25*(ll + lr + ul + ur);
        Vector up = 0.5 * (ul + ur ) - center_near;
        Vector left = 0.5 * (ul + ll) - center_near;


        // compute the corners of the far plane
        ll = frustum.applyMap(Vec3d(min.x(), min.y(), max.z()));
        lr = frustum.applyMap(Vec3d(max.x(), min.y(), max.z()));
        ul = frustum.applyMap(Vec3d(min.x(), max.y(), max.z()));
        ur = frustum.applyMap(Vec3d(max.x(), max.y(), max.z()));
        Vector center_far = 0.25*(ll + lr + ul + ur);

        Vector direction = center_far - center_near;
        Real depth = direction.length();
        direction.normalize();
        Real gamma = frustum.getTaper() - 1.;
        Real zNear = 2.*left.length() / gamma;
        Real zFar = zNear + depth;
        Vector translation = frustum.getAffineMap()->applyMap(Vec3d(0,0,0));
        Vector position = translation - zNear*direction;

        initFromCamera(position, direction, up, left, zNear, zFar);
    }

    /// rescales the Frustum to fit the unit coordinates. Note
    /// this method employs linear interpolation and clamps the
    /// unit coordinates to [0;1]!
    void rescaleLerp(const Point &unit_min, const Point &unit_max)
    {
        assert(this->isValid());
        PlaneType tmp;
        // rescaling in z (only positions are modified!)
        tmp      = mFace[0].lerpPoints(mFace[1],  unit_min[2]);//front
        mFace[1] = mFace[1].lerpPoints(mFace[0],1-unit_max[2]);//back
        mFace[0] = tmp;
        /*
           mFace[1].flip();
           tmp = mFace[0].lerp(mFace[1],unit_min[2]);//front
           mFace[0].flip();
           mFace[1].flip();
           mFace[1] = mFace[0].lerp(mFace[1],unit_max[2]);//back
           mFace[0]=tmp;//front
           if (tmp0 != mFace[0]) std::cerr << "Error0: " << tmp0 << " != " << mFace[0] << std::endl;
           if (tmp1 != mFace[1]) std::cerr << "Error1: " << tmp1 << " != " << mFace[1] << std::endl;
           */
        // rescaling in x
        mFace[3].flip();
        tmp = mFace[2].lerp(mFace[3],unit_min[0]);//left
        mFace[2].flip();
        mFace[3].flip();
        mFace[3] = mFace[2].lerp(mFace[3],unit_max[0]);//right
        mFace[2]=tmp;//left

        // rescalig in y
        mFace[5].flip();
        tmp = mFace[5].lerp(mFace[4],unit_max[1]);//up
        mFace[4].flip();
        mFace[5].flip();
        mFace[5] = mFace[5].lerp(mFace[4],unit_min[1]);//down
        mFace[4]=tmp;//up
    }

    /// rescales the Frustum to fit the unit coordinates. Note
    /// this method employs spherical interpolation and clamps the
    /// unit coordinates to [0;1]!
    void rescaleSlerp(const Point &unit_min, const Point &unit_max)
    {
        assert(this->isValid());
        PlaneType tmp;
        // rescaling in z (only positions are modified!)
        tmp      = mFace[0].lerpPoints(mFace[1],  unit_min[2]);//front
        mFace[1] = mFace[1].lerpPoints(mFace[0],1-unit_max[2]);//back
        mFace[0] = tmp;
        /*
           mFace[1].flip();
           PlaneType tmp = mFace[0].slerp(mFace[1],unit_min[2]);//front
           mFace[0].flip();
           mFace[1].flip();
           mFace[1] = mFace[0].slerp(mFace[1],unit_max[2]);//back
           mFace[0] = tmp;//front
           */
        // rescaling in x
        mFace[3].flip();
        tmp = mFace[2].slerp(mFace[3],unit_min[0]);//left
        mFace[2].flip();
        mFace[3].flip();
        mFace[3] = mFace[2].slerp(mFace[3],unit_max[0]);//right
        mFace[2] = tmp;//left

        // rescalig in y
        mFace[5].flip();
        tmp = mFace[5].slerp(mFace[4],unit_max[1]);//up
        mFace[4].flip();
        mFace[5].flip();
        mFace[5] = mFace[5].slerp(mFace[4],unit_min[1]);//down
        mFace[4] = tmp;//up
    }

    inline bool operator==(const Frustum &other) const
    {
        return mFace[0]==other.mFace[0] && mFace[1]==other.mFace[1]
            && mFace[2]==other.mFace[2] && mFace[3]==other.mFace[3]
            && mFace[4]==other.mFace[4] && mFace[5]==other.mFace[5];
    }

    inline bool operator!=(const Frustum &other) const
    {
        return !(*this == other);
    }

    /// @return true if point is inside Frustum, i.e. inside all
    /// six Planes.
    inline bool isInside(const Point &pt) const
    {
        return mFace[0].isInside(pt) && mFace[1].isInside(pt)
            && mFace[2].isInside(pt) && mFace[3].isInside(pt)
            && mFace[4].isInside(pt) && mFace[5].isInside(pt);
    }

    /// @return true if point is outside Frustum, i.e. outside one
    /// of the six Planes.
    inline bool isOutside(const Point &pt) const
    {
        return mFace[0].isOutside(pt) || mFace[1].isOutside(pt)
            || mFace[2].isOutside(pt) || mFace[3].isOutside(pt)
            || mFace[4].isOutside(pt) || mFace[5].isOutside(pt);
    }

    /// @return true if the Frustum is valid, i.e. all six plane
    /// are valid.
    // NB THIS IS NOT A GOOD ENOUGH TEST!!!!!
    inline bool isValid() const
    {
        return mFace[0].isValid() && mFace[1].isValid()
            && mFace[2].isValid() && mFace[3].isValid()
            && mFace[4].isValid() && mFace[5].isValid();
    }

    /// @return copy of one of the 6 half-planes defined by n
    inline PlaneType getFace(int n) const {
        assert(n>=0 && n<6);
        return mFace[n];
    }
    inline PlaneType getPlane(int n) const {return this->getFace(n);}

    /// @return defines one of the 6 half-planes defined by n
    inline void setFace(int n, const PlaneType &P) {
        assert(n>=0 && n<6);
        mFace[n]=P;
    }
    inline void setPlane(int n, const PlaneType &P) {this->setFace(n,P);}

    /// @return reference to one of the 6 half-planes defined by n
    inline PlaneType& face(int n) {
        assert(n>=0 && n<6);
        return mFace[n];
    }
    inline PlaneType& plane(int n) {return face(n);}

    inline static std::string getFaceName(int i) {
        switch(i) {
            case 0 : return "front"; break;
            case 1 : return "back "; break;
            case 2 : return "left "; break;
            case 3 : return "right"; break;
            case 4 : return "upper"; break;
            case 5 : return "lower"; break;
        }
    }

    /// @brief Intersection test against ray
    ///
    /// @note Assumes (six) planes of the frustum forms a convex polyhedron!
    ///
    /// @param ray,t0,t1  t0 = near, t1 = far
    ///
    /// @return  0 if ray does not intersect the frustum at all, i.e. is completely outside!
    ///
    /// @return -1 if ray is completely inside and never intersects the planes
    /// of the frustum, i.e. origin is inside and back-face is beyond ray.tmax
    ///
    /// @return  1 if ray intersects once. There are two such cases:
    /// 1) origin is outside and ray hits front face once (i.e. t0>ray.tmin && t1==ray.tmax)
    /// 2) origin is inside and ray hits back face once (i.e. t0==ray.tmin && t1<ray.tmax)
    ///
    /// @return  2 if ray intersects twice, i.e. origin outside and hitting back face
    /// (i.e t0>ray.tmin and t1<ray.tmax)
    int intersects(const Ray<Real> &ray, Real &t0, Real &t1) const
    {
        t0 = ray.getMinTime();//near
        t1 = ray.getMaxTime();//far
        for (int p=0; p<6; ++p) {// Test each plane of the frustum
            const Real angl = mFace[p].getNorm().dot(ray.getDirection());
            const Real dist = mFace[p].getNorm().dot(ray.getOrigin())-mFace[p].getDist();
            if (isExactlyEqual(angl, 0.0)) {//ray is parallel to plane
                if (dist > 0) return 0;//missed - ray origin is outside plane
            } else {// ray not parallel to plane
                const Real t = -dist / angl ;
                if ( angl < 0 ) {// front face - t is a near point
                    if ( t > t1 ) return 0;// missed - never reached frustum
                    if ( t > t0 ) t0=t;// hit near face, update t0
                } else {// back face - t is a far point - never reached frustum
                    if ( t < t0 ) return 0;//missed - never reached frustum
                    if ( t < t1 ) t1=t;// hit far face, update t1
                }
            }
        }
        // survived all tests so either it hit or it's completely inside
        int hits=0;// origin inside and back face beyond tmax
        if (t0>ray.getMinTime()) ++hits;// outside, hitting front face
        if (t1<ray.getMaxTime()) ++hits;// inside, hitting back face
        //fprintf(stderr,"t0=%f t1=%f hits=%i\n",t0,t1,hits);
        return hits>0 ? hits : -1;
    }

    /// @return true if ray intersects one (or more) of the six
    /// planes of the Frustum OR is completely inside
    bool intersects(const Ray<Real> &ray) const
    {
        Real t0, t1;
        return this->intersects(ray,t0,t1)!=0;
    }

    /// @return true if ray is completely inside the Frustum
    bool isInside(const Ray<Real> &ray) const
    {
        Real t0, t1;
        return this->intersects(ray,t0,t1)==-1;
    }

private:

    /// Constructs a Frustum from the indexToWorld transfrom in a Grid.
    /// GridType (or TransformType) is assumed to have a
    /// indexToWorld(Vector) method! A complicating aspect is that
    /// linear- and frustrum-transforms employ different handedness!
    void init(const Transform& transform)
    {
        const Point P[7]={transform.indexToWorld(Vector(0,0,0)),// 0 origin-near
            transform.indexToWorld(Vector(1,0,0)),// 1 x-near
            transform.indexToWorld(Vector(0,1,0)),// 2 y-near
            transform.indexToWorld(Vector(1,1,0)),// 3 xy-near
            transform.indexToWorld(Vector(0,0,1)),// 4 origin-far
            transform.indexToWorld(Vector(1,0,1)),// 5 x-far
            transform.indexToWorld(Vector(0,1,1))};//6 y-far
        // are unit-vectors left or right handed?
        mFace[0]=PlaneType(P[0],P[1],P[2]);//front (near) face, N = U_x x U_y
        if (mFace[0].getNorm().dot(P[4]-P[0]) >= 0) {//Right-hand,i.e. LinearTransforms
            mFace[0].flip();                   //front (near) face
            //mFace[1]=PlaneType(P[4],P[5],P[6]);//back  (far)
            mFace[2]=PlaneType(P[0],P[4],P[2]);//left
            mFace[3]=PlaneType(P[1],P[3],P[5]);//right
            mFace[4]=PlaneType(P[2],P[6],P[3]);//up
            mFace[5]=PlaneType(P[0],P[1],P[4]);//down
        } else {//Left-hand unit-vectors, i.e. FrustumTransForms
            //mFace[1]=PlaneType(P[4],P[6],P[5]);//back  (far)
            mFace[2]=PlaneType(P[0],P[2],P[4]);//left
            mFace[3]=PlaneType(P[1],P[5],P[3]);//right
            mFace[4]=PlaneType(P[2],P[3],P[6]);//up
            mFace[5]=PlaneType(P[0],P[4],P[1]);//down
        }
        mFace[1].set(P[4],-mFace[0].getNorm());//back (far) has opposit normal of front
    }

    void initFromCamera(const Point  &position, //origin = camera position
            const Vector &direction,//z-axis = unit-vector from origin to center
            const Vector &up, //x-axis (=center - upper edge of near plane)
            const Vector &left,//y-axis = center - left edge of near plane
            Real zNear, Real zFar)//two plane offsets along z-axis
    {

        // Center of the near plane.
        const Vector center = direction*zNear;

        // Get the four basis vectors along the frustum edges.
        const Vector base[4]={center - up - left,// lower right base
            center - up + left,// lower left base
            center + up - left,// upper right base
            center + up + left};//upper left base

        const Vector pos1 = position + base[1];// lower left vertex
        const Vector pos2 = position + base[2];// upper right vertex

        mFace[0].set(pos2,-direction);//front face
        mFace[1].set(position+direction*zFar,direction);//back face

        typedef typename PlaneType::Normalize NormalizedPlane;
        mFace[2].set(pos1, base[3].cross(base[1]), NormalizedPlane());//left face
        mFace[3].set(pos2, base[0].cross(base[2]), NormalizedPlane());//right face

        mFace[4].set(pos2, base[2].cross(base[3]), NormalizedPlane());//up face
        mFace[5].set(pos1, base[1].cross(base[0]), NormalizedPlane());//down face
    }


}; // class Frustum


template <typename Real>
inline std::ostream &operator<<(std::ostream &os, const Frustum<Real> &f) {
    for (int i=0; i<6; ++i) os << "Frustum " << f.getFaceName(i)
        << " face: " << f.getPlane(i) << std::endl;
    return os;
}

/// @brief LegacyFrustum GeClass used at DreamWorks for converting old vdb files.
class LegacyFrustum
{
public:

   

     LegacyFrustum(std::istream& is) {
         // Frist read in the old transform's base class.
            // the "extents"
            Coord tmpMin, tmpMax;
            is.read(reinterpret_cast<char*>(&tmpMin), sizeof(Coord::ValueType) * 3);
            is.read(reinterpret_cast<char*>(&tmpMax), sizeof(Coord::ValueType) * 3);
            
            // set the extents
            mExtents = CoordBBox(tmpMin, tmpMax);

            // read the old-frustum class member data
            //Mat4d tmpW2C; 
            Mat4d tmpW2C, tmpC2S, tmpS2C, tmpWorldToLocal;
            Mat4d tmpS2U, tmpXYLocalToUnit, tmpZLocalToUnit;
            Real tmpWindow[6];
            Real tmpPadding;
           
            //Mat4d  tmpXYUnitToLocal, tmpZUnitToLocal
            
            // read in each matrix.
            is.read(reinterpret_cast<char*>(&tmpW2C),
                    sizeof(Mat4d::value_type) * Mat4d::size * Mat4d::size);
            is.read(reinterpret_cast<char*>(&mC2W),
                    sizeof(Mat4d::value_type) * Mat4d::size * Mat4d::size);
            is.read(reinterpret_cast<char*>(&tmpC2S),
              sizeof(Mat4d::value_type) * Mat4d::size * Mat4d::size);
            is.read(reinterpret_cast<char*>(&tmpS2C),
                    sizeof(Mat4d::value_type) * Mat4d::size * Mat4d::size);    
            is.read(reinterpret_cast<char*>(&tmpWorldToLocal),
                    sizeof(Mat4d::value_type) * Mat4d::size * Mat4d::size);
            is.read(reinterpret_cast<char*>(&mLocalToWorld),
                    sizeof(Mat4d::value_type) * Mat4d::size * Mat4d::size);
            
            is.read(reinterpret_cast<char*>(&tmpWindow[0]), sizeof(Real));
            is.read(reinterpret_cast<char*>(&tmpWindow[1]), sizeof(Real));
            is.read(reinterpret_cast<char*>(&tmpWindow[2]), sizeof(Real));
            is.read(reinterpret_cast<char*>(&tmpWindow[3]), sizeof(Real));
            is.read(reinterpret_cast<char*>(&tmpWindow[4]), sizeof(Real));
            is.read(reinterpret_cast<char*>(&tmpWindow[5]), sizeof(Real));
            
            is.read(reinterpret_cast<char*>(&tmpPadding), sizeof(Real));
            
            is.read(reinterpret_cast<char*>(&tmpS2U),
                    sizeof(Mat4d::value_type) * Mat4d::size * Mat4d::size);
            is.read(reinterpret_cast<char*>(&mXYUnitToLocal),
                    sizeof(Mat4d::value_type) * Mat4d::size * Mat4d::size);
            is.read(reinterpret_cast<char*>(&tmpXYLocalToUnit),
                    sizeof(Mat4d::value_type) * Mat4d::size * Mat4d::size);
            is.read(reinterpret_cast<char*>(&mZUnitToLocal),
                    sizeof(Mat4d::value_type) * Mat4d::size * Mat4d::size);
            is.read(reinterpret_cast<char*>(&tmpZLocalToUnit),
                    sizeof(Mat4d::value_type) * Mat4d::size * Mat4d::size);


            mNearPlane = tmpWindow[4];
            mFarPlane  = tmpWindow[5];
            
            // Look up the world space corners of the
            // frustum grid.
            mFrNearOrigin = unitToLocalFrustum(Vec3R(0,0,0)); 
            mFrFarOrigin = unitToLocalFrustum(Vec3R(0,0,1));   
        
            Vec3d frNearXTip = unitToLocalFrustum(Vec3R(1,0,0)); 
            Vec3d frNearYTip = unitToLocalFrustum(Vec3R(0,1,0)); 
            mFrNearXBasis = frNearXTip - mFrNearOrigin;
            mFrNearYBasis = frNearYTip - mFrNearOrigin;
            
            Vec3R frFarXTip = unitToLocalFrustum(Vec3R(1,0,1)); 
            Vec3R frFarYTip = unitToLocalFrustum(Vec3R(0,1,1)); 
            mFrFarXBasis = frFarXTip - mFrFarOrigin;
            mFrFarYBasis = frFarYTip - mFrFarOrigin;
            
    }

    ~LegacyFrustum(){};

    const Mat4d& getCamXForm() const {return mC2W; }

    double getDepth() const {return (mFarPlane - mNearPlane); } 
    double getTaper() const {
              
        return   getNearPlaneWidth() / getFarPlaneWidth(); 
    }

    double getNearPlaneWidth() const {
        double nearPlaneWidth  = (unitToWorld(Vec3d(0,0,0)) - unitToWorld(Vec3d(1,0,0))).length();
        return nearPlaneWidth;
    }
    
    double getFarPlaneWidth() const {
        double farPlaneWidth = (unitToWorld(Vec3d(0,0,1)) - unitToWorld(Vec3d(1,0,1))).length();
        return farPlaneWidth;
    }

    double getNearPlaneDist() const { return mNearPlane; }
    
    const CoordBBox& getBBox() const {return mExtents; }
    
    Vec3d unitToWorld(const Vec3d& in) const {return mLocalToWorld.transform( unitToLocal(in) ); }
    
private:
    LegacyFrustum(){};
         
    Vec3d unitToLocal(const Vec3d& U) const { 
        
        // We first find the local space coordinates
        // of the unit point projected onto the near
        // and far planes of the frustum by using a 
        // linear combination of the planes basis vectors
        Vec3d nearLS = ( U[0] * mFrNearXBasis ) + ( U[1] * mFrNearYBasis ) + mFrNearOrigin;
        Vec3d farLS  = ( U[0] * mFrFarXBasis  ) + ( U[1] * mFrFarYBasis  ) + mFrFarOrigin;
        
        // then we lerp the two ws points in frustum z space
        return U[2] * farLS + ( 1.0 - U[2] ) * nearLS;    
    }

    Vec3d unitToLocalFrustum(const Vec3d& u) const {
        Vec3d fzu = mZUnitToLocal.transformH(u);
        Vec3d fu = u;
        fu[2] = fzu.z();
        return mXYUnitToLocal.transformH(fu);
    }

    
private:

    Mat4d mC2W, mLocalToWorld, mXYUnitToLocal, mZUnitToLocal;
    CoordBBox mExtents;
    Vec3d mFrNearXBasis, mFrNearYBasis, mFrFarXBasis, mFrFarYBasis;
    Vec3d mFrNearOrigin, mFrFarOrigin;
    double mNearPlane, mFarPlane;
    
};


} // namespace math
} // namespace OPENVDB_VERSION_NAME
} // namespace openvdb

#endif // OPENVDB_MATH_FRUSTUM_HAS_BEEN_INCLUDED

// Copyright (c) 2012 DreamWorks Animation LLC
// All rights reserved. This software is distributed under the
// Mozilla Public License 2.0 ( http://www.mozilla.org/MPL/2.0/ )
