///////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2012 DreamWorks Animation LLC
//
// All rights reserved. This software is distributed under the
// Mozilla Public License 2.0 ( http://www.mozilla.org/MPL/2.0/ )
//
// Redistributions of source code must retain the above copyright
// and license notice and the following restrictions and disclaimer.
//
// *     Neither the name of DreamWorks Animation nor the names of
// its contributors may be used to endorse or promote products derived
// from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
// IN NO EVENT SHALL THE COPYRIGHT HOLDERS' AND CONTRIBUTORS' AGGREGATE
// LIABILITY FOR ALL CLAIMS REGARDLESS OF THEIR BASIS EXCEED US$250.00.
//
///////////////////////////////////////////////////////////////////////////
//
/// @author Ken Museth
///
/// @file Filter.h

#ifndef OPENVDB_TOOLS_FILTER_HAS_BEEN_INCLUDED
#define OPENVDB_TOOLS_FILTER_HAS_BEEN_INCLUDED

#include <tbb/parallel_reduce.h>
#include <tbb/parallel_for.h>
#include <boost/bind.hpp>
#include <boost/function.hpp>
#include <boost/scoped_ptr.hpp>
#include <openvdb/openvdb.h>
#include <openvdb/Types.h>
#include <openvdb/math/Math.h>
#include <openvdb/math/Stencils.h>
#include <openvdb/math/Transform.h>
#include <openvdb/tree/LeafManager.h>
#include <openvdb/util/NullInterrupter.h>
#include <openvdb/Grid.h>


namespace openvdb {
OPENVDB_USE_VERSION_NAMESPACE
namespace OPENVDB_VERSION_NAME {
namespace tools {

/// @brief Filtering of VDB volumes
template<typename GridT, typename InterruptT = util::NullInterrupter>
class Filter
{
public:
    typedef GridT                                  GridType;
    typedef typename GridType::TreeType            TreeType;
    typedef typename TreeType::LeafNodeType        LeafType;
    typedef typename LeafType::ValueType           ValueType;
    typedef typename tree::LeafManager<TreeType>   LeafManagerType;
    typedef typename LeafManagerType::RangeType    RangeType;
    typedef typename LeafManagerType::BufferType   BufferType;

    /// Constructor
    explicit Filter(GridT& grid, InterruptT* interrupt = NULL) :
        mIsMaster(true), mGrid(grid), mTask(0),
        mLeafs(new LeafManagerType(grid.tree())), mInterrupter(interrupt)
    {
    }
    /// Shallow copy constructor called by tbb::parallel_for() threads during filtering
    Filter(const Filter& other) :
        mIsMaster(false), mGrid(other.mGrid), mTask(other.mTask),
        mLeafs(other.mLeafs), mInterrupter(other.mInterrupter)
    {
    }
    virtual ~Filter()
    {
        if ( mIsMaster ) delete mLeafs;
    }

    /// @brief One iteration of a fast separable mean-value (i.e. box) filter.
    void mean(int width = 1, bool serial = false);

    /// @brief One iteration of a fast separable gaussian filter.
    ///
    /// @note This is approximated as 4 iterations of a separable mean filter
    /// which typically leads an approximation that's better than 95%!
    void gaussian(int width = 1, bool serial = false);

    /// @brief One iteration of a median-value filter
    ///
    /// @note This filter is not separable and is hence relatively slow!
    void median(int width = 1, bool serial = false);

    /// Offsets (i.e. adds) a constant value to all active voxels.
    void offset(float offset, bool serial = false);

    /// @brief Used internally by tbb::parallel_for()
    ///
    /// @note Never call this method directly!
    void operator()(const RangeType& r) const
    {
        if (mTask) mTask(const_cast<Filter*>(this), r);
        else OPENVDB_THROW(ValueError, "task is undefined - call median(), mean(), etc.");
    }

private:
    typedef typename boost::function<void (Filter*, const RangeType&)> FuncType;

    void cook(bool serial, bool swapBuffers);

    // Private filter methods called by tbb::parallel_for threads
    void doBoxX(const RangeType&, Int32);
    void doBoxY(const RangeType&, Int32);
    void doBoxZ(const RangeType&, Int32);
    void doMedian(const RangeType&, int);
    void doOffset(const RangeType&, float);
    void startInterrupter(const char* msg);
    void endInterrupter();
    void checkInterrupter();

    const bool        mIsMaster;
    GridType&         mGrid;
    FuncType          mTask;
    LeafManagerType*  mLeafs;
    InterruptT*       mInterrupter;

}; // end of Filter class

////////////////////////////////////////

template<typename GridT, typename InterruptT>
inline void
Filter<GridT, InterruptT>::mean(int width, bool serial)
{
    this->startInterrupter("Applying mean filter");
    width = std::max(1, width);

    mLeafs->rebuildAuxBuffers(1, serial);

    mTask = boost::bind(&Filter::doBoxX, _1, _2, width);
    this->cook(serial, true);

    mTask = boost::bind(&Filter::doBoxY, _1, _2, width);
    this->cook(serial, true);

    mTask = boost::bind(&Filter::doBoxZ, _1, _2, width);
    this->cook(serial, true);
    this->endInterrupter();
}

template<typename GridT, typename InterruptT>
inline void
Filter<GridT, InterruptT>::gaussian(int width, bool serial)
{
    this->startInterrupter("Applying gaussian filter");
    width = std::max(1, width);

    mLeafs->rebuildAuxBuffers(1, serial);

    for (int n=0; n<4; ++n) {
        mTask = boost::bind(&Filter::doBoxX, _1, _2, width);
        this->cook(serial, true);

        mTask = boost::bind(&Filter::doBoxY, _1, _2, width);
        this->cook(serial, true);

        mTask = boost::bind(&Filter::doBoxZ, _1, _2, width);
        this->cook(serial, true);
    }
    this->endInterrupter();
}


template<typename GridT, typename InterruptT>
inline void
Filter<GridT, InterruptT>::median(int width, bool serial)
{
    this->startInterrupter("Applying median filter");
    mLeafs->rebuildAuxBuffers(1, serial);
    mTask = boost::bind(&Filter::doMedian, _1, _2, std::max(1, width));
    this->cook(serial, true);
    this->endInterrupter();
}

template<typename GridT, typename InterruptT>
inline void
Filter<GridT, InterruptT>::offset(float value, bool serial)
{
    this->startInterrupter("Applying offset");
    mLeafs->removeAuxBuffers();
    mTask = boost::bind(&Filter::doOffset, _1, _2, value);
    this->cook(serial, false);
    this->endInterrupter();
}

////////////////////////////////////////


/// Private method to perform the task (serial or threaded) and
/// subsequently swap the leaf buffers.
template<typename GridT, typename InterruptT>
inline void
Filter<GridT, InterruptT>::cook(bool serial, bool swapBuffers)
{
    if (serial) {
        (*this)(mLeafs->getRange());
    } else {
        tbb::parallel_for(mLeafs->getRange(), *this);
    }
    if (swapBuffers) mLeafs->swapLeafBuffer(1, serial);
}

/// X convolution of a separable box filter
template<typename GridT, typename InterruptT>
inline void
Filter<GridT, InterruptT>::doBoxX(const RangeType& range, Int32 w)
{
    this->checkInterrupter();
    const ValueType frac = ValueType(1)/ValueType(2*w+1);
    typename GridT::ConstAccessor acc = mGrid.getConstAccessor();
    for (size_t n=range.begin(), nLast = range.end(); n != nLast; ++n) {
        const LeafType& leaf = mLeafs->leaf(n);
        BufferType& buffer   = mLeafs->getBuffer(n, 1);
        for (typename LeafType::ValueOnCIter iter = leaf.cbeginValueOn(); iter; ++iter) {
            ValueType sum = zeroVal<ValueType>();
            math::Coord xyz = iter.getCoord();
            for (Int32 x = xyz.x()-w, xLast = xyz.x()+w; x <= xLast; ++x) {
                sum += acc.getValue(xyz.setX(x));
            }
            buffer.setValue(iter.pos(), sum*frac);
        }
    }
}

/// Y convolution of a separable box filter
template<typename GridT, typename InterruptT>
inline void
Filter<GridT, InterruptT>::doBoxY(const RangeType& range, Int32 w)
{
    this->checkInterrupter();
    const ValueType frac = ValueType(1)/ValueType(2*w+1);
    typename GridT::ConstAccessor acc = mGrid.getConstAccessor();
    for (size_t n=range.begin(), nLast = range.end(); n != nLast; ++n) {
        const LeafType& leaf = mLeafs->leaf(n);
        BufferType& buffer   = mLeafs->getBuffer(n, 1);
        for (typename LeafType::ValueOnCIter iter = leaf.cbeginValueOn(); iter; ++iter) {
            ValueType sum = zeroVal<ValueType>();
            math::Coord xyz = iter.getCoord();
            for (Int32 y = xyz.y()-w, yLast = xyz.y()+w; y <= yLast; ++y) {
                sum += acc.getValue(xyz.setY(y));
            }
            buffer.setValue(iter.pos(), sum*frac);
        }
    }
}

/// Z convolution of a separable box filter
template<typename GridT, typename InterruptT>
inline void
Filter<GridT, InterruptT>::doBoxZ(const RangeType& range, Int32 w)
{
    this->checkInterrupter();
    const ValueType frac = ValueType(1)/ValueType(2*w+1);
    typename GridT::ConstAccessor acc = mGrid.getConstAccessor();
    for (size_t n=range.begin(), nLast = range.end(); n != nLast; ++n) {
        const LeafType& leaf = mLeafs->leaf(n);
        BufferType& buffer   = mLeafs->getBuffer(n, 1);
        for (typename LeafType::ValueOnCIter iter = leaf.cbeginValueOn(); iter; ++iter) {
            ValueType sum = zeroVal<ValueType>();
            math::Coord xyz = iter.getCoord();
            for (Int32 z = xyz.z()-w, zLast = xyz.z()+w; z <= zLast; ++z) {
                sum += acc.getValue(xyz.setZ(z));
            }
            buffer.setValue(iter.pos(), sum*frac);
        }
    }
}

/// Performs simple but slow median-value diffusion
template<typename GridT, typename InterruptT>
inline void
Filter<GridT, InterruptT>::doMedian(const RangeType& range, int width)
{
    this->checkInterrupter();
    typename math::DenseStencil<GridType> stencil(mGrid, width);//creates local cache!
    for (size_t n=range.begin(), e=range.end(); n != e; ++n) {
        const LeafType& leaf = mLeafs->leaf(n);
        BufferType& buffer   = mLeafs->getBuffer(n, 1);
        for (typename LeafType::ValueOnCIter iter = leaf.cbeginValueOn(); iter; ++iter) {
            stencil.moveTo(iter);
            buffer.setValue(iter.pos(), stencil.median());
        }
    }
}

/// Offsets the values by a constant
template<typename GridT, typename InterruptT>
inline void
Filter<GridT, InterruptT>::doOffset(const RangeType& range, float floatVal)
{
    const ValueType value = static_cast<ValueType>(floatVal);
    for (size_t n=range.begin(), e=range.end(); n != e; ++n) mLeafs->leaf(n).addValue(value);
}

template<typename GridT, typename InterruptT>
inline void
Filter<GridT, InterruptT>::startInterrupter(const char* msg)
{
    if (mInterrupter) mInterrupter->start(msg);
}

template<typename GridT, typename InterruptT>
inline void
Filter<GridT, InterruptT>::endInterrupter()
{
    if (mInterrupter) mInterrupter->end();
}

template<typename GridT, typename InterruptT>
inline void
Filter<GridT, InterruptT>::checkInterrupter()
{
    if (util::wasInterrupted(mInterrupter)) tbb::task::self().cancel_group_execution();
}

} // namespace tools
} // namespace OPENVDB_VERSION_NAME
} // namespace openvdb

#endif // OPENVDB_TOOLS_FILTER_HAS_BEEN_INCLUDED

// Copyright (c) 2012 DreamWorks Animation LLC
// All rights reserved. This software is distributed under the
// Mozilla Public License 2.0 ( http://www.mozilla.org/MPL/2.0/ )
