//---------------------------------------------------------------------------
//
// Project: OpenWalnut ( http://www.openwalnut.org )
//
// Copyright 2009 OpenWalnut Community, BSV@Uni-Leipzig and CNCF@MPI-CBS
// For more information see http://www.openwalnut.org/copying
//
// This file is part of OpenWalnut.
//
// OpenWalnut is free software: you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// OpenWalnut is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public License
// along with OpenWalnut. If not, see <http://www.gnu.org/licenses/>.
//
//---------------------------------------------------------------------------

#include <iostream>
#include <map>
#include <string>
#include <vector>

#include <boost/thread.hpp>
#include <boost/regex.hpp>
#include <boost/filesystem.hpp>

#include <QtGui/QApplication>
#include <QtGui/QCloseEvent>
#include <QtGui/QDockWidget>
#include <QtGui/QFileDialog>
#include <QtGui/QIcon>
#include <QtGui/QMenu>
#include <QtGui/QMenuBar>
#include <QtGui/QMessageBox>
#include <QtGui/QTextEdit>
#include <QtGui/QShortcut>
#include <QtGui/QSlider>
#include <QtGui/QVBoxLayout>
#include <QtGui/QWidget>
#include <QtCore/QSettings>
#include <QtCore/QUrl>
#include <QtGui/QInputDialog>

#ifndef QT4GUI_NOWEBKIT
    #include <QtWebKit/QWebView>
#endif

#include "core/WVersion.h"   // NOTE: this file is auto-generated by CMAKE

#include "controlPanel/WPropertyBoolWidget.h"
#include "controlPanel/WQtControlPanel.h"
#include "core/common/WColor.h"
#include "core/common/WIOTools.h"
#include "core/common/WPathHelper.h"
#include "core/common/WProjectFileIO.h"
#include "core/dataHandler/WDataHandler.h"
#include "core/dataHandler/WDataSetFibers.h"
#include "core/dataHandler/WDataSetSingle.h"
#include "core/dataHandler/WDataSetPoints.h"
#include "core/dataHandler/WEEG2.h"
#include "core/graphicsEngine/WGEZoomTrackballManipulator.h"
#include "core/graphicsEngine/WROIBox.h"
#include "core/kernel/WDataModule.h"
#include "core/kernel/WKernel.h"
#include "core/kernel/WModule.h"
#include "core/kernel/WModuleCombiner.h"
#include "core/kernel/WModuleCombinerTypes.h"
#include "core/kernel/WProjectFile.h"
#include "core/kernel/WROIManager.h"
#include "core/kernel/WSelectionManager.h"
#include "events/WEventTypes.h"
#include "events/WModuleCrashEvent.h"
#include "events/WModuleReadyEvent.h"
#include "events/WModuleRemovedEvent.h"
#include "events/WOpenCustomDockWidgetEvent.h"
#include "events/WCloseCustomDockWidgetEvent.h"
#include "events/WLoadFinishedEvent.h"
#include "events/WLogEvent.h"
#include "guiElements/WQtPropertyBoolAction.h"
#include "WQtMessagePopup.h"
#include "WQt4Gui.h"
#include "WQtCombinerToolbar.h"
#include "WQtCustomDockWidget.h"
#include "WQtGLDockWidget.h"
#include "WQtNavGLWidget.h"
#include "WSettingAction.h"
#include "WSettingMenu.h"
#include "WQtMessageDialog.h"

#include "WMainWindow.h"
#include "WMainWindow.moc"

WMainWindow::WMainWindow( QSplashScreen* splash ):
    QMainWindow(),
    m_splash( splash ),
    m_currentCompatiblesToolbar( NULL ),
    m_iconManager()
{
    setAcceptDrops( true ); // enable drag and drop events
}

WMainWindow::~WMainWindow()
{
    // cleanup
}

/**
 * Create a distinct separator.
 *
 * \param parent the parent
 *
 * \return the separator
 */
QAction* createSeperator( QWidget* parent )
{
    QAction* separator = new QAction( parent );
    separator->setSeparator( true );
    return separator;
}

void WMainWindow::setupGUI()
{
    wlog::info( "WMainWindow" ) << "Setting up GUI";

    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    // Setting setup
    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    WSettingAction* hideMenuAction = new WSettingAction( this, "qt4gui/showMenu",
                                                               "Show Menubar",
                                                               "Allows you to hide the menu. Can be restored using CTRL-M.",
                                                               true,
                                                               false,
                                                               QKeySequence( Qt::CTRL + Qt::Key_M ) );

    WSettingAction* showNavWidgets = new WSettingAction( this, "qt4gui/showNavigationWidgets",
                                                               "Show Navigation Views",
                                                               "Disables the navigation views completely. This can lead to a speed-up and is "
                                                               "recommended for those who do not need them.",
                                                               false,
                                                               true    // this requires a restart
                                                       );
    m_autoDisplaySetting = new WSettingAction( this, "qt4gui/useAutoDisplay",
                                                     "Auto-Display",
                                                     "If enabled, the best matching module is automatically added if some data was loaded.",
                                                     true );
    m_sliderMinMaxEditSetting = new WSettingAction( this, std::string( "qt4gui/" ) +  std::string( "sliderMinMaxEdit" ),
                                                    "Slider Min/Max Editing",
                                                    "If enabled, the maximum and minimum values of slider can be edited.",
                                                    false );

    WSettingAction* mtViews = new WSettingAction( this, "qt4gui/ge/multiThreadedViewer",
                                                        "Multi-Threaded Views",
                                                        "If enabled, the graphic windows are rendered in different threads. This can speed-up "
                                                        "rendering on machines with multiple cores. WARNING: can lead to crashes sometimes.",
                                                        false,
                                                        true // require restart
                                                );
    // NOTE: the multi-threading feature needs to be activated BEFORE the first viewer is created. To ensure this we do it here.
    WGraphicsEngine::getGraphicsEngine()->setMultiThreadedViews( mtViews->get() );

    // set the log-level setting.
    // NOTE: see WQt4Gui which reads the setting.
    QList< QString > logOptions;
    logOptions.push_back( "Debug" );
    logOptions.push_back( "Info" );
    logOptions.push_back( "Warning" );
    logOptions.push_back( "Error" );
    WSettingMenu* logLevels = new WSettingMenu( this, "qt4gui/logLevel",
                                                      "Log-Level",
                                                      "Allows one to set the log verbosity.",
                                                      1,    // info is the default
                                                      logOptions
                                              );
    connect( logLevels, SIGNAL( change( unsigned int ) ), this, SLOT( handleLogLevelUpdate( unsigned int ) ) );

    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    // GUI setup
    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    // We need several icons later in the GUI. Build some mappings:
    m_iconManager.addMapping( "logo", "openwalnut" );
    m_iconManager.addMapping( "axial icon", "orientation_axial" );
    m_iconManager.addMapping( "coronal icon", "orientation_coronal" );
    m_iconManager.addMapping( "sagittal icon", "orientation_sagittal" );
    m_iconManager.addMapping( "DefaultModuleIcon", "default" );
    m_iconManager.addMapping( "missingModule", "question" );
    m_iconManager.addMapping( "view", "camera" );
    m_iconManager.addMapping( "moduleCrashed", "error" );
    m_iconManager.addMapping( "moduleBusy", "busy" );
    m_iconManager.addMapping( "saveProject", "save" );
    m_iconManager.addMapping( "newProject", "new" );

    if( objectName().isEmpty() )
    {
        setObjectName( QString::fromUtf8( "MainWindow" ) );
    }

    // NOTE: this only is an initial size. The state reloaded from QSettings will set it to the value the user had last session.
    resize( 1024, 768 );
    setWindowIcon( m_iconManager.getIcon( "logo" ) );
    std::string windowHeading =  std::string( "OpenWalnut " ) + std::string( W_VERSION );
    setWindowTitle( QApplication::translate( "MainWindow", windowHeading.c_str(), 0, QApplication::UnicodeUTF8 ) );

    setDockOptions( QMainWindow::AnimatedDocks |  QMainWindow::AllowNestedDocks | QMainWindow::AllowTabbedDocks );

    //network Editor
    m_networkEditor = NULL;
    m_networkEditor = new WQtNetworkEditor( this );
    m_networkEditor->setFeatures( QDockWidget::AllDockWidgetFeatures );

    // strangely, the QGraphics* objects do not properly forward drag/drop events. We need to explicitly handle them.
    connect( m_networkEditor->getView(), SIGNAL( dragDrop( QDropEvent* ) ),
             this, SLOT( handleDrop( QDropEvent* ) ) );

    // the control panel instance is needed for the menu
    m_controlPanel = new WQtControlPanel( this );
    m_controlPanel->setFeatures( QDockWidget::AllDockWidgetFeatures );
    m_controlPanel->addSubject( "Default Subject" );

    // add all docks
    addDockWidget( Qt::RightDockWidgetArea, m_controlPanel->getModuleDock() );
    if( m_networkEditor )
    {
        addDockWidget( Qt::RightDockWidgetArea, m_networkEditor );
    }

    addDockWidget( Qt::RightDockWidgetArea, m_controlPanel->getColormapperDock() );
    addDockWidget( Qt::RightDockWidgetArea, m_controlPanel->getRoiDock() );

    // the message dock:
    m_messageDock = new WQtMessageDock( "Messages", this );
    addDockWidget( Qt::RightDockWidgetArea, m_messageDock );

    // tabify those panels by default
    if( m_networkEditor )
    {
        tabifyDockWidget( m_networkEditor, m_controlPanel->getModuleDock() );
    }
    tabifyDockWidget( m_controlPanel->getModuleDock(), m_controlPanel->getColormapperDock() );
    tabifyDockWidget( m_controlPanel->getColormapperDock(), m_controlPanel->getRoiDock() );
    tabifyDockWidget( m_controlPanel->getRoiDock(), m_messageDock );

    m_glDock = new QMainWindow();
    m_glDock->setObjectName( "GLDock" );
    m_glDock->setDockOptions( QMainWindow::AnimatedDocks |  QMainWindow::AllowNestedDocks | QMainWindow::AllowTabbedDocks );
    m_glDock->setDocumentMode( true );
    setCentralWidget( m_glDock );
    m_mainGLDock = new WQtGLDockWidget( "Main View", "3D View", m_glDock );
    // activate effects for this view by default
    m_mainGLDock->getGLWidget()->getViewer()->setEffectsActiveDefault();
    m_mainGLDock->getGLWidget()->setSizePolicy( QSizePolicy::Expanding, QSizePolicy::Expanding );
    m_mainGLDock->restoreSettings();
    m_mainGLWidget = m_mainGLDock->getGLWidget();
    m_glDock->addDockWidget( Qt::RightDockWidgetArea, m_mainGLDock );
    connect( m_mainGLWidget.get(), SIGNAL( renderedFirstFrame() ), this, SLOT( handleGLVendor() ) );

    addDockWidget( Qt::RightDockWidgetArea, m_controlPanel );

    // by default, the module editor should be in front
    if( m_networkEditor )
    {
        m_networkEditor->raise();
    }
    else
    {
        m_controlPanel->getModuleDock()->raise();
    }

    // NOTE: we abuse the gl widgets first frame event to handle startup news.
    connect( m_mainGLWidget.get(), SIGNAL( renderedFirstFrame() ), this, SLOT( handleStartMessages() ) );
    connect( m_mainGLWidget.get(), SIGNAL( renderedFirstFrame() ), this, SLOT( closeSplash() ) );

    m_permanentToolBar = new WQtToolBar( "Standard Toolbar", this );
    addToolBar( Qt::TopToolBarArea, m_permanentToolBar );
    m_permanentToolBar->setVisible( false );

    m_loadButton = new QAction( m_iconManager.getIcon( "load" ), "Load Dataset or Project", m_permanentToolBar );
    m_loadButton->setShortcut( QKeySequence(  QKeySequence::Open ) );
    QAction* roiButton = new QAction( m_iconManager.getIcon( "add_roi" ), "ROI", m_permanentToolBar );
    m_saveAction = new QAction( m_iconManager.getIcon( "saveProject" ), "Save Project", m_permanentToolBar );

    connect( m_loadButton, SIGNAL(  triggered( bool ) ), this, SLOT( openLoadDialog() ) );
    connect( roiButton, SIGNAL(  triggered( bool ) ), this, SLOT( newRoi() ) );
    connect( m_saveAction, SIGNAL( triggered( bool ) ), this, SLOT( projectSaveAll() ) );

    m_loadButton->setToolTip( "Load a dataset or project from file" );
    roiButton->setToolTip( "Insert a new ROI" );
    m_saveAction->setToolTip( "Save current project to file" );

    // we want the upper most tree item to be selected. This helps to make the always compatible modules
    // show up in the tool bar from the beginning. And ... it doesn't hurt.
    m_controlPanel->selectUpperMostEntry();

    // NOTE: Please be aware that not every menu needs a shortcut key. If you add a shortcut, you should use one of the
    // QKeySequence::StandardKey defaults and avoid ambiguities like Ctrl-C for the configure dialog is not the best choice as Ctrl-C, for the
    // most users is the Copy shortcut.

    m_menuBar = new QMenuBar( this );

    // hide menu?
    m_menuBar->setVisible( hideMenuAction->get() );
    connect( hideMenuAction, SIGNAL( change( bool ) ), m_menuBar, SLOT( setVisible( bool ) ) );
    addAction( hideMenuAction );

    QMenu* fileMenu = m_menuBar->addMenu( "File" );

    m_newAction = fileMenu->addAction( m_iconManager.getIcon( "newProject" ),
                                       "New Project",
                                       this,
                                       SLOT( newProject() ),
                                       QKeySequence( Qt::CTRL + Qt::Key_N ) );
    fileMenu->addAction( m_loadButton );
    m_saveMenu = fileMenu->addMenu( m_iconManager.getIcon( "saveProject" ), "Save Project" );
    m_saveMenu->addAction( "Save Project", this, SLOT( projectSaveAll() ), QKeySequence::Save );
    m_saveMenu->addAction( "Save Modules Only", this, SLOT( projectSaveModuleOnly() ) );
    m_saveMenu->addAction( "Save Camera Only", this, SLOT( projectSaveCameraOnly() ) );
    // saveMenu->addAction( "Save ROIs Only", this, SLOT( projectSaveROIOnly() ) );
    m_saveAction->setMenu( m_saveMenu );

    fileMenu->addSeparator();
    // TODO(all): If all distributions provide a newer QT version we should use QKeySequence::Quit here
    //fileMenu->addAction( m_iconManager.getIcon( "quit" ), "Quit", this, SLOT( close() ), QKeySequence( QKeySequence::Quit ) );
    m_quitAction = fileMenu->addAction( m_iconManager.getIcon( "quit" ), "Quit", this, SLOT( close() ),  QKeySequence( Qt::CTRL + Qt::Key_Q ) );

    // This QAction stuff is quite ugly and complicated some times ... There is no nice constructor which takes name, slot keysequence and so on
    // directly -> set shortcuts, and some further properties using QAction's interface

    m_settingsAction = new QAction( "Settings", this );
    m_settingsAction->setIcon( m_iconManager.getIcon( "configure" ) );
    m_settingsMenu = m_menuBar->addMenu( "Settings" );
    m_viewMenu = m_settingsMenu->addMenu( "View" );
    m_viewMenu->addAction( hideMenuAction );
    m_viewMenu->addSeparator();
    m_viewMenu->addAction( showNavWidgets );
    m_viewMenu->addSeparator();
    m_viewMenu->addMenu( m_permanentToolBar->getStyleMenu() );
    m_settingsMenu->addAction( m_autoDisplaySetting );
    m_settingsMenu->addAction( m_sliderMinMaxEditSetting );
    m_settingsMenu->addAction( m_controlPanel->getModuleConfig().getConfigureAction() );
    m_settingsMenu->addSeparator();
    m_settingsMenu->addAction( mtViews );
    m_settingsMenu->addSeparator();
    m_settingsMenu->addMenu( logLevels );
    m_settingsMenu->addSeparator();
    m_settingsAction->setMenu( m_settingsMenu );

    QAction* controlPanelTrigger = m_controlPanel->toggleViewAction();
    QList< QKeySequence > controlPanelShortcut;
    controlPanelShortcut.append( QKeySequence( Qt::Key_F9 ) );
    controlPanelTrigger->setShortcuts( controlPanelShortcut );
    this->addAction( controlPanelTrigger );  // this enables the action even if the menu bar is invisible

    m_helpAction = new QAction( "Help", this );
    m_helpAction->setIcon( m_iconManager.getIcon( "help" ) );
    connect( m_helpAction, SIGNAL( triggered() ), this, SLOT( openOpenWalnutHelpDialog() ) );
    m_helpMenu = m_menuBar->addMenu( "Help" );
    m_helpMenu->addAction( m_iconManager.getIcon( "help" ), "OpenWalnut Help", this, SLOT( openOpenWalnutHelpDialog() ),
                           QKeySequence( QKeySequence::HelpContents ) );
    m_helpMenu->addAction( m_iconManager.getIcon( "logo" ), "Welcome to OpenWalnut", this, SLOT( showWelcomeDialog() ) );
    m_helpMenu->addSeparator();
    m_helpMenu->addAction( m_iconManager.getIcon( "logo" ), "About OpenWalnut", this, SLOT( openAboutDialog() ) );
    m_helpMenu->addAction(  m_iconManager.getIcon( "qt" ), "About Qt", this, SLOT( openAboutQtDialog() ) );
    m_helpAction->setMenu( m_helpMenu );
    connect( m_helpAction, SIGNAL( triggered( bool ) ), this, SLOT( openOpenWalnutHelpDialog() ) );

    setMenuBar( m_menuBar );

    // initially 3 navigation views
    {
        if( showNavWidgets->get() )
        {
            m_navAxial = boost::shared_ptr< WQtNavGLWidget >( new WQtNavGLWidget( "Axial View", "Axial View", this, "Axial Slice",
                                                                                  m_mainGLWidget.get() ) );
            m_navAxial->setFeatures( QDockWidget::AllDockWidgetFeatures );
            m_navAxial->setSliderProperty( WKernel::getRunningKernel()->getSelectionManager()->getPropAxialPos() );
            m_navAxial->getGLWidget()->setCameraManipulator( WQtGLWidget::NO_OP );

            m_glDock->addDockWidget( Qt::LeftDockWidgetArea, m_navAxial.get() );

            m_navCoronal = boost::shared_ptr< WQtNavGLWidget >( new WQtNavGLWidget( "Coronal View", "Coronal View", this, "Coronal Slice",
                                                                                    m_mainGLWidget.get() ) );
            m_navCoronal->setFeatures( QDockWidget::AllDockWidgetFeatures );
            m_navCoronal->setSliderProperty( WKernel::getRunningKernel()->getSelectionManager()->getPropCoronalPos() );
            m_navCoronal->getGLWidget()->setCameraManipulator( WQtGLWidget::NO_OP );

            m_glDock->addDockWidget( Qt::LeftDockWidgetArea, m_navCoronal.get() );

            m_navSagittal =
                boost::shared_ptr< WQtNavGLWidget >( new WQtNavGLWidget( "Sagittal View", "Sagittal View", this, "Sagittal Slice",
                                                                         m_mainGLWidget.get() ) );
            m_navSagittal->setFeatures( QDockWidget::AllDockWidgetFeatures );
            m_navSagittal->setSliderProperty( WKernel::getRunningKernel()->getSelectionManager()->getPropSagittalPos() );
            m_navSagittal->getGLWidget()->setCameraManipulator( WQtGLWidget::NO_OP );

            m_glDock->addDockWidget( Qt::LeftDockWidgetArea, m_navSagittal.get() );
        }
    }

    // create the show/hide actions using the selection manager's props
    WQtPropertyBoolAction* showAxial = new WQtPropertyBoolAction( WKernel::getRunningKernel()->getSelectionManager()->getPropAxialShow(),
                                                                  m_permanentToolBar );
    showAxial->setToolTip( "Toggle axial slice" );
    showAxial->setText( "Toggle Axial Slice" );
    showAxial->setIcon( m_iconManager.getIcon( "axial icon" ) );

    WQtPropertyBoolAction* showCoronal = new WQtPropertyBoolAction( WKernel::getRunningKernel()->getSelectionManager()->getPropCoronalShow(),
                                                                    m_permanentToolBar );
    showCoronal->setToolTip( "Toggle coronal slice" );
    showCoronal->setText( "Toggle Coronal Slice" );
    showCoronal->setIcon( m_iconManager.getIcon( "coronal icon" ) );

    WQtPropertyBoolAction* showSagittal = new WQtPropertyBoolAction( WKernel::getRunningKernel()->getSelectionManager()->getPropSagittalShow(),
                                                                     m_permanentToolBar );
    showSagittal->setToolTip( "Toggle sagittal slice" );
    showSagittal->setText( "Toggle Sagittal Slice" );
    showSagittal->setIcon( m_iconManager.getIcon( "sagittal icon" ) );

    // setup permanent toolbar
    m_permanentToolBar->addAction( m_newAction );
    m_permanentToolBar->addAction( m_loadButton );
    m_permanentToolBar->addAction( m_saveAction );
    m_permanentToolBar->addSeparator();
    m_permanentToolBar->addAction( roiButton );
    m_permanentToolBar->addSeparator();
    m_permanentToolBar->addAction( showAxial );
    m_permanentToolBar->addAction( showCoronal );
    m_permanentToolBar->addAction( showSagittal );
    m_permanentToolBar->addSeparator();

    // set the according actions to the toolbars
    m_networkEditor->addTitleAction( m_newAction );
    m_networkEditor->addTitleAction( m_loadButton );
    m_networkEditor->addTitleAction( m_saveAction );
    m_networkEditor->addTitleSeperator();
    m_networkEditor->addTitleAction( m_settingsAction, true );

    // allow the control panel to complete setup
    m_controlPanel->completeGuiSetup();

    // add the roi actions to roi dock title
    m_controlPanel->getRoiDock()->addTitleAction( roiButton );
    m_controlPanel->getRoiDock()->addTitleAction( m_controlPanel->getRoiDeleteAction() );

    // after creating the GUI, restore its saved state
    restoreSavedState();

    // this ensures that there is always at least the main window visible. Removing this might cause a freeze on startup if the user has closed
    // all gl widgets during the last session.
    m_mainGLDock->setVisible( true );
}

void WMainWindow::autoAdd( boost::shared_ptr< WModule > module, std::string proto, bool onlyOnce )
{
    // if only one module should be added, and there already is one --- skip.
    if( onlyOnce && !WKernel::getRunningKernel()->getRootContainer()->getModules( proto ).empty() )
    {
        return;
    }

    // get the prototype.
    if( !WKernel::getRunningKernel()->getRootContainer()->applyModule( module, proto, true ) )
    {
        WLogger::getLogger()->addLogMessage( "Auto Display active but module " + proto + " could not be added.",
                                             "GUI", LL_ERROR );
    }
}

void WMainWindow::moduleSpecificCleanup( boost::shared_ptr< WModule > /* module */ )
{
    // called for each removed module. Use this to undo modifications done due to added modules (moduleSpecificSetup)
}

void WMainWindow::moduleSpecificSetup( boost::shared_ptr< WModule > module )
{
    // Add all special handlings here. This method is called whenever a module is marked "ready". You can set up the gui for special modules,
    // load certain modules for datasets and so on.

    // The Data Modules also play an special role. To have modules being activated when certain data got loaded, we need to hook it up here.
    bool useAutoDisplay = m_autoDisplaySetting->get();
    if( useAutoDisplay && module->getType() == MODULE_DATA )
    {
        WLogger::getLogger()->addLogMessage( "Auto Display active and Data module added. The proper module will be added.",
                                             "GUI", LL_DEBUG );

        // data modules contain an member denoting the real data type. Currently we only have one data module and a not very modulated data
        // structures.
        boost::shared_ptr< WDataModule > dataModule = boost::static_pointer_cast< WDataModule >( module );

        // grab data and identify type
        if( dataModule->getDataSet()->isA< WDataSetSingle >() && dataModule->getDataSet()->isTexture() )
        {
            // it is a dataset single
            // load a nav slice module if a WDataSetSingle is available!?
            autoAdd( module, "Navigation Slices", true );
        }
        else if( dataModule->getDataSet()->isA< WDataSetFibers >() )
        {
            // it is a fiber dataset -> add the FiberDisplay module
            autoAdd( module, "Fiber Display" );
        }
        else if( dataModule->getDataSet()->isA< WEEG2 >() )
        {
            // it is a eeg dataset -> add the eegView module
            autoAdd( module, "EEG View" );
        }
        else if( dataModule->getDataSet()->isA< WDataSetPoints >() )
        {
            // it is a point dataset -> add the point render module
            autoAdd( module, "Point Renderer" );
        }
    }
}

void WMainWindow::setCompatiblesToolbar( WQtCombinerToolbar* toolbar )
{
    if( m_currentCompatiblesToolbar )
    {
        delete m_currentCompatiblesToolbar;
    }
    m_currentCompatiblesToolbar = toolbar;

    if( !toolbar )
    {
        // ok, reset the toolbar
        // So create a dummy to permanently reserve the space
        m_currentCompatiblesToolbar = new WQtCombinerToolbar( this );
    }

    // we want to keep the tool-button styles in sync
    m_currentCompatiblesToolbar->setToolButtonStyle( m_permanentToolBar->toolButtonStyle() );
    connect( m_permanentToolBar, SIGNAL( toolButtonStyleChanged( Qt::ToolButtonStyle ) ),
             m_currentCompatiblesToolbar, SLOT( setToolButtonStyle( Qt::ToolButtonStyle ) ) );

    // and the position of the toolbar
    toolbar->setVisible( false );
    addToolBar( Qt::TopToolBarArea, m_currentCompatiblesToolbar );
}

WQtCombinerToolbar* WMainWindow::getCompatiblesToolbar()
{
    return m_currentCompatiblesToolbar;
}

WQtControlPanel* WMainWindow::getControlPanel()
{
    return m_controlPanel;
}

WQtNetworkEditor* WMainWindow::getNetworkEditor()
{
    return m_networkEditor;
}

bool WMainWindow::projectSave( const std::vector< boost::shared_ptr< WProjectFileIO > >& writer )
{
    QString lastPath = WQt4Gui::getSettings().value( "LastProjectSavePath", "" ).toString();
    QString selected = QFileDialog::getSaveFileName( this, "Save Project as", lastPath,
                                                     "Project File (*.owproj *.owp)" );
    if( selected == "" )
    {
        return false;
    }

    // extract path and save to settings
    boost::filesystem::path p( selected.toStdString() );
    WQt4Gui::getSettings().setValue( "LastProjectSavePath", QString::fromStdString( p.parent_path().string() ) );

    bool success = true;
    std::string filename = ( selected ).toStdString();

    // append owp if suffix is not present, yet
    if( filename.rfind( ".owp" ) != filename.size() - 4
     && filename.rfind( ".owproj" ) != filename.size() - 7 )
    {
        filename += ".owp";
    }

    boost::shared_ptr< WProjectFile > proj = boost::shared_ptr< WProjectFile >(
            new WProjectFile( filename )
    );

    try
    {
        // This call is synchronous.
        if( writer.empty() )
        {
            proj->save();
        }
        else
        {
            proj->save( writer );
        }
    }
    catch( const std::exception& e )
    {
        QString title = "Problem while saving project file.";
        QString message = "<b>Problem while saving project file.</b><br/><br/><b>File:  </b>" + selected +
                          "<br/><b>Message:  </b>" + QString::fromStdString( e.what() );
        QMessageBox::critical( this, title, message );
        success = false;
    }
    return success;
}

bool WMainWindow::projectSaveAll()
{
    std::vector< boost::shared_ptr< WProjectFileIO > > w;
    // an empty list equals "all"
    return projectSave( w );
}

bool WMainWindow::projectSaveCameraOnly()
{
    std::vector< boost::shared_ptr< WProjectFileIO > > w;
    w.push_back( WProjectFile::getCameraWriter() );
    return projectSave( w );
}

bool WMainWindow::projectSaveROIOnly()
{
    std::vector< boost::shared_ptr< WProjectFileIO > > w;
    w.push_back( WProjectFile::getROIWriter() );
    return projectSave( w );
}

bool WMainWindow::projectSaveModuleOnly()
{
    std::vector< boost::shared_ptr< WProjectFileIO > > w;
    w.push_back( WProjectFile::getModuleWriter() );
    return projectSave( w );
}

void WMainWindow::newProject()
{
    WKernel::getRunningKernel()->getRootContainer()->removeAll();
    WDataHandler::getDataHandler()->clear();
}

void WMainWindow::openLoadDialog()
{
    QString lastPath = WQt4Gui::getSettings().value( "LastOpenPath", "" ).toString();

    // build filter list
    // NOTE: Qt Doc says we need to separate multiple filters by ";;"
    QString filters;
    filters = QString( "Known file types (*.cnt *.edf *.asc *.nii *.nii.gz *.vtk *.fib *.owproj *.owp *.fdg);;" )
            + QString( "Simple Project File (*.owproj *.owp);;" )
            + QString( "EEG files (*.cnt *.edf *.asc);;" )
            + QString( "NIfTI (*.nii *.nii.gz);;" )
            + QString( "Fibers (*.fib);;" )
            + QString( "Clusters (*.fdg);;" );
    for( std::size_t k = 0; k < WKernel::getRunningKernel()->getScriptEngine()->getNumInterpreters(); ++k )
    {
        filters += QString::fromStdString( WKernel::getRunningKernel()->getScriptEngine()->getInterpreter( k )->getName() + " (*"
                   + WKernel::getRunningKernel()->getScriptEngine()->getInterpreter( k )->getExtension() + ")" );
        filters += QString( ";;" );
    }
    filters += QString( "Any files (*)" );

    QStringList filenames = QFileDialog::getOpenFileNames( this, "Open Data, Project or Script", lastPath, filters );
    if( filenames.empty() )
    {
        return;
    }

    // extract path and save to settings
    boost::filesystem::path p( filenames[0].toStdString() );
    WQt4Gui::getSettings().setValue( "LastOpenPath", QString::fromStdString( p.parent_path().string() ) );

    std::vector< std::string > loadDataFilenames;
    QStringList::const_iterator constIterator;
    for( constIterator = filenames.constBegin(); constIterator != filenames.constEnd(); ++constIterator )
    {
        boost::filesystem::path fn( ( *constIterator ).toLocal8Bit().constData() );
        std::string suffix = getSuffix( fn );

        // is this a project file?
        if( ( suffix == ".owp" ) || ( suffix == ".owproj" ) )
        {
            asyncProjectLoad( fn.string() );
        }
        else
        {
            // this is not a project. So we assume it is a data file or script
            boost::shared_ptr< WScriptInterpreter > scriptInterpreter =
                    WKernel::getRunningKernel()->getScriptEngine()->getInterpreterByFileExtension( suffix );

            if( scriptInterpreter )
            {
                scriptInterpreter->executeFileAsync( fn.string() );
            }
            else
            {
                loadDataFilenames.push_back( fn.string() );
            }
        }
    }

    m_loaderSignal( loadDataFilenames );
}

void WMainWindow::asyncProjectLoad( std::string filename )
{
    WProjectFile::SPtr proj( new WProjectFile( filename, boost::bind( &WMainWindow::slotLoadFinished, this, _1, _2, _3 ) ) );
    proj->load();
}

void WMainWindow::slotLoadFinished( boost::filesystem::path file, std::vector< std::string > errors, std::vector< std::string > warnings )
{
    // as this function might be called from outside the gui thread, use an event:
    QCoreApplication::postEvent( this, new WLoadFinishedEvent( file, errors, warnings ) );

    if( errors.size() )
    {
        wlog::warn( "MainWindow" ) << "Async load error occurred. Informing user.";
    }
}

void WMainWindow::openAboutQtDialog()
{
    QMessageBox::aboutQt( this, "About Qt" );
}

void WMainWindow::openAboutDialog()
{
    std::string filename( WPathHelper::getDocPath().string() + "/openwalnut-qt4/OpenWalnutAbout.html" );
    std::string content = readFileIntoString( filename );
    std::string windowHeading =  std::string( "About OpenWalnut " ) + std::string( W_VERSION );
    QMessageBox::about( this, windowHeading.c_str(), content.c_str() );
}

void WMainWindow::openOpenWalnutHelpDialog()
{
    std::string filename( WPathHelper::getDocPath().string() + "/openwalnut-qt4/OpenWalnutHelp.html" );

#ifndef QT4GUI_NOWEBKIT
    std::string content = readFileIntoString( filename );

    QWidget* window = new QWidget( this, Qt::Window );
    window->setWindowTitle( "OpenWalnut Help" );
    // specify intial layout
    QVBoxLayout *layout = new QVBoxLayout( window );
    window->setLayout( layout );
    window->resize( 500, 500 );

    window->show();

    QWebView *view = new QWebView( this );
    QString location( QString( "file://" ) + WPathHelper::getDocPath().string().c_str() + "/openwalnut-qt4/" );
    view->setHtml( content.c_str(), QUrl( location  ) );
    view->show();
    layout->addWidget( view );
#else
    QMessageBox::information( this, "Help", QString::fromStdString( "Sorry! Your version of OpenWalnut was not compiled with embedded help. "
                                                                    "To open the help pages in your browser, use this link: <a href=" +
                                                                    filename + ">Help</a>." ) );
#endif
}

void WMainWindow::openNotImplementedDialog()
{
    QMessageBox::information( this, "Not yet implemented!",
                              "This functionality is planned for future versions of OpenWalnut. "
                              "It is not yet implemented." );
}

boost::signals2::signal1< void, std::vector< std::string > >* WMainWindow::getLoaderSignal()
{
    return &m_loaderSignal;
}

WIconManager*  WMainWindow::getIconManager()
{
    return &m_iconManager;
}

void WMainWindow::closeEvent( QCloseEvent* e )
{
    // use some "Really Close?" Dialog here
    bool reallyClose = true;

    // handle close event
    if( reallyClose )
    {
        m_splash->show();
        m_splash->showMessage( "Shutting down" );

        saveWindowState();

        // signal everybody to shut down properly.
        m_splash->showMessage( "Shutting down kernel. Waiting for modules to finish." );
        WKernel::getRunningKernel()->finalize();

        // now nobody acesses the osg anymore
        m_splash->showMessage( "Shutting down GUI." );

        // clean up gl widgets
        m_mainGLDock->close();
        if( m_navAxial )
        {
            m_navAxial->close();
        }
        if( m_navCoronal )
        {
            m_navCoronal->close();
        }
        if( m_navSagittal )
        {
            m_navSagittal->close();
        }

        // delete CustomDockWidgets
        boost::mutex::scoped_lock lock( m_customDockWidgetsLock );
        for( std::map< std::string, boost::shared_ptr< WQtCustomDockWidget > >::iterator it = m_customDockWidgets.begin();
             it != m_customDockWidgets.end(); ++it )
        {
            it->second->close();
        }
        //m_customDockWidgetsLock.unlock();

        // finally close
        e->accept();
    }
    else
    {
        e->ignore();
    }
}

void WMainWindow::customEvent( QEvent* event )
{
    if( event->type() == WOpenCustomDockWidgetEvent::CUSTOM_TYPE )
    {
        // OpenCustomDockWidgetEvent
        WOpenCustomDockWidgetEvent* ocdwEvent = static_cast< WOpenCustomDockWidgetEvent* >( event );
        std::string title = ocdwEvent->getTitle();

        boost::shared_ptr< WQtCustomDockWidget > widget;

        boost::mutex::scoped_lock lock( m_customDockWidgetsLock );
        if( m_customDockWidgets.count( title ) == 0 )
        {
            // create new custom dock widget
            widget = boost::shared_ptr< WQtCustomDockWidget >(
                new WQtCustomDockWidget( title, m_glDock, ocdwEvent->getProjectionMode() ) );
            m_glDock->addDockWidget( Qt::BottomDockWidgetArea, widget.get() );

            // restore state and geometry
            m_glDock->restoreDockWidget( widget.get() );
            widget->restoreSettings();

            // store it in CustomDockWidget list
            m_customDockWidgets.insert( make_pair( title, widget ) );
        }
        else
        {
            widget = m_customDockWidgets[title];
            widget->increaseUseCount();
        }

        ocdwEvent->getFlag()->set( widget );
        boost::dynamic_pointer_cast< QDockWidget >( widget )->toggleViewAction()->activate( QAction::Trigger );
    }
    else if( event->type() == WCloseCustomDockWidgetEvent::CUSTOM_TYPE )
    {
        WCloseCustomDockWidgetEvent* closeEvent = static_cast< WCloseCustomDockWidgetEvent* >( event );
        boost::mutex::scoped_lock lock( m_customDockWidgetsLock );
        if( m_customDockWidgets.count( closeEvent->getTitle() ) > 0 )
        {
            if( m_customDockWidgets[closeEvent->getTitle()]->decreaseUseCount() )
            {
                // custom dock widget should be deleted
                m_customDockWidgets.erase( closeEvent->getTitle() );
            }
        }
    }
    else
    {
        // other event
        QMainWindow::customEvent( event );
    }
}

void WMainWindow::reportError( QWidget* parent, QString title, QString message )
{
    WQtMessagePopup* m = new WQtMessagePopup( parent, title, message, LL_ERROR );
    m->show();
    m_messageDock->addMessage( title, message, LL_ERROR );
}

void WMainWindow::reportWarning( QWidget* parent, QString title, QString message )
{
    WQtMessagePopup* m = new WQtMessagePopup( parent, title, message, LL_WARNING );
    m->show();
    m_messageDock->addMessage( title, message, LL_WARNING );
}

void WMainWindow::reportInfo( QWidget* parent, QString title, QString message )
{
    WQtMessagePopup* m = new WQtMessagePopup( parent, title, message, LL_INFO );
    m->show();
    m_messageDock->addMessage( title, message, LL_INFO );
}

bool WMainWindow::event( QEvent* event )
{
    // a module got associated with the root container -> add it to the list
    if( event->type() == WQT_READY_EVENT )
    {
        // convert event to ready event
        WModuleReadyEvent* e1 = dynamic_cast< WModuleReadyEvent* >( event );     // NOLINT
        if( e1 )
        {
            moduleSpecificSetup( e1->getModule() );
        }
    }

    // push the log message to the message dock
    if( event->type() == WQT_LOG_EVENT )
    {
        WLogEvent* e1 = dynamic_cast< WLogEvent* >( event );     // NOLINT
        if( e1 && getMessageDock() )
        {
            getMessageDock()->addLogMessage( e1->getEntry() );
        }
    }

    if( event->type() == WQT_CRASH_EVENT )
    {
        // convert event to ready event
        WModuleCrashEvent* e1 = dynamic_cast< WModuleCrashEvent* >( event );     // NOLINT
        if( e1 )
        {
            QString title = "Module \"" + QString::fromStdString( e1->getModule()->getName() ) + "\" caused a problem.";
            QString message = QString::fromStdString( e1->getMessage() );
            reportError( this, title, message );
        }
    }

    if( event->type() == WQT_MODULE_REMOVE_EVENT )
    {
        // convert event to ready event
        WModuleRemovedEvent* e1 = dynamic_cast< WModuleRemovedEvent* >( event );     // NOLINT
        if( e1 )
        {
            moduleSpecificCleanup( e1->getModule() );
        }
    }

    if( event->type() == WQT_LOADFINISHED )
    {
        // convert event
        WLoadFinishedEvent* e1 = dynamic_cast< WLoadFinishedEvent* >( event );
        if( e1 )
        {
            if( e1->getErrors().size() || e1->getWarnings().size() )
            {
                size_t curErrCount = 0;
                const size_t maxErrCount = 5;
                std::string errors = "<ul>";
                for( std::vector< std::string >::const_iterator iter = e1->getErrors().begin(); iter != e1->getErrors().end(); ++iter )
                {
                    errors += "<li> " + *iter;
                    curErrCount++;

                    if( ( curErrCount == maxErrCount ) && ( e1->getErrors().size() > maxErrCount ) )
                    {
                        size_t errDiff = e1->getErrors().size() - curErrCount;
                        errors += "<li> ... and " + string_utils::toString( errDiff ) + " more errors.";
                        break;
                    }
                }
                errors += "</ul>";

                size_t curWarnCount = 0;
                const size_t maxWarnCount = 5;
                std::string warnings = "<ul>";
                for( std::vector< std::string >::const_iterator iter = e1->getWarnings().begin(); iter != e1->getWarnings().end(); ++iter )
                {
                    warnings += "<li> " + *iter;
                    curWarnCount++;

                    if( ( curWarnCount == maxWarnCount ) && ( e1->getWarnings().size() > maxWarnCount ) )
                    {
                        size_t warnDiff = e1->getWarnings().size() - curWarnCount;
                        warnings += "<li> ... and " + string_utils::toString( warnDiff ) + " more warnings.";
                        break;
                    }
                }
                warnings += "</ul>";

                if( curWarnCount && curErrCount )   // Errors and warnings
                {
                    reportError( this, "There where errors and warnings during load.",
                                             "Errors occurred during load of \"" + QString::fromStdString( e1->getFilename() ) + "\". "
                                             "The loader tried to apply as much as possible, ignoring the erroneous data. The first errors where:"
                                             + QString::fromStdString( errors ) +
                                             "Warnings occurred during load of \"" + QString::fromStdString( e1->getFilename() ) + "\". "
                                             + QString::fromStdString( warnings )
                               );
                }
                else if( curWarnCount && !curErrCount ) // only warnings
                {
                    reportWarning( this, "There where warnings during load.",
                                             "Warnings occurred during load of \"" + QString::fromStdString( e1->getFilename() ) + "\". "
                                             + QString::fromStdString( warnings )
                                 );
                }
                else if( !curWarnCount && curErrCount ) // only errors
                {
                    reportError( this, "There where errors during load.",
                                             "Errors occurred during load of \"" + QString::fromStdString( e1->getFilename() ) + "\". "
                                             "The loader tried to apply as much as possible, ignoring the erroneous data. The first errors where:"
                                             + QString::fromStdString( errors )
                               );
                }
            }
        }
    }

    return QMainWindow::event( event );
}

boost::shared_ptr< WQtCustomDockWidget > WMainWindow::getCustomDockWidget( std::string title )
{
    boost::mutex::scoped_lock lock( m_customDockWidgetsLock );
    boost::shared_ptr< WQtCustomDockWidget > out = m_customDockWidgets.count( title ) > 0 ?
        m_customDockWidgets[title] :
        boost::shared_ptr< WQtCustomDockWidget >();
    return out;
}


void WMainWindow::closeCustomDockWidget( std::string title )
{
    QCoreApplication::postEvent( this, new WCloseCustomDockWidgetEvent( title ) );
}

void WMainWindow::newRoi()
{
    // do nothing if we can not get
    WPosition crossHairPos = WKernel::getRunningKernel()->getSelectionManager()->getCrosshair()->getPosition();
    WPosition minROIPos = crossHairPos - WPosition( 10., 10., 10. );
    WPosition maxROIPos = crossHairPos + WPosition( 10., 10., 10. );

    if( m_controlPanel->getFirstRoiInSelectedBranch().get() == NULL )
    {
        osg::ref_ptr< WROIBox > newRoi = osg::ref_ptr< WROIBox >( new WROIBox( minROIPos, maxROIPos ) );
        WKernel::getRunningKernel()->getRoiManager()->addRoi( newRoi );
    }
    else
    {
        osg::ref_ptr< WROIBox > newRoi = osg::ref_ptr< WROIBox >( new WROIBox( minROIPos, maxROIPos ) );
        WKernel::getRunningKernel()->getRoiManager()->addRoi( newRoi, m_controlPanel->getFirstRoiInSelectedBranch() );
    }
}

void WMainWindow::restoreSavedState()
{
    wlog::info( "MainWindow" ) << "Restoring window state.";

    restoreGeometry( WQt4Gui::getSettings().value( "MainWindowGeometry", "" ).toByteArray() );
    restoreState( WQt4Gui::getSettings().value( "MainWindowState", "" ).toByteArray() );

    m_glDock->restoreGeometry( WQt4Gui::getSettings().value( "GLDockWindowGeometry", "" ).toByteArray() );
    m_glDock->restoreState( WQt4Gui::getSettings().value( "GLDockWindowState", "" ).toByteArray() );

    if( m_navAxial )
    {
        m_navAxial->restoreSettings();
    }
    if( m_navCoronal )
    {
        m_navCoronal->restoreSettings();
    }
    if( m_navSagittal )
    {
        m_navSagittal->restoreSettings();
    }
}

void WMainWindow::saveWindowState()
{
    wlog::info( "MainWindow" ) << "Saving window state.";

    // this saves the window state to some common location on the target OS in user scope.
    WQt4Gui::getSettings().setValue( "MainWindowState", saveState() );
    WQt4Gui::getSettings().setValue( "GLDockWindowState", m_glDock->saveState() );

    // NOTE: Qt Doc says that saveState also saves geometry. But this somehow is wrong (at least for 4.6.3)
    WQt4Gui::getSettings().setValue( "MainWindowGeometry", saveGeometry() );
    WQt4Gui::getSettings().setValue( "GLDockWindowGeometry", m_glDock->saveGeometry() );

    m_messageDock->saveSettings();
}

QSettings& WMainWindow::getSettings()
{
    return WQt4Gui::getSettings();
}

void WMainWindow::setSetting( std::string key, std::string value )
{
    WMainWindow::getSettings().setValue( QString::fromStdString( key ), QString::fromStdString( value ) );
}

void WMainWindow::handleLogLevelUpdate( unsigned int logLevel )
{
    WLogger::getLogger()->setDefaultLogLevel( static_cast< LogLevel >( logLevel ) );
}

void WMainWindow::handleGLVendor()
{
    // WARNING: never put blocking code here, as it might freeze the mainGLWidget.
    std::string vendor = string_utils::toLower( m_mainGLWidget->getViewer()->getOpenGLVendor() );

    // is this a mesa card?
    if( vendor.find( "mesa" ) != std::string::npos )
    {
        QString msg = "<b>Warning:</b> Your graphics card is powered by the Mesa OpenGL implementation. OpenWalnut does not support Mesa "
                      "officially, since Mesa has some severe problems with GLSL shaders. You can still use OpenWalnut, but you should be "
                      "aware that Mesa can freeze OpenWalnut. Ensure you have the latest version of Mesa installed to avoid problems.";
        QLabel* l = new QLabel( msg );
        l->setWordWrap( true );
        l->setMinimumWidth( 640 );

        WQtMessageDialog* msgDia = new WQtMessageDialog( "MesaWarning", "Mesa Warning", l, getSettings(), this );
        msgDia->show();
    }

    // is this a mesa card?
    if( ( vendor.find( "Chromium" ) != std::string::npos ) ||
        ( vendor.find( "Humper" ) != std::string::npos ) )
    {
        QString msg = "<b>Warning:</b> You seem to use OpenWalnut from inside a virtual machine. Graphics acceleration on these virtual machines"
                      " is often limited. OpenWalnut might not properly work in your setup.";
        QLabel* l = new QLabel( msg );
        l->setWordWrap( true );
        l->setMinimumWidth( 640 );

        WQtMessageDialog* msgDia = new WQtMessageDialog( "VMChromiumWarning", "Virtual Machine Warning", l, getSettings(), this );
        msgDia->show();
    }
}

void WMainWindow::showWelcomeDialog( bool force )
{
    // Load welcome file
    std::string filename( WPathHelper::getDocPath().string() + "/openwalnut-qt4/OpenWalnutWelcome.html" );
    std::string content = readFileIntoString( filename );

    // gen ID for it using version (allows showing release/welcome message for each new release)
    QString msgID = "OpenWalnutWelcome" + QString( W_LIB_VERSION );

    // replace OW_VERSION
    boost::regex reg1( "%OW_LIB_VERSION%" );
    boost::regex reg2( "%OW_VERSION%" );
    content = boost::regex_replace( content, reg1, std::string( W_LIB_VERSION ) );
    content = boost::regex_replace( content, reg2, std::string( W_VERSION ) );

    QWidget* w = NULL;
#ifndef QT4GUI_NOWEBKIT
    QWebView* view = new QWebView( this );
    view->setHtml( QString::fromStdString( content ) );
    view->setMinimumWidth( 640 );
    view->page()->setLinkDelegationPolicy( QWebPage::DelegateExternalLinks );
    w = view;
#else
    QLabel* l = new QLabel( QString::fromStdString( content ) );
    l->setWordWrap( true );
    l->setMinimumWidth( 640 );
    w = l;
#endif

    WQtMessageDialog* msgDia = new WQtMessageDialog( msgID, "Welcome to OpenWalnut", w, getSettings(), this );
    msgDia->show( force );
}

void WMainWindow::handleStartMessages()
{
    // only show welcome dialog for now
    showWelcomeDialog( false );
}

void WMainWindow::handleDrop( QDropEvent* event )
{
    if( event->mimeData()->hasUrls() )
    {
        std::vector < std::string > projects;
        std::vector < std::string > filenames;
        std::vector < std::string > unsupported;
        foreach( QUrl url, event->mimeData()->urls() )
        {
            QString path =  url.toLocalFile();
            QFileInfo info( path );
            QString suffix =  info.completeSuffix();
            if( suffix.endsWith( "cnt" )
              || suffix.endsWith( "edf" )
              || suffix.endsWith( "asc" )
              || suffix.endsWith( "nii" )
              || suffix.endsWith( "nii.gz" )
              || suffix.endsWith( "fib" )
              || suffix.endsWith( "fdg" )
              || suffix.endsWith( "vtk" ) )
            {
                filenames.push_back( path.toStdString() );
            }
            else
            {
                if( suffix == "owp" || suffix == "owproj" )
                {
                    projects.push_back( path.toStdString() );
                }
                else
                {
                    unsupported.push_back( path.toStdString() );
                }
            }
        }
        if( projects.size() > 0 )
        {
            for( size_t i = 0; i < projects.size(); ++i )
            {
                asyncProjectLoad( projects[ i ] );
            }
            event->accept();
        }
        if( filenames.size() > 0 )
        {
            m_loaderSignal( filenames );
            event->accept();
        }
        if( unsupported.size() > 0 )
        {
            QString message = QString() +
                "The following files are not supported as standard data types by OpenWalnut at the moment:<br>";
            for( size_t i = 0; i < unsupported.size(); ++i )
            {
                message += QString::fromStdString( unsupported[ i ] ) + QString( "<br>" );
            }
            message += "There may be additional modules supporting them.<br>"
                "All other files have been loaded and should be visible in the module "
                "browser and network editor.";
            QMessageBox::information( this, "Not yet implemented!",
                    message
                    );
        }
    }
}

void WMainWindow::dropEvent( QDropEvent* event )
{
    handleDrop( event );
    QMainWindow::dropEvent( event );
}

bool WMainWindow::isDropAcceptable( const QMimeData* mimeData )
{
    if( mimeData->hasUrls() )
    {
        foreach( QUrl url, mimeData->urls() )
        {
            QString path =  url.toLocalFile();
            QFileInfo info( path );
            QString suffix =  info.completeSuffix();
            if( suffix.endsWith( "cnt" )
              || suffix.endsWith( "edf" )
              || suffix.endsWith( "asc" )
              || suffix.endsWith( "nii" )
              || suffix.endsWith( "nii.gz" )
              || suffix.endsWith( "fib" )
              || suffix.endsWith( "fdg" )
              || suffix.endsWith( "vtk" )
              || suffix.endsWith( "owp" )
              || suffix.endsWith( "owproj" ) )
            {
                return true;
            }
        }
    }

    return false;
}

void WMainWindow::dragMoveEvent( QDragMoveEvent* event )
{
    if( WMainWindow::isDropAcceptable( event->mimeData() ) )
    {
        event->acceptProposedAction();
    }
    QMainWindow::dragMoveEvent( event );
}

void WMainWindow::dragEnterEvent( QDragEnterEvent* event )
{
    if( WMainWindow::isDropAcceptable( event->mimeData() ) )
    {
        event->acceptProposedAction();
    }
    QMainWindow::dragEnterEvent( event );
}

void WMainWindow::closeSplash()
{
    m_splash->finish( this );
}

QSplashScreen* WMainWindow::getSplash() const
{
    return m_splash;
}

void WMainWindow::addGlobalMenu( QWidget* widget )
{
    widget->addAction( createSeperator( this ) );
    widget->addAction( m_newAction );
    widget->addAction( m_loadButton );
    widget->addAction( m_saveAction );
    widget->addAction( createSeperator( this ) );
    widget->addAction( m_settingsAction );
    widget->addAction( createSeperator( this ) );
    widget->addAction( m_helpAction );
    widget->addAction( createSeperator( this ) );
    widget->addAction( m_quitAction );
}

const WQtMessageDock* WMainWindow::getMessageDock() const
{
    return m_messageDock;
}

WQtMessageDock* WMainWindow::getMessageDock()
{
    return m_messageDock;
}
