
/*
 * Copyright (c) Abraham vd Merwe <abz@blio.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *	  notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *	  notice, this list of conditions and the following disclaimer in the
 *	  documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of other contributors
 *	  may be used to endorse or promote products derived from this software
 *	  without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <inttypes.h>
#include <stdlib.h>
#include <errno.h>
#include <ctype.h>

#if defined(GETHOSTBYNAME) || defined(GETSERVBYNAME)
#include <netdb.h>
#endif	/* #if defined(GETHOSTBYNAME) || defined(GETSERVBYNAME) */

#include <ber/ber.h>

#include <debug/memory.h>

#include <tinysnmp/tinysnmp.h>
#include <tinysnmp/manager/snmp.h>

#include "atou16.h"
#include "bprintf.h"
#include "router.h"

#define ARRAYSIZE(x) (sizeof (x) / sizeof ((x)[0]))
#define PRINTABLE(x) (isprint (x) || isspace (x))

static int get_interface_descriptions (router_t *router)
{
   uint32_t ifDescr[11] = { 10, 43, 6, 1, 2, 1, 2, 2, 1, 2, 0 };
   uint32_t *oid = ifDescr;
   snmp_value_t *value = NULL;
   interface_t *tmp;
   int i;

   for (tmp = router->iface; tmp != NULL; tmp = tmp->next)
	 {
		oid[10] = tmp->idx;

		if ((value = snmp_get (&router->agent,&oid,1)) == NULL)
		  return (-1);

		if (value->type != BER_OCTET_STRING)
		  {
			 snmp_free_values (&value,1);
			 for (tmp = router->iface; tmp != NULL; tmp = tmp->next)
			   if (tmp->descr != NULL)
				 mem_free (tmp->descr);
			 errno = EINVAL;
			 return (-1);
		  }

		if ((tmp->descr = (char *) mem_alloc ((value->data.OCTET_STRING.len + 1) * sizeof (char))) == NULL)
		  {
			 snmp_free_values (&value,1);
			 for (tmp = router->iface; tmp != NULL; tmp = tmp->next)
			   if (tmp->descr != NULL)
				 mem_free (tmp->descr);
			 errno = ENOMEM;
			 return (-1);
		  }

		memset (tmp->descr,'\0',(value->data.OCTET_STRING.len + 1) * sizeof (char));

		for (i = 0; i < value->data.OCTET_STRING.len; i++)
		  tmp->descr[i] = PRINTABLE (value->data.OCTET_STRING.buf[i]) ? value->data.OCTET_STRING.buf[i] : '?';

		snmp_free_values (&value,1);
	 }

   return (0);
}

static int get_interface_aliases (router_t *router)
{
   uint32_t ifAlias[12] = { 11, 43, 6, 1, 2, 1, 31, 1, 1, 1, 18, 0 };
   uint32_t *oid = ifAlias;
   snmp_value_t *value = NULL;
   interface_t *tmp;
   int i;

   for (tmp = router->iface; tmp != NULL; tmp = tmp->next)
	 {
		oid[11] = tmp->idx;

		/* aliases are optional */
		if ((value = snmp_get (&router->agent,&oid,1)) == NULL)
		  continue;

		if (value->type == BER_NULL)
		  {
			 snmp_free_values (&value,1);
			 continue;
		  }

		if (value->type != BER_OCTET_STRING)
		  {
			 snmp_free_values (&value,1);
			 for (tmp = router->iface; tmp != NULL; tmp = tmp->next)
			   if (tmp->alias != NULL)
				 mem_free (tmp->alias);

			 errno = EINVAL;
			 return (-1);
		  }

		if ((tmp->alias = (char *) mem_alloc ((value->data.OCTET_STRING.len + 1) * sizeof (char))) == NULL)
		  {
			 snmp_free_values (&value,1);
			 for (tmp = router->iface; tmp != NULL; tmp = tmp->next)
			   if (tmp->alias != NULL)
				 mem_free (tmp->alias);
			 errno = ENOMEM;
			 return (-1);
		  }

		memset (tmp->alias,'\0',(value->data.OCTET_STRING.len + 1) * sizeof (char));

		for (i = 0; i < value->data.OCTET_STRING.len; i++)
		  tmp->alias[i] = PRINTABLE (value->data.OCTET_STRING.buf[i]) ? value->data.OCTET_STRING.buf[i] : '?';

		snmp_free_values (&value,1);

		if (tmp->alias[0] == '\0')
		  {
			 mem_free (tmp->alias);
			 tmp->alias = NULL;
		  }
	 }

   return (0);
}

static void destroy_interface_list (interface_t **i)
{
   interface_t *tmp;
   while (*i != NULL)
	 {
		tmp = *i, *i = (*i)->next;
		if (tmp->descr != NULL) mem_free (tmp->descr);
		if (tmp->alias != NULL) mem_free (tmp->alias);
		mem_free (tmp);
	 }
}

static int get_num_interfaces (snmp_agent_t *agent)
{
   int32_t n;
   uint32_t *oid,ifNumber[9] = { 8, 43, 6, 1, 2, 1, 2, 1, 0 };
   snmp_value_t *value = NULL;

   oid = ifNumber;

   if ((value = snmp_get (agent,&oid,1)) == NULL)
	 return (-1);

   if (value->type != BER_INTEGER)
	 {
		snmp_free_values (&value,1);
		errno = EINVAL;
		return (-1);
	 }

   n = value->data.INTEGER;

   snmp_free_values (&value,1);

   return (n);
}

static interface_t *create_interface_list (snmp_agent_t *agent)
{
   int i,saved,n;
   uint32_t *oid,ifIndex[11] = { 9, 43, 6, 1, 2, 1, 2, 2, 1, 1 };
   snmp_next_value_t *next = NULL;
   interface_t *node,*iface = NULL;

   if ((n = get_num_interfaces (agent)) < 0)
	 return (NULL);

   oid = ifIndex;

   for (i = 0; i < n; i++)
	 {
		if ((next = snmp_get_next (agent,&oid,1)) == NULL)
		  {
			 saved = errno;
			 if (iface != NULL) destroy_interface_list (&iface);
			 errno = saved;
			 return (NULL);
		  }

		if (next->value.type != BER_INTEGER || next->oid[0] != 10 || memcmp (next->oid + 1,oid + 1,36))
		  {
			 snmp_free_next_values (&next,1);
			 if (iface != NULL) destroy_interface_list (&iface);
			 errno = EINVAL;	/* any better error for non-compliant MIB? */
			 return (NULL);
		  }

		memcpy (oid,next->oid,44);

		if ((node = (interface_t *) mem_alloc (sizeof (interface_t))) == NULL)
		  {
			 saved = errno;
			 snmp_free_next_values (&next,1);
			 if (iface != NULL) destroy_interface_list (&iface);
			 errno = saved;
			 return (NULL);
		  }

		node->idx = next->value.data.INTEGER;
		node->descr = NULL;
		node->alias = NULL;
		node->next = NULL;

		if (iface != NULL)
		  {
			 interface_t *cur = iface;
			 while (cur->next != NULL) cur = cur->next;
			 cur->next = node;
		  }
		else iface = node;

		snmp_free_next_values (&next,1);
	 }

   if (iface == NULL) errno = 0;

   return (iface);
}

/*
 * Initialize the router structure used by other functions.
 */
int router_open (router_t *router,const char *hostname,const char *service,const char *community,uint32_t timeout,uint32_t retries)
{
   struct in_addr addr;
   struct sockaddr_in sock;
#ifdef GETHOSTBYNAME
   struct hostent *host;
#endif	/* #ifdef GETHOSTBYNAME */
#ifdef GETSERVBYNAME
   struct servent *serv;
#endif	/* #ifdef GETSERVBYNAME */
   uint16_t port;
   uint32_t *oid,sysName[9] = { 8, 43, 6, 1, 2, 1, 1, 5, 0 };
   snmp_value_t *value;
   int i,saved;

   if (inet_aton (hostname,&addr))
	 {
		sock.sin_addr.s_addr = addr.s_addr;
		if ((router->hostname = bprintf ("%d.%d.%d.%d",
										 (addr.s_addr & 0xff000000) >> 24,
										 (addr.s_addr & 0x00ff0000) >> 16,
										 (addr.s_addr & 0x0000ff00) >> 8,
										 (addr.s_addr & 0x000000ff))) == NULL)
		  return (-1);
	 }
#ifdef GETHOSTBYNAME
   else if ((host = gethostbyname (hostname)) != NULL)
	 {
		if ((router->hostname = (char *) mem_alloc ((strlen (host->h_name) + 1) * sizeof (char))) == NULL)
		  return (-1);

		strcpy (router->hostname,host->h_name);

		memcpy (&sock.sin_addr,(struct in_addr *) host->h_addr,sizeof (struct in_addr));
	 }
#endif	/* #ifdef GETHOSTBYNAME */
   else
	 {
		errno = EINVAL;
		return (-1);
	 }

   if (service == NULL)
	 {
		sock.sin_port = htons (161);
#ifdef GETSERVBYNAME
		if ((serv = getservbyname ("snmp","udp")) != NULL)
		  sock.sin_port = serv->s_port;
#endif	/* #ifdef GETSERVBYNAME */
	 }
   else
	 {
		if (!atou16 (service,&port))
		  sock.sin_port = htons (port);
#ifdef GETSERVBYNAME
		else if ((serv = getservbyname (service,"udp")) != NULL || (serv = getservbyname ("snmp","udp")) != NULL)
		  sock.sin_port = serv->s_port;
#endif	/* #ifdef GETSERVBYNAME */
		else sock.sin_port = htons (161);
	 }

   sock.sin_family = PF_INET;

   if (snmp_open (&router->agent,&sock,community,timeout,retries) < 0)
	 {
		saved = errno;
		mem_free (router->hostname);
		errno = saved;
		return (-1);
	 }

   oid = sysName;
   router->name = NULL;

   do
	 {
		if ((value = snmp_get (&router->agent,&oid,1)) != NULL)
		  if (value->type != BER_OCTET_STRING)
			{
			   snmp_free_values (&value,1);
			   value = NULL;
			}

		if (value != NULL)
		  {
			 if (!value->data.OCTET_STRING.len)
			   {
				  snmp_free_values (&value,1);
				  break;
			   }

			 if ((router->name = (char *) mem_alloc ((value->data.OCTET_STRING.len + 1) * sizeof (char))) == NULL)
			   {
				  snmp_free_values (&value,1);
				  break;
			   }

			 memset (router->name,'\0',(value->data.OCTET_STRING.len + 1) * sizeof (char));

			 for (i = 0; i < value->data.OCTET_STRING.len; i++)
			   router->name[i] = PRINTABLE (value->data.OCTET_STRING.buf[i]) ? value->data.OCTET_STRING.buf[i] : '?';

			 snmp_free_values (&value,1);
		  }
	 } while (0);

   if (router->name == NULL)
	 {
		if ((router->name = (char *) mem_alloc ((strlen (hostname) + 1) * sizeof (char))) == NULL)
		  {
			 saved = errno;
			 snmp_close (&router->agent);
			 mem_free (router->hostname);
			 errno = saved;
			 return (-1);
		  }
		strcpy (router->name,hostname);
	 }

   if ((router->iface = create_interface_list (&router->agent)) == NULL)
	 {
		if (!errno) return (0);
		saved = errno;
		snmp_close (&router->agent);
		mem_free (router->hostname);
		mem_free (router->name);
		errno = saved;
		return (-1);
	 }

   if (get_interface_descriptions (router) < 0 || get_interface_aliases (router) < 0)
	 {
		saved = errno;
		snmp_close (&router->agent);
		destroy_interface_list (&router->iface);
		mem_free (router->hostname);
		mem_free (router->name);
		errno = saved;
		return (-1);
	 }

   router->ignore_admin_down = 0;

   return (0);
}

/*
 * Query router and update interface counters.
 */
int router_update (router_t *router)
{
   snmp_value_t *value = NULL;
   uint32_t ifInOctets[11]    = { 10, 43, 6, 1, 2, 1, 2, 2, 1, 10, 0 };
   uint32_t ifOutOctets[11]   = { 10, 43, 6, 1, 2, 1, 2, 2, 1, 16, 0 };
   uint32_t ifInDiscards[11]  = { 10, 43, 6, 1, 2, 1, 2, 2, 1, 13, 0 };
   uint32_t ifOutDiscards[11] = { 10, 43, 6, 1, 2, 1, 2, 2, 1, 19, 0 };
   uint32_t ifInErrors[11]    = { 10, 43, 6, 1, 2, 1, 2, 2, 1, 14, 0 };
   uint32_t ifOutErrors[11]   = { 10, 43, 6, 1, 2, 1, 2, 2, 1, 20, 0 };
   uint32_t ifOperStatus[11]  = { 10, 43, 6, 1, 2, 1, 2, 2, 1, 8,  0 };
   uint32_t ifAdminStatus[11] = { 10, 43, 6, 1, 2, 1, 2, 2, 1, 7,  0 };
   uint32_t ifSpeed[11]       = { 10, 43, 6, 1, 2, 1, 2, 2, 1, 5,  0 };
   uint32_t *status[2] =
	 {
		ifOperStatus,
		ifAdminStatus
	 };
   uint32_t *counters[6 + 1] =
	 {
		ifInOctets,
		ifOutOctets,
		ifInDiscards,
		ifOutDiscards,
		ifInErrors,
		ifOutErrors,
		ifSpeed			/* well, not all of them... (; */
	 };
   int i,up;
   interface_t *tmp;

   /* update counters and status */
   for (tmp = router->iface; tmp != NULL; tmp = tmp->next)
	 {
		up = 1;

		for (i = 0; i < ARRAYSIZE (status); i++) status[i][10] = tmp->idx;

		if ((value = snmp_get (&router->agent,status,ARRAYSIZE (status))) == NULL)
		  return (-1);

		for (i = 0; i < ARRAYSIZE (status); i++)
		  {
			 if (value[i].type != BER_INTEGER)
			   {
				  type_mismatch:
				  snmp_free_values (&value,1);
				  errno = EINVAL;
				  return (-1);
			   }

			 tmp->status[i] = value[i].data.INTEGER;
			 if (tmp->status[i] != IF_STATUS_UP)
			   tmp->status[i] = IF_STATUS_DOWN, up = 0;
		  }

		mem_free (value);

		if (up)
		  {
			 for (i = 0; i < ARRAYSIZE (counters); i++) counters[i][10] = tmp->idx;

			 if ((value = snmp_get (&router->agent,counters,ARRAYSIZE (counters))) == NULL)
			   return (-1);

			 for (i = 0; i < ARRAYSIZE (counters) - 1; i++)
			   if (value[i].type != BER_Counter32)
				 goto type_mismatch;

			 if (value[ARRAYSIZE (counters) - 1].type != BER_Gauge32)
			   goto type_mismatch;

			 tmp->octets[IF_PACKET_IN]    = value[0].data.Counter32;
			 tmp->octets[IF_PACKET_OUT]   = value[1].data.Counter32;
			 tmp->discards[IF_PACKET_IN]  = value[2].data.Counter32;
			 tmp->discards[IF_PACKET_OUT] = value[3].data.Counter32;
			 tmp->errors[IF_PACKET_IN]    = value[4].data.Counter32;
			 tmp->errors[IF_PACKET_OUT]   = value[5].data.Counter32;
			 tmp->speed                   = value[6].data.Gauge32;

			 mem_free (value);
		  }

		tmp->valid = 1;
	 }

   return (0);
}

/*
 * Remove the specified interface from the list of router interfaces.
 */
int router_purge_interface (router_t *router,const char *descr)
{
   interface_t *cur = router->iface,*prev = NULL;

   while (cur != NULL)
	 {
		if (!strcmp (cur->descr,descr))
		  {
			 if (prev != NULL && cur->next != NULL)
			   prev->next = cur->next;
			 else if (prev == NULL)
			   router->iface = cur->next;
			 else prev->next = NULL;
			 mem_free (cur->descr);
			 if (cur->alias != NULL) mem_free (cur->alias);
			 mem_free (cur);
			 return (0);
		  }
		else prev = cur, cur = cur->next;
	 }

   return (-1);
}

/*
 * Close the connection and free memory allocated by router_open().
 */
void router_close (router_t *router)
{
   snmp_close (&router->agent);
   destroy_interface_list (&router->iface);
   mem_free (router->name);
   mem_free (router->hostname);
}

