
/*
 * Copyright (c) Abraham vd Merwe <abz@blio.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *	  notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *	  notice, this list of conditions and the following disclaimer in the
 *	  documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of other contributors
 *	  may be used to endorse or promote products derived from this software
 *	  without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdarg.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <signal.h>
#include <sys/time.h>
#include <sys/types.h>
#include <fcntl.h>
#include <ctype.h>

#include <debug/memory.h>
#include <debug/log.h>

#include "router.h"
#include "db.h"
#include "atou32.h"
#include "atou16.h"
#include "config.h"
#include "html.h"

#define printable(x) (isprint (x) || isspace (x))

typedef struct
{
   const char *logfile;		/* location of log file				*/
   int loglevel;			/* logging system verbosity			*/
   int daemon;				/* are we a daemon?					*/
   const char *configfile;	/* location of configuration file	*/
} options_t;

static volatile int finished = 0;
static volatile int rotated = 0;

static void catch_signals (int sig)
{
   if (sig == SIGUSR1)
	 rotated = 1;
   else
	 finished = 1;
}

static void print_settings (options_t *o,config_t *c)
{
   int i;
   const char *str[] = { "quiet", "errors", "warnings", "normal", "verbose", "debug", "noisy" };

   log_printf (LOG_DEBUG,
			   "[options]\n"
			   "   logfile: %s\n"
			   "  loglevel: %s\n"
			   "    daemon: %s\n"
			   "configfile: %s\n"
			   "\n",
			   o->logfile,
			   str[o->loglevel],
			   (o->daemon ? "yes" : "no"),
			   o->configfile);

   log_printf (LOG_DEBUG,
			   "[config]\n"
			   "database: %s\n"
			   "  htdocs: %s\n"
			   " pidfile: %s\n"
			   "     uid: %u\n"
			   "     gid: %u\n"
			   "\n",
			   c->database,
			   c->htdocs,
			   c->pidfile,
			   c->uid,
			   c->gid);

   for (i = 0; i < c->n; i++)
	 {
		int j;
		log_printf (LOG_NOISY,
					"[config.router %d]\n"
					"         hostname: %s\n"
					"          service: %s\n"
					"        community: %s\n"
					"          timeout: %u ms\n"
					"          retries: %u\n"
					"ignore_admin_down: %s\n"
					"           ignore:",
					i,
					c->router[i].hostname,
					c->router[i].service,
					c->router[i].community,
					c->router[i].timeout,
					c->router[i].retries,
					c->router[i].ignore_admin_down ? "true" : "false");

		for (j = 0; j < c->router[i].n_ignore; j++)
		  log_printf (LOG_NOISY," \"%s\"",c->router[i].ignore[j]);

		log_printf (LOG_NOISY,
					"\n            order:");
		for (j = 0; j < c->router[i].n_order; j++)
		  log_printf (LOG_NOISY," \"%s\"",c->router[i].order[j]);

		log_printf (LOG_NOISY,
					"\n            alias:");
		for (j = 0; j < c->router[i].n_alias; j++)
		  log_printf (LOG_NOISY," \"%s\"",c->router[i].alias[j]);

		log_printf (LOG_NOISY,"\n\n");
	 }
}

static void print_router (router_t *router,int idx)
{
   interface_t *node;

   log_printf (LOG_DEBUG,
			   "[router %d]\n"
			   "         hostname: %s\n"
			   "             name: %s\n"
			   "ignore_admin_down: %s\n"
			   "\n",
			   idx,
			   router->hostname,
			   router->name,
			   router->ignore_admin_down ? "true" : "false");

   for (node = router->iface; node != NULL; node = node->next)
	 {
		log_printf (LOG_NOISY,
					"[router.interface %d]\n"
					"   descr: %s\n"
					"   alias: %s\n"
					"   valid: %d\n"
					"  octets: in %-20" PRIu32 "out %" PRIu32 "\n"
					"disarded: in %-20" PRIu32 "out %" PRIu32 "\n"
					"  errors: in %-20" PRIu32 "out %" PRIu32 "\n"
					"  status: admin %s, operational %s\n"
					"   speed: %" PRIu32 "\n"
					"\n",
					node->idx,
					node->descr,
					node->alias,
					node->valid,
					node->octets[IF_PACKET_IN],node->octets[IF_PACKET_OUT],
					node->discards[IF_PACKET_IN],node->discards[IF_PACKET_OUT],
					node->errors[IF_PACKET_IN],node->errors[IF_PACKET_OUT],
					node->status[IF_ADMIN_STATUS] == IF_STATUS_UP ? "up(1)" : "down(2)",
					node->status[IF_OPER_STATUS] == IF_STATUS_UP ? "up(1)" : "down(2)",
					node->speed);
	 }
}

#define usage(p) help_stub(p,0)
#define help(p) help_stub(p,1)
static void help_stub (const char *progname,int verbose)
{
   fprintf (stderr,
			"usage: %s [OPTIONS] <configfile>\n"
			"       %s -h | --help\n",
			progname,progname);

   if (verbose)
	 {
		fprintf (stderr,
				 "\n"
				 "where OPTIONS can be any of the following:\n"
				 "\n"
				 "%s"
				 "\n"
				 "    -f | --logfile=FILENAME     Location of log file. By default, messages will be\n"
				 "                                logged to the console.\n"
				 "\n"
				 "    -l | --loglevel=LEVEL       The following levels can be specified (in increasing\n"
				 "                                order of verbosity):\n"
				 "\n"
				 "                                    quiet     No messages will be logged\n"
				 "                                    errors    Only errors will be logged\n"
				 "                                    warnings  Warnings & errors will be logged\n"
				 "                                    normal    Default messages and all of the above\n"
				 "                                    verbose   Verbose messages will also be logged\n"
				 "                                    debug     Log debug messages as well\n"
				 "                                    noisy     Noisy debug messages will be logged as well\n"
				 "\n"
				 "                                If a log file is specified, ORCA defaults to ``verbose'',\n"
				 "                                otherwise, ``noisy'' is selected.\n"
				 "\n"
				 "    -d | --daemon               Run in the background. You need to specify a log file if\n"
				 "                                you want to use this option.\n"
				 "\n"
				 "    -h | --help                 Show this help message.\n"
				 "\n"
				 "    <configfile>                Location of ORCA configuration file.\n"
				 "\n",
				 progname);
	 }

   exit (EXIT_FAILURE);
}

static void parse_options (options_t *opts,int argc,char *argv[])
{
   const char *progname;
   int i;

   (progname = strrchr (argv[0],'/')) ? progname++ : (progname = argv[0]);

   /* is the user requesting help? */
   if (argc == 2 && (!strcmp (argv[1],"-h") || !strcmp (argv[1],"--help")))
	 help (progname);

   if (argc < 2) usage (progname);

   opts->logfile = NULL;
   opts->loglevel = -1;
   opts->daemon = 0;

   for (i = 1; i < argc - 1; i++)
	 {
		if (!strcmp (argv[i],"-d") || !strcmp (argv[i],"--daemon"))
		  opts->daemon = 1;
		else if (!strcmp (argv[i],"-f") || !strncmp (argv[i],"--logfile=",10))
		  {
			 char *arg = strlen (argv[i]) == 2 ? argv[++i] : strchr (argv[i],'=') + 1;
			 if (i == argc - 1 || *arg == '\0') usage (progname);
			 opts->logfile = arg;
		  }
		else if (!strcmp (argv[i],"-l") || !strncmp (argv[i],"--loglevel=",11))
		  {
			 char *arg = strlen (argv[i]) == 2 ? argv[++i] : strchr (argv[i],'=') + 1;
			 if (i == argc - 1 || *arg == '\0') usage (progname);

			 if (!strcmp (arg,"quiet"))
			   opts->loglevel = LOG_QUIET;
			 else if (!strcmp (arg,"errors"))
			   opts->loglevel = LOG_ERROR;
			 else if (!strcmp (arg,"warnings"))
			   opts->loglevel = LOG_WARNING;
			 else if (!strcmp (arg,"normal"))
			   opts->loglevel = LOG_NORMAL;
			 else if (!strcmp (arg,"verbose"))
			   opts->loglevel = LOG_VERBOSE;
			 else if (!strcmp (arg,"debug"))
			   opts->loglevel = LOG_DEBUG;
			 else if (!strcmp (arg,"noisy"))
			   opts->loglevel = LOG_NOISY;
			 else usage (progname);
		  }
		else usage (progname);
	 }

   opts->configfile = argv[i];

   if (opts->daemon && opts->logfile == NULL)
	 usage (progname);

   if (opts->loglevel == -1)
	 opts->loglevel = opts->logfile == NULL ? LOG_NOISY : LOG_VERBOSE;
}

static const char *create_main_index (const char *htdocs,const router_t *router,int n)
{
   int i = 0,found = 0;
   interface_t *tmp = NULL;
   char *src,*dest;

   while (!found && i < n)
	 {
		tmp = router[i].iface;
		while (!found && tmp != NULL)
		  {
			 if (!router[i].ignore_admin_down || tmp->status[IF_ADMIN_STATUS] == IF_STATUS_UP)
			   {
				  found = 1;
				  break;
			   }
			 tmp = tmp->next;
		  }
	 }

   if (!found) return ("No routers with configured interfaces");

   if ((dest = (char *) mem_alloc ((strlen (htdocs) + 12) * sizeof (char))) == NULL)
	 return ("Out of memory");

   if ((src = (char *) mem_alloc ((strlen (htdocs) + strlen (router[i].hostname) + 40) * sizeof (char))) == NULL)
	 {
		mem_free (dest);
		return ("Out of memory");
	 }

   strcpy (dest,htdocs);
   if (*dest != '\0') strcat (dest,"/");
   sprintf (src,"%s%s/%d/index.html",dest,router[i].hostname,tmp->idx);
   strcat (dest,"index.html");

   log_printf (LOG_DEBUG,"linking %s to %s\n",src,dest);

   unlink (dest);
   if (symlink (src,dest) < 0)
	 {
		mem_free (src);
		mem_free (dest);
		return ("Couldn't create symlink");
	 }

   mem_free (src);
   mem_free (dest);

   return (NULL);
}

static int create_pidfile (const char *filename)
{
   int fd,result;
   char str[32];

   if ((fd = creat (filename,0644)) < 0) return (-1);

   sprintf (str,"%u\n",getpid ());

   if ((result = write (fd,str,strlen (str))) != strlen (str))
	 {
		int saved = result < 0 ? errno : EIO;
		close (fd);
		errno = saved;
		return (-1);
	 }

   close (fd);

   return (0);
}

static void order_interfaces (router_t *router,char *order[],size_t n)
{
   size_t i,j,k;
   interface_t *tmp,*cur1,*cur2,*prev1,*prev2;
   int found;

   for (i = 0; i < n; i++)
	 {
		found = 0;
		for (j = 0, tmp = router->iface; tmp != NULL; j++, tmp = tmp->next)
		  if (tmp->descr != NULL && !strcmp (order[i],tmp->descr))
			{
			   found = 1;
			   break;
			}

		if (found)
		  {
			 /* swap interface i and j around */

			 for (cur1 = router->iface, prev1 = NULL, k = 0; cur1->next != NULL && k < i; k++)
			   prev1 = cur1, cur1 = cur1->next;

			 for (cur2 = router->iface, prev2 = NULL, k = 0; cur2->next != NULL && k < j; k++)
			   prev2 = cur2, cur2 = cur2->next;

			 if (prev1 != NULL) prev1->next = cur2;
			 if (prev2 != NULL) prev2->next = cur1;

			 tmp = cur1->next, cur1->next = cur2->next, cur2->next = tmp;

			 if (prev1 == NULL) router->iface = cur2;
			 if (prev2 == NULL) router->iface = cur1;
		  }
		else log_printf (LOG_WARNING,"router %s doesn't have an interface called %s\n",router->name,order[i]);
	 }
}

static void change_interface_aliases (router_t *router,char *alias[],size_t n)
{
   size_t i = 0;
   interface_t *tmp;
   char *newalias;

   for (tmp = router->iface; tmp != NULL && i < n; tmp = tmp->next)
	 if (!router->ignore_admin_down || tmp->status[IF_ADMIN_STATUS] == IF_STATUS_UP)
	   {
		  if ((newalias = (char *) mem_alloc ((strlen (alias[i]) + 1) * sizeof (char))) != NULL)
			{
			   if (tmp->alias != NULL) mem_free (tmp->alias);
			   strcpy (newalias,alias[i]);
			   tmp->alias = newalias;
			}
		  else log_printf (LOG_WARNING,
						   "couldn't allocate memory for router %s, interface %s override (%s): %s\n"
						   "reverting to old interface alias\n",
						   router->name,tmp->descr,alias[i],strerror (errno));
		  i++;
	   }
}

int main (int argc,char *argv[])
{
   options_t opts;
   config_t conf;
   router_t *router;
   const char *msg;
   int i,flags;
   struct timeval then,now;
   long snoozy;
   interface_t *tmp;

   /* parse command-line options */
   parse_options (&opts,argc,argv);

   flags = opts.logfile == NULL ? LOG_HAVE_COLORS : LOG_DEBUG_PREFIX_ONLY;

   /* initialize memory detection and logging subsystems */
   mem_open (NULL);
   if (log_open (opts.logfile,opts.loglevel,flags | LOG_DETECT_FLOODING) < 0)
	 {
		fprintf (stderr,"unable to initialize logging system: %s\n",strerror (errno));
		exit (EXIT_FAILURE);
	 }

   atexit (log_close);
   atexit (mem_close);

   /* parse configuration file */
   if (config_parse (&conf,opts.configfile) < 0)
	 exit (EXIT_FAILURE);

   /* print configuration and command-line options */
   print_settings (&opts,&conf);

   /* catch all the important signals */
   signal (SIGHUP,SIG_IGN);
   signal (SIGINT,catch_signals);
   signal (SIGTERM,catch_signals);
   signal (SIGUSR1,catch_signals);
   signal (SIGUSR2,SIG_IGN);
   signal (SIGTSTP,SIG_IGN);

   if (opts.daemon)
	 {
		/* fork and become a daemon */
		if (daemon (0,0) < 0)
		  {
			 log_printf (LOG_ERROR,"unable to daemonize myself: %s\n",strerror (errno));
			 exit (EXIT_FAILURE);
		  }

		/* now create the pidfile */
		if (create_pidfile (conf.pidfile) < 0)
		  {
			 log_printf (LOG_ERROR,"unable to create pid file %s: %s\n",conf.pidfile,strerror (errno));
			 exit (EXIT_FAILURE);
		  }
	 }

   /* change ownership of log file */
   if (opts.logfile != NULL && chown (opts.logfile,conf.uid,conf.gid) < 0)
	 {
		log_printf (LOG_ERROR,"unable to change ownership of log file %s to uid %u, gid %u: %s\n",opts.logfile,conf.uid,conf.gid,strerror (errno));
		exit (EXIT_FAILURE);
	 }

   /* change ownership of orca library path */
   if (chown (conf.database,conf.uid,conf.gid) < 0)
	 {
		log_printf (LOG_ERROR,"unable to change ownership of RRD database directory %s to uid %u, gid %u: %s\n",
					conf.database,
					conf.uid,
					conf.gid,
					strerror (errno));
		exit (EXIT_FAILURE);
	 }

   /* change ownership of orca htdocs dir */
   if (chown (conf.htdocs,conf.uid,conf.gid) < 0)
	 {
		log_printf (LOG_ERROR,"unable to change ownership of web page directory %s to uid %u, gid %u: %s\n",
					conf.htdocs,
					conf.uid,
					conf.gid,
					strerror (errno));
		exit (EXIT_FAILURE);
	 }

   /* set process gid */
   if (setregid (conf.gid,conf.gid) < 0)
	 {
		log_printf (LOG_ERROR,"unable to change process group id to %u\n",conf.gid);
		exit (EXIT_FAILURE);
	 }

   /* set process uid */
   if (setreuid (conf.uid,conf.uid) < 0)
	 {
		log_printf (LOG_ERROR,"unable to change process user id to %u\n",conf.uid);
		exit (EXIT_FAILURE);
	 }

   /* allocate memory for router structures */
   if ((router = (router_t *) mem_alloc (sizeof (router_t) * conf.n)) == NULL)
	 {
		log_printf (LOG_ERROR,"couldn't allocate memory for temporary storage: %s\n",strerror (errno));
		config_destroy (&conf);
		exit (EXIT_FAILURE);
	 }

   /* initialize those router structures */
   for (i = 0; i < conf.n; i++)
	 {
		int j;

		if (router_open (&router[i],
						 conf.router[i].hostname,
						 conf.router[i].service,
						 conf.router[i].community,
						 conf.router[i].timeout,
						 conf.router[i].retries) < 0 && errno)
		  {
			 log_printf (LOG_ERROR,"unable to query router %s: initialization failed: %s\n",conf.router[i].hostname,strerror (errno));
			 config_destroy (&conf);
			 for (j = 0; j < i; j++) router_close (&router[j]);
			 mem_free (router);
			 exit (EXIT_FAILURE);
		  }

		for (j = 0; j < conf.router[i].n_ignore; j++)
		  if (router_purge_interface (&router[i],conf.router[i].ignore[j]) < 0)
			log_printf (LOG_WARNING,"router %s does not have an interface called %s\n",router[i].name,conf.router[i].ignore[j]);

		if (router[i].iface == NULL)
		  {
			 log_printf (LOG_ERROR,"router %s seems to have no interfaces\n",conf.router[i].hostname);
			 for (j = 0; j <= i; j++) router_close (&router[j]);
			 config_destroy (&conf);
			 mem_free (router);
			 exit (EXIT_FAILURE);
		  }

		order_interfaces (&router[i],conf.router[i].order,conf.router[i].n_order);

		router[i].ignore_admin_down = conf.router[i].ignore_admin_down;

		/* retrieve SNMP information from routers */
		if (router_update (&router[i]) < 0)
		  {
			 interface_t *tmp;
			 log_printf (LOG_WARNING,"unable to query router %s: update failed: %s\n",router[i].name,strerror (errno));
			 for (tmp = router[i].iface; tmp != NULL; tmp = tmp->next) tmp->valid = 0;
		  }

		change_interface_aliases (&router[i],conf.router[i].alias,conf.router[i].n_alias);

		print_router (&router[i],i);
	 }

   /* create RRD databases for each router (if necessary) */
   for (i = 0; i < conf.n; i++)
	 if ((msg = db_create (conf.database,&router[i])) != NULL)
	   {
		  log_printf (LOG_ERROR,"unable to create RRD databases: %s\n",msg);
		  for (i = 0; i < conf.n; i++) router_close (&router[i]);
		  config_destroy (&conf);
		  mem_free (router);
		  exit (EXIT_FAILURE);
	   }

   /* main loop */
   while (!finished)
	 {
		gettimeofday (&then,NULL);

		if (rotated)
		  {
			 if (log_reset () < 0) break;
			 rotated = 0;
		  }

		log_printf (LOG_VERBOSE,"Retrieving SNMP information from routers\n");

		for (i = 0; i < conf.n; i++)
		  if (router_update (&router[i]) < 0)
			{
			   log_printf (LOG_WARNING,"unable to query router %s: updated failed: %s\n",router[i].name,strerror (errno));
			   for (tmp = router[i].iface; tmp != NULL; tmp = tmp->next) tmp->valid = 0;
			}

		log_printf (LOG_VERBOSE,"Updating RRD databases\n");

		for (i = 0; i < conf.n; i++)
		  if ((msg = db_update (conf.database,&router[i])) != NULL)
			log_printf (LOG_WARNING,"unable to update RRD databases for router %s: %s\n",router[i].name,msg);

		log_printf (LOG_VERBOSE,"Generating graphs\n");

		for (i = 0; i < conf.n; i++)
		  if ((msg = html_generate (router,conf.n,i,conf.database,conf.htdocs,conf.documentroot)) != NULL)
			log_printf (LOG_WARNING,"%s\n",msg);

		if ((msg = create_main_index (conf.htdocs,router,conf.n)) != NULL)
		  log_printf (LOG_WARNING,"While trying to create %s/index.html: %s\n",conf.htdocs,msg);

		gettimeofday (&now,NULL);

		snoozy = 300 - (now.tv_sec - then.tv_sec);
		if (snoozy < 0) snoozy = 0;

		log_printf (LOG_VERBOSE,"Sleeping for %ld minutes",snoozy / 60);
		if (snoozy % 60) log_printf (LOG_VERBOSE,", %ld seconds",snoozy % 60);
		log_printf (LOG_VERBOSE,"\n");

		sleep (snoozy);
	 }

   for (i = 0; i < conf.n; i++) router_close (&router[i]);
   mem_free (router);
   config_destroy (&conf);

   /* this means the log rotation failed. no point in complaining since everything is screwed anyway */
   if (!finished) exit (EXIT_FAILURE);

   exit (EXIT_SUCCESS);
}

