/*
 * schematic.c
 *
 *
 * Copyright (C) 2000  Richard Hult
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include "schematic.h"
#include "node-store.h"
#include "load-schematic.h"
#include "settings.h"
#include "sim-settings.h"
#include "simulation.h"

struct _SchematicPriv {
	char           *title;
	char           *filename;
	char           *netlist_filename;

	/*
	 * Data for various dialogs.
	 */
	gpointer        settings;
	gpointer        sim_settings;
	gpointer        simulation;

	GList          *items;

	NodeStore      *store;
	GHashTable     *symbols;
	GHashTable     *refdes_values;
//	NetList        *netlist;

	double          zoom;

	/* FIXME: Make an object. */ 
	GList *log;
};

typedef enum {
	REFDATA_SINGLE,
	REFDATA_RANGE,
	REFDATA_MIN,
	REFDATA_MAX
} RefDataType;

typedef struct {
	RefDataType type;
	union {
		int nr;
		struct { int min; int max; } range;
	} u;
} RefData; 

enum {
	TITLE_CHANGED,
	ITEM_DATA_ADDED,
	LOG_UPDATED,
	DOT_ADDED,
	DOT_REMOVED,
	LAST_SCHEMATIC_DESTROYED,
	LAST_SIGNAL
};

static void schematic_init		(Schematic *schematic);
static void schematic_class_init	(SchematicClass	*klass);
static void schematic_shutdown          (GtkObject *object);
static void schematic_destroy           (GtkObject *object);

static void item_data_destroy_callback (GtkObject *data, Schematic *sm);
static int  schematic_get_lowest_available_refdes (Schematic *schematic, char *prefix);
static void schematic_set_lowest_available_refdes (Schematic *schematic, char *prefix, int num);

static GtkObjectClass *parent_class = NULL;
static guint schematic_signals[LAST_SIGNAL] = { 0 };

static GList *schematic_list = NULL;
static int schematic_count_ = 0;

GtkType
schematic_get_type (void)
{
	static GtkType schematic_type = 0;
	
	if (!schematic_type) {
		static const GtkTypeInfo schematic_info = {
			"Schematic",
			sizeof (Schematic),
			sizeof (SchematicClass),
			(GtkClassInitFunc) schematic_class_init,
			(GtkObjectInitFunc) schematic_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};
		
		schematic_type = gtk_type_unique (gtk_object_get_type (), &schematic_info);
	}
	
	return schematic_type;
}

static void
schematic_class_init (SchematicClass *klass)
{
	GtkObjectClass *object_class;
	
	object_class = (GtkObjectClass*) klass;
	
	parent_class = gtk_type_class (gtk_object_get_type ());

	schematic_signals[TITLE_CHANGED] = 
		gtk_signal_new ("title_changed",
				GTK_RUN_FIRST,
				object_class->type,
				0,
				gtk_marshal_NONE__STRING,
				GTK_TYPE_NONE, 1, GTK_TYPE_STRING);

	schematic_signals[LAST_SCHEMATIC_DESTROYED] = 
		gtk_signal_new ("last_schematic_destroyed",
				GTK_RUN_FIRST,
				object_class->type,
				0,
				gtk_marshal_NONE__NONE,
				GTK_TYPE_NONE, 0);

	schematic_signals[ITEM_DATA_ADDED] = 
		gtk_signal_new ("item_data_added",
				GTK_RUN_FIRST,
				object_class->type,
				0,
				gtk_marshal_NONE__POINTER,
				GTK_TYPE_NONE, 1, GTK_TYPE_POINTER);

	schematic_signals[DOT_ADDED] = 
		gtk_signal_new ("dot_added",
				GTK_RUN_FIRST,
				object_class->type,
				0,
				gtk_marshal_NONE__POINTER,
				GTK_TYPE_NONE, 1, GTK_TYPE_POINTER);

	schematic_signals[DOT_REMOVED] = 
		gtk_signal_new ("dot_removed",
				GTK_RUN_FIRST,
				object_class->type,
				0,
				gtk_marshal_NONE__POINTER,
				GTK_TYPE_NONE, 1, GTK_TYPE_POINTER);

	schematic_signals[LOG_UPDATED] = 
		gtk_signal_new ("log_updated",
				GTK_RUN_FIRST,
				object_class->type,
				0,
				gtk_marshal_NONE__NONE,
				GTK_TYPE_NONE, 0);
	
	gtk_object_class_add_signals (object_class, schematic_signals, LAST_SIGNAL);
	
	object_class->shutdown = schematic_shutdown;
	object_class->destroy = schematic_destroy;
}

static void
dot_added_callback (NodeStore *store, SheetPos *pos, Schematic *schematic)
{
	g_return_if_fail (schematic != NULL);
	g_return_if_fail (IS_SCHEMATIC (schematic));

	gtk_signal_emit (GTK_OBJECT (schematic), schematic_signals[DOT_ADDED], pos);
}

static void
dot_removed_callback (NodeStore *store, SheetPos *pos, Schematic *schematic)
{
	g_return_if_fail (schematic != NULL);
	g_return_if_fail (IS_SCHEMATIC (schematic));

	gtk_signal_emit (GTK_OBJECT (schematic), schematic_signals[DOT_REMOVED], pos);
}

static void
schematic_init (Schematic *schematic)
{
	SchematicPriv *priv;

	priv = schematic->priv = g_new0 (SchematicPriv, 1);

	priv->symbols = g_hash_table_new (g_str_hash, g_str_equal); // FIXME: use own str_equal (lib::sym)
	priv->refdes_values = g_hash_table_new (g_str_hash, g_str_equal);
	priv->store = node_store_new ();

	gtk_signal_connect_while_alive (
		GTK_OBJECT (priv->store), 
		"dot_added",
		dot_added_callback,
		schematic,
		GTK_OBJECT (schematic));

	gtk_signal_connect_while_alive (
		GTK_OBJECT (priv->store), 
		"dot_removed", 
		dot_removed_callback,
		schematic,
		GTK_OBJECT (schematic));

	priv->sim_settings = sim_settings_new (schematic);
	priv->settings = settings_new (schematic);
	priv->simulation = simulation_new (schematic);

	priv->filename = NULL;
	priv->netlist_filename = NULL;
}

Schematic *
schematic_new (void)
{
	Schematic *schematic;

	schematic = gtk_type_new (schematic_get_type ());

	schematic_count_++;
	schematic_list = g_list_prepend (schematic_list, schematic);

	return schematic;
}

static void
schematic_shutdown (GtkObject *object)
{
	Schematic *schematic = SCHEMATIC (object);

	schematic_count_--;
	schematic_list = g_list_remove (schematic_list, schematic);

	if (schematic_count_ == 0)
		gtk_signal_emit_by_name (GTK_OBJECT (schematic), "last_schematic_destroyed", NULL);

	GTK_OBJECT_CLASS (parent_class)->shutdown (object);
}

static void
schematic_destroy (GtkObject *object)
{
	Schematic *sm;

	sm = SCHEMATIC (object);
	g_free (sm->priv->simulation);
	g_hash_table_destroy (sm->priv->symbols);
	g_hash_table_destroy (sm->priv->refdes_values);
	
	GTK_OBJECT_CLASS (parent_class)->destroy (object);
}

/*
 *
 * Get/set functions.
 *
 */

char *
schematic_get_title (Schematic *schematic)
{
	g_return_val_if_fail (schematic != NULL, NULL);
	g_return_val_if_fail (IS_SCHEMATIC (schematic), NULL);

	return schematic->priv->title;
}

void
schematic_set_title (Schematic *schematic, char *title)
{
	g_return_if_fail (schematic != NULL);
	g_return_if_fail (IS_SCHEMATIC (schematic));

	g_free (schematic->priv->title);
	schematic->priv->title = g_strdup (title);

	gtk_signal_emit_by_name (GTK_OBJECT (schematic), "title_changed", schematic->priv->title);
}

char *
schematic_get_filename (Schematic *schematic)
{
	g_return_val_if_fail (schematic != NULL, NULL);
	g_return_val_if_fail (IS_SCHEMATIC (schematic), NULL);

	return schematic->priv->filename;
}

void
schematic_set_filename (Schematic *schematic, char *filename)
{
	g_return_if_fail (schematic != NULL);
	g_return_if_fail (IS_SCHEMATIC (schematic));

	g_free (schematic->priv->filename);
	schematic->priv->filename = g_strdup (filename);

//	gtk_signal_emit_by_name (GTK_OBJECT (schematic), "filename_changed", schematic->priv->filename);
}

char *
schematic_get_netlist_filename (Schematic *schematic)
{
	g_return_val_if_fail (schematic != NULL, NULL);
	g_return_val_if_fail (IS_SCHEMATIC (schematic), NULL);

	return schematic->priv->netlist_filename;
}

void
schematic_set_netlist_filename (Schematic *schematic, char *filename)
{
	g_return_if_fail (schematic != NULL);
	g_return_if_fail (IS_SCHEMATIC (schematic));

	g_free (schematic->priv->netlist_filename);
	schematic->priv->netlist_filename = g_strdup (filename);

//	gtk_signal_emit_by_name (GTK_OBJECT (schematic), "netlist_filename_changed", schematic->priv->filename);
}

double
schematic_get_zoom (Schematic *schematic)
{
	g_return_val_if_fail (schematic != NULL, 1.0);
	g_return_val_if_fail (IS_SCHEMATIC (schematic), 1.0);

	return schematic->priv->zoom;
}

/*
 * FIXME: different zoom level on views...
 */
void
schematic_set_zoom (Schematic *schematic, double zoom)
{
	g_return_if_fail (schematic != NULL);
	g_return_if_fail (IS_SCHEMATIC (schematic));

	schematic->priv->zoom = zoom;

//	gtk_signal_emit_by_name (GTK_OBJECT (schematic), "zoom_changed", schematic->priv->zoom);
}

NodeStore *
schematic_get_store (Schematic *schematic)
{
	g_return_val_if_fail (schematic != NULL, NULL);
	g_return_val_if_fail (IS_SCHEMATIC (schematic), NULL);

	return schematic->priv->store;
}

gpointer 
schematic_get_settings (Schematic *schematic)
{
	g_return_val_if_fail (schematic != NULL, NULL);
	g_return_val_if_fail (IS_SCHEMATIC (schematic), NULL);

	return schematic->priv->settings;
}

gpointer
schematic_get_sim_settings (Schematic *schematic)
{
	g_return_val_if_fail (schematic != NULL, NULL);
	g_return_val_if_fail (IS_SCHEMATIC (schematic), NULL);

	return schematic->priv->sim_settings;
}

gpointer
schematic_get_simulation (Schematic *schematic)
{
	g_return_val_if_fail (schematic != NULL, NULL);
	g_return_val_if_fail (IS_SCHEMATIC (schematic), NULL);

	return schematic->priv->simulation;
}

/*
 * Actually prepends but that doesn't matter here.
 * This is an internal storage only.
 */
void
schematic_log_append (Schematic *schematic, const char *message)
{
	g_return_if_fail (schematic != NULL);
	g_return_if_fail (IS_SCHEMATIC (schematic));

	schematic->priv->log = g_list_prepend (schematic->priv->log, g_strdup (message));
}

void
schematic_log_show (Schematic *schematic)
{
	g_return_if_fail (schematic != NULL);
	g_return_if_fail (IS_SCHEMATIC (schematic));

	gtk_signal_emit (GTK_OBJECT (schematic), schematic_signals[LOG_UPDATED]);
}

void 
schematic_log_clear (Schematic *schematic) 
{
	GList *log;

	g_return_if_fail (schematic != NULL);
	g_return_if_fail (IS_SCHEMATIC (schematic));

	for (log = schematic->priv->log; log; log = log->next)
 		g_free (log->data);

	g_list_free (schematic->priv->log);
	schematic->priv->log = NULL;
} 

GList *
schematic_get_log_text (Schematic *schematic)
{
	g_return_val_if_fail (schematic != NULL, NULL);
	g_return_val_if_fail (IS_SCHEMATIC (schematic), NULL);

	return schematic->priv->log;
}

int
schematic_count (void)
{
	return schematic_count_;
}

Schematic *
schematic_read (char *fname)
{
	Schematic *new_sm;
	int ret;

	g_return_val_if_fail (fname != NULL, NULL);

	if (!g_file_exists (fname)) {
		new_sm = schematic_new ();
		schematic_set_filename (new_sm, fname);
		return new_sm;
	}

	new_sm = schematic_new ();
	ret = schematic_parse_xml_file (new_sm, fname);

	if (ret) {
		gtk_object_destroy (GTK_OBJECT (new_sm)); // was schematic_destroy () ??
		new_sm = NULL;
	}

	return new_sm;
}

void
schematic_add_item (Schematic *sm, ItemData *data)
{
	NodeStore *store;
	char *prefix, *refdes;
	int num;

	g_return_if_fail (sm != NULL);
	g_return_if_fail (IS_SCHEMATIC (sm));
	g_return_if_fail (data != NULL);
	g_return_if_fail (IS_ITEM_DATA (data));

	store = sm->priv->store;
	gtk_object_set (GTK_OBJECT (data), "store", store, NULL);
	item_data_register (data);

	/*
	 * Some items need a reference designator. Find a good one.
	 */
	prefix = item_data_get_refdes_prefix (data);
	if (prefix != NULL) {
		num = schematic_get_lowest_available_refdes (sm, prefix);
		refdes = g_strdup_printf ("%s%d", prefix, num);
		item_data_set_property (data, "refdes", refdes);

		schematic_set_lowest_available_refdes (sm, prefix, num + 1);

		g_free (prefix);
		g_free (refdes);
	}

	sm->priv->items = g_list_prepend (sm->priv->items, data);
	gtk_signal_connect_while_alive (GTK_OBJECT (data), "destroy", item_data_destroy_callback, sm, GTK_OBJECT (sm));

	gtk_signal_emit (GTK_OBJECT (sm), schematic_signals[ITEM_DATA_ADDED], data);
}

void
schematic_parts_foreach (Schematic *schematic, ForeachItemDataFunc func, gpointer user_data)
{
	GList *list;

	g_return_if_fail (schematic != NULL);
	g_return_if_fail (IS_SCHEMATIC (schematic));
	
	if (func == NULL)
		return;

	for (list = node_store_get_parts (schematic->priv->store); list; list = list->next) {
		func (list->data, user_data);
	}
}
	
void
schematic_wires_foreach (Schematic *schematic, ForeachItemDataFunc func, gpointer user_data)
{
	GList *list;

	g_return_if_fail (schematic != NULL);
	g_return_if_fail (IS_SCHEMATIC (schematic));
	
	if (func == NULL)
		return;

	for (list = node_store_get_wires (schematic->priv->store); list; list = list->next) {
		func (list->data, user_data);
	}
}

void
schematic_items_foreach (Schematic *schematic, ForeachItemDataFunc func, gpointer user_data)
{
	GList *list;

	g_return_if_fail (schematic != NULL);
	g_return_if_fail (IS_SCHEMATIC (schematic));
	
	if (func == NULL)
		return;

	for (list = schematic->priv->items; list; list = list->next) {
		func (list->data, user_data);
	}
}

GList *
schematic_get_items (Schematic *sm)
{
	g_return_val_if_fail (sm != NULL, NULL);
	g_return_val_if_fail (IS_SCHEMATIC (sm), NULL);

	return sm->priv->items;
}

static void
item_data_destroy_callback (GtkObject *data, Schematic *sm)
{
	if (sm->priv) {
		sm->priv->items = g_list_remove (sm->priv->items, data);
	}
}

static int
schematic_get_lowest_available_refdes (Schematic *schematic, char *prefix)
{
	gpointer key, value;

	g_return_val_if_fail (schematic != NULL, -1);
	g_return_val_if_fail (IS_SCHEMATIC (schematic), -1);
	g_return_val_if_fail (prefix != NULL, -1);

	if (g_hash_table_lookup_extended (schematic->priv->refdes_values, prefix, &key, &value)) {
		return GPOINTER_TO_INT (value);
	} else {
		return 1;
	}
}

static void
schematic_set_lowest_available_refdes (Schematic *schematic, char *prefix, int num)
{
	gpointer key, value;

	g_return_if_fail (schematic != NULL);
	g_return_if_fail (IS_SCHEMATIC (schematic));
	g_return_if_fail (prefix != NULL);

	/*
	 * If there already is a key, use it, otherwise copy the prefix and use as key.
	 */
	if (!g_hash_table_lookup_extended (schematic->priv->refdes_values, prefix, &key, &value))
		key = g_strdup (prefix);
	
	g_hash_table_insert (schematic->priv->refdes_values, key, GINT_TO_POINTER (num));	
}

/*
static int
schematic_get_lowest_available_refdes2 (Schematic *schematic, char *prefix)
{
	RefData *data;

	g_return_val_if_fail (schematic != NULL, -1);
	g_return_val_if_fail (IS_SCHEMATIC (schematic), -1);
	g_return_val_if_fail (prefix != NULL, -1);

	data = g_hash_table_lookup(schematic->priv->refdes_values, prefix);
	if (data == NULL) {
		return 1;
	} else {
		int result;

		switch (data->type) {
		REF_DATA_SINGLE:
			break;
		REF_DATA_RANGE:
			break;
		REF_DATA_MIN:
			break;
		REF_DATA_MAX:
			break;
		default:
			result = 1;
		}
		return result;
	}

}
*/

