/*
  Copyright (C) 2003 <ryu@gpul.org>

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


 * Copyright (C) 2004 Mekensleep
 *
 *	Mekensleep
 *	24 rue vieille du temple
 *	75004 Paris
 *       licensing@mekensleep.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors:
 *  Cedric PINSON <cpinson@freesheep.org>
 *  Loic Dachary <loic@gnu.org>
 *  Igor Kravtchenko <igor@obraz.net>
 *
 */

#include <cal3d/model.h>

#include <osg/NodeCallback>
#include <osg/Material>
#include <osg/Texture2D>
#include <osg/Timer>

#include <osg/ShapeDrawable>

#include <osgCal/Model>
#include <osgCal/SubMeshSoftware>
#include <osgCal/SubMeshHardware>

#include "config.h"

#define OSGCAL_MAX_BONES_PER_MESH 26
static char VERTEX_PROGRAM_STR[]= 
"!!ARBvp1.0\n"\
"PARAM constant = { 1, 3, 0, 0 };\n"\
"TEMP R0, R1, R2, R3, R4, R5;\n"\
"ADDRESS A0;\n"\
"ATTRIB texCoord = vertex.attrib[8];\n"\
"ATTRIB normal = vertex.attrib[2];\n"\
"ATTRIB index = vertex.attrib[3];\n"\
"ATTRIB weight = vertex.attrib[1];\n"\
"ATTRIB position = vertex.attrib[0];\n"\
"# PARAM worldViewProjMatrix[4] = { state.matrix.mvp };\n"\
"PARAM mViewInverse[4] = { state.matrix.modelview.invtrans };\n"\
"PARAM mView[4] = { state.matrix.modelview };\n"\
"PARAM mProj[4] = { state.matrix.projection };\n"\
"PARAM mTex[4] = { state.matrix.texture[0] };\n"\
"PARAM diffuse = state.material.diffuse;\n"\
"PARAM ambient = state.material.ambient;\n"\
"# its a position light;\n"\
"PARAM lightDir = state.light[0].position;\n" \
"PARAM matrix[78] = { program.local[0..77] };\n"\
"\n"\
/*
"MOV result.texcoord[0].xy, texCoord.xyxx;	\n"\
*/
"DPH result.texcoord[0].x, texCoord.xyzw, mTex[0]; \n"\
"DPH result.texcoord[0].y, texCoord.xyzw, mTex[1]; \n"\
"DPH result.texcoord[0].z, texCoord.xyzw, mTex[2]; \n"\
"DPH result.texcoord[0].w, texCoord.xyzw, mTex[3]; \n"\
"\n"\
"MUL R4, index, constant.y;	\n"\
"\n"\
"ARL A0.x, R4.y;\n"\
"DP3 R0.x, matrix[A0.x].xyzx, normal.xyzx;\n"\
"DP3 R0.y, matrix[A0.x + 1].xyzx, normal.xyzx;\n"\
"DP3 R0.z, matrix[A0.x + 2].xyzx, normal.xyzx;\n"\
"MUL R1.yzw, R0.xxyz, weight.y;\n"\
"\n"\
"ARL A0.x, R4.x;\n"\
"DP3 R0.x, matrix[A0.x].xyzx, normal.xyzx;\n"\
"DP3 R0.y, matrix[A0.x + 1].xyzx, normal.xyzx;\n"\
"DP3 R0.z, matrix[A0.x + 2].xyzx, normal.xyzx;\n"\
"MAD R1.yzw, R0.xxyz, weight.x, R1.yyzw;\n"\
"\n"\
"# compute normal with m and weight 3 add the result to R1\n"\
"#ARL A0.x, R4.z;\n"\
"#DP3 R0.x, matrix[A0.x], normal;\n"\
"#DP3 R0.y, matrix[A0.x + 1], normal;\n"\
"#DP3 R0.z, matrix[A0.x + 2], normal;\n"\
"#MAD R1, R0, weight.z, R1;\n"\
"#\n"\
"# compute normal with m and weight 4 add the result to R1\n"\
"#ARL A0.x, R4.w;\n"\
"#DP3 R0.x, matrix[A0.x], normal;\n"\
"#DP3 R0.y, matrix[A0.x + 1], normal;\n"\
"#DP3 R0.z, matrix[A0.x + 2], normal;\n"\
"#MAD R1, R0, weight.w, R1;\n"\
"#\n"\
"DP3 R0.x, R1.yzwy, R1.yzwy;\n"\
"RSQ R0.x, R0.x;\n"\
"MUL R0.xyz, R0.x, R1.yzwy;\n"\

"# rotate the normal in modelview result in R2\n"\
"DP3 R5.x, R0, mViewInverse[0];\n"\
"DP3 R5.y, R0, mViewInverse[1];\n"\
"DP3 R5.z, R0, mViewInverse[2];\n"\
"#MOV R5, R0 ;\n"\

"# MOV R0, R5;\n"\
"# MUL R0, R0, constant.z;\n"\

"ARL A0.x, R4.w;\n"\
"DPH R0.x, position.xyzx, matrix[A0.x];\n"\
"DPH R0.y, position.xyzx, matrix[A0.x + 1];\n"\
"DPH R0.z, position.xyzx, matrix[A0.x + 2];\n"\
"\n"\
"ARL A0.x, R4.z;\n"\
"DPH R3.x, position.xyzx, matrix[A0.x];\n"\
"DPH R3.y, position.xyzx, matrix[A0.x + 1];\n"\
"DPH R3.z, position.xyzx, matrix[A0.x + 2];\n"\
"\n"\
"ARL A0.x, R4.y;\n"\
"DPH R1.y, position.xyzx, matrix[A0.x];\n"\
"DPH R1.z, position.xyzx, matrix[A0.x + 1];\n"\
"DPH R1.w, position.xyzx, matrix[A0.x + 2];\n"\
"MUL R2.xyz, R1.yzwy, weight.y;\n"\
"\n"\
"ARL A0.x, R4.x;\n"\
"DPH R1.x, position.xyzx, matrix[A0.x];\n"\
"DPH R1.y, position.xyzx, matrix[A0.x + 1];\n"\
"DPH R1.z, position.xyzx, matrix[A0.x + 2];\n"\
"\n"\
"MAD R1.xyz, R1.xyzx, weight.x, R2.xyzx;\n"\
"MAD R1.xyz, R3.xyzx, weight.z, R1.xyzx;\n"\
"MAD R0.xyz, R0.xyzx, weight.w, R1.xyzx;\n"\
"\n"\
"DPH R3.x, R0.xyzx, mView[0];\n"\
"DPH R3.y, R0.xyzx, mView[1];\n"\
"DPH R3.z, R0.xyzx, mView[2];\n"\
"DPH R3.w, R0.xyzx, mView[3];\n"\

"SUB R2, lightDir,R3 ;\n" \
"DPH result.position.x, R3.xyzx, mProj[0];\n"\
"DPH result.position.y, R3.xyzx, mProj[1];\n"\
"DPH result.position.z, R3.xyzx, mProj[2];\n"\
"DPH result.position.w, R3.xyzx, mProj[3];\n"\

"MOV R0,R5;\n"\
"MOV R3,R2;\n"\

"# DP3 R1.x, lightDir.xyzx, lightDir.xyzx;\n"\
"DP3 R1.x, R3.xyzx, R3.xyzx;\n"\
"RSQ R1.x, R1.x;\n"\
"# MUL R2.xyz, R1.x, lightDir.xyzx;\n"\
"MUL R2.xyz, R1.x, R3.xyzx;\n"\
"DP3 R0.x, R0.xyzx, R2.xyzx;\n"\
"MAX R0.x, R0.x, constant.z;\n"\
"#ADD R0, R0.x, ambient;\n"\
"MAD result.color.front.primary, R0.x, diffuse, ambient;\n"\
"\n"\

"END\n";

using namespace osgCal;

class CalUpdateCallback: public osg::NodeCallback 
{
public:

  CalUpdateCallback() : previous(timer.tick()) {}

  virtual void operator()(osg::Node* node, osg::NodeVisitor* nv)
  {
    Model* model = dynamic_cast<Model*>(node);
    osg::Timer_t current = timer.tick();
    float deltaTime = (float)timer.delta_s(previous, current);

	CalModel * pCalModel = model->getCalModel();
	pCalModel->getAbstractMixer()->updateAnimation(deltaTime);
	pCalModel->getAbstractMixer()->updateSkeleton();
	
	model->update();
    
    previous = current;
    
	traverse(node,nv);
    node->dirtyBound();
  }

private:

  osg::Timer timer;
  osg::Timer_t previous;

};

Model::Model() {
  setUseVertexProgram(false);
  setUseColorOrTexture(true);
  for (int i=0;i<6;i++)
    mVbo[i]=0;
}

Model::Model(const Model& model, const osg::CopyOp& copyop) :
  osg::Geode(model, copyop),
  _useVertexProgram(model._useVertexProgram),
  _useColorOrTexture(model._useColorOrTexture),
  _coreModel(model._coreModel)
{
  if(_coreModel.valid())
    _calModel.create(getCalCoreModel());
}

Model::~Model()
{
  if(_coreModel.valid())
    _coreModel->garbageCollect();
  _calModel.destroy();

  for (std::vector<int>::iterator i=_activeMeshes.begin();i!=_activeMeshes.end();i++)
    _coreModel->SubUsingMeshId(*i);

  osg::Drawable::Extensions* drExt = osg::Drawable::getExtensions(0,true);
  if (mVbo[0])
    drExt->glDeleteBuffers(6, mVbo);
}

bool Model::setCoreModel(CoreModel* coreModel) {
  _coreModel = coreModel;
  return _calModel.create(_coreModel->getCalCoreModel());
}

bool Model::setUseVertexProgram(bool useVertexProgram)
{
  if(useVertexProgram) {
    osg::VertexProgram::Extensions* vpExt = osg::VertexProgram::getExtensions(0,true);
    if(!vpExt || !vpExt->isVertexProgramSupported())
      useVertexProgram = false;
  }

  return _useVertexProgram = useVertexProgram;
}

bool Model::create(void) 
{
  if(_activeMeshes.empty()) {
    // By default all attached meshes are active.
    for(int coreMeshId = 0; coreMeshId < getCalCoreModel()->getCoreMeshCount(); coreMeshId++) {
      if(getCalModel()->getMesh(coreMeshId)) {
	invertUVs(coreMeshId);
	_activeMeshes.push_back(coreMeshId);
	_coreModel->AddUsingMeshId(coreMeshId);
      }
    }
    // All mesh are used for collisions/bounding boxes
    _collisionMeshes = _activeMeshes;
  }

  _calModel.update(0);

  setUpdateCallback(new CalUpdateCallback());

#if 0
  // TRICK for poker3d
  // test try a better way to do it
  // the bbox is used to make center of all element at the same location
  // then during sort of elements for transparency all elements are at the same
  // position from camera and are always displayed in the same order
  _bbox.set(osg::Vec3(-45,0,-200)*0.75,osg::Vec3(45,192,-50)*0.75);
#endif

  if(getUseVertexProgram())
    return createHardware();
  else
    return createSoftware();

  return true;
}

static int coreMeshId2MeshId(CalModel* calModel, int coreMeshId) {
  CalMesh* mesh = calModel->getMesh(coreMeshId);
  if(mesh == 0) {
    CalError::setLastError(CalError::NULL_BUFFER, __FILE__, __LINE__);
    return -1;
  }
  int meshId = -1;
  for(std::vector<CalMesh *>::iterator meshIt = calModel->getVectorMesh().begin();
      meshIt != calModel->getVectorMesh().end();
      meshIt++) {
    meshId++;
    if(*meshIt == mesh)
      break;
  }

  return meshId;
}

bool Model::createHardware(void) 
{
  float *tmpVertexBuffer = new float[OSGCAL_MAX_VERTEX_PER_MODEL*3];
  float *tmpWeightBuffer = new float[OSGCAL_MAX_VERTEX_PER_MODEL*4];
  float *tmpMatrixIndexBuffer = new float[OSGCAL_MAX_VERTEX_PER_MODEL*4];
  float *tmpNormalBuffer = new float[OSGCAL_MAX_VERTEX_PER_MODEL*3];
  float *tmpTexCoordBuffer = new float[OSGCAL_MAX_VERTEX_PER_MODEL*2];
  CalIndex *tmpIndexBuffer = new CalIndex[OSGCAL_MAX_VERTEX_PER_MODEL*3];
    
  CalCoreModel* calCoreModel = getCalCoreModel();
  mHardwareModel.create(calCoreModel);

  mHardwareModel.setVertexBuffer((char*)tmpVertexBuffer,3*sizeof(float));
  mHardwareModel.setNormalBuffer((char*)tmpNormalBuffer,3*sizeof(float));
  mHardwareModel.setWeightBuffer((char*)tmpWeightBuffer,4*sizeof(float));
  mHardwareModel.setMatrixIndexBuffer((char*)tmpMatrixIndexBuffer,4*sizeof(float));
  mHardwareModel.setTextureCoordNum(1);
  mHardwareModel.setTextureCoordBuffer(0,(char*)tmpTexCoordBuffer,2*sizeof(float));
  mHardwareModel.setIndexBuffer(tmpIndexBuffer);
  mHardwareModel.setCoreMeshIds(_activeMeshes);
  mHardwareModel.load( 0, 0, OSGCAL_MAX_BONES_PER_MESH);

  // the index index in pIndexBuffer are relative to the begining of the hardware mesh,
  // we make them relative to the begining of the buffer.

  int hardwareMeshId;
  for(hardwareMeshId = 0; hardwareMeshId < mHardwareModel.getHardwareMeshCount(); hardwareMeshId++) {

    mHardwareModel.selectHardwareMesh(hardwareMeshId);

    int faceId;
    for(faceId = 0; faceId < mHardwareModel.getFaceCount(); faceId++) {
      tmpIndexBuffer[faceId*3+0+ mHardwareModel.getStartIndex()]+=mHardwareModel.getBaseVertexIndex();
      tmpIndexBuffer[faceId*3+1+ mHardwareModel.getStartIndex()]+=mHardwareModel.getBaseVertexIndex();
      tmpIndexBuffer[faceId*3+2+ mHardwareModel.getStartIndex()]+=mHardwareModel.getBaseVertexIndex();
    }
  }

  int nvtx=mHardwareModel.getTotalVertexCount();


  unsigned int* useVbo=0;
  osg::Drawable::Extensions* drExt = osg::Drawable::getExtensions(0,true);

  drExt->glGenBuffers(6, mVbo);
  drExt->glBindBuffer(GL_ARRAY_BUFFER_ARB, mVbo[0]);
  drExt->glBufferData(GL_ARRAY_BUFFER_ARB, 
		      nvtx*3*sizeof(float),
		      (const void*)tmpVertexBuffer,
		      GL_STATIC_DRAW_ARB);

  drExt->glBindBuffer(GL_ARRAY_BUFFER_ARB, mVbo[1]);
  drExt->glBufferData(GL_ARRAY_BUFFER_ARB, 
		      nvtx*4*sizeof(float),
		      (const void*)tmpWeightBuffer,
		      GL_STATIC_DRAW_ARB);

  drExt->glBindBuffer(GL_ARRAY_BUFFER_ARB, mVbo[2]);
  drExt->glBufferData(GL_ARRAY_BUFFER_ARB,
		      nvtx*3*sizeof(float),
		      (const void*)tmpNormalBuffer,
		      GL_STATIC_DRAW_ARB);
  
  drExt->glBindBuffer(GL_ARRAY_BUFFER_ARB, mVbo[3]);
  drExt->glBufferData(GL_ARRAY_BUFFER_ARB, 
		      nvtx*4*sizeof(float),
		      (const void*)tmpMatrixIndexBuffer,
		      GL_STATIC_DRAW_ARB);

  drExt->glBindBuffer(GL_ARRAY_BUFFER_ARB, mVbo[4]);
  drExt->glBufferData(GL_ARRAY_BUFFER_ARB, 
		      nvtx*2*sizeof(float),
		      (const void*)tmpTexCoordBuffer, 
		      GL_STATIC_DRAW_ARB);

  drExt->glBindBuffer(GL_ELEMENT_ARRAY_BUFFER_ARB, mVbo[5]);
  drExt->glBufferData(GL_ELEMENT_ARRAY_BUFFER_ARB, 
		      mHardwareModel.getTotalFaceCount()*3*sizeof(CalIndex),
		      (const void*)tmpIndexBuffer, 
		      GL_STATIC_DRAW_ARB);

  drExt->glBindBuffer(GL_ELEMENT_ARRAY_BUFFER_ARB,0);
  drExt->glBindBuffer(GL_ARRAY_BUFFER_ARB,0);
  useVbo=mVbo;

//   static osg::ref_ptr<osg::VertexProgram> vp=0;
//   if (!mVP.get()) {
    _vp = new osg::VertexProgram;
    _vp->setVertexProgram(VERTEX_PROGRAM_STR);
//   }
  osg::ref_ptr<osg::Vec3Array> vertices = new osg::Vec3Array(nvtx,(osg::Vec3*)tmpVertexBuffer); //,
  osg::ref_ptr<osg::Vec3Array> normals = new osg::Vec3Array(nvtx,(osg::Vec3*)tmpNormalBuffer); //,
  osg::ref_ptr<osg::Vec4Array> weights = new osg::Vec4Array(nvtx,(osg::Vec4*)tmpWeightBuffer); //,
  osg::ref_ptr<osg::Vec4Array> matrixs = new osg::Vec4Array(nvtx,(osg::Vec4*)tmpMatrixIndexBuffer); //,
  osg::ref_ptr<osg::Vec2Array> uvs = new osg::Vec2Array(nvtx,(osg::Vec2*)tmpTexCoordBuffer); //,


  // process hardware meshes
  for(hardwareMeshId = 0; hardwareMeshId < mHardwareModel.getHardwareMeshCount(); hardwareMeshId++) {
    mHardwareModel.selectHardwareMesh(hardwareMeshId);

    const CalHardwareModel::CalHardwareMesh& hardwareMesh = mHardwareModel.getVectorHardwareMesh()[hardwareMeshId];
    int coreMeshId = hardwareMesh.meshId; // incorrect naming in hardwareMesh, the meshId is really a coreMeshId
    bool supportsPrimitiveFunctor = isCollisionMesh(coreMeshId);
    bool invisible = isInvisibleMesh(coreMeshId);
    if(!invisible && !supportsPrimitiveFunctor) {
      SubMeshHardware* submesh = new SubMeshHardware(&_calModel, hardwareMeshId);
      CalCoreMesh* coreMesh = calCoreModel->getCoreMesh(coreMeshId);

      submesh->setName(coreMesh->getName());
      int indexInBuffer=mHardwareModel.getStartIndex();
      submesh->InitHardwareMesh(vertices,
				weights,
				normals,
				matrixs,
				uvs,
				//                                tmpIndexBuffer + mHardwareModel.getStartIndex(),
				&tmpIndexBuffer[indexInBuffer],
				mHardwareModel.getFaceCount()*3,
				_vp.get(),
				&mHardwareModel,
				useVbo,
				indexInBuffer*sizeof(CalIndex));
      setupMaterial(submesh, _calModel.getMesh(coreMeshId)->getSubmesh(hardwareMesh.submeshId));
      _coreMeshId2Drawables[coreMeshId].push_back(submesh);
      addDrawable(submesh);
      submesh->_staticbbox=_bbox;
      osg::Vec3 min=_bbox._min;
      osg::Vec3 max=_bbox._max;
      min[2]-=hardwareMeshId/(1.0*mHardwareModel.getHardwareMeshCount());
      max[2]-=hardwareMeshId/(1.0*mHardwareModel.getHardwareMeshCount());
      submesh->_staticbbox=osg::BoundingBox(min,max);

#if 0 // display bbox
      osg::ShapeDrawable* sh=new osg::ShapeDrawable;
//       std::cout << "Box " << submesh->_staticbbox._min << " - " <<  submesh->_staticbbox._max <<std::endl;
      osg::Box* bb=new osg::Box(_bbox.center(),_bbox.center()[0]-_bbox.xMin(),_bbox.center()[1]-_bbox.yMin(),_bbox.center()[2]-_bbox.zMin());
      sh->setShape(bb);
      addDrawable(sh);
#endif
    }
  }

  // process collid meshes
  for(std::vector<int>::iterator coreMeshId = _activeMeshes.begin(); coreMeshId != _activeMeshes.end(); coreMeshId++) {
    bool supportsPrimitiveFunctor = isCollisionMesh(*coreMeshId);
    bool invisible = isInvisibleMesh(*coreMeshId);
    if(invisible || supportsPrimitiveFunctor) {
      int meshId = coreMeshId2MeshId(&_calModel, *coreMeshId);
      if(meshId < 0) {
	CalError::setLastError(CalError::INDEX_BUILD_FAILED, __FILE__, __LINE__, "meshId not found");
	return false;
      }
      if(!createSubMeshSoftware(meshId, *coreMeshId))
	return false;
    }
  }
    
  delete [] tmpVertexBuffer;
  delete [] tmpNormalBuffer;
  delete [] tmpWeightBuffer;
  delete [] tmpMatrixIndexBuffer;
  delete [] tmpTexCoordBuffer;
  delete [] tmpIndexBuffer;

  return true;
}

bool Model::createSoftware(void) 
{
  CalRenderer *pCalRenderer = _calModel.getRenderer();
  if(pCalRenderer->beginRendering()) {
    for(std::vector<int>::iterator coreMeshId = _activeMeshes.begin(); coreMeshId != _activeMeshes.end(); coreMeshId++) {
      int meshId = coreMeshId2MeshId(&_calModel, *coreMeshId);
      if(meshId < 0) {
	CalError::setLastError(CalError::INDEX_BUILD_FAILED, __FILE__, __LINE__, "meshId not found");
	return false;
      }
      if(!createSubMeshSoftware(meshId, *coreMeshId))
	return false;
    }
  }
  pCalRenderer->endRendering();

  return true;
}

bool Model::createSubMeshSoftware(int meshId, int coreMeshId)
{
  CalRenderer *pCalRenderer = _calModel.getRenderer();
  Drawables submeshes;
  bool supportsPrimitiveFunctor = isCollisionMesh(coreMeshId);
  bool invisible = isInvisibleMesh(coreMeshId);
  const std::string& name = getCalCoreModel()->getCoreMesh(coreMeshId)->getName();
  int submeshCount = pCalRenderer->getSubmeshCount(meshId);
  for(int submeshId = 0; submeshId < submeshCount; submeshId++) {
    pCalRenderer->selectMeshSubmesh(meshId, submeshId);
    CalSubmesh* calSubmesh = pCalRenderer->m_pSelectedSubmesh;
    SubMeshSoftware* submesh = new SubMeshSoftware(calSubmesh, pCalRenderer);
    submesh->setName(name);
    submesh->setInvisible(invisible);
    submesh->setSupportsPrimitiveFunctor(supportsPrimitiveFunctor);
    submesh->create();
    if(!invisible) {
      if(!setupMaterial(submesh, calSubmesh))
	return false;

      submesh->_staticbbox=_bbox;
      osg::Vec3 min=_bbox._min;
      osg::Vec3 max=_bbox._max;
      min[2]-=submeshId/(1.0*submeshCount);
      max[2]-=submeshId/(1.0*submeshCount);
      submesh->_staticbbox=osg::BoundingBox(min,max);
    }
    addDrawable(submesh);
    submeshes.push_back(submesh);
  }
  _coreMeshId2Drawables[coreMeshId] = submeshes;

  return true;
}

void Model::update(void) 
{
  for(unsigned int i = 0; i < getNumDrawables(); i++) {
    osg::Drawable* drawable = getDrawable(i);
    SubMeshHardware* hardware = dynamic_cast<SubMeshHardware*>(drawable);
    if(hardware) {
      hardware->update();
    } else {
      SubMeshSoftware* software = dynamic_cast<SubMeshSoftware*>(drawable);
      if(software)
	software->update();
      else {
	CalError::setLastError(CalError::NULL_BUFFER, __FILE__, __LINE__, "unexpected drawable type");
      }
    }
  }
}

inline void Color2Vec4(const CalCoreMaterial::Color& color, osg::Vec4& vec4) {
  vec4[0] = color.red / 255.f;
  vec4[1] = color.green / 255.f;
  vec4[2] = color.blue / 255.f;
  vec4[3] = color.alpha == 0 ? 1.f : (color.alpha / 255.f);
}

bool Model::setupMaterial(osg::Drawable* drawable, CalSubmesh* calSubmesh)
{
  int coreMaterialId = calSubmesh->getCoreMaterialId();
  if(coreMaterialId < 0) {
    CalError::setLastError(CalError::INDEX_BUILD_FAILED, __FILE__, __LINE__);
    return false;
  }
  
  CalCoreMaterial* calCoreMaterial = getCalCoreModel()->getCoreMaterial(coreMaterialId);
  if(calCoreMaterial == 0) {
    CalError::setLastError(CalError::NULL_BUFFER, __FILE__, __LINE__);
    return false;
  }

  osg::StateSet *set = drawable->getOrCreateStateSet();

  osg::Material *material = new osg::Material();
  set->setAttributeAndModes(material, osg::StateAttribute::ON);

  if(!getUseColorOrTexture() || calCoreMaterial->getMapCount() <= 0) {
    osg::Vec4 materialColor;

    // set the material ambient color
    Color2Vec4(calCoreMaterial->getAmbientColor(), materialColor);
    material->setAmbient(osg::Material::FRONT_AND_BACK, materialColor);

    // set the material diffuse color
    Color2Vec4(calCoreMaterial->getDiffuseColor(), materialColor);
    material->setDiffuse(osg::Material::FRONT_AND_BACK, materialColor);

    // set the material specular color
    Color2Vec4(calCoreMaterial->getSpecularColor(), materialColor);
    material->setSpecular(osg::Material::FRONT_AND_BACK, materialColor);
  }

  // set the material shininess factor
  float shininess = calCoreMaterial->getShininess();
  material->setShininess(osg::Material::FRONT_AND_BACK, shininess);

  if(isPrivateTexture(coreMaterialId)) {
    CoreModel::Textures2D* textures = getCoreModel()->getTextures2D(coreMaterialId);
    if(textures == 0) return false;
    CoreModel::Textures2D private_textures;
    for(CoreModel::Textures2D::iterator j = textures->begin();
	j != textures->end();
	j++) {
      osg::Texture2D* texture = dynamic_cast<osg::Texture2D*>((*j)->clone(osg::CopyOp::SHALLOW_COPY));
      private_textures.push_back(texture);
    }
    _coreMaterialId2PrivateTextures[coreMaterialId] = private_textures;
  }
  CoreModel::Textures2D* textures = getTextures2D(coreMaterialId);
  if(textures == 0) return false;

  int channel = 0;
  for(CoreModel::Textures2D::iterator i = textures->begin();
      i != textures->end();
      i++) {
    osg::Texture2D* texture = i->get();
    if(texture == 0) {
      CalError::setLastError(CalError::NULL_BUFFER, __FILE__, __LINE__);
      return false;
    }
    set->setTextureAttributeAndModes(channel, texture, osg::StateAttribute::ON);
    channel++;
    break; // because we don't manage multitexture yet !!! so use the first texture 
  }

  return true;
}

Model::Textures2D* Model::getTextures2D(const std::string& materialName)
{
  int coreMaterialId = getCalCoreModel()->getCoreMaterialId(materialName);
  if(coreMaterialId < 0) {
    CalError::setLastError(CalError::INDEX_BUILD_FAILED, __FILE__, __LINE__);
    return 0;
  } else
    return getTextures2D(coreMaterialId);
}

Model::Textures2D* Model::getTextures2D(int coreMaterialId)
{
  if(coreMaterialId < 0) {
    CalError::setLastError(CalError::INDEX_BUILD_FAILED, __FILE__, __LINE__);
    return 0;
  }

  if(_coreMaterialId2PrivateTextures.find(coreMaterialId) != _coreMaterialId2PrivateTextures.end()) {
    return &_coreMaterialId2PrivateTextures[coreMaterialId];
  } else
    return getCoreModel()->getTextures2D(coreMaterialId);
}

Model::Drawables* Model::getDrawables(const std::string& meshName)
{
  int coreMeshId = getCalCoreModel()->getCoreMeshId(meshName);
  if(coreMeshId < 0) {
    CalError::setLastError(CalError::INDEX_BUILD_FAILED, __FILE__, __LINE__);
    return 0;
  } else
    return getDrawables(coreMeshId);
}

bool Model::hasDrawables(int coreMeshId)
{
  if(coreMeshId < 0) {
    CalError::setLastError(CalError::INDEX_BUILD_FAILED, __FILE__, __LINE__);
    return false;
  }

  return _coreMeshId2Drawables.find(coreMeshId) != _coreMeshId2Drawables.end();
}

Model::Drawables* Model::getDrawables(int coreMeshId)
{
  if(coreMeshId < 0) {
    CalError::setLastError(CalError::INDEX_BUILD_FAILED, __FILE__, __LINE__);
    return 0;
  }

  if(_coreMeshId2Drawables.find(coreMeshId) != _coreMeshId2Drawables.end()) {
    return &_coreMeshId2Drawables[coreMeshId];
  } else {
    CalError::setLastError(CalError::INDEX_BUILD_FAILED, __FILE__, __LINE__);
    return 0;
  }
}

bool Model::setPrivateTextures(const std::vector<std::string>& privateTextures)
{
  if(!_coreModel.valid())
    return false;

  bool status = true;
  _privateTextures.clear();
  for(std::vector<std::string>::const_iterator i = privateTextures.begin();
      i != privateTextures.end();
      i++) {
    const std::string& name = *i;
    int coreMaterialId = getCalCoreModel()->getCoreMaterialId(name);
    if(coreMaterialId >= 0)
      _privateTextures.push_back(coreMaterialId);
    else
      status = false;
  }
  return status;
}

bool Model::bindMaterials(const std::string& meshName, const std::vector<std::string>& materials)
{
  int coreMeshId = getCalCoreModel()->getCoreMeshId(meshName);
  if(coreMeshId < 0) {
    CalError::setLastError(CalError::INDEX_BUILD_FAILED, __FILE__, __LINE__);
    return false;
  }

  CalMesh* calMesh = _calModel.getMesh(coreMeshId);
  if(calMesh == 0) return false;
  
  if(materials.size() != (unsigned int)calMesh->getSubmeshCount()) {
    CalError::setLastError(CalError::INDEX_BUILD_FAILED, __FILE__, __LINE__);
    return false;
  }

  for(int submeshId = 0; submeshId < calMesh->getSubmeshCount(); submeshId++) {
    int materialId = getCalCoreModel()->getCoreMaterialId(materials[submeshId]);
    if(materialId < 0) {
      CalError::setLastError(CalError::INDEX_BUILD_FAILED, __FILE__, __LINE__);
      return false;
    }
    calMesh->getSubmesh(submeshId)->setCoreMaterialId(materialId);
  }

  return true;
}

bool Model::setActiveMesh(const std::string& meshName)
{
  int coreMeshId = getCalCoreModel()->getCoreMeshId(meshName);

  if(coreMeshId < 0) {
    const CoreModel::Name2Filename& meshName2Filename = getCoreModel()->getMeshName2Filename();
    if(meshName2Filename.find(meshName) == meshName2Filename.end()) {
      CalError::setLastError(CalError::INDEX_BUILD_FAILED, __FILE__, __LINE__);
      return false;
    }
    const std::string& filename = meshName2Filename.find(meshName)->second;
  
    coreMeshId = getCalCoreModel()->loadCoreMesh(filename, meshName);

    if(coreMeshId >= 0) invertUVs(coreMeshId);
  }

  if(coreMeshId < 0)
    return false;

  _activeMeshes.push_back(coreMeshId);
  _calModel.attachMesh(coreMeshId);
  _coreModel->AddUsingMeshId(coreMeshId);
  return true;
}

void Model::invertUVs(int coreMeshId)
{
  CalCoreMesh* calCoreMesh = getCalCoreModel()->getCoreMesh(coreMeshId);
  int count = calCoreMesh->getCoreSubmeshCount();
  for(int coreSubmeshId = 0; coreSubmeshId < count; coreSubmeshId++) {
    CalCoreSubmesh* coreSubmesh = calCoreMesh->getCoreSubmesh(coreSubmeshId);
    std::vector<std::vector<CalCoreSubmesh::TextureCoordinate> >& uv = coreSubmesh->getVectorVectorTextureCoordinate();
    for(unsigned int n = 0; n < uv.size(); n++)
      for(unsigned int iuv = 0; iuv < uv[n].size(); iuv++)
	uv[n][iuv].v=1.0-uv[n][iuv].v;
  }
}

bool Model::setCollisionMeshNames(const std::vector<std::string>& meshNames)
{
  _collisionMeshes.clear();
  _collisionMeshNames.clear();
  for(std::vector<std::string>::const_iterator i = meshNames.begin();
      i != meshNames.end();
      i++)
    if(!setCollisionMesh(*i))
      return false;
  return true;
}

bool Model::setCollisionMesh(const std::string& meshName)
{
  int coreMeshId = getCalCoreModel()->getCoreMeshId(meshName);
  if(coreMeshId < 0) {
    CalError::setLastError(CalError::INDEX_BUILD_FAILED, __FILE__, __LINE__);
    return false;
  }

  _collisionMeshNames.push_back(meshName);
  _collisionMeshes.push_back(coreMeshId);
  return true;
}

bool Model::setInvisibleMesh(const std::string& meshName)
{
  int coreMeshId = getCalCoreModel()->getCoreMeshId(meshName);
  if(coreMeshId < 0) {
    CalError::setLastError(CalError::INDEX_BUILD_FAILED, __FILE__, __LINE__);
    return false;
  }

  _invisibleMeshes.push_back(coreMeshId);
  return true;
}

bool Model::find(const std::vector<int>& vector, int item) {
  for(std::vector<int>::const_iterator i = vector.begin();
      i != vector.end();
      i++)
    if(*i == item)
      return true;
  return false;
}
