
/******************************************************************************
**
**  Copyright (C) 2005 Brian Wotring.
**
**  This program is free software; you can redistribute it and/or
**  modify it, however, you cannot sell it.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
**
**  You should have received a copy of the license attached to the
**  use of this software.  If not, view a current copy of the license
**  file here:
**
**      http://www.hostintegrity.com/osiris/LICENSE
**
******************************************************************************/

/*****************************************************************************
**
**  File:    error.c
**  Date:    June 2, 2002
**
**  Author:  Brian Wotring
**  Purpose: central error reference for all osiris applications.
**
******************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <string.h>

#include <sys/types.h>
#include <time.h>

#include "utilities.h"
#include "error.h"

struct e_keywords error_keywords[] =
{
    { "unknown error occurred.", OSI_ERROR_UNKNOWN },

    { "no error", OSI_OK },
    { "no error.", MESSAGE_OK },
    { "no error.", OSI_DB_OK },
    { "no error.", OSI_FILE_OK },

    { "null argument.", OSI_ERROR_NULL_ARGUMENT },

    { "error reading message.", OSI_ERROR_MESSAGE_READ },
    { "message header too short.", OSI_ERROR_MESSAGE_HEADER_TOO_SHORT },
    { "no data in message.", OSI_ERROR_MESSAGE_NO_DATA },
    { "message payload too large.", OSI_ERROR_MESSAGE_PAYLOAD_TOO_LARGE },
    { "message payload too short.", OSI_ERROR_MESSAGE_PAYLOAD_TOO_SHORT },
    { "error setting message payload.", OSI_ERROR_SETTING_MESSAGE_PAYLOAD },
    { "error sending message.", OSI_ERROR_SENDING_MESSAGE },

    { "socket was closed.", OSI_ERROR_SOCKET_CLOSED },
    { "unable to connect to host.", OSI_ERROR_UNABLE_TO_CONNECT },
    { "no response from host.", OSI_ERROR_NO_RESPONSE },
    { "invalid response.", OSI_ERROR_INVALID_RESPONSE },

    { "start scan failed.", OSI_ERROR_SCAN_START_FAILED },
    { "no config present.", OSI_ERROR_NO_CONFIG },
    { "daemon not idle.", OSI_ERROR_DAEMON_NOT_IDLE },
    { "no scanning process exists.", OSI_ERROR_NO_SCANNING_PROCESS },

    { "no privileges.", OSI_ERROR_NO_PRIVLEDGES },

    { "unable to read host listing.", OSI_ERROR_READ_HOSTS_FAILED },
    { "no host specified.", OSI_ERROR_NO_HOST_SPECIFIED },
    { "no configuration file specified.", OSI_ERROR_NO_CONFIG_SPECIFIED },
    { "no database specified.", OSI_ERROR_NO_DB_SPECIFIED },
    { "no filters", OSI_ERROR_NO_FILTERS },
    { "unable to read notification filters.", OSI_ERROR_READ_FILTERS_FAILED },

    { "no hosts.", OSI_ERROR_NO_HOSTS },

    { "no configs exist for host.", OSI_ERROR_NO_CONFIGS },
    { "specified host doesn't exist.", OSI_ERROR_HOST_DOES_NOT_EXIST },
    { "specified configuration file doesn't exist.",
      OSI_ERROR_CONFIG_DOES_NOT_EXIST },
    { "specified database doesn't exist.", OSI_ERROR_DATABASE_DOES_NOT_EXIST },

    { "configuration file is empty.", OSI_ERROR_CONFIG_IS_EMPTY },
    { "host exists.", OSI_ERROR_HOST_EXISTS },
    { "configuration file exists.", OSI_ERROR_CONFIG_EXISTS },

    { "unknown message type.", OSI_ERROR_UNKNOWN_MESSAGE_TYPE },
    { "unable to initialize database.", OSI_ERROR_DB_INIT_FAILED },
    { "unable to create database.", OSI_ERROR_DB_CREATE_FAILED },
    { "unable to open database.", OSI_ERROR_DB_OPEN_FAILED },
    { "database not opened.", OSI_ERROR_DB_NOT_OPENED },
    { "database write error.", OSI_ERROR_DB_STORING },
    { "invalid database path.", OSI_ERROR_DB_PATH },
    { "database read error.", OSI_ERROR_DB_RETRIEVING },
    { "data too large to store.", OSI_ERROR_DB_DATA_TOO_LARGE },
    { "database cursor error.", OSI_ERROR_DB_DATA_TOO_LARGE },
    { "last item.", OSI_ERROR_DB_LAST_ITEM },
    { "item exists.", OSI_ERROR_DB_ITEM_EXISTS },
    { "no error records exist.", OSI_ERROR_DB_NO_ERRORS },
    { "no filters exist.", OSI_ERROR_DB_NO_FILTERS },
    { "database item not found.", OSI_ERROR_DB_ITEM_NOT_FOUND },
    { "no base-db exists, host not initialized?", OSI_ERROR_BASE_DB_NOT_FOUND },
    { "file not found.", OSI_ERROR_FILE_NOT_FOUND },
    { "path not found.", OSI_ERROR_PATH_NOT_FOUND },
    { "access denied.", OSI_ERROR_ACCESS_DENIED },
    { "write protected.", OSI_ERROR_WRITE_PROTECTED },
    { "sharing violation.", OSI_ERROR_SHARING_VIOLATION },
    { "directory not empty.", OSI_ERROR_DIRECTORY_NOT_EMPTY },
    { "disk full!", OSI_ERROR_DISK_FULL },
    { "invalid path", OSI_ERROR_INVALID_PATH },
    { "i/o error.", OSI_ERROR_IO_ERROR },
    { "file exists.", OSI_ERROR_FILE_EXISTS },

    { "unable to create file.", OSI_ERROR_UNABLE_TO_CREATE_FILE },
    { "unable to open directory.", OSI_ERROR_UNABLE_TO_OPEN_DIRECTORY },
    { "unable to stat file.", OSI_ERROR_UNABLE_TO_STAT_FILE },
    { "unable to read file.", OSI_ERROR_UNABLE_TO_READ_FILE },
    { "unable to write file.", OSI_ERROR_UNABLE_TO_WRITE_FILE },
    { "unable to compute file checksum.", OSI_ERROR_UNABLE_TO_CSUM_FILE },

    { "unable to create process.", OSI_ERROR_UNABLE_TO_CREATE_PROCESS },
    { "scanner was killed.", OSI_ERROR_SCANNER_WAS_KILLED },
    { "scanner was stopped.", OSI_ERROR_SCANNER_WAS_STOPPED },
    { "session key negotiation failed.",
                        OSI_ERROR_SESSION_KEY_NEGOTIATION_FAILED },
    { "no user login specified.", OSI_ERROR_NO_USER_LOGIN_SPECIFIED },
    { "invalid user login.", OSI_ERROR_SET_USER_LOGIN_FAILED },
    { "error reading user database.", OSI_ERROR_READ_USERS_FAILED },
    { "no users.", OSI_ERROR_NO_USERS },
    { "user does not exist.", OSI_ERROR_USER_DOES_NOT_EXIST },
    { "unable to delete user.", OSI_ERROR_DELETE_USER_FAILED },

    { NULL, 0 }
};


char * osi_get_name_for_error( osi_uint64 error )
{
    int index;
    char *name = "?";

    for( index = 0; error_keywords[index].word != NULL; index++ )
    {
        if( error == error_keywords[index].type )
        {
            name = error_keywords[index].word;
            break;
        }
    }

    return name;
}

void wrap_error( OSI_ERROR *error )
{
    if( error != NULL )
    {
        error->type = OSI_HTONLL( error->type );
        error->time = OSI_HTONLL( error->time );
    }
}

void unwrap_error( OSI_ERROR *error )
{
    if( error != NULL )
    {
        error->type = OSI_NTOHLL( error->type );
        error->time = OSI_NTOHLL( error->time );
    }
}


void print_osi_error( OSI_ERROR *error )
{
    time_t error_time;
    char *time;

    if( error == NULL )
    {
        fprintf( stderr, "<empty error>\n" );
        return;
    }

    /* get time as a string. */

    error_time = (time_t)error->time;

    if( error_time != 0 )
    {
        time = ctime( &error_time );
    }

    else
    {
        time = "<no timestamp>";
    }

    if( time != NULL )
    {
        if( time[strlen(time)-1] == '\n' )
        {
            time[strlen(time)-1] = '\0';
        }
    }

    fprintf( stdout, "[%llu] [%s] ", error->type, time );

    if( ( error->message != NULL ) && ( strlen( error->message ) > 0 ) )
    {
        fprintf( stdout, "[ %s ]", error->message );
    }

    else
    {
        fprintf( stdout, "[ %s ]", osi_get_name_for_error( error->type ) );
    }

    fprintf( stdout, "\n" );
}

void pack_error( OSI_ERROR *error, char *buffer, int buffer_size )
{
    char *temp = buffer;

    if( ( error == NULL ) || ( buffer == NULL ) || ( buffer_size <= 0 ) )
    {
        return;
    }

    memcpy( temp, &error->type, sizeof( error->type ) );
    temp += sizeof( error->type );
    memcpy( temp, &error->time, sizeof( error->time ) );
    temp += sizeof( error->time );
    memcpy( temp, error->message, sizeof( error->message ) );
}

void unpack_error( OSI_ERROR *error, char *buffer, int buffer_size )
{
    char *temp = buffer;

    if( ( error == NULL ) || ( buffer == NULL ) || ( buffer_size <= 0 ) )
    {
        return;
    }

    memcpy( &error->type, temp, sizeof( error->type ) );
    temp += sizeof( error->type );
    memcpy( &error->time, temp, sizeof( error->time ) );
    temp += sizeof( error->time );
    memcpy( error->message, temp, sizeof( error->message ) );
}

