
/******************************************************************************
**
**  Copyright (C) 2005 Brian Wotring.
**
**  This program is free software; you can redistribute it and/or
**  modify it, however, you cannot sell it.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
**
**  You should have received a copy of the license attached to the
**  use of this software.  If not, view a current copy of the license
**  file here:
**
**      http://www.hostintegrity.com/osiris/LICENSE
**
******************************************************************************/

/*****************************************************************************
**
**  File:    mod_kmods.c
**  Date:    January 1, 2004
**
**  Author:  Brian Wotring
**  Purpose: platform specific methods for reading active kernel mod info.
**
******************************************************************************/

#ifndef WIN32
#include "config.h"
#endif

static const char *MODULE_NAME = "mod_kmods";

#if defined(SYSTEM_DARWIN)
#include <mach/mach.h>
#include <mach/mach_error.h>
#include <mach/mach_host.h>
#endif

#if defined(SYSTEM_FREEBSD)
#include <unistd.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/module.h>
#include <sys/linker.h>
#endif

#if defined(SYSTEM_SUNOS) || defined(SYSTEM_SOLARIS)
#include <sys/modctl.h>
extern int modctl( int action, ... );
#endif

#if defined(SYSTEM_OPENBSD)
#include <sys/param.h>
#include <sys/ioctl.h>
#include <sys/conf.h>
#include <sys/mount.h>
#include <sys/lkm.h>
#include <errno.h>
#include <fcntl.h>
#include <unistd.h>
#define _PATH_LKM "/dev/lkm"
#define POINTERSIZE     ((int)(2 * sizeof(void*)))

static char *type_names[] =
{
    "SYSCALL",
    "VFS",
    "DEV",
    "EXEC",
    "MISC"
};
#endif

#include "libosiris.h"
#include "libfileapi.h"
#include "rootpriv.h"
#include "common.h"
#include "version.h"

#include "scanner.h"
#include "logging.h"

#ifdef WIN32

void process_services_windows( SCANNER *scanner )
{
 
    unsigned int i;
    unsigned char buf[64000];

    SC_HANDLE shandle;
    DWORD bytes_needed = 0;
    DWORD num_services = 0;
    DWORD resume_handle = 0;
    ENUM_SERVICE_STATUS *status;
    LPENUM_SERVICE_STATUS enums = (ENUM_SERVICE_STATUS *)buf;

    shandle = OpenSCManager( NULL, NULL, SC_MANAGER_ENUMERATE_SERVICE );
 
 
    if( shandle == NULL ) 
    {
        log_error( "unable to open services database." );
        return;
    }


    if(  EnumServicesStatus( shandle,
                             SERVICE_WIN32,
                             SERVICE_STATE_ALL,
                             enums,
                             sizeof(buf),
                             &bytes_needed,
                             &num_services,
                             &resume_handle ) == FALSE )

    {
        log_error( "unable to enumerate services list." );
        return;
    }

    status = enums;

    for( i = 0; i < num_services; i++ )
    {
        if( status != NULL )
        {
            char *state = NULL;
            SCAN_RECORD_TEXT_1 record;

            initialize_scan_record( (SCAN_RECORD *)&record,
                                    SCAN_RECORD_TYPE_TEXT_1 );

            osi_strlcpy( record.module_name, MODULE_NAME,
                         sizeof( record.module_name ) );

            osi_snprintf( record.name, sizeof( record.name ),
                          "service:%s", status->lpServiceName );

            switch ( status->ServiceStatus.dwCurrentState )
            {
                case SERVICE_CONTINUE_PENDING:
                    state = "continue_pending";
                    break;

                case SERVICE_PAUSE_PENDING:
                    state = "pause_pending";
                    break;

                case SERVICE_PAUSED:
                    state = "paused";
                    break;

                case SERVICE_RUNNING:
                    state = "running";
                    break;

                case SERVICE_START_PENDING:
                    state = "start_pending";
                    break;

                case SERVICE_STOP_PENDING:
                    state = "stop_pending";
                    break;

                case SERVICE_STOPPED:
                    state = "stopped";
                    break;

                default:
                    state = "unknown";
            }

            osi_snprintf( record.data, sizeof( record.data ),
                          "service:%s;dname:%s;status:%s",
                          status->lpServiceName,
                          status->lpDisplayName,
                          state );

            send_scan_data( scanner, (SCAN_RECORD *)&record );
            status++;
        }

        else
        {
            break;
        }
    }

    CloseServiceHandle( shandle );
}

#endif

#if defined(SYSTEM_FREEBSD)

void process_kmods_freebsd( SCANNER *scanner )
{
    SCAN_RECORD_TEXT_1 record;
    int fileid = 0;

    for( fileid = kldnext(0); fileid > 0; fileid = kldnext( fileid ) )
    {
        struct kld_file_stat stat;
        int modid;

        stat.version = sizeof(struct kld_file_stat);

        if( kldstat( fileid, &stat ) < 0 )
        {
            log_error( "error reading kernel mod file." );
            continue;
        }

        else
        {
            initialize_scan_record( (SCAN_RECORD *)&record,
                                    SCAN_RECORD_TYPE_TEXT_1 );

            osi_strlcpy( record.module_name, MODULE_NAME,
                         sizeof( record.module_name ) );

            osi_snprintf( record.name, sizeof( record.name ), "kern:%d",
                          stat.id ); 

            osi_snprintf( record.data, sizeof( record.data ),
                          "refs:%d;size:%x;name:%s;", stat.refs,
                          stat.size, stat.name );

            osi_strlcat( record.data, "modules:", sizeof( record.data ) );

            /* now get the modules under these link files. */

            for( modid = kldfirstmod( fileid ); modid > 0;
                 modid = modfnext( modid ) )
            {
                struct module_stat stat;

                stat.version = sizeof(struct module_stat);

                if( modstat( modid, &stat ) < 0 )
                {
                    continue;
                }

                else
                {
                    osi_strlcat( record.data, stat.name,
                                 sizeof( record.data ) );
                    
                    osi_strlcat( record.data, ", ", sizeof( record.data ) );
                }

            } /* end get module list. */

            send_scan_data( scanner, (SCAN_RECORD *)&record );
        }
    }
}

#endif

#if defined(SYSTEM_OPENBSD)

void process_kmods_openbsd( SCANNER *scanner )
{
    int modnum = 0;
    int devfd;

    char name[MAXLKMNAME];
    struct lmc_stat sbuf;

    FILE *file;
    SCAN_RECORD_TEXT_1 record;

    if( ( file = rootpriv_fopen( _PATH_LKM ) ) == NULL )
    {
        log_error( "error opening kernel mod file (%s)", _PATH_LKM );
        return;
    }

    devfd = fileno( file );

    /* Start at 0 and work up until we receive EINVAL. */

    while(1)
    {
        bzero( &name, sizeof( name ) );
        bzero( &sbuf, sizeof( sbuf ) );

        sbuf.id = modnum;
        sbuf.name = name;

        if( ioctl( devfd, LMSTAT, &sbuf ) == -1 )
        {
            if( errno == ENOENT )
            {
                continue;
            }

            else
            {
                goto exit_gracefully;
            }
        }

        initialize_scan_record( (SCAN_RECORD *)&record,
                                SCAN_RECORD_TYPE_TEXT_1 );

        osi_strlcpy( record.module_name, MODULE_NAME,
                     sizeof( record.module_name ) );

        osi_snprintf( record.name, sizeof( record.name ), "kern:%d", sbuf.id );

        osi_snprintf( record.data, sizeof( record.data ),
                      "type=%s;id=%d;size=%lx;ver=%ld;name=%s",
                      type_names[sbuf.type], sbuf.id, (long)sbuf.size,
                      (long)sbuf.ver, sbuf.name );

        send_scan_data( scanner, (SCAN_RECORD *)&record );
        modnum++;
    }

exit_gracefully:

    close( devfd );
}

#endif

#if defined(SYSTEM_SUNOS) || defined(SYSTEM_SOLARIS)

void process_kmods_solaris( SCANNER *scanner )
{
    int result = 0;
    int id = -1;
    int index = 0;
    int p;

    SCAN_RECORD_TEXT_1 record;
    struct modinfo m;

    memset( &m, 0, sizeof( m ) );

    m.mi_info = ( MI_INFO_ALL | MI_INFO_CNT );
    m.mi_id = id;

    result = modctl( MODINFO, id , &m );

    initialize_scan_record( (SCAN_RECORD *)&record,
                            SCAN_RECORD_TYPE_TEXT_1 );

    osi_strlcpy( record.module_name, MODULE_NAME,
                 sizeof( record.module_name ) );

    while( result == 0 )
    {
        for( index = 0; index < MODMAXLINK; index++ )
        {
            m.mi_name[MODMAXNAMELEN-1] = 0;

            initialize_scan_record( (SCAN_RECORD *)&record,
                                    SCAN_RECORD_TYPE_TEXT_1 );

            osi_snprintf( record.name, sizeof( record.name ),
                          "kern:%d,%d", m.mi_id, index );

            osi_strlcpy( record.module_name, MODULE_NAME,
                         sizeof( record.module_name ) );

            if( m.mi_msinfo[index].msi_linkinfo[0] == 0 )
            {
		        osi_snprintf( record.data, sizeof( record.data ),
				     "id=%d;loadcount=%d;state=%d;name=%s",
		           m.mi_id, m.mi_loadcnt, m.mi_state, m.mi_name );

        		send_scan_data( scanner, (SCAN_RECORD *)&record );
                break;
            }

            p = m.mi_msinfo[index].msi_p0;

            m.mi_msinfo[index].msi_linkinfo[MODMAXNAMELEN-1] = 0;

            osi_snprintf( record.data, sizeof( record.data ),
          "id=%d;loadcount=%d;base=%x;size=%x;state=%d;info=%d;name=%s;desc=%s",
                          m.mi_id, m.mi_loadcnt, m.mi_base, m.mi_size,
                          m.mi_state, p, m.mi_name,
                          m.mi_msinfo[index].msi_linkinfo );

            send_scan_data( scanner, (SCAN_RECORD *)&record );
        }

        id = m.mi_id;
        result = modctl( MODINFO, id, &m );
    }
}

#endif

#if defined(SYSTEM_LINUX)

#define LINUX_PROC_FILE "/proc/modules"

void process_kmods_linux( SCANNER *scanner )
{
    FILE *file;
    char line[100];
    char *data;

    SCAN_RECORD_TEXT_1 record;

    if( ( file = osi_fopen( LINUX_PROC_FILE , "r", 0 ) ) == NULL )
    {
        log_error( "unable to open proc file (%s).", LINUX_PROC_FILE );
        return;
    }

    while( ( data = fgets( line, sizeof( line ), file ) ) != NULL )
    {
        char name[100] = "";
        char size[20] = "";
        char used[20] = "";
        char address[100] = "";

        char *temp = NULL;

        temp = get_token( line, name, sizeof(name) );

        if( strlen( name ) == 0 )
        {
            continue;
        }

        initialize_scan_record( (SCAN_RECORD *)&record,
                                SCAN_RECORD_TYPE_TEXT_1 );

        osi_strlcpy( record.module_name, MODULE_NAME,
                     sizeof( record.module_name ) );

        /* store name in record. */

        osi_strlcpy( record.name, "kern:", sizeof( record.name ) );
        osi_strlcat( record.name, name, sizeof( record.name ) );        

        /* get and data, we don't need the "used by count".   */
        /* the format is <name> <size> <usedby> - <address>...*/

        temp = get_token( temp, size, sizeof(size) );
        temp = get_token( temp, used, sizeof(used) );
	
        osi_snprintf( record.data, sizeof(record.data), "%s %s - %s",
                      name, size, temp );

        send_scan_data( scanner, (SCAN_RECORD *)&record );
    }

    fclose( file );
}
#endif

#if defined(SYSTEM_DARWIN)

/* all of this code taken from kextstat.c from the darwin source. */

static kmod_info_t * kmod_lookup(
    kmod_info_t * kmods,
    unsigned int kmod_count,
    int kmod_id)
{
    kmod_info_t * found_kmod = 0;  
    unsigned int i;

    for (i = 0; i < kmod_count; i++) {
        kmod_info_t * this_kmod = &kmods[i];
        if (this_kmod->id == kmod_id) {
            found_kmod = this_kmod;
            goto finish;
        }
    }

finish:
    return found_kmod;
}

static int kmod_ref_compare(const void * a, const void * b)
{
    kmod_reference_t * r1 = (kmod_reference_t *)a;
    kmod_reference_t * r2 = (kmod_reference_t *)b;
    /*  these are load indices, not CFBundleIdentifiers */
    /* sorting high-low. */
    return ((int)r2->info - (int)r1->info);
}

static int kmod_compare(const void * a, const void * b)
{
    kmod_info_t * k1 = (kmod_info_t *)a;
    kmod_info_t * k2 = (kmod_info_t *)b;
    /*  these are load indices, not CFBundleIdentifiers */
    return (k1->id - k2->id);
}

void process_kmods_darwin( SCANNER *scanner )
{
    port_t host_port = PORT_NULL;
    kmod_info_t * kmod_list;
    kmod_info_t * this_kmod;
    int kmod_bytecount;  
    int kmod_count;
    kern_return_t mach_result = KERN_SUCCESS;
    kmod_reference_t * kmod_ref;
    int ref_count;
    int i, j;

    SCAN_RECORD_TEXT_1 record;

    if( scanner == NULL )
    {
        return;
    }
    
   /* Get the list of loaded kmods from the kernel.
    */
    host_port = mach_host_self();
    mach_result = kmod_get_info( host_port, (void *)&kmod_list,
                                 &kmod_bytecount);

    if (mach_result != KERN_SUCCESS)
    {

        log_error( "unable to get list of loaded kexts from kernel!" );
        goto finish;
    }

   /* kmod_get_info() doesn't return a proper count so we have
    * to scan the array checking for a NULL next pointer.
    */
    this_kmod = kmod_list;
    kmod_count = 0;

    while (this_kmod)
    {
        kmod_count++;
        this_kmod = (this_kmod->next) ? (this_kmod + 1) : 0;
    }

   /* rebuild the reference lists from their serialized pileup
    * after the list of kmod_info_t structs.
    */
    this_kmod = kmod_list;
    kmod_ref = (kmod_reference_t *)(kmod_list + kmod_count);
    while (this_kmod) {

       /* How many refs does this kmod have? Again, kmod_get_info ovverrides
        * a field. Here what is the actual reference list in the kernel becomes
        * the count of references tacked onto the end of the kmod_info_t list.
        */
        ref_count = (int)this_kmod->reference_list;
        if (ref_count) {
            this_kmod->reference_list = kmod_ref;

            for (i = 0; i < ref_count; i++) {
                int foundit = 0;
                for (j = 0; j < kmod_count; j++) {
                   /* kmod_get_info() made each kmod_info_t struct's .next field
                    * point to itself IN KERNEL SPACE, so this is a sort of id
                    * for the reference list. Here we replace the ref's
                    * info field, a here-useless KERNEL SPACE ADDRESS,
                    * with the list id of the kmod_info_t struct.
                    * Gross, gross hack.
                    */
                    if (kmod_ref->info == kmod_list[j].next) {
                        kmod_ref->info = (kmod_info_t *)kmod_list[j].id;
                        foundit++;
                        break;
                    }
                }

               /* If we didn't find it, that's because the last entry's next
                * pointer is SET TO ZERO to signal the end of the kmod_info_t
                * list, even though the same field is used for other purposes
                * in every other entry in the list. So set the ref's info
                * field to the id of the last entry in the list.
                */
                if (!foundit) {
                    kmod_ref->info =
                        (kmod_info_t *)kmod_list[kmod_count - 1].id;
                }

                kmod_ref++;
            }

           /* Sort the references in descending order of reference index.
            */
            qsort(this_kmod->reference_list, ref_count,
                  sizeof(kmod_reference_t), kmod_ref_compare);

           /* Patch up the links between ref structs and move on to the
            * next one.
            */
            for (i = 0; i < ref_count - 1; i++) {
                this_kmod->reference_list[i].next =
                    &this_kmod->reference_list[i+1];
            }
            this_kmod->reference_list[ref_count - 1].next = 0;
        }
        this_kmod  = (this_kmod->next) ? (this_kmod + 1) : 0;
    }

    if (!kmod_count) {
        goto finish;
    }

    qsort(kmod_list, kmod_count, sizeof(kmod_info_t), kmod_compare);

   /* Now print out what was found.
    */
    this_kmod = kmod_list;
    for (i=0; i < kmod_count; i++, this_kmod++)
    {
        char buffer[sizeof(record.data)];

        if (!this_kmod->size)
        {
            continue;
        }

        initialize_scan_record( (SCAN_RECORD *)&record,
                                SCAN_RECORD_TYPE_TEXT_1 );

        osi_strlcpy( record.module_name, MODULE_NAME,
                     sizeof( record.module_name ) );

        /* store name and data in record. */

        osi_strlcpy( record.name, "kern:", sizeof( record.name ) );
        osi_strlcat( record.name, this_kmod->name, sizeof( record.name ) );

        osi_snprintf( buffer, sizeof(buffer),
                      "%5d %4d %-10p %-10p %-10p %s (%s)",
                      this_kmod->id,
                      this_kmod->reference_count,
                      (void *)this_kmod->address,
                      (void *)this_kmod->size,
                      (void *)(this_kmod->size - this_kmod->hdr_size),
                      this_kmod->name,
                      this_kmod->version);

        kmod_ref = this_kmod->reference_list;
        if (kmod_ref) {
            int printed_brace = 0;
            while (kmod_ref) {
                kmod_info_t * ref_info =
                    kmod_lookup(kmod_list, kmod_count, (int)kmod_ref->info);

                if (ref_info && (ref_info->address))
                {
                    char tmp[10];
                
                    osi_snprintf( tmp, sizeof(tmp), " %s%d",
                                  (!printed_brace ? "<" : "" ),
                                  (int)kmod_ref->info );

                    osi_strlcat( buffer, tmp, sizeof( buffer ) );
                    printed_brace = 1;
                }

                kmod_ref = kmod_ref->next;
            }

            if (printed_brace)
            {
                osi_strlcat( buffer, ">", sizeof( buffer ) );
            }
        }

        osi_strlcpy( record.data, buffer, sizeof( record.data ) );
        send_scan_data( scanner, (SCAN_RECORD *)&record );
    }

finish:

   /* Dispose of the host port to prevent security breaches and port
    * leaks. We don't care about the kern_return_t value of this
    * call for now as there's nothing we can do if it fails.
    */
    if (PORT_NULL != host_port) {
        mach_port_deallocate(mach_task_self(), host_port);
    }

    if (kmod_list) {
        vm_deallocate(mach_task_self(), (vm_address_t)kmod_list,
            kmod_bytecount);
    }
}


#endif

void mod_kmods( SCANNER *scanner )
{
#if defined(SYSTEM_LINUX)
    process_kmods_linux( scanner );
#endif

#if defined(SYSTEM_DARWIN)
    process_kmods_darwin( scanner );
#endif

#if defined(SYSTEM_FREEBSD)
    process_kmods_freebsd( scanner );
#endif

#if defined(SYSTEM_OPENBSD)
    process_kmods_openbsd( scanner );
#endif

#if defined(SYSTEM_SUNOS) || defined(SYSTEM_SOLARIS)
    process_kmods_solaris( scanner );
#endif

#ifdef WIN32
    process_services_windows( scanner );
#endif
}

