///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/**
 * \file SimpleShapeObject.h
 * \brief Contains the definition of the Core::SimpleShapeObject class.
 */

#ifndef __OVITO_SIMPLE_SHAPE_OBJECT_H
#define __OVITO_SIMPLE_SHAPE_OBJECT_H

#include <core/Core.h>
#include <core/scene/bezier/BezierShape.h>
#include "../SceneObject.h"

namespace Core {

/**
 * \brief Base class for all simple shape objects that are based on a BezierShape.
 *
 * Subclasses must implements the buildShape() method.
 *
 * \author Alexander Stukowski
 */
class CORE_DLLEXPORT SimpleShapeObject : public SceneObject
{
protected:

	/// \brief Default constructor.
	/// \param isLoading Indicates whether the object is being loaded from a file.
	///                  This parameter is only used by the object serialization system.
	SimpleShapeObject(bool isLoading) : SceneObject(isLoading), shapeValidity(TimeNever) {}

public:

	/// \brief Makes the object render itself into a viewport.
	/// \param time The animation time at which to render the object
	/// \param contextNode The node context used to render the object.
	/// \param vp The viewport to render in.
	///
	/// This implementation just renders the BezierShape produced by buildShape().
	virtual void renderObject(TimeTicks time, ObjectNode* contextNode, Viewport* vp);

	/// \brief Asks the object whether it should show in a rendered image.
	virtual bool isRenderable() { return false; }

	/// \brief Computes the bounding box of the object.
	/// \param time The animation time for which the bounding box should be computed.
	/// \param contextNode The scene node to which this scene object belongs to.
	/// \return The bounding box of the object in local object coordinates.
	///
	/// This implementation returns the bounding box of the BezierShape produced by buildShape().
	virtual Box3 boundingBox(TimeTicks time, ObjectNode* contextNode) {
		if(!shapeValidity.contains(time))
			buildShape(time, shape, shapeValidity);
		return shape.boundingBox();
	}

	/// \brief This asks the object whether it supports the conversion to another object type.
	/// \param objectClass The destination type. This must be a SceneObject derived class.
	/// \return \c true if this object can be converted to the requested type given by \a objectClass or any sub-class thereof.
	///         \c false if the conversion is not possible.
	///
	/// This implementation allows the conversion to a ShapeObject.
	virtual bool canConvertTo(PluginClassDescriptor* objectClass);

	/// \brief Lets the object convert itself to another object type.
	/// \param objectClass The destination type. This must be a SceneObject derived class.
	/// \param time The time at which to convert the object.
	/// \return The newly created object or \c NULL if no conversion is possible.
	///
	/// This implementation handles the conversion to a ShapeObject.
	virtual SceneObject::SmartPtr convertTo(PluginClassDescriptor* objectClass, TimeTicks time);

protected:

	/// This method is called when a reference target changes.
	virtual bool onRefTargetMessage(RefTarget* source, RefTargetMessage* msg) {
		if(msg->type() == REFTARGET_CHANGED) {
			// Invalidate shape cache when a parameter has changed.
			shapeValidity.setEmpty();
		}
		return SceneObject::onRefTargetMessage(source, msg);
	}

	/// Is called when the value of a reference field of this RefMaker changes.
	virtual void onRefTargetReplaced(const PropertyFieldDescriptor& field, RefTarget* oldTarget, RefTarget* newTarget) {
		// Invalidate shape cache when a parameter controller has been replaced.
		shapeValidity.setEmpty();
        SceneObject::onRefTargetReplaced(field, oldTarget, newTarget);
	}

	/// Is called when a reference target has been added to a list reference field of this RefMaker.
	virtual void onRefTargetInserted(const PropertyFieldDescriptor& field, RefTarget* newTarget, int listIndex) {
        SceneObject::onRefTargetInserted(field, newTarget, listIndex);
		// Invalidate shape cache when a parameter controller has been added.
		shapeValidity.setEmpty();
	}

	/// Is called when a reference target has been removed from a list reference field of this RefMaker.
	virtual void onRefTargetRemoved(const PropertyFieldDescriptor& field, RefTarget* oldTarget, int listIndex) {
        SceneObject::onRefTargetRemoved(field, oldTarget, listIndex);
		// Invalidate shape cache when a parameter controller has been removed.
		shapeValidity.setEmpty();
	}

	/// Is called when the value of a non-animatable property field of this RefMaker has changed.
	virtual void onPropertyFieldValueChanged(const PropertyFieldDescriptor& field) {
        SceneObject::onPropertyFieldValueChanged(field);
		// Invalidate shape cache when a parameter has changed.
		shapeValidity.setEmpty();
	}

protected:

	/// \brief Builds the BezierShape representation of this shape object.
	/// \param[in] time The animation time at which to generate the shape.
	/// \param[out] shape The shape object that should be filled by the method. Please note
	///                  that the BezierShape instance passed to this method can contain the old shape from the last
	///                  call to buildShape(), so the method has to clear it first.
	/// \param[out] shapeValidity The method implementation stores the validty interval of the generated shape in this variable.
	///
	/// The SimpleShapeObject class calls this virtual method to let the sub-class generate the shape based on the internal
	/// parameters of the object.
	virtual void buildShape(TimeTicks time, BezierShape& shape, TimeInterval& shapeValidity) = 0;

private:

	/// The validity interval of the cached shape.
	TimeInterval shapeValidity;

	/// The cached shape generated by this object.
	BezierShape shape;

	Q_OBJECT
	DECLARE_ABSTRACT_PLUGIN_CLASS(SimpleShapeObject)
};


};

#endif // __OVITO_SIMPLE_SHAPE_OBJECT_H
