
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: odk_event.c 1574 2006-11-19 20:48:44Z mschwerin $
 *
 */

#include "config.h"

#include <assert.h>

#include "heap.h"
#include "logger.h"
#include "scheduler.h"

#include "odk.h"
#include "odk_joystick.h"
#include "odk_lirc.h"
#include "odk_private.h"

/* 
 * ***************************************************************************
 * Name:            event_handlers_destroy
 * Access:          private
 *
 * Description:     Frees an event handler.
 * ***************************************************************************
 */
static void
event_handlers_destroy (void *event_handler_p)
{
    odk_event_handler_t *eh = (odk_event_handler_t *) event_handler_p;
    ho_free (eh);
}


static int
swap_handler (void *h1_p, void *h2_p)
{
    odk_event_handler_t *h1 = (odk_event_handler_t *) h1_p;
    odk_event_handler_t *h2 = (odk_event_handler_t *) h2_p;

    return (h1->priority < h2->priority);
}


/* 
 * ***************************************************************************
 * Name:            odk_register_event_handler
 * Access:          public
 *
 * Description:     Registers an event handler.
 * ***************************************************************************
 */
void
odk_add_event_handler (odk_t * odk, event_handler_t eh, void *data,
                       odk_event_handler_priority_t priority)
{
    assert (eh);

    odk_event_handler_t *cur = l_list_first (odk->event_handlers);
    while (cur) {
        if (cur->event_handler == eh) {
            return;
        }
        cur = l_list_next (odk->event_handlers, cur);
    }

    odk_event_handler_t *handler = ho_new (odk_event_handler_t);
    handler->event_handler = eh;
    handler->event_handler_data = data;
    handler->priority = priority;

    l_list_append (odk->event_handlers, handler);
    l_list_sort (odk->event_handlers, swap_handler);
}


/* 
 * ***************************************************************************
 * Name:            odk_remove_event_handler
 * Access:          public
 *
 * Description:     Removes an event handler.
 * ***************************************************************************
 */
void
odk_del_event_handler (odk_t * odk, event_handler_t eh)
{
    assert (eh);

    odk_event_handler_t *cur = l_list_first (odk->event_handlers);
    while (cur) {
        if (cur->event_handler == eh) {
            l_list_remove (odk->event_handlers, cur);
            ho_free (cur);
            return;
        }
        cur = l_list_next (odk->event_handlers, cur);
    }
}


/* 
 * ***************************************************************************
 * Name:            odk_get_forward_events_to_xine
 * Access:          public
 *
 * Description:     Are events supposed to be forwarded to the xine engine.
 * ***************************************************************************
 */
bool
odk_get_forward_events_to_xine (odk_t * odk)
{
    return odk->forward_events_to_xine;
}


/* 
 * ***************************************************************************
 * Name:            odk_set_forward_events_to_xine
 * Access:          public
 *
 * Description:     Are events supposed to be forwarded to the xine engine.
 * ***************************************************************************
 */
void
odk_set_forward_events_to_xine (odk_t * odk, bool forward)
{
    odk->forward_events_to_xine = forward;
}


/* 
 * ***************************************************************************
 * Name:            odk_xine_event_send
 * Access:          public
 *
 * Description:     Send an event to the xine engine.
 * ***************************************************************************
 */
void
odk_xine_event_send (odk_t * odk, bool force, int type)
{
    if (!odk->forward_events_to_xine && !force)
        return;

    xine_event_t ev;
    ev.type = type;
    ev.data = NULL;
    ev.data_length = 0;
    xine_event_send (odk->win->stream, &ev);
}


/* 
 * ***************************************************************************
 * Name:            odk_oxine_event_send
 * Access:          public
 *
 * Description:     Sends an event to the registered event handler.
 * ***************************************************************************
 */
void
odk_oxine_event_send (odk_t * odk, oxine_event_t * event)
{
    odk_event_handler_t *handler = l_list_first (odk->event_handlers);
    while (handler && !((event->type == OXINE_EVENT_KEY)
                        && (event->source.key == OXINE_KEY_NULL))) {
        handler->event_handler (handler->event_handler_data, event);
        handler = l_list_next (odk->event_handlers, handler);
    }
}


/* 
 * ***************************************************************************
 * Name:            scale_mouse_position
 * Access:          private
 *
 * Description:     Scales the mouse position to the current video size.
 * ***************************************************************************
 */
static void
scale_mouse_position (odk_t * odk, int *x, int *y)
{
    /* Scale position. */
    x11_rectangle_t rect;
    rect.x = *x;
    rect.y = *y;
    rect.w = 0;
    rect.h = 0;

    if (xine_port_send_gui_data (odk->win->video_port,
                                 XINE_GUI_SEND_TRANSLATE_GUI_TO_VIDEO,
                                 (void *) &rect) == -1) {
        return;
    }

    *x = rect.x;
    *y = rect.y;
}


/* 
 * ***************************************************************************
 * Name:            odk_event_handler
 * Access:          private
 *
 * Description:     Handles events coming from the current video window and
 *                  the input plugins (LIRC, joystick).
 * ***************************************************************************
 */
static void
odk_event_handler (void *odk_p, oxine_event_t * event)
{
    odk_t *odk = (odk_t *) odk_p;

    switch (event->type) {
    case OXINE_EVENT_OUTPUT_FORMAT_CHANGED:
        odk_osd_adapt_size (odk, NULL);
        break;
    case OXINE_EVENT_BUTTON:
    case OXINE_EVENT_MOTION:
        /* Send an event to the xine engine. */
        if (odk->forward_events_to_xine) {
            int vx = event->data.pos.x;
            int vy = event->data.pos.y;
            scale_mouse_position (odk, &vx, &vy);

            xine_input_data_t inp;
            inp.x = vx;
            inp.y = vy;
            inp.button = event->source.button;

            xine_event_t xine_event;
            xine_event.type = event->type;
            xine_event.data = &inp;
            xine_event.data_length = sizeof (inp);

            xine_event_send (odk->win->stream, &xine_event);
        }

        /* Scale x and y to video size. */
        if (!odk->is_unscaled_osd_available) {
            int vx = event->data.pos.x;
            int vy = event->data.pos.y;
            scale_mouse_position (odk, &vx, &vy);

            event->data.pos.x = vx;
            event->data.pos.y = vy;
        }

        /* Scale x any y to OSD size. */
        {
            event->data.pos.x =
                (int) ((double) (event->data.pos.x - odk->osd.x_off) /
                       odk->osd.hscale);
            event->data.pos.y =
                (int) ((double) (event->data.pos.y - odk->osd.y_off) /
                       odk->osd.vscale);
        }

        break;
    case OXINE_EVENT_KEY:
        /* Send an event to the xine engine. */
        if (odk->forward_events_to_xine
            && (event->source.key < OXINE_KEY_NULL)) {
            xine_event_t xine_event;

            xine_event.stream = odk->win->stream;
            xine_event.type = event->source.key;
            xine_event.data = NULL;
            xine_event.data_length = 0;

#ifdef HAVE_VDR
            /* map vdr events */
            switch (xine_event.type) {
            case XINE_EVENT_INPUT_MENU2:
                if (odk_current_is_vdr (odk))
                    xine_event.type = XINE_EVENT_VDR_CHANNELS;
                break;
            case XINE_EVENT_INPUT_MENU3:
                if (odk_current_is_vdr (odk))
                    xine_event.type = XINE_EVENT_VDR_SCHEDULE;
                break;
            case XINE_EVENT_INPUT_MENU4:
                if (odk_current_is_vdr (odk))
                    xine_event.type = XINE_EVENT_VDR_RECORDINGS;
                break;
            case XINE_EVENT_INPUT_MENU5:
                if (odk_current_is_vdr (odk))
                    xine_event.type = XINE_EVENT_VDR_RED;
                break;
            case XINE_EVENT_INPUT_MENU6:
                if (odk_current_is_vdr (odk))
                    xine_event.type = XINE_EVENT_VDR_GREEN;
                break;
            case XINE_EVENT_INPUT_MENU7:
                if (odk_current_is_vdr (odk))
                    xine_event.type = XINE_EVENT_VDR_YELLOW;
                break;
            }
#endif

            xine_event_send (odk->win->stream, &xine_event);
        }
        break;
    default:
        break;
    }

    /* Send the event to the frontend. */
    odk_oxine_event_send (odk, event);
}


/* 
 * ***************************************************************************
 * Name:            xine_event_cb
 * Access:          private
 *
 * Description:     Handles events coming from the xine engine.
 * ***************************************************************************
 */
static void
xine_event_cb (void *odk_p, const xine_event_t * xine_event)
{
    odk_t *odk = (odk_t *) odk_p;

    switch (xine_event->type) {
    case XINE_EVENT_FRAME_FORMAT_CHANGE:
        odk_osd_adapt_size (odk, (xine_format_change_data_t *)
                            xine_event->data);
        break;
    case XINE_EVENT_UI_PLAYBACK_FINISHED:
        if (odk->current_mode == ODK_MODE_NULL)
            return;
        if (odk->current_mode == ODK_MODE_LOGO)
            return;
        if (xine_event->stream == odk->background_stream)
            return;
        if (odk_current_is_image (odk))
            return;

        /* Restart the animation stream */
        if (xine_event->stream == odk->animation_stream) {
            xine_play (odk->animation_stream, 0, 0);
            return;
        }

        /* The main stream has ended. */
        if (xine_event->stream == odk->win->stream) {
            /* If we have alternative streams waiting we play one of them. */
            if (playlist_length (odk->current_alternatives) > 0) {
                /* FIXME: We currently clear the alternatives when we stop
                 *        the current stream in odk_play. This means we
                 *        only try the first alternative. */
                playlist_t *alts = odk->current_alternatives;
                playitem_t *alt = playlist_first (alts);
                while (alt) {
                    oxine_event_t ev;
                    ev.type = OXINE_EVENT_WAITING_FOR_ALTERNATIVE;
                    odk_oxine_event_send (odk, &ev);
                    if (odk_play (odk, alt->mrl, NULL, odk->current_mode))
                        break;
                    alt = playlist_next (alts, alt);
                }

                /* If we were not able to play one of the alternative
                 * streams, we send a stop event. */
                if (!alt) {
                    oxine_event_t ev;
                    ev.type = OXINE_EVENT_PLAYBACK_ERROR;
                    odk_oxine_event_send (odk, &ev);
                }
            }

            /* Else we inform the frontend about the ending of the stream. */
            else {
                odk->current_mode = ODK_MODE_NULL;
                oxine_event_t ev;
                ev.type = xine_event->type;
                odk_oxine_event_send (odk, &ev);
            }
        }
        break;
    case XINE_EVENT_UI_CHANNELS_CHANGED:
        /* If current mode is still ODK_MODE_NULL we ignore this event, as
         * we're going to send a OXINE_EVENT_PLAYBACK_STARTED event after
         * setting the new mode in odk_mutexed_play anyway. */
        if (odk->current_mode == ODK_MODE_NULL)
            return;

        /* We use this event to find out if the meta info has changed
         * without a new track starting (e.g. web-radio). */
        if (xine_event->stream == odk->win->stream) {
            char *current_title = odk_get_meta_info (odk, META_INFO_TITLE);

            if (current_title && !odk->current_title) {
                odk->current_title = current_title;
                oxine_event_t ev;
                ev.type = OXINE_EVENT_PLAYBACK_STARTED;
                odk_oxine_event_send (odk, &ev);
            }

            else if (!current_title && odk->current_title) {
                ho_free (odk->current_title);
                odk->current_title = NULL;
                oxine_event_t ev;
                ev.type = OXINE_EVENT_PLAYBACK_STARTED;
                odk_oxine_event_send (odk, &ev);
            }

            else if (current_title
                     && odk->current_title
                     && (strcmp (current_title, odk->current_title) != 0)) {
                ho_free (odk->current_title);
                odk->current_title = current_title;
                oxine_event_t ev;
                ev.type = OXINE_EVENT_PLAYBACK_STARTED;
                odk_oxine_event_send (odk, &ev);
            }

            else if (current_title) {
                ho_free (current_title);
            }
        }
        break;
    case XINE_EVENT_PROGRESS:
#ifdef DEBUG
        if (xine_event->stream == odk->win->stream) {
            xine_progress_data_t *pevent =
                (xine_progress_data_t *) xine_event->data;
            debug ("%s [%d%%]", pevent->description, pevent->percent);
        }
#endif
        break;
    case XINE_EVENT_MRL_REFERENCE:
        /* If we receive an alternative stream, we add it to a list of
         * alternatives. When receiving the first alternative stream, we
         * send an event telling the frontend, that we are waiting for the
         * stream. */
        if (xine_event->stream == odk->win->stream) {
            xine_mrl_reference_data_t *ref =
                (xine_mrl_reference_data_t *) xine_event->data;

            debug ("Got alternative MRL [%d]: '%s'",
                   ref->alternative, ref->mrl);

            playlist_insert (odk->current_alternatives,
                             ref->alternative, ref->mrl, ref->mrl);
        }
        break;
    default:
        break;
    }
}


/* 
 * ***************************************************************************
 * Name:            odk_listen_to
 * Access:          public
 *
 * Description:     Tells odk to listen to an input plugin.
 * ***************************************************************************
 */
void
odk_listen_to (odk_t * odk, const char *id)
{
#ifdef HAVE_LIRC
    if (!strcmp (id, "lirc")) {
        if (!odk->lirc)
            odk->lirc = start_lirc (odk, odk_event_handler);
    }
#endif /* HAVE_LIRC */
#ifdef HAVE_JOYSTICK
    if (!strcmp (id, "joystick")) {
        if (!odk->joystick)
            odk->joystick = start_joystick (odk, odk_event_handler);
    }
#endif /* HAVE_JOYSTICK */
}


/* 
 * ***************************************************************************
 * Name:            odk_event_init
 * Access:          private
 *
 * Description:     Initialize ODK event handling.
 * ***************************************************************************
 */
void
odk_event_init (odk_t * odk)
{
#ifdef DEBUG
    assert (odk);
    assert (odk->win);
    assert (odk->win->stream);
    assert (odk->background_stream);
    assert (odk->animation_stream);
#endif

    odk->event_handlers = l_list_new ();
    odk->forward_events_to_xine = 0;
    odk->win->set_event_handler (odk->win, odk_event_handler, odk);

    xine_event_queue_t *q0 = xine_event_new_queue (odk->win->stream);
    xine_event_queue_t *q1 = xine_event_new_queue (odk->background_stream);
    xine_event_queue_t *q2 = xine_event_new_queue (odk->animation_stream);

    xine_event_create_listener_thread (q0, xine_event_cb, odk);
    xine_event_create_listener_thread (q1, xine_event_cb, odk);
    xine_event_create_listener_thread (q2, xine_event_cb, odk);
}


/* 
 * ***************************************************************************
 * Name:            odk_event_free
 * Access:          private
 *
 * Description:     Free ODK event handling.
 * ***************************************************************************
 */
void
odk_event_free (odk_t * odk)
{
#ifdef HAVE_LIRC
    if (odk->lirc)
        stop_lirc (odk->lirc);
    odk->lirc = NULL;
#endif
#ifdef HAVE_JOYSTICK
    if (odk->joystick)
        stop_joystick (odk->joystick);
    odk->joystick = NULL;
#endif
    if (odk->event_handlers)
        l_list_free (odk->event_handlers, event_handlers_destroy);
    odk->event_handlers = NULL;
}
