
/*
 * Copyright (C) 2002-2003 Stefan Holst
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: disc.h 2032 2007-05-14 17:18:57Z mschwerin $
 *
 */

#ifndef HAVE_DISC_H
#define HAVE_DISC_H
#ifdef __cplusplus
extern "C" {
#endif

#include "disc_hal.h"

#include "oxine.h"
#include "types.h"


/// The possible types of volumes oxine handles.
typedef enum {
    /// The device type is not known.
    VOLUME_TYPE_UNKNOWN,
    /// It's an audio CD.
    VOLUME_TYPE_AUDIO_CD,
    /// It's a (super) video CD.
    VOLUME_TYPE_VIDEO_CD,
    /// It's a video DVD.
    VOLUME_TYPE_VIDEO_DVD,
    /// It's a data CDROM.
    VOLUME_TYPE_CDROM,
    /// It's some other mountable volume.
    VOLUME_TYPE_MOUNTABLE,
    /// It's a blank disc (CD or DVD).
    VOLUME_TYPE_BLANK,
} volume_type_t;


/// Returns the mountpoint of the specified device.
/**
 * @param device                A device file.
 * @return                      The mountpoint of <code>device</code>. The
 *                              returned string must be freed.
 */
char *volume_get_mountpoint (const char *device);


/// Returns the title of the specified device.
/**
 * @param device                A device file.
 * @return                      The title of <code>device</code>. The returned
 *                              string must be freed.
 */
char *volume_get_title (const char *device);


/// Returns the type of the specified device.
/**
 * @param device                A device file.
 * @return                      The type of <code>device</code>.
 */
volume_type_t volume_get_type (const char *device);


/// Returns TRUE if the specified device is mounted.
/**
 * @param device                A device file.
 */
bool volume_is_mounted (const char *device);


/// Returns TRUE if the specified device is present.
/**
 * @param device                A device file.
 */
bool volume_is_present (const char *device);


/// Mounts the specified device.
/**
 * @param device                A device file.
 * @return                      TRUE if successfull.
 */
bool volume_mount (const char *device);


/// Unmounts the specified device.
/**
 * @param device                A device file.
 * @return                      TRUE if successfull.
 */
bool volume_umount (const char *device);


/// Returns the title of the specified device.
/**
 * @param device                A device file.
 * @return                      The title of <code>device</code>. The returned
 *                              string must be freed.
 */
char *drive_get_title (const char *device);


/// Returns TRUE if the specified device is ejected.
/**
 * @param device                A device file.
 */
bool drive_is_ejected (const char *device);


/// Eject the specified device.
/**
 * @param device                A device file.
 * @return                      TRUE if successfull.
 */
bool drive_eject (const char *device);


/// Injects the specified device.
/**
 * @param device                A device file.
 * @return                      TRUE if successfull.
 */
bool drive_inject (const char *device);


#ifdef __cplusplus
}
#endif
#endif
