
/*
 * Copyright (C) 2006 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: disc_hal.c 2578 2007-07-22 13:29:54Z mschwerin $
 *
 */
#include "config.h"

#include <assert.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <xine.h>

#include "disc.h"
#include "disc_hal.h"
#include "environment.h"
#include "extractor.h"
#include "heap.h"
#include "i18n.h"
#include "logger.h"
#include "odk.h"
#include "oxine.h"
#include "utils.h"

#ifdef HAVE_HAL

#include <glib.h>

#include <libhal.h>
#include <libhal-storage.h>

#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>

extern oxine_t *oxine;

static LibHalContext *hal_ctx = NULL;
static pthread_t thread;


/// Removes all items in the playlist that refer to a specific device or mountpoint.
static void
playlist_clean (playlist_t * playlist,
                const char *device, const char *mountpoint)
{
    playlist_lock (playlist);
    {
        playitem_t *item = playlist_first (playlist);
        while (item) {
            playitem_t *next = playlist_next (playlist, item);
            if (item->mrl) {
                if (device && strstr (item->mrl, device)) {
                    playlist_remove (playlist, item);
                }
                else if (mountpoint && strstr (item->mrl, mountpoint)) {
                    playlist_remove (playlist, item);
                }
            }
            item = next;
        }
    }
    playlist_unlock (playlist);
}


/// Recursively walks through a filelist and clears it.
static void
filelist_clean (filelist_t * filelist)
{
    if (!filelist)
        return;

    fileitem_t *item = filelist_first (filelist);
    while (item) {
        filelist_clean (item->child_list);
        item = filelist_next (filelist, item);
    }

    filelist_clear (filelist);
}


static bool
hal_add_drive (LibHalDrive * drive)
{
    LibHalDriveType drive_type = libhal_drive_get_type (drive);

    /* We're not interested in non-removable discs. */
    if ((drive_type == LIBHAL_DRIVE_TYPE_DISK)
        && !libhal_drive_is_hotpluggable (drive))
        return false;

    /* Who cares about floppies? */
    if (drive_type == LIBHAL_DRIVE_TYPE_FLOPPY)
        return false;

    const char *drive_udi = libhal_drive_get_udi (drive);
    const char *drive_device = libhal_drive_get_device_file (drive);
    char *drive_title = drive_get_title (drive_device);

    char *item_mrl = ho_strdup (drive_udi);
    char *item_title = ho_strdup_printf ("[%s: %s]", drive_title,
                                         _("No Disc"));
    fileitem_t *item = filelist_add (oxine->hal_volume_list, item_title,
                                     item_mrl, FILE_TYPE_DIRECTORY);
    item->device = ho_strdup (drive_device);
    item->drive_udi = ho_strdup (drive_udi);

    playlist_clean (oxine->ro_playlist, drive_device, NULL);
    playlist_clean (oxine->rw_playlist, drive_device, NULL);

    ho_free (drive_title);
    ho_free (item_mrl);
    ho_free (item_title);

    return true;
}


/**
 * Adds a volume to the list of volumes. The parameter autoplay_volume is
 * necessary, so we don't autoplay on startup.
 */
static bool
hal_add_volume (LibHalDrive * drive, LibHalVolume * volume,
                bool autoplay_volume)
{
    LibHalDriveType drive_type = libhal_drive_get_type (drive);

    /* We're not interested in non-removable discs. */
    if ((drive_type == LIBHAL_DRIVE_TYPE_DISK)
        && !libhal_drive_is_hotpluggable (drive))
        return false;

    /* Who cares about floppies? */
    if (drive_type == LIBHAL_DRIVE_TYPE_FLOPPY)
        return false;

    const char *drive_udi = libhal_drive_get_udi (drive);
    const char *volume_udi = libhal_volume_get_udi (volume);
    const char *volume_device = libhal_volume_get_device_file (volume);
    const char *volume_mountpoint = libhal_volume_get_mount_point (volume);

    volume_type_t vol_type = volume_get_type (volume_device);

    char *item_mrl = NULL;
    char *item_thumbnail = NULL;
    char *item_title = volume_get_title (volume_device);
    fileitem_type_t item_type = FILE_TYPE_UNKNOWN;
    switch (vol_type) {
    case VOLUME_TYPE_AUDIO_CD:
        item_mrl = ho_strdup_printf ("cdda://%s", volume_device);
        item_type = FILE_TYPE_CDDA_VFOLDER;
        item_thumbnail = ho_strdup (OXINE_VISUALDIR "/logo_device_cdda.png");
        break;
    case VOLUME_TYPE_VIDEO_CD:
        item_mrl = ho_strdup_printf ("vcd://%s", volume_device);
        item_type = FILE_TYPE_REGULAR;
        item_thumbnail = ho_strdup (OXINE_VISUALDIR "/logo_device_vcd.png");
        break;
    case VOLUME_TYPE_VIDEO_DVD:
        item_mrl = ho_strdup_printf ("dvd://%s", volume_device);
        item_type = FILE_TYPE_REGULAR;
        item_thumbnail = ho_strdup (OXINE_VISUALDIR "/logo_device_dvd.png");
        break;
    case VOLUME_TYPE_CDROM:
    case VOLUME_TYPE_MOUNTABLE:
        if (volume_mountpoint) {
            item_mrl = ho_strdup (volume_mountpoint);
        }
        else {
            item_mrl = ho_strdup (volume_udi);
        }
        item_type = FILE_TYPE_MOUNTPOINT;
        break;
    case VOLUME_TYPE_BLANK:
        item_mrl = ho_strdup (volume_udi);
        item_type = FILE_TYPE_REGULAR;
        break;
    default:
        warn ("Unknown device type '%s'!", volume_udi);
        goto out_free;
    }

    /* We always add the new drive to the HAL volume list. */
    fileitem_t *item = filelist_add (oxine->hal_volume_list, item_title,
                                     item_mrl, item_type);
    item->device = ho_strdup (volume_device);
    item->drive_udi = ho_strdup (drive_udi);
    item->volume_udi = ho_strdup (volume_udi);
    item->thumbnail_mrl = item_thumbnail;

    /* Depending on the current policy we play or do nothing. */
    int autoplay_action = 0;
    int autoplay_policy = config_get_number ("hal.auto.policy");
    playlist_t *playlist = oxine->ro_playlist;

    switch (vol_type) {
    case VOLUME_TYPE_AUDIO_CD:
        playlist = oxine->rw_playlist;
        autoplay_action = config_get_number ("hal.auto.action.cdda");
        break;
    case VOLUME_TYPE_VIDEO_CD:
        playlist = oxine->ro_playlist;
        autoplay_action = config_get_number ("hal.auto.action.vcd");
        break;
    case VOLUME_TYPE_VIDEO_DVD:
        playlist = oxine->ro_playlist;
        autoplay_action = config_get_number ("hal.auto.action.dvd");
        break;
    case VOLUME_TYPE_CDROM:
    case VOLUME_TYPE_MOUNTABLE:
        playlist = oxine->rw_playlist;
        autoplay_action =
            config_get_number ("hal.auto.action.volume.mountable");
        if (autoplay_action && autoplay_volume) {
            volume_mount (volume_device);
        }
        break;
    case VOLUME_TYPE_BLANK:
        break;
    default:
        assert (false);
        break;
    }

    if ((autoplay_action == 1) && autoplay_volume) {
        playlist_lock (playlist);
        if (odk_current_is_playback_mode (oxine->odk)) {
            /* Replace current playlist */
            if (autoplay_policy == 1) {
                playlist_clear (playlist);
                playlist_add_fileitem (playlist, item);
                playlist_play_first (playlist);
            }
            /* Add to end of playlist */
            else if (autoplay_policy == 2) {
                playlist_add_fileitem (playlist, item);
            }
        }
        else {
            playlist_clear (playlist);
            playlist_add_fileitem (playlist, item);
            playlist_play_first (playlist);
        }
        playlist_unlock (playlist);
    }
#ifdef HAVE_EXTRACTOR
    else if ((autoplay_action == 2) && autoplay_volume) {
        fileitem_t *items[1];
        items[0] = item;
        filelist_expand (item);
        extractor_start (1, items);
    }
#endif
    else {
        filelist_expand (item);
    }

  out_free:
    ho_free (item_mrl);
    ho_free (item_title);

    return true;
}


static fileitem_t *
find_item_by_hal_volume_udi (const char *udi)
{
    fileitem_t *item = filelist_first (oxine->hal_volume_list);
    while (item) {
        if (item->volume_udi && (strcmp (item->volume_udi, udi) == 0)) {
            return item;
        }
        item = filelist_next (oxine->hal_volume_list, item);
    }

    return NULL;
}


static fileitem_t *
find_item_by_hal_drive_udi (const char *udi)
{
    fileitem_t *item = filelist_first (oxine->hal_volume_list);
    while (item) {
        if (item->drive_udi && (strcmp (item->drive_udi, udi) == 0)) {
            return item;
        }
        item = filelist_next (oxine->hal_volume_list, item);
    }

    return NULL;
}


static void
hal_device_added (LibHalContext * hal_ctx, const char *udi)
{
    bool repaint = false;

    filelist_lock (oxine->hal_volume_list);

    /* If a device with the capability 'volume' (meaning it's a volume) is
     * added, we add a volume entry to the list. */
    if (libhal_device_query_capability (hal_ctx, udi, "volume", NULL)) {
        LibHalDrive *drive = NULL;
        LibHalVolume *volume = NULL;

        const char *volume_udi = udi;
        volume = libhal_volume_from_udi (hal_ctx, udi);
        if (!volume) {
            error (_("Failed to get volume from UDI '%s'!"), volume_udi);
            goto volume_add_out;
        }

        const char *drive_udi = libhal_volume_get_storage_device_udi (volume);
        drive = libhal_drive_from_udi (hal_ctx, drive_udi);
        if (!drive) {
            error (_("Failed to get drive from UDI '%s'!"), drive_udi);
            goto volume_add_out;
        }

        if (hal_add_volume (drive, volume, true)) {
            /* If there is an item in the list with the correct drive UDI, that
             * does not have a volume UDI, it must is removed. */
            fileitem_t *drive_item = find_item_by_hal_drive_udi (drive_udi);
            if (drive_item && !drive_item->volume_udi) {
                filelist_remove (oxine->hal_volume_list, drive_item);
            }

            debug ("A volume was added:");
            debug ("drive = '%s'", drive_udi);
            debug ("  volume = '%s'", udi);

            repaint = true;
        }

      volume_add_out:
        if (drive != NULL) {
            libhal_drive_free (drive);
        }
        if (volume != NULL) {
            libhal_volume_free (volume);
        }
    }

    /* If a device with the capability 'storage' (meaning it's a drive) is
     * added, we add a drive entry to the list. */
    else if (libhal_device_query_capability (hal_ctx, udi, "storage", NULL)) {
        const char *drive_udi = udi;
        LibHalDrive *drive = libhal_drive_from_udi (hal_ctx, drive_udi);
        if (!drive) {
            error (_("Failed to get drive from UDI '%s'!"), drive_udi);
            goto drive_add_out;
        }

        debug ("A drive was added:");
        debug ("drive = '%s'", drive_udi);

        if (hal_add_drive (drive)) {
            repaint = true;
        }

      drive_add_out:
        if (drive != NULL) {
            libhal_drive_free (drive);
        }
    }

    if (repaint) {
        filelist_sort (oxine->hal_volume_list, NULL);
    }

    filelist_unlock (oxine->hal_volume_list);

    if (repaint) {
        oxine_event_t ev;
        ev.type = OXINE_EVENT_GUI_REPAINT;
        odk_oxine_event_send (oxine->odk, &ev);
    }
}


static void
hal_device_removed (LibHalContext * hal_ctx, const char *udi)
{
    bool repaint = false;

    filelist_lock (oxine->hal_volume_list);

    fileitem_t *drive_item = find_item_by_hal_drive_udi (udi);
    fileitem_t *volume_item = find_item_by_hal_volume_udi (udi);

    /* If there is an entry in the list with this UDI as it's volume UDI
     * (meaning it was a volume that was removed), we remove that entry. */
    if (volume_item) {
        char *drive_udi = ho_strdup (volume_item->drive_udi);

        char *device = volume_item->device;
        char *mountpoint = volume_item->mrl;

        /* Remove any items from the playlists that were on this volume. */
        playlist_clean (oxine->ro_playlist, device, mountpoint);
        playlist_clean (oxine->rw_playlist, device, mountpoint);

        /* This is a bit of a nuissance! If we're currently displaying the
         * child_list of the item we're about to remove that child_list cannot
         * be freed as the reference counter is still > 0. But as the disc is
         * no longer in the drive we definitely do not want the user to see
         * it's contents anymore. So we have to clear it before removing this
         * item. */
        filelist_clean (volume_item->child_list);
        filelist_remove (oxine->hal_volume_list, volume_item);

        debug ("A volume was removed:");
        debug ("  volume = '%s'", udi);

        /* If there is no volume of this drive left in the list, we add the
         * drive. */
        if (!find_item_by_hal_drive_udi (drive_udi)) {
            LibHalDrive *drive = libhal_drive_from_udi (hal_ctx, drive_udi);
            if (drive) {
                hal_add_drive (drive);
                libhal_drive_free (drive);
            }
        }

        ho_free (drive_udi);
        repaint = true;
    }

    /* If there is no entry in the list with this UDI as it's volume UDI, but
     * there is an entry with this UDI as it's drive UDI (meaning it was a
     * drive that was removed), we remove that entry. */
    else if (drive_item) {
        filelist_remove (oxine->hal_volume_list, drive_item);

        debug ("A drive was removed:");
        debug ("  drive = '%s'", udi);

        repaint = true;
    }

    if (repaint) {
        filelist_sort (oxine->hal_volume_list, NULL);
    }

    filelist_unlock (oxine->hal_volume_list);

    if (repaint) {
        oxine_event_t ev;
        ev.type = OXINE_EVENT_GUI_REPAINT;
        odk_oxine_event_send (oxine->odk, &ev);
    }
}


static void
hal_device_property_modified (LibHalContext * hal_ctx,
                              const char *udi, const char *key,
                              dbus_bool_t is_removed, dbus_bool_t is_added)
{
    bool repaint = false;

    if (!libhal_device_query_capability (hal_ctx, udi, "volume", NULL))
        return;

    filelist_lock (oxine->hal_volume_list);

    LibHalVolume *volume = libhal_volume_from_udi (hal_ctx, udi);
    if (!volume) {
        error (_("Failed to get volume from UDI '%s'!"), udi);
        goto out_free;
    }

    /* Try to find the volume in our list of volumes. */
    fileitem_t *volume_item = find_item_by_hal_volume_udi (udi);
    if (!volume_item) {
        goto out_free;
    }

    /* When a volume is unmounted (but not removed) we do not remove it from
     * the list of removable devices. But we do have to clear it's child_list.
     * And we have to remove any items in the playlists that refer to this
     * volume. */
    if (!libhal_volume_is_mounted (volume)) {
        char *device = volume_item->device;
        char *mountpoint = volume_item->mrl;

        playlist_clean (oxine->ro_playlist, device, mountpoint);
        playlist_clean (oxine->rw_playlist, device, mountpoint);

        filelist_clean (volume_item->child_list);

        repaint = true;
    }

  out_free:
    if (volume) {
        libhal_volume_free (volume);
    }

    if (repaint) {
        filelist_sort (oxine->hal_volume_list, NULL);
    }

    filelist_unlock (oxine->hal_volume_list);

    if (repaint) {
        oxine_event_t ev;
        ev.type = OXINE_EVENT_GUI_REPAINT;
        odk_oxine_event_send (oxine->odk, &ev);
    }
}


static void
hal_update_all (void)
{
    DBusError error;
    dbus_error_init (&error);

    filelist_lock (oxine->hal_volume_list);

    int num_drives;
    char **drives = libhal_find_device_by_capability (hal_ctx, "storage",
                                                      &num_drives, &error);
    int i;
    for (i = 0; i < num_drives; i++) {
        debug ("drive = '%s'", drives[i]);
        LibHalDrive *drive = libhal_drive_from_udi (hal_ctx, drives[i]);

        if (drive == NULL) {
            continue;
        }

        int num_volumes_added = 0;
        int num_volumes;
        char **volumes = libhal_drive_find_all_volumes (hal_ctx, drive,
                                                        &num_volumes);
        int j;
        for (j = 0; j < num_volumes; j++) {
            debug ("  volume = '%s'", volumes[j]);
            LibHalVolume *volume = libhal_volume_from_udi (hal_ctx,
                                                           volumes[j]);
            if (hal_add_volume (drive, volume, false)) {
                num_volumes_added++;
            }

            libhal_volume_free (volume);
        }

        if (num_volumes_added == 0) {
            hal_add_drive (drive);
        }

        libhal_free_string_array (volumes);
        libhal_drive_free (drive);
    }
    libhal_free_string_array (drives);

    filelist_sort (oxine->hal_volume_list, NULL);

    filelist_unlock (oxine->hal_volume_list);

    oxine_event_t ev;
    ev.type = OXINE_EVENT_GUI_REPAINT;
    odk_oxine_event_send (oxine->odk, &ev);
}


static void *
hal_monitor_thread (void *data)
{
    info (_("Successfully started HAL monitor thread."));
    debug ("HAL monitor thread: 0x%X", (int) pthread_self ());

    DBusError error;
    DBusConnection *dbus_connection;

    if (hal_get_context () == NULL) {
        error (_("Failed to connect to HAL daemon!"));
        return false;
    }

    dbus_error_init (&error);
    dbus_connection = dbus_bus_get (DBUS_BUS_SYSTEM, &error);
    if (dbus_error_is_set (&error)) {
        error (_("Failed to connect to the D-BUS system bus: %s!"),
               error.message);
        dbus_error_free (&error);
        return false;
    }
    dbus_connection_setup_with_g_main (dbus_connection, NULL);

    libhal_ctx_set_dbus_connection (hal_ctx, dbus_connection);
    libhal_ctx_set_device_added (hal_ctx, hal_device_added);
    libhal_ctx_set_device_removed (hal_ctx, hal_device_removed);
    libhal_ctx_set_device_property_modified (hal_ctx,
                                             hal_device_property_modified);

    if (!libhal_ctx_init (hal_ctx, &error)) {
        error ("Failed to initialize HAL context: %s!", error.message);
        dbus_error_free (&error);
        return false;
    }
    libhal_device_property_watch_all (hal_ctx, &error);

    hal_update_all ();

    GMainLoop *loop = g_main_loop_new (NULL, FALSE);
    g_main_loop_run (loop);

    pthread_exit (NULL);
    return NULL;
}


bool
hal_monitor_free (void)
{
    DBusError error;

    dbus_error_init (&error);
    if (hal_ctx && !libhal_ctx_shutdown (hal_ctx, &error)) {
        error (_("Failed to shutdown HAL connection: %s!"), error.message);
        dbus_error_free (&error);
        return false;
    }

    if (hal_ctx && !libhal_ctx_free (hal_ctx)) {
        error (_("Unable to free HAL context!"));
        return false;
    }

    hal_ctx = NULL;

    info (_("Successfully stopped HAL monitor thread."));

    return true;
}


bool
hal_monitor_start (void)
{
    if (pthread_create (&thread, NULL, hal_monitor_thread, NULL) != 0) {
        error (_("Could not create HAL monitor thread: %s!"),
               strerror (errno));
    }

    return true;
}


LibHalContext *
hal_get_context (void)
{
    if (!hal_ctx) {
        hal_ctx = libhal_ctx_new ();
    }

    return hal_ctx;
}


char *
hal_volume_get_udi_from_device (const char *device)
{
    if (hal_get_context () == NULL) {
        error (_("Failed to connect to HAL daemon!"));
        return NULL;
    }

    LibHalVolume *volume = libhal_volume_from_device_file (hal_ctx, device);
    if (!volume) {
        error (_("Failed to get volume from device '%s'!"), device);
        return NULL;
    }

    const char *udi = libhal_volume_get_udi (volume);
    char *res = udi ? ho_strdup (udi) : NULL;
    libhal_volume_free (volume);

    return res;
}


char *
hal_volume_get_label_from_device (const char *device)
{
    if (hal_get_context () == NULL) {
        error (_("Failed to connect to HAL daemon!"));
        return NULL;
    }

    LibHalVolume *volume = libhal_volume_from_device_file (hal_ctx, device);
    if (!volume) {
        error (_("Failed to get volume from device '%s'!"), device);
        return NULL;
    }

    const char *label = libhal_volume_get_label (volume);
    char *res = label ? ho_strdup (label) : NULL;
    libhal_volume_free (volume);

    return res;
}


bool
hal_volume_is_mountable (const char *device)
{
    if (hal_get_context () == NULL) {
        error (_("Failed to connect to HAL daemon!"));
        return false;
    }

    LibHalVolume *volume = libhal_volume_from_device_file (hal_ctx, device);
    if (!volume) {
        error (_("Failed to get volume from device '%s'!"), device);
        return false;
    }

    bool res = false;
    if (libhal_volume_get_fsusage (volume) ==
        LIBHAL_VOLUME_USAGE_MOUNTABLE_FILESYSTEM)
        res = true;

    libhal_volume_free (volume);

    return res;
}


bool
hal_volume_is_mounted (const char *device)
{
    if (hal_get_context () == NULL) {
        error (_("Failed to connect to HAL daemon!"));
        return false;
    }

    LibHalVolume *volume = libhal_volume_from_device_file (hal_ctx, device);
    if (!volume) {
        error (_("Failed to get volume from device '%s'!"), device);
        return false;
    }

    bool res = libhal_volume_is_mounted (volume);

    libhal_volume_free (volume);

    return res;
}

#endif /* HAVE_HAL */
