
/*
 * Copyright (C) 2006 Joel Klinghed
 * Copyright (C) 2007 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: disc_image.c 2637 2007-08-17 12:44:34Z mschwerin $
 *
 */
#include "config.h"

#include <dirent.h>
#include <errno.h>
#include <stddef.h>
#include <stdio.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "disc_image.h"
#include "filelist.h"
#include "heap.h"
#include "i18n.h"
#include "logger.h"
#include "oxine.h"
#include "utils.h"

#ifdef HAVE_LIBCDIO

#include <cdio/cdio.h>
#include <cdio/cd_types.h>
#include <cdio/iso9660.h>

typedef enum {
    IMG_UNKNOWN,
    IMG_ERROR,
    IMG_AUDIO,
    IMG_DVD,
    IMG_DATA,
    IMG_MIXED,
    IMG_VCD,
    IMG_SVCD,
    IMG_CVD
} img_type_t;

static img_type_t read_img (const char *filename, driver_id_t driver,
                            char **volume_id);
static img_type_t read_iso (const char *filename, char **volume_id);

void
disc_image_detect (filelist_t * filelist, const char *mrl)
{
    bool try_iso = true;
    img_type_t img_type = IMG_UNKNOWN;
    char *add_mrl = NULL;
    char *add_title = NULL;;
    char *mrl_title = NULL;
    char *volume_id = NULL;

    if (!mrl)
        return;
    if (!file_exists (mrl))
        return;

    /* If it's a .bin file we try to find the .cue file that belongs. If we
     * find that, we exit because the .cue file will be found and used anyway. */
    if (has_extension (mrl, "bin")) {
        char *cuefile = ho_strdup (mrl);
        memcpy (cuefile + strlen (cuefile) - 3, "cue", 3);
        if (file_exists (cuefile)) {
            ho_free (cuefile);
            return;
        }
        else {
            ho_free (cuefile);
            warn ("Could not find .CUE for '%s'!", mrl);
        }
    }

    if (has_extension (mrl, "toc")) {
        img_type = read_img (mrl, DRIVER_CDRDAO, &volume_id);
        try_iso = false;
    }

    else if (has_extension (mrl, "cue")) {
        img_type = read_img (mrl, DRIVER_BINCUE, &volume_id);
        try_iso = false;
    }

    else if (has_extension (mrl, "nrg")) {
        img_type = read_img (mrl, DRIVER_NRG, &volume_id);
    }

    if ((img_type == IMG_UNKNOWN) || (img_type == IMG_ERROR)) {
        if (try_iso) {
            img_type = read_iso (mrl, &volume_id);
        }
    }

    /* This is a hack! As many distributions have version of libcdio that do
     * not support large files DVD's may not be correctly recognized. To
     * circumvent this we assume that any disc image that is larger than 1GB
     * is a video DVD. */
    if ((img_type == IMG_UNKNOWN) || (img_type == IMG_ERROR)) {
        off_t size = get_filesize (mrl);
        if (size > 1024 * 1024 * 1024) {
            img_type = IMG_DVD;
        }
    }

    mrl_title = create_title (mrl);

    switch (img_type) {
    case IMG_ERROR:
    case IMG_UNKNOWN:
    case IMG_MIXED:
    case IMG_CVD:
    case IMG_DATA:
        warn (_("Unable to determine type of disc image '%s'."), mrl);
        break;
    case IMG_AUDIO:
        add_mrl = ho_strdup_printf ("cdda://%s", mrl);
        add_title = ho_strdup_printf ("[%s: %s]", _("Audio ISO"), mrl_title);
        break;
    case IMG_VCD:
    case IMG_SVCD:
        add_mrl = ho_strdup_printf ("vcd://%s", mrl);
        add_title = ho_strdup_printf ("[%s: %s]", _("Video CD ISO"),
                                      volume_id ? volume_id : mrl_title);
        break;
    case IMG_DVD:
        add_mrl = ho_strdup_printf ("dvd://%s", mrl);
        add_title = ho_strdup_printf ("[%s: %s]", _("Video DVD ISO"),
                                      volume_id ? volume_id : mrl_title);
        break;
    }

    if (add_mrl && add_title
        && is_file_allowed (add_mrl, filelist->allowed_filetypes)) {
        filelist_add (filelist, add_title, add_mrl, FILE_TYPE_REGULAR);
    }

    ho_free (add_mrl);
    ho_free (add_title);
    ho_free (mrl_title);
    ho_free (volume_id);
}


img_type_t
read_img (const char *filename, driver_id_t driver, char **volume_id)
{
    CdIo_t *cd;
    int first_track;
    int tracks;
    int num_audio = 0;
    int num_data = 0;
    int first_audio = -1;
    int first_data = -1;
    int t;
    iso9660_pvd_t pvd;
    img_type_t img_type = IMG_UNKNOWN;

    memset (&pvd, 0, sizeof (iso9660_pvd_t));

    cd = cdio_open (filename, driver);
    if (!cd) {
        error (_("Could not open '%s': %s!"), filename, _("Unknown error"));
        return IMG_ERROR;
    }

    first_track = cdio_get_first_track_num (cd);
    tracks = cdio_get_num_tracks (cd);

    for (t = first_track; t <= tracks; t++) {
        if (cdio_get_track_format (cd, t) == TRACK_FORMAT_AUDIO) {
            ++num_audio;
            if (first_audio == -1)
                first_audio = t;
        }
        else {
            ++num_data;
            if (first_data == -1)
                first_data = t;
        }
    }

    if (num_data == 0) {
        if (num_audio == 0) {
            debug ("Found no tracks on image %s.", filename);
            img_type = IMG_ERROR;
        }
        else {
            /* only audio tracks == AUDIO CD */
            img_type = IMG_AUDIO;
        }
    }
    else {
        lsn_t start_track;
        lsn_t data_start = 0;
        cdio_fs_anal_t fs = 0;
        cdio_iso_analysis_t cdio_iso_analysis;
        bool got_pvd = false;
        bool got_type = false;

        memset (&cdio_iso_analysis, 0, sizeof (cdio_iso_analysis));

        for (t = first_data; t <= tracks; t++) {
            lsn_t lsn;
            track_format_t track_format;

            track_format = cdio_get_track_format (cd, t);

            switch (track_format) {
            case TRACK_FORMAT_AUDIO:
            case TRACK_FORMAT_ERROR:
                /* skip track */
                continue;
            case TRACK_FORMAT_CDI:
            case TRACK_FORMAT_XA:
            case TRACK_FORMAT_DATA:
            case TRACK_FORMAT_PSX:
                break;
            }

            lsn = cdio_get_track_lsn (cd, t);
            start_track = (t == 1) ? 0 : lsn;

            if (t == first_data) {
                data_start = start_track;
            }

            if (!got_pvd) {
                if (!iso9660_fs_read_pvd (cd, &pvd)) {
                    /* Track didn't have PVD - it's OK tho, maybe next has */
                }
                else {
                    *volume_id = ho_malloc (ISO_MAX_VOLUME_ID + 1);
                    memcpy (*volume_id, pvd.volume_id, ISO_MAX_VOLUME_ID);
                    (*volume_id)[ISO_MAX_VOLUME_ID] = '\0';
                    trim_whitespace (*volume_id);
                    got_pvd = true;
                }
            }

            /* skip tracks which belong to the current walked session */
            if (start_track < data_start + cdio_iso_analysis.isofs_size)
                continue;

            fs = cdio_guess_cd_type (cd, start_track, t, &cdio_iso_analysis);

            if ((fs & CDIO_FS_ANAL_VIDEOCD) && (num_audio == 0)) {
                img_type = IMG_VCD;
                got_type = true;
            }
            else if ((fs & CDIO_FS_ANAL_SVCD)) {
                img_type = IMG_SVCD;
                got_type = true;
            }
            else if ((fs & CDIO_FS_ANAL_CVD)) {
                img_type = IMG_CVD;
                got_type = true;
            }

            if (!((CDIO_FSTYPE (fs) == CDIO_FS_ISO_9660)
                  || (CDIO_FSTYPE (fs) == CDIO_FS_ISO_HFS)
                  || (CDIO_FSTYPE (fs) == CDIO_FS_ISO_9660_INTERACTIVE)))
                break;

            if (got_type && got_pvd)
                break;
        }

        if (!got_type) {
            if (num_audio > 0) {
                img_type = IMG_MIXED;
            }
            else {
                img_type = IMG_DATA;
            }
        }

        if (!got_pvd) {
            *volume_id = ho_strdup ("");
        }
    }

    cdio_destroy (cd);

    return img_type;
}


img_type_t
read_iso (const char *filename, char **volume_id)
{
    iso9660_t *iso;
    iso9660_pvd_t pvd;
    iso9660_stat_t *stat;
    img_type_t img_type = IMG_UNKNOWN;

    memset (&pvd, 0, sizeof (iso9660_pvd_t));

    iso = iso9660_open_fuzzy_ext (filename, ISO_EXTENSION_NONE, 16);
    if (!iso) {
        error (_("Could not open '%s': %s!"), filename, _("Unknown error"));
        return IMG_ERROR;
    }

    if (!iso9660_ifs_read_pvd (iso, &pvd)) {
        debug ("Could not read PVD in '%s'.", filename);
        iso9660_close (iso);
        return IMG_UNKNOWN;
    }

    if ((stat = iso9660_ifs_stat (iso, "VIDEO_TS/VIDEO_TS.IFO;1"))) {
        img_type = IMG_DVD;
    }
    else if ((stat = iso9660_ifs_stat (iso, "VCD/INFO.VCD;1"))) {
        img_type = IMG_VCD;
    }
    else if ((stat = iso9660_ifs_stat (iso, "SVCD/INFO.VCD;1"))) {
        img_type = IMG_SVCD;
    }
    else if ((stat = iso9660_ifs_stat (iso, "SVCD/INFO.SVD;1"))) {
        img_type = IMG_SVCD;
    }
    else {
        img_type = IMG_DATA;
    }

    *volume_id = ho_malloc (ISO_MAX_VOLUME_ID + 1);
    memcpy (*volume_id, pvd.volume_id, ISO_MAX_VOLUME_ID);
    (*volume_id)[ISO_MAX_VOLUME_ID] = '\0';
    trim_whitespace (*volume_id);

    free (stat);
    iso9660_close (iso);

    return img_type;
}

#endif /* HAVE_LIBCDIO */
