
/*
 * Copyright (C) 2002-2003 Stefan Holst
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: mediamarks.c 2606 2007-07-25 09:51:03Z mschwerin $
 *
 */
#include "config.h"

#include <errno.h>
#include <fcntl.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "codeset.h"
#include "environment.h"
#include "filelist.h"
#include "heap.h"
#include "i18n.h"
#include "logger.h"
#include "mediamarks.h"
#include "utils.h"
#include "xmlparser.h"

#define OXINE_MEDIAMARKS_XML    "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n"
#define OXINE_MEDIAMARKS_DTD    "<!DOCTYPE oxine_mediamarks SYSTEM \"http://oxine.sf.net/DTD/mediamarks.dtd\">\n\n"


static void
read_entry (const char *xml_mrl, filelist_t * filelist, xml_node_t * node)
{
    char *title = NULL;
    char *mrl = NULL;
    char *thumbnail_mrl = NULL;
    fileitem_type_t type = FILE_TYPE_UNKNOWN;
    int listened_count = 0;

    char *codeset = get_system_encoding ();
    recode_t *xr = recode_init ("utf8", codeset);

    while (node) {
        if (strcasecmp (node->name, "title") == 0) {
            title = recode (xr, node->data);
        }

        else if (strcasecmp (node->name, "thumbnail") == 0) {
            thumbnail_mrl =
                recode (xr, xml_parser_get_property (node, "href"));
        }

        else if (strcasecmp (node->name, "mrl") == 0) {
            mrl = recode (xr, xml_parser_get_property (node, "href"));

            const char *entry_type = xml_parser_get_property (node, "type");
            if (entry_type) {
                if (strcasecmp (entry_type, "regular") == 0) {
                    type = FILE_TYPE_REGULAR;
                }

                else if (strcasecmp (entry_type, "directory") == 0) {
                    type = FILE_TYPE_DIRECTORY;
                }

                else if (strcasecmp (entry_type, "mountpoint") == 0) {
                    type = FILE_TYPE_MOUNTPOINT;
                }

                else if (strcasecmp (entry_type, "mediamarks") == 0) {
                    type = FILE_TYPE_MEDIAMARKS;
                }

                else if (strcasecmp (entry_type, "playlist_m3u") == 0) {
                    type = FILE_TYPE_PLAYLIST_M3U;
                }

                else if (strcasecmp (entry_type, "playlist_pls") == 0) {
                    type = FILE_TYPE_PLAYLIST_PLS;
                }
            }
        }

        else if (strcasecmp (node->name, "listened_count") == 0) {
            listened_count = atoi (node->data);
        }

        else {
            error (_("Unknown name '%s' for node!"), node->name);
        }

        node = node->next;
    }

    recode_done (xr);
    ho_free (codeset);

    if (mrl) {
        fileitem_t *fileitem = filelist_add (filelist, title, mrl, type);
        fileitem->listened_count = listened_count;
        if (thumbnail_mrl) {
            fileitem->thumbnail_mrl = ho_strdup (thumbnail_mrl);
        }
    }
    else {
        error (_("Found an entry that did not contain a MRL!"));
    }

    ho_free (mrl);
    ho_free (title);
    ho_free (thumbnail_mrl);
}


static void
read_nodes (const char *xml_mrl, filelist_t * filelist, xml_node_t * node)
{
    while (node) {
        if (!strcasecmp (node->name, "entry")) {
            read_entry (xml_mrl, filelist, node->child);
        }

        else {
            error (_("Unknown name '%s' for node!"), node->name);
        }
        node = node->next;
    }
}


static bool
_mediamarks_read (filelist_t * filelist, const char *rootnode_name,
                  l_swap_cb_t swap_cb)
{
    bool result = false;
    bool sort = true;

    int size = 0;
    char *buffer = read_entire_file (filelist->mrl, &size);
    if (!buffer) {
        goto buffer_free;
    }

    xml_node_t *rootnode;
    xml_parser_init (buffer, size, XML_PARSER_CASE_INSENSITIVE);
    if (xml_parser_build_tree (&rootnode) < 0) {
        error (_("Parsing '%s' failed!"), filelist->mrl);
        goto buffer_free;
    }

    if (strcasecmp (rootnode->name, rootnode_name) != 0) {
        error (_("Root node of '%s' must be '%s'!"),
               filelist->mrl, rootnode_name);
        goto parser_free;
    }

    const char *str_sort = xml_parser_get_property (rootnode, "sort");
    if (str_sort && (strcasecmp (str_sort, "false") == 0)) {
        sort = false;
    }

    read_nodes (filelist->mrl, filelist, rootnode->child);

    if (sort) {
        filelist_sort (filelist, swap_cb);
    }

    result = true;
  parser_free:
    xml_parser_free_tree (rootnode);
  buffer_free:
    ho_free (buffer);

    return result;
}


bool
mediamarks_read (filelist_t * filelist)
{
    return _mediamarks_read (filelist, "oxine_mediamarks", NULL);
}


bool
mediamarks_write (filelist_t * filelist)
{
    FILE *f = fopen (filelist->mrl, "w");
    if (f == NULL) {
        error (_("Could not open '%s': %s!"),
               filelist->mrl, strerror (errno));
        return false;
    }

    char *codeset = get_system_encoding ();
    recode_t *xr = recode_init (codeset, "utf8");

    fprintf (f, OXINE_MEDIAMARKS_XML);
    fprintf (f, OXINE_MEDIAMARKS_DTD);

    fprintf (f, "<!--\n\n");
    fprintf (f, "This file was automatically created by oxine.\n");
    fprintf (f, "If you want to add other entries feel free to do so.\n\n");
    fprintf (f, "An example mediamarks file can be found at\n");
    fprintf (f, "%s/mediamarks_example.xml\n\n", OXINE_DATADIR);
    fprintf (f, "-->\n\n");

    fprintf (f, "<oxine_mediamarks>\n");
    fileitem_t *item = filelist_first (filelist);
    while (item) {
        if (item->type == FILE_TYPE_DIRECTORY) {
            char *esc_str0 = xml_escape (item->title,
                                         xr, XML_ESCAPE_NO_QUOTE);
            char *esc_str1 = xml_escape (item->mrl,
                                         xr, XML_ESCAPE_DOUBLE_QUOTE);
            fprintf (f, "  <entry>\n");
            fprintf (f, "    <title>%s</title>\n", esc_str0);
            fprintf (f, "    <mrl href=\"%s\" />\n", esc_str1);
            fprintf (f, "  </entry>\n");
            ho_free (esc_str0);
            ho_free (esc_str1);
        }

        item = filelist_next (filelist, item);
    }
    fprintf (f, "</oxine_mediamarks>\n");
    fclose (f);

    recode_done (xr);
    ho_free (codeset);

    return true;
}


static bool
favorites_swap_cb (void *d1, void *d2)
{
    fileitem_t *f1 = (fileitem_t *) d1;
    fileitem_t *f2 = (fileitem_t *) d2;

    return (f2->listened_count > f1->listened_count);
}


bool
favorites_read (filelist_t * filelist)
{
    return _mediamarks_read (filelist, "oxine_favorites", favorites_swap_cb);
}
