
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: menu_filelist.c 2603 2007-07-25 08:37:30Z mschwerin $
 *
 */
#include "config.h"

#include <assert.h>
#include <errno.h>
#include <math.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include "disc.h"
#include "download.h"
#include "environment.h"
#include "extractor.h"
#include "filelist.h"
#include "heap.h"
#include "i18n.h"
#include "logger.h"
#include "mediamarks.h"
#include "mediamarks_favorites.h"
#include "meta_info.h"
#include "oxine.h"
#include "playlist.h"
#include "shoutcast.h"
#include "scheduler.h"
#include "utils.h"
#include "vdr.h"
#include "youtube.h"

#include "menu_base.h"
#include "menu_extractor.h"
#include "menu_filelist.h"
#include "menu_main.h"
#include "menu_playback.h"
#include "menu_playlist.h"

extern oxine_t *oxine;

static otk_cb_t show_last_filelist_menu_cb = show_menu_media;

static l_list_t *prev_filelist = NULL;
static l_list_t *next_filelist = NULL;

static filelist_t *current_filelist = NULL;
static filelist_t *toplevel_filelist = NULL;
static filelist_t *extras_filelist = NULL;
static filelist_t *favorites_filelist = NULL;

static otk_widget_t *button_up = NULL;
static otk_widget_t *button_prev = NULL;
static otk_widget_t *button_next = NULL;
static otk_widget_t *button_home = NULL;

static int thumbnail_job = 0;

#define FILELIST_BUTTONS_NUM 8
static otk_widget_t *filelist_menu_buttons[FILELIST_BUTTONS_NUM];
static otk_widget_t *filelist_menu_list = NULL;
static otk_widget_t *filelist_menu_window = NULL;
static otk_widget_t *filelist_menu_title = NULL;
#ifdef HAVE_OSD_IMAGE
static otk_widget_t *filelist_menu_image_large = NULL;
static otk_widget_t *filelist_menu_image_small = NULL;
#endif

/*
 * ***************************************************************************
 * Some prototypes of methods declared in this file
 * ***************************************************************************
 */
static void filelist_menu_update_list (void);
static void filelist_menu_update_buttons (void);
static void filelist_menu_update_title (void);

static void filelist_menu_set_filelist (filelist_t * newlist);
static bool filelist_menu_get_selected (fileitem_t *** items,
                                        int *num_selected);

#ifdef HAVE_OSD_IMAGE
static void filelist_menu_thumbnail_job (void *entry_cb_data);
#endif

static void filelist_focus_enter_cb (void *entry_cb_data);
static void filelist_focus_leave_cb (void *entry_cb_data);

static void filelist_select_cb (void *entry_cb_data);
static void filelist_activate_cb (void *entry_cb_data);
static void filelist_remove_cb (void *entry_cb_data);

static void filelist_playall_cb (void *entry_cb_data);
static void filelist_playall_recursive_cb (void *entry_cb_data);

static void add_selected_cb (void *p);
static void play_selected_cb (void *p);
static void remove_selected_cb (void *p);
static void copy_selected_cb (void *p);


static bool
current_is_toplevel (void)
{
    return (current_filelist == toplevel_filelist);
}


static bool
current_is_favorites (void)
{
    return (current_filelist == favorites_filelist);
}


static bool
current_is_playlists (void)
{
    return (strcmp (current_filelist->mrl, get_dir_oxine_playlists ()) == 0);
}


#ifdef HAVE_VDR
static bool
current_is_vdr_recordings (void)
{
    fileitem_t *parent = current_filelist->parent_item;
    if (!parent) {
        return false;
    }
    return (parent->type == FILE_TYPE_VDR_RECORDINGS_VFOLDER);
}


static bool
current_is_vdr_timers (void)
{
    fileitem_t *parent = current_filelist->parent_item;
    if (!parent) {
        return false;
    }
    return (parent->type == FILE_TYPE_VDR_TIMERS_VFOLDER);
}
#endif


#ifdef HAVE_SHOUTCAST
static bool
current_is_shoutcast (void)
{
    fileitem_t *parent = current_filelist->parent_item;
    if (!parent) {
        return false;
    }
    return (parent->type == FILE_TYPE_SHOUTCAST_VFOLDER);
}
#endif


#ifdef HAVE_YOUTUBE
static bool
current_is_youtube (void)
{
    fileitem_t *parent = current_filelist->parent_item;
    if (!parent) {
        return false;
    }
    return (parent->type == FILE_TYPE_YOUTUBE_VFOLDER);
}
#endif


/*
 * ***************************************************************************
 * Methods that are part of the filelist GUI.
 * ***************************************************************************
 */
static void
filelist_menu_update_buttons (void)
{
    if (!filelist_menu_is_current_menu ()) {
        return;
    }

    bool has_prev = (l_list_length (prev_filelist) > 0);
    bool has_next = (l_list_length (next_filelist) > 0);

    otk_widget_set_enabled (button_up, !current_is_toplevel ());
    otk_widget_set_enabled (button_home, !current_is_toplevel ());
    otk_widget_set_enabled (button_prev, has_prev);
    otk_widget_set_enabled (button_next, has_next);

    int num_selected;
    fileitem_t **items;
    filelist_menu_get_selected (&items, &num_selected);

    /* Play the selected items */
#ifdef HAVE_IMAGE_PLAYBACK
    if (is_current_menu (show_menu_image)) {
        otk_button_set_text (filelist_menu_buttons[0], _("View"));
    }
    else {
        otk_button_set_text (filelist_menu_buttons[0], _("Play"));
    }
#endif
    {
        bool enabled = (num_selected > 0);
        enabled &= (current_filelist != toplevel_filelist);
#ifdef HAVE_VDR
        enabled &= !current_is_vdr_timers ();
#endif
        otk_widget_set_enabled (filelist_menu_buttons[0], enabled);
    }

    /* Load saved playlists */
    if (current_is_playlists ()) {
        bool enabled = (num_selected == 1);
        otk_widget_set_enabled (filelist_menu_buttons[1], enabled);
        otk_button_set_text (filelist_menu_buttons[1], _("Load"));
    }

    /* Add to playlist */
    else {
        bool enabled = (num_selected > 0);
        enabled &= (current_filelist != toplevel_filelist);
#ifdef HAVE_SHOUTCAST
        enabled &= !current_is_shoutcast ();
#endif
#ifdef HAVE_VDR
        enabled &= !current_is_vdr_recordings ();
        enabled &= !current_is_vdr_timers ();
#endif
#ifdef HAVE_YOUTUBE
        enabled &= !current_is_youtube ();
#endif
        otk_widget_set_enabled (filelist_menu_buttons[1], enabled);
        otk_button_set_text (filelist_menu_buttons[1], _("Add"));
    }

    /* Reset the favorites list */
    if (current_is_favorites ()) {
        bool enabled = (filelist_length (current_filelist) > 0);
        otk_widget_set_enabled (filelist_menu_buttons[2], enabled);
        otk_button_set_text (filelist_menu_buttons[2], _("Reset"));
    }

    /* Remove saved playlists */
    else if (current_is_playlists ()) {
        bool enabled = (num_selected > 0);
        otk_widget_set_enabled (filelist_menu_buttons[2], enabled);
        otk_button_set_text (filelist_menu_buttons[2], _("Remove"));
    }

#ifdef HAVE_VDR
    /* Remove VDR recordings */
    else if (current_is_vdr_recordings ()) {
        bool enabled = (num_selected > 0);
        otk_widget_set_enabled (filelist_menu_buttons[2], enabled);
        otk_button_set_text (filelist_menu_buttons[2], _("Remove"));
    }

    /* Remove VDR timers */
    else if (current_is_vdr_timers ()) {
        bool enabled = (num_selected > 0);
        otk_widget_set_enabled (filelist_menu_buttons[2], enabled);
        otk_button_set_text (filelist_menu_buttons[2], _("Remove"));
    }
#endif

    /* Remove regular files */
    else {
        bool enabled = (num_selected > 0);
        enabled &= !current_is_toplevel ();
#ifdef HAVE_SHOUTCAST
        enabled &= !current_is_shoutcast ();
#endif

        /**
         * @todo Implement removing for regular files.
         */
        enabled = false;
        otk_widget_set_enabled (filelist_menu_buttons[2], enabled);
        otk_button_set_text (filelist_menu_buttons[2], _("Remove"));
    }

    /* Extract DVD's, video and audio CD's */
#ifdef HAVE_HAL
    int i;
    bool contains_disc = false;
    for (i = 0; i < num_selected; i++) {
        contains_disc |= starts_with (items[i]->mrl, "dvd:");
        contains_disc |= starts_with (items[i]->mrl, "vcd:");
        contains_disc |= starts_with (items[i]->mrl, "cdda:");
    }
    if (contains_disc && (num_selected == 1)) {
        otk_widget_set_enabled (filelist_menu_buttons[3], true);
        otk_button_set_text (filelist_menu_buttons[3], _("Extract"));
    }
    else
#endif
    {
        bool enabled = (num_selected > 0);
        enabled &= !current_is_toplevel ();
        enabled &= !current_is_favorites ();
        enabled &= !current_is_playlists ();
#ifdef HAVE_VDR
        enabled &= !current_is_vdr_recordings ();
        enabled &= !current_is_vdr_timers ();
#endif
#ifdef HAVE_SHOUTCAST
        enabled &= !current_is_shoutcast ();
#endif

        /**
         * @todo Implement copying for regular files.
         */
        enabled = false;
        otk_widget_set_enabled (filelist_menu_buttons[3], enabled);
        otk_button_set_text (filelist_menu_buttons[3], _("Copy"));
    }

    /* Current title */
    {
        bool enabled = odk_current_is_playback_mode (oxine->odk);
        otk_widget_set_visible (filelist_menu_buttons[7], enabled);
#ifdef HAVE_OSD_IMAGE
        otk_widget_set_visible (filelist_menu_image_small, enabled);
        otk_widget_set_visible (filelist_menu_image_large, !enabled);
#endif
    }

    ho_free (items);
}


static void
filelist_menu_addto_list (fileitem_t * item)
{
    otk_widget_t *w;

    char *device = NULL;
#ifdef HAVE_HAL
    if (item->device) {
        device = item->device;
    }
#endif

    /* If this entry refers to a DVD and is not an image on disc, we use the
     * DVD disc callback. */
    if (starts_with (item->mrl, "dvd://")
        && mrl_filename_is_device (item->mrl)) {
        w = otk_listentry_new (filelist_menu_list, item->title,
                               play_dvd_cb, device,
                               filelist_select_cb, item, NULL, NULL);
    }
    /* If this entry refers to a video CD and is not an image on disc, we use
     * the video CD disc callback. */
    else if (starts_with (item->mrl, "vcd://")
             && mrl_filename_is_device (item->mrl)) {
        w = otk_listentry_new (filelist_menu_list, item->title,
                               play_vcd_cb, device,
                               filelist_select_cb, item, NULL, NULL);
    }
#ifdef HAVE_VDR
    else if (item->type == FILE_TYPE_VDR_TIMER) {
        w = otk_listentry_new (filelist_menu_list, item->title,
                               NULL, NULL,
                               filelist_select_cb, item,
                               filelist_remove_cb, item);
    }
#endif
    /* Else we use the normal activate callback. */
    else {
        w = otk_listentry_new (filelist_menu_list, item->title,
                               filelist_activate_cb, item,
                               filelist_select_cb, item,
                               filelist_remove_cb, item);
    }

    otk_widget_set_focus_callbacks (w, filelist_focus_enter_cb, item,
                                    filelist_focus_leave_cb, item);
}


static void
filelist_menu_update_list (void)
{
    if (!filelist_menu_is_current_menu ()) {
        return;
    }

    /* If we're currently not in the toplevel filelist and the filelist does
     * NOT have a parent we're probably in a removable disc, that has been
     * removed (see disc.c). To rescue ourself of segmentation faults we jump
     * to the toplevel filelist and clear the prev and next lists. */
    if (!current_is_toplevel ()
        && (current_filelist->parent_list == NULL)) {
        filelist_menu_set_filelist (toplevel_filelist);
        l_list_clear (prev_filelist, NULL);
        l_list_clear (next_filelist, NULL);
    }

    otk_list_clear (filelist_menu_list);

    filelist_lock (current_filelist);

    /* If the current filelist has regular files in it, we add a
     * play-all-entries entry to the top of the list. */
    if (config_get_bool ("misc.play_all_metatrack.use")
        && !current_is_toplevel ()
        && !current_is_favorites ()
        && (strcmp (current_filelist->mrl, "/") != 0)
        && (strcmp (current_filelist->mrl, get_dir_home ()) != 0)
        && (strcmp (current_filelist->mrl, get_dir_oxine_playlists ()) != 0)) {
        bool has_regular_entries = false;
        bool has_child_list_entries = false;

        fileitem_t *item = filelist_first (current_filelist);
        while (item) {
            switch (item->type) {
            case FILE_TYPE_REGULAR:
                has_regular_entries = true;
                break;
            case FILE_TYPE_DIRECTORY:
            case FILE_TYPE_MOUNTPOINT:
            case FILE_TYPE_MEDIAMARKS:
                has_child_list_entries = true;
                break;
            default:
                break;
            }
            item = filelist_next (current_filelist, item);
        }

        if (has_child_list_entries) {
            char *title = _("Play all titles in this folder "
                            "and all subfolders...");
            otk_widget_t *w = otk_listentry_new (filelist_menu_list, title,
                                                 filelist_playall_recursive_cb,
                                                 current_filelist,
                                                 NULL, NULL, NULL, NULL);
            otk_widget_set_focus_callbacks (w, filelist_focus_enter_cb, item,
                                            filelist_focus_leave_cb, item);
        }
        else if (has_regular_entries) {
            char *title = _("Play all titles in this folder...");
            otk_widget_t *w = otk_listentry_new (filelist_menu_list, title,
                                                 filelist_playall_cb,
                                                 current_filelist,
                                                 NULL, NULL, NULL, NULL);
            otk_widget_set_focus_callbacks (w, filelist_focus_enter_cb, item,
                                            filelist_focus_leave_cb, item);
        }
    }

    /* next we add all the entries from the extras filelist */
    if (current_is_toplevel () && extras_filelist) {
        filelist_lock (extras_filelist);
        fileitem_t *item = filelist_first (extras_filelist);
        while (item) {
            filelist_menu_addto_list (item);
            item = filelist_next (extras_filelist, item);
        }
        filelist_unlock (extras_filelist);
    }

    /* next we add all the entries from the current filelist */
    if (current_is_favorites ()) {
        int i = 0;
        fileitem_t *item = filelist_first (current_filelist);
        for (i = 0; (i < 20) && item; i++) {
            filelist_menu_addto_list (item);
            item = filelist_next (current_filelist, item);
        }
    }
    else {
        fileitem_t *item = filelist_first (current_filelist);
        while (item) {
            filelist_menu_addto_list (item);
            item = filelist_next (current_filelist, item);
        }
    }

    /* last we add all removable drives currently available */
#ifdef HAVE_HAL
    if (current_is_toplevel ()
        && (filelist_trylock (oxine->hal_volume_list) != EBUSY)) {
        fileitem_allowed_t allowed = toplevel_filelist->allowed_filetypes;
        fileitem_t *item = filelist_first (oxine->hal_volume_list);
        while (item) {
            bool add = false;
            /* We only allow audio CDs in the audio menu and DVDs and VCDs
             * in the video menu. */
            add |= is_file_allowed (item->mrl, allowed);
            /* Mountable discs (e.g. external HD, CDROM) are allowed in all
             * menus. */
            add |= (item->type == FILE_TYPE_MOUNTPOINT);
            /* We only add volumes. Empty drives are not show in this list
             * as they are not really of interest to a user. */
            add &= (item->volume_udi != NULL);

            if (add) {
                filelist_menu_addto_list (item);
            }
            item = filelist_next (oxine->hal_volume_list, item);
        }
        filelist_unlock (oxine->hal_volume_list);
    }
#endif

    otk_list_set_pos (filelist_menu_list, current_filelist->top_position);
    otk_list_set_focus (filelist_menu_list, current_filelist->cur_position);

    filelist_unlock (current_filelist);
}


static void
filelist_menu_update_title (void)
{
    if (!filelist_menu_is_current_menu ()) {
        return;
    }

    if (current_filelist == toplevel_filelist) {
        otk_label_set_text (filelist_menu_title, _("Choose an entry..."));
    }
    else {
        char *tmp = ho_strdup (current_filelist->title);
        char *title = tmp;

        if (title[0] == '[') {
            title++;
        }
        if (title[strlen (title) - 1] == ']') {
            title[strlen (title) - 1] = '\0';
        }

        otk_label_set_text (filelist_menu_title, title);
        ho_free (tmp);
    }
}


static void
filelist_menu_set_filelist (filelist_t * newlist)
{
    filelist_t *prev = (filelist_t *) l_list_last (prev_filelist);
    filelist_t *next = (filelist_t *) l_list_last (next_filelist);

    /* We are returning to the directory on top of the prev stack. We 
     * remove it from the prev stack and add the current directory to the 
     * next stack. */
    if (newlist == prev) {
        l_list_remove (prev_filelist, newlist);
        l_list_append (next_filelist, current_filelist);
    }

    /* We are returning to the directory on top of the next stack. We 
     * remove it from the next stack and add the current directory to the 
     * prev stack. */
    else if (newlist == next) {
        l_list_remove (next_filelist, newlist);
        l_list_append (prev_filelist, current_filelist);
    }

    /* We are entering a directory that is neither on top of the prev stack
     * nor on top of the next stack. We clear the next stack and add the
     * current directory to the prev stack. */
    else {
        l_list_clear (next_filelist, NULL);
        l_list_append (prev_filelist, current_filelist);
    }


    /* We update the position in the list. */
    if (otk_list_get_length (filelist_menu_list) > 0) {
        current_filelist->top_position =
            otk_list_get_pos (filelist_menu_list);
        current_filelist->cur_position =
            otk_list_get_focus (filelist_menu_list);
    }
    filelist_ref_set (&current_filelist, newlist);

#ifdef HAVE_OSD_IMAGE
    /* We cancel any thumbnail job that might be running. */
    cancel_job (thumbnail_job);
    /* We schedule a new thumbnail job. */
    thumbnail_job = schedule_job (100, filelist_menu_thumbnail_job, NULL);
#endif

    filelist_menu_update_list ();
    filelist_menu_update_title ();
    filelist_menu_update_buttons ();

    show_user_interface (oxine);
}


static void
show_download_progress_info (void *p)
{
    create_new_window (false, true);

    int w = 600;
    int h = 140;
    int x = (odk_osd_get_width (oxine->odk) - w) / 2;
    int y = (odk_osd_get_height (oxine->odk) - h) / 2;

    otk_border_new (oxine->otk, x, y, w, h);

    otk_label_new (oxine->otk, x + (w / 2), y + 50, 560,
                   OTK_ALIGN_CENTER | OTK_ALIGN_BOTTOM, _("Please wait..."));

    otk_widget_t *slider =
        otk_slider_new (oxine->otk, x + 20, y + 70, 560, 40, 0, 20, false,
                        OTK_SLIDER_HORIZONTAL, "%3.f %%", 1.0, false, false,
                        0, 1, 0, 100, download_get_percent, NULL, NULL, NULL);
    otk_widget_set_updated (slider, true);

    set_current_menu (show_download_progress_info, NULL);

    show_user_interface (NULL);
    show_menu_background (NULL);
}


static void
filelist_remove_cb (void *entry_cb_data)
{
    fileitem_t *fileitem = (fileitem_t *) entry_cb_data;

    if (strcmp (current_filelist->mrl, get_dir_oxine_playlists ()) == 0) {
        info (_("Removing '%s'."), fileitem->mrl);
        if (unlink (fileitem->mrl) == 0) {
            filelist_remove (current_filelist, fileitem);
            filelist_menu_set_filelist (current_filelist);
        }
        else {
            error (_("Could not remove '%s': %s!"),
                   fileitem->mrl, strerror (errno));
        }
    }
#ifdef HAVE_VDR
    else if (fileitem->type == FILE_TYPE_VDR_RECORDING) {
        info (_("Removing '%s'."), fileitem->mrl);
        if (vdr_recording_remove (fileitem->mrl)) {
            /* After removing an entry we have to reread the entries, as the
             * numbers of the recordings may have changed. */
            fileitem = fileitem->parent_list->parent_item;
            filelist_expand (fileitem);
            filelist_menu_set_filelist (fileitem->child_list);
        }
        else {
            error (_("Could not remove '%s': %s!"),
                   fileitem->mrl, _("SVDRP error"));
        }
    }
    else if (fileitem->type == FILE_TYPE_VDR_TIMER) {
        info (_("Removing '%s'."), fileitem->mrl);
        if (vdr_timer_remove (fileitem->mrl)) {
            /* After removing an entry we have to reread the entries, as the
             * numbers of the timers may have changed. */
            fileitem = fileitem->parent_list->parent_item;
            filelist_expand (fileitem);
            filelist_menu_set_filelist (fileitem->child_list);
        }
        else {
            error (_("Could not remove '%s': %s!"),
                   fileitem->mrl, _("SVDRP error"));
        }
    }
#endif
#ifdef HAVE_HAL
    else if (fileitem->parent_list == oxine->hal_volume_list) {
        drive_eject (fileitem->device);
    }
#endif
    else {
        todo ("Implement removing of 'normal' files.");
    }
}


/*
 * This is the callback for a click on an item in the list.
 */
static void
filelist_select_cb (void *entry_cb_data)
{
    filelist_menu_update_buttons ();
    show_user_interface (oxine);
}


static void
filelist_please_wait (void *p)
{
    otk_label_set_text (filelist_menu_title, _("Please wait..."));
    otk_widget_set_enabled (button_up, false);
    otk_widget_set_enabled (button_home, false);
    otk_widget_set_enabled (button_prev, false);
    otk_widget_set_enabled (button_next, false);
    otk_list_clear (filelist_menu_list);
}


/*
 * This is the callback for a doubleclick on an item in the list.
 */
static void
filelist_activate_cb (void *entry_cb_data)
{
    fileitem_t *fileitem = (fileitem_t *) entry_cb_data;

    current_filelist->top_position = otk_list_get_pos (filelist_menu_list);
    current_filelist->cur_position = otk_list_get_focus (filelist_menu_list);

    /* For lists that may take a while to load we set the title to 'Please
     * wait...' and clear the list, so that the user gets a feedback that
     * something is happening. So we don't see a flashing of the list when a
     * list does load quickly we wait a few milliseconds before clearing the
     * list. */
    int job_id = 0;
    switch (fileitem->type) {
#ifdef HAVE_SHOUTCAST
    case FILE_TYPE_SHOUTCAST_STATION:
        please_wait ();
        break;
#endif
#ifdef HAVE_YOUTUBE
    case FILE_TYPE_YOUTUBE_VIDEO:
        if (config_get_bool ("streaming.youtube.download_to_cache")) {
            show_download_progress_info (oxine);
        }
        else {
            please_wait ();
        }
        break;
#endif
#ifdef HAVE_TVLINKS
    case FILE_TYPE_TVLINKS_VIDEO:
        if (config_get_bool ("streaming.youtube.download_to_cache")) {
            show_download_progress_info (oxine);
        }
        else {
            please_wait ();
        }
        break;
#endif
    case FILE_TYPE_DIRECTORY:
    case FILE_TYPE_MOUNTPOINT:
    case FILE_TYPE_MEDIAMARKS:
    case FILE_TYPE_FAVORITES:
#ifdef HAVE_VDR
    case FILE_TYPE_VDR_RECORDINGS_VFOLDER:
    case FILE_TYPE_VDR_TIMERS_VFOLDER:
#endif
#ifdef HAVE_SHOUTCAST
    case FILE_TYPE_SHOUTCAST_VFOLDER:
#endif
#ifdef HAVE_YOUTUBE
    case FILE_TYPE_YOUTUBE_VFOLDER:
#endif
#ifdef HAVE_TVLINKS
    case FILE_TYPE_TVLINKS_VFOLDER:
#endif
        job_id = schedule_job (100, filelist_please_wait, NULL);
        break;
    default:
        break;
    }

    /* Make sure the fileitem is expanded. */
    filelist_expand (fileitem);
    cancel_job (job_id);

    switch (fileitem->type) {
    case FILE_TYPE_DIRECTORY:
    case FILE_TYPE_MOUNTPOINT:
    case FILE_TYPE_MEDIAMARKS:
    case FILE_TYPE_FAVORITES:
    case FILE_TYPE_CDDA_VFOLDER:
#ifdef HAVE_VDR
    case FILE_TYPE_VDR_RECORDINGS_VFOLDER:
    case FILE_TYPE_VDR_TIMERS_VFOLDER:
#endif
#ifdef HAVE_SHOUTCAST
    case FILE_TYPE_SHOUTCAST_VFOLDER:
#endif
#ifdef HAVE_YOUTUBE
    case FILE_TYPE_YOUTUBE_VFOLDER:
#endif
#ifdef HAVE_TVLINKS
    case FILE_TYPE_TVLINKS_VFOLDER:
#endif
        if (fileitem->child_list->wait_for_password) {
            /* Do nothing. */
        }
        else if (fileitem->child_list->error) {
            if (otk_list_get_length (filelist_menu_list) == 0) {
                filelist_menu_update_list ();
                filelist_menu_update_buttons ();
            }
            otk_label_set_text (filelist_menu_title,
                                fileitem->child_list->error);
        }
        else {
            filelist_menu_set_filelist (fileitem->child_list);
        }
        break;
        /* Handle all playlists. */
    case FILE_TYPE_PLAYLIST_M3U:
    case FILE_TYPE_PLAYLIST_PLS:
    case FILE_TYPE_PLAYLIST_OXP:
        {
            playlist_clear (oxine->rw_playlist);
            playlist_load (oxine->rw_playlist, fileitem->mrl);
            playitem_t *current = playlist_get_current (oxine->rw_playlist);
            if (!current) {
                playlist_play_first (oxine->rw_playlist);
            }
            else {
                playlist_play_item (oxine->rw_playlist, current);
            }
        }
        break;
    case FILE_TYPE_REGULAR:
#ifdef HAVE_SHOUTCAST
    case FILE_TYPE_SHOUTCAST_STATION:
#endif
#ifdef HAVE_YOUTUBE
    case FILE_TYPE_YOUTUBE_VIDEO:
#endif
#ifdef HAVE_TVLINKS
    case FILE_TYPE_TVLINKS_VIDEO:
#endif
        playlist_clear (oxine->rw_playlist);
        playlist_add_fileitem (oxine->rw_playlist, fileitem);
        playlist_play_first (oxine->rw_playlist);
        break;
#ifdef HAVE_VDR
        /* Handle VDR recordings */
    case FILE_TYPE_VDR_RECORDING:
        if (!vdr_recording_play (fileitem->mrl)) {
            show_message_dialog (backto_menu_cb, NULL,
                                 backto_menu_cb, NULL,
                                 DIALOG_OK_CANCEL, NULL,
                                 _("Could not play title!"));
        }
        else {
            play_vdr_cb (oxine);
        }
        break;
#endif
    default:
        break;
    }
}


#ifdef HAVE_OSD_IMAGE
static void
filelist_menu_update_thumbnail (const char *thumbnail_mrl)
{
    char *mrl = NULL;
    if (thumbnail_mrl) {
        if (is_downloadable (thumbnail_mrl)) {
            mrl = download_to_cache (thumbnail_mrl, NULL, false);
        }
        else {
            mrl = ho_strdup (thumbnail_mrl);
        }

        if (!file_exists (mrl)) {
            ho_free (mrl);
        }
    }

    otk_image_set_mrl (filelist_menu_image_small, mrl);
    otk_image_set_mrl (filelist_menu_image_large, mrl);

    ho_free (mrl);
}


static void
filelist_menu_thumbnail_preload (fileitem_t * fileitem)
{
    if (!fileitem) {
        return;
    }

    char *mrl = NULL;
    const char *thumbnail_mrl = fileitem_get_thumbnail (fileitem);

    if (thumbnail_mrl) {
        if (is_downloadable (thumbnail_mrl)) {
            mrl = download_to_cache (thumbnail_mrl, NULL, false);
        }
        else {
            mrl = ho_strdup (thumbnail_mrl);
        }

        if (!file_exists (mrl)) {
            ho_free (mrl);
        }
    }

    if (mrl) {
        int a = ODK_ALIGN_CENTER | ODK_ALIGN_VCENTER;
        int x = filelist_menu_image_large->x;
        int y = filelist_menu_image_large->y;
        int w = filelist_menu_image_large->w;
        int h = filelist_menu_image_large->h;

        if (odk_current_is_playback_mode (oxine->odk)) {
            x = filelist_menu_image_small->x;
            y = filelist_menu_image_small->y;
            w = filelist_menu_image_small->w;
            h = filelist_menu_image_small->h;
        }

        odk_osd_preload_image (oxine->odk, mrl, x, y, w, h, true, 0, a);
    }

    ho_free (mrl);
}


static void
filelist_menu_thumbnail_set (fileitem_t * fileitem)
{
    const char *mrl = fileitem_get_thumbnail (fileitem);
    if (!mrl) {
        mrl = filelist_get_thumbnail (current_filelist);
    }
    filelist_menu_update_thumbnail (mrl);

    /* We try and preload the thumbnail images of the next and of the
     * previous item in the list. */
    if (fileitem) {
        filelist_t *filelist = fileitem->parent_list;
        filelist_menu_thumbnail_preload (filelist_next (filelist, fileitem));
        filelist_menu_thumbnail_preload (filelist_prev (filelist, fileitem));
    }
}


static void
filelist_menu_thumbnail_job (void *p)
{
    fileitem_t *fileitem = (fileitem_t *) p;

    if (filelist_menu_is_current_menu ()) {
        oxine_event_t ev;
        ev.type = OXINE_EVENT_THUMBNAIL_UPDATE;
        ev.data.user_data = fileitem;
        odk_oxine_event_send (oxine->odk, &ev);
    }
}
#endif /* HAVE_OSD_IMAGE */


static void
filelist_focus_enter_cb (void *entry_cb_data)
{
#ifdef HAVE_OSD_IMAGE
    fileitem_t *item = (fileitem_t *) entry_cb_data;

    /* We cancel any thumbnail job that might be running. */
    cancel_job (thumbnail_job);
    /* We want to wait for 200ms before showing the thumbnail. This is a
     * good idea, because if the user scrolls down a list quickly, showing
     * the thumbnail at once would slow down scrolling. */
    thumbnail_job = schedule_job (300, filelist_menu_thumbnail_job, item);
#endif /* HAVE_OSD_IMAGE */
}


static void
filelist_focus_leave_cb (void *entry_cb_data)
{
#ifdef HAVE_OSD_IMAGE
    /* We cancel any thumbnail job that might be running. */
    cancel_job (thumbnail_job);
    /* If available we show the thumbnail of the current directory. */
    thumbnail_job = schedule_job (600, filelist_menu_thumbnail_job, NULL);

    filelist_menu_update_title ();
#endif /* HAVE_OSD_IMAGE */
}


static void
filelist_up_cb (void *p)
{
    if (current_filelist) {
        filelist_t *newlist = current_filelist->parent_list;
        if (current_filelist->parent_list == extras_filelist) {
            newlist = toplevel_filelist;
        }
#ifdef HAVE_HAL
        else if (current_filelist->parent_list == oxine->hal_volume_list) {
            newlist = toplevel_filelist;
        }
#endif
        filelist_menu_set_filelist (newlist);
    }
}


static void
filelist_home_cb (void *p)
{
    if (current_filelist != toplevel_filelist) {
        filelist_menu_set_filelist (toplevel_filelist);
    }
}


static void
filelist_prev_cb (void *p)
{
    filelist_t *last = (filelist_t *) l_list_last (prev_filelist);
    if (last) {
        filelist_menu_set_filelist (last);
    }
}


static void
filelist_next_cb (void *p)
{
    filelist_t *last = (filelist_t *) l_list_last (next_filelist);
    if (last) {
        filelist_menu_set_filelist (last);
    }
}


static void
filelist_menu_show_list (void)
{
    int x = odk_osd_get_width (oxine->odk) - 20 - 4 * 46 + 6;
    int y = 100;

    filelist_menu_title =
        otk_label_new (oxine->otk, 220, 117, x - 240,
                       OTK_ALIGN_LEFT | OTK_ALIGN_VCENTER,
                       _("Choose an entry..."));
    otk_widget_set_font (filelist_menu_title, "sans", 32);

    button_prev =
        otk_vector_button_new (oxine->otk, x, y, 40, 35,
                               OSD_VECTOR_ARROW_LEFT, 20, 20,
                               filelist_prev_cb, oxine);
    x += 46;
    button_next =
        otk_vector_button_new (oxine->otk, x, y, 40, 35,
                               OSD_VECTOR_ARROW_RIGHT, 20, 20,
                               filelist_next_cb, oxine);
    x += 46;
    button_up =
        otk_vector_button_new (oxine->otk, x, y, 40, 35,
                               OSD_VECTOR_ARROW_UP, 20, 20,
                               filelist_up_cb, oxine);
    x += 46;
    button_home =
        otk_vector_button_new (oxine->otk, x, y, 40, 35,
                               OSD_VECTOR_HOME, 20, 20,
                               filelist_home_cb, oxine);

    int w = odk_osd_get_width (oxine->odk) - 240;
    int h = odk_osd_get_height (oxine->odk) - 160;
    filelist_menu_list =
        otk_list_new (oxine->otk, 220, 140, w, h, 30, 33, true, true,
                      OTK_LIST_SELECTION_MULTIPLE, oxine);
}


static void
filelist_menu_event_handler (void *p, oxine_event_t * event)
{
    if (!filelist_menu_is_current_menu ()) {
        return;
    }

#ifdef HAVE_OSD_IMAGE
    if (event->type == OXINE_EVENT_THUMBNAIL_UPDATE) {
        fileitem_t *fileitem = (fileitem_t *) event->data.user_data;
        filelist_menu_thumbnail_set (fileitem);
    }
#endif

    if (event->type != OXINE_EVENT_KEY) {
        return;
    }

    switch (event->source.key) {
    case OXINE_KEY_HOME:
        filelist_home_cb (oxine);
        event->source.key = OXINE_KEY_NULL;
        break;
    case OXINE_KEY_BACK:
        if (current_filelist != toplevel_filelist) {
            filelist_up_cb (oxine);
        }
        else {
            show_menu_main (oxine);
        }
        event->source.key = OXINE_KEY_NULL;
        break;
    case OXINE_KEY_PREV:
        filelist_prev_cb (oxine);
        event->source.key = OXINE_KEY_NULL;
        break;
    case OXINE_KEY_NEXT:
        filelist_next_cb (oxine);
        event->source.key = OXINE_KEY_NULL;
        break;
    case OXINE_KEY_INSERT:
        add_selected_cb (oxine);
        event->source.key = OXINE_KEY_NULL;
        break;
    default:
        break;
    }
}


static bool
filelist_menu_get_selected (fileitem_t *** items, int *num_selected)
{
    *items = (fileitem_t **) otk_list_get_selected (filelist_menu_list,
                                                    num_selected);

    return (items && num_selected);
}


/*
 * ***************************************************************************
 * Methods that are part the music and video menu
 * ***************************************************************************
 */
static void
add_selected_cb (void *p)
{
    int i;
    int num_selected;
    fileitem_t **fileitems;

    if (!filelist_menu_get_selected (&fileitems, &num_selected)) {
        return;
    }

    for (i = 0; i < num_selected; i++) {
        filelist_expand (fileitems[i]);
        playlist_add_fileitem (oxine->rw_playlist, fileitems[i]);
        if (favorites_filelist && (fileitems[i]->type == FILE_TYPE_DIRECTORY)) {
            favorites_add_directory (favorites_filelist,
                                     fileitems[i]->title, fileitems[i]->mrl);
        }
    }

    ho_free (fileitems);
    show_menu_playlist (oxine);
}


static void
play_selected_cb (void *p)
{
    int i;
    int num_selected;
    fileitem_t **fileitems;

    if (!filelist_menu_get_selected (&fileitems, &num_selected)) {
        return;
    }

    for (i = 0; i < num_selected; i++) {
#ifdef HAVE_VDR
        if (fileitems[i]->type == FILE_TYPE_VDR_RECORDING) {
            if (vdr_recording_play (fileitems[i]->mrl)) {
                play_vdr_cb (oxine);
            }
            goto free_fileitems;
        }
#endif
    }

    playlist_clear (oxine->rw_playlist);
    for (i = 0; i < num_selected; i++) {
        filelist_expand (fileitems[i]);
        playlist_add_fileitem (oxine->rw_playlist, fileitems[i]);
        if (favorites_filelist && (fileitems[i]->type == FILE_TYPE_DIRECTORY)) {
            favorites_add_directory (favorites_filelist,
                                     fileitems[i]->title, fileitems[i]->mrl);
        }
    }
    playlist_play_first (oxine->rw_playlist);

#ifdef HAVE_VDR
  free_fileitems:
#endif
    ho_free (fileitems);
}


static void
copy_selected_cb (void *p)
{
    int num_selected;
    fileitem_t **fileitems;

    if (!filelist_menu_get_selected (&fileitems, &num_selected)) {
        return;
    }
#ifdef HAVE_EXTRACTOR
    if (extractor_start (num_selected, fileitems)) {
        show_menu_extractor (oxine);
    }
    else {
        show_message_dialog (backto_menu_cb, oxine, NULL, NULL, DIALOG_OK,
                             NULL, _("The extractor is currently busy!"));
    }
#endif

    ho_free (fileitems);
}


static void
remove_selected_cb (void *p)
{
    /* If we're in the favorites filelist, the remove button is a reset
     * button that removes all items from the favorites filelist. */
    if (current_is_favorites ()) {
        if (unlink (get_file_favorites ()) == 0) {
            filelist_clear (favorites_filelist);
        }
        else {
            error (_("Could not remove '%s': %s!"),
                   get_file_favorites (), strerror (errno));
        }
    }

    /* In all other lists we get all selected items and call the remove
     * callback for all of them. */
    else {
        int num_selected;
        fileitem_t **fileitems;

        if (filelist_menu_get_selected (&fileitems, &num_selected)) {
            int i = 0;
            for (; i < num_selected; i++) {
                filelist_remove_cb (fileitems[i]);
            }
            ho_free (fileitems);
        }
    }

    filelist_menu_set_filelist (current_filelist);
}


static void
filelist_playall_recursive_cb (void *entry_cb_data)
{
    filelist_t *filelist = (filelist_t *) entry_cb_data;

    please_wait ();
    playlist_clear (oxine->rw_playlist);

    fileitem_t *fileitem = filelist_first (filelist);
    while (fileitem) {
        filelist_expand (fileitem);
        playlist_add_fileitem (oxine->rw_playlist, fileitem);
        fileitem = filelist_next (filelist, fileitem);
    }

    if (favorites_filelist) {
        char *title = create_title (filelist->mrl);
        favorites_add_directory (favorites_filelist, title, filelist->mrl);
        ho_free (title);
    }

    playlist_play_first (oxine->rw_playlist);
}


static void
filelist_playall_cb (void *entry_cb_data)
{
    filelist_t *filelist = (filelist_t *) entry_cb_data;

    please_wait ();
    playlist_clear (oxine->rw_playlist);

    fileitem_t *fileitem = filelist_first (filelist);
    while (fileitem) {
        if (fileitem->type == FILE_TYPE_REGULAR) {
            playlist_add_fileitem (oxine->rw_playlist, fileitem);
        }
        fileitem = filelist_next (filelist, fileitem);
    }

    if (favorites_filelist) {
        char *title = create_title (filelist->mrl);
        favorites_add_directory (favorites_filelist, title, filelist->mrl);
        ho_free (title);
    }

    playlist_play_first (oxine->rw_playlist);
}


static void
filelist_menu_show_gui (const char *background, otk_cb_t menu_cb)
{
    if (filelist_menu_window) {
        otk_set_current_window (oxine->otk, filelist_menu_window);

        current_filelist->top_position =
            otk_list_get_pos (filelist_menu_list);
        current_filelist->cur_position =
            otk_list_get_focus (filelist_menu_list);

        goto out_show;
    }

    odk_add_event_handler (oxine->odk, filelist_menu_event_handler, oxine,
                           EVENT_HANDLER_PRIORITY_NORMAL);

    filelist_menu_window = create_new_window (true, true);

    int h = 0;
    int x = 20;
    int y = 100;
    filelist_menu_buttons[0] =
        otk_text_button_new (oxine->otk, x, y, 180, 35, _("Play"),
                             play_selected_cb, oxine);
    y += 40;
    filelist_menu_buttons[1] =
        otk_text_button_new (oxine->otk, x, y, 180, 35, _("Add"),
                             add_selected_cb, oxine);
    y += 40;
    filelist_menu_buttons[2] =
        otk_text_button_new (oxine->otk, x, y, 180, 35, _("Remove"),
                             remove_selected_cb, oxine);
    y += 40;
    filelist_menu_buttons[3] =
        otk_text_button_new (oxine->otk, x, y, 180, 35, _("Copy"),
                             copy_selected_cb, oxine);
    y += 40;
    filelist_menu_buttons[4] =
        otk_text_button_new (oxine->otk, x, y, 180, 35, _("Eject"),
                             eject_cb, oxine);
    y += 50;
    filelist_menu_buttons[5] =
        otk_text_button_new (oxine->otk, x, y, 180, 35, _("Playlist"),
                             show_menu_playlist, oxine);
    y += 40;
    filelist_menu_buttons[6] =
        otk_text_button_new (oxine->otk, x, y, 180, 35, _("Mainmenu"),
                             show_menu_main, oxine);
    y += 40;
    filelist_menu_buttons[7] =
        otk_text_button_new (oxine->otk, x, y, 180, 35, _("Current title"),
                             show_menu_playback, oxine);

#ifdef HAVE_OSD_IMAGE
    h = (odk_osd_get_height (oxine->odk) - 30 - y);
    y = (y + 10 + (h / 2));
    filelist_menu_image_large =
        otk_image_new (oxine->otk, 110, y, 180, h, NULL,
                       ODK_ALIGN_CENTER | ODK_ALIGN_VCENTER, true);
    h -= 40;
    y += 20;
    filelist_menu_image_small =
        otk_image_new (oxine->otk, 110, y, 180, h, NULL,
                       ODK_ALIGN_CENTER | ODK_ALIGN_VCENTER, true);
#endif /* HAVE_OSD_IMAGE */

    filelist_menu_show_list ();
  out_show:
    show_last_filelist_menu_cb = menu_cb;
    if (odk_current_is_logo_mode (oxine->odk)) {
        set_playback_ended_menu (menu_cb, NULL);
    }
    set_backto_menu (menu_cb, NULL);
    set_current_menu (menu_cb, NULL);

    filelist_menu_update_list ();
    filelist_menu_update_buttons ();
    filelist_menu_update_title ();

    show_user_interface (NULL);
    show_menu_background (background);
}


static void
filelist_menu_mediamarks_create (void *p)
{
    otk_cb_t cb = (otk_cb_t) p;
    assert (cb);

    filelist_t *filelist = NULL;
    filelist_ref_set (&filelist,
                      filelist_new (NULL, NULL, NULL, ALLOW_FILES_ALL));

    {
        const char *mrl = get_dir_home ();
        char *title = create_title (mrl);
        filelist_add (filelist, title, mrl, FILE_TYPE_DIRECTORY);
        ho_free (title);
    }

    {
        const char *mrl = "/";
        char *title = create_title (mrl);
        filelist_add (filelist, title, mrl, FILE_TYPE_DIRECTORY);
        ho_free (title);
    }

    ho_free (filelist->mrl);
    filelist->mrl = ho_strdup (get_file_mediamarks_video ());
    if (!file_exists (filelist->mrl)) {
        mediamarks_write (filelist);
    }

    ho_free (filelist->mrl);
    filelist->mrl = ho_strdup (get_file_mediamarks_music ());
    if (!file_exists (filelist->mrl)) {
        mediamarks_write (filelist);
    }
#ifdef HAVE_IMAGE_PLAYBACK
    ho_free (filelist->mrl);
    filelist->mrl = ho_strdup (get_file_mediamarks_image ());
    if (!file_exists (filelist->mrl)) {
        mediamarks_write (filelist);
    }
#endif

    ho_free (filelist->mrl);
    filelist->mrl = ho_strdup (get_file_mediamarks_media ());
    if (!file_exists (filelist->mrl)) {
        mediamarks_write (filelist);
    }

    filelist_ref_set (&filelist, NULL);

    config_set_bool ("misc.ask_missing_mediamarks", false);
    config_save ();

    cb (NULL);
}


static void
filelist_menu_mediamarks_ignore (void *p)
{
    otk_cb_t cb = (otk_cb_t) p;
    assert (cb);

    config_set_bool ("misc.ask_missing_mediamarks", false);
    config_save ();

    cb (NULL);
}


static bool
filelist_menu_mediamarks_check (const char *background,
                                const char *mediamarks,
                                const char *menu_name, otk_cb_t menu_cb)
{
    if (file_exists (mediamarks)) {
        return false;
    }
    if (!config_get_bool ("misc.ask_missing_mediamarks")) {
        return false;
    }

    char *msg =
        ho_strdup_printf (_
                          ("The %s mediamarks file could not be found.\n"
                           "Do you want me to create an example file?\n\n"
                           "If you do not know what to do please have\n"
                           "a look at the oxine documentation! If you do\n"
                           "not care one way or the other select 'No'!"),
menu_name);
    show_message_dialog (filelist_menu_mediamarks_create, menu_cb,
                         filelist_menu_mediamarks_ignore, menu_cb,
                         DIALOG_YES_NO, background, msg);
    ho_free (msg);

    return true;
}


static bool
filelist_menu_enter (const char *background, const char *mediamarks,
                     int allowed_filetypes, otk_cb_t menu_cb)
{
    /* If the last filelist menu we entered is the one we want to enter now
     * and if there is a current filelist, we need not do anything here. */
    if ((show_last_filelist_menu_cb == menu_cb) && current_filelist) {
        return false;
    }

    otk_window_keep (filelist_menu_window, 0);
    filelist_menu_window = NULL;

    show_message_dialog (NULL, NULL, NULL, NULL, DIALOG_PLAIN,
                         background, _("Please wait..."));

    filelist_ref_set (&toplevel_filelist,
                      filelist_new (NULL, NULL, mediamarks,
                                    allowed_filetypes));

    filelist_ref_set (&extras_filelist,
                      filelist_new (NULL, NULL, NULL, allowed_filetypes));

    filelist_ref_set (&current_filelist, toplevel_filelist);
    filelist_ref_set (&favorites_filelist, NULL);

    if (prev_filelist) {
        l_list_clear (prev_filelist, NULL);
    }
    else {
        prev_filelist = l_list_new ();
    }
    if (next_filelist) {
        l_list_clear (next_filelist, NULL);
    }
    else {
        next_filelist = l_list_new ();
    }

    if (!mediamarks_read (toplevel_filelist)) {
        info (_("You should create a file containing "
                "mediamarks at '%s'!"), mediamarks);
        {
            const char *mrl = get_dir_home ();
            char *title = create_title (mrl);
            filelist_add (toplevel_filelist, title, mrl, FILE_TYPE_DIRECTORY);
            ho_free (title);
        }
        {
            const char *mrl = "/";
            char *title = create_title (mrl);
            filelist_add (toplevel_filelist, title, mrl, FILE_TYPE_DIRECTORY);
            ho_free (title);
        }
    }

    return true;
}


void
show_menu_music (void *p)
{
    const char *background = "menu_music.png";
    const char *mediamarks = get_file_mediamarks_music ();

    if (filelist_menu_mediamarks_check (background, mediamarks,
                                        _("music"), show_menu_music)) {
        return;
    }

    if (filelist_menu_enter (background, mediamarks,
                             ALLOW_FILES_MUSIC, show_menu_music)) {
        if (config_get_bool ("playlist.metafolder.show")) {
            const char *mrl = get_dir_oxine_playlists ();
            char *title = create_title (mrl);
            filelist_add (extras_filelist, title, mrl, FILE_TYPE_DIRECTORY);
            ho_free (title);
        }
#ifdef HAVE_SHOUTCAST
        if (config_get_bool ("streaming.shoutcast.radio.enable")) {
            const char *mrl = config_get_string ("streaming.shoutcast."
                                                 "radio.location");
            char *title = ho_strdup_printf ("[%s]", _("SHOUTcast"));
            fileitem_t *item = filelist_add (extras_filelist, title, mrl,
                                             FILE_TYPE_SHOUTCAST_VFOLDER);
            item->thumbnail_mrl =
                ho_strdup (OXINE_VISUALDIR "/logo_shoutcast.png");
            ho_free (title);
        }
#endif
        if (config_get_bool ("misc.favorites.enable")) {
            const char *mrl = get_file_favorites ();
            char *title = create_title (mrl);
            fileitem_t *item = filelist_add (extras_filelist, title, mrl,
                                             FILE_TYPE_FAVORITES);
            ho_free (title);
            filelist_expand (item);
            filelist_ref_set (&favorites_filelist, item->child_list);
        }

        filelist_sort (extras_filelist, NULL);
    }

    filelist_menu_show_gui (background, show_menu_music);
}


void
show_menu_video (void *p)
{
    const char *background = "menu_video.png";
    const char *mediamarks = get_file_mediamarks_video ();

    if (filelist_menu_mediamarks_check (background, mediamarks,
                                        _("video"), show_menu_video)) {
        return;
    }

    if (filelist_menu_enter (background, mediamarks,
                             ALLOW_FILES_VIDEO, show_menu_video)) {
        if (config_get_bool ("playlist.metafolder.show")) {
            const char *mrl = get_dir_oxine_playlists ();
            char *title = create_title (mrl);
            filelist_add (extras_filelist, title, mrl, FILE_TYPE_DIRECTORY);
            ho_free (title);
        }
#ifdef HAVE_VDR
        if ((config_get_number ("television.type") == 2)
            && config_get_bool ("television.enable")
            && config_get_bool ("television.recordings_show")) {
            const char *mrl = FILELIST_VDR_RECORDINGS_MRL;
            char *title = ho_strdup_printf ("[%s]",
                                            _("Television Recordings"));
            fileitem_t *item = filelist_add (extras_filelist, title, mrl,
                                             FILE_TYPE_VDR_RECORDINGS_VFOLDER);
            item->thumbnail_mrl = ho_strdup (OXINE_VISUALDIR "/logo_vdr.png");
            ho_free (title);
        }
        if ((config_get_number ("television.type") == 2)
            && config_get_bool ("television.enable")
            && config_get_bool ("television.timers_show")) {
            const char *mrl = FILELIST_VDR_TIMERS_MRL;
            char *title = ho_strdup_printf ("[%s]",
                                            _("Television Timers"));
            fileitem_t *item = filelist_add (extras_filelist, title, mrl,
                                             FILE_TYPE_VDR_TIMERS_VFOLDER);
            item->thumbnail_mrl = ho_strdup (OXINE_VISUALDIR "/logo_vdr.png");
            ho_free (title);
        }
#endif
#ifdef HAVE_SHOUTCAST
        if (config_get_bool ("streaming.shoutcast.tv.enable")) {
            const char *mrl = config_get_string ("streaming.shoutcast."
                                                 "tv.location");
            char *title = ho_strdup_printf ("[%s]", _("SHOUTcast"));
            fileitem_t *item = filelist_add (extras_filelist, title, mrl,
                                             FILE_TYPE_SHOUTCAST_VFOLDER);
            item->thumbnail_mrl =
                ho_strdup (OXINE_VISUALDIR "/logo_shoutcast.png");
            ho_free (title);
        }
#endif
#ifdef HAVE_YOUTUBE
        if (config_get_bool ("streaming.youtube.enable")) {
            const char *mrl = FILELIST_YOUTUBE_MRL;
            char *title = ho_strdup_printf ("[%s]", _("YouTube"));
            fileitem_t *item = filelist_add (extras_filelist, title, mrl,
                                             FILE_TYPE_YOUTUBE_VFOLDER);
            item->thumbnail_mrl =
                ho_strdup (OXINE_VISUALDIR "/logo_youtube.png");
            ho_free (title);
        }
#endif
#ifdef HAVE_TVLINKS
        if (config_get_bool ("streaming.tvlinks.enable")) {
            const char *mrl = FILELIST_TVLINKS_MRL;
            char *title = ho_strdup_printf ("[%s]", _("TV Links"));
            fileitem_t *item = filelist_add (extras_filelist, title, mrl,
                                             FILE_TYPE_TVLINKS_VFOLDER);
            item->thumbnail_mrl =
                ho_strdup (OXINE_VISUALDIR "/logo_tvlinks_4.png");
            ho_free (title);
        }
#endif

        filelist_sort (extras_filelist, NULL);
    }

    filelist_menu_show_gui (background, show_menu_video);
}


#ifdef HAVE_IMAGE_PLAYBACK
void
show_menu_image (void *p)
{
    const char *background = "menu_image.png";
    const char *mediamarks = get_file_mediamarks_image ();

    if (filelist_menu_mediamarks_check (background, mediamarks,
                                        _("image"), show_menu_image)) {
        return;
    }

    if (filelist_menu_enter (background, mediamarks,
                             ALLOW_FILES_IMAGE, show_menu_image)) {
        filelist_sort (extras_filelist, NULL);
    }

    filelist_menu_show_gui (background, show_menu_image);
}
#endif


void
show_menu_media (void *p)
{
    const char *background = "menu_media.png";
    const char *mediamarks = get_file_mediamarks_media ();

    if (filelist_menu_mediamarks_check (background, mediamarks,
                                        _("media"), show_menu_media)) {
        return;
    }

    if (filelist_menu_enter (background, mediamarks,
                             ALLOW_FILES_MULTIMEDIA, show_menu_media)) {
        filelist_sort (extras_filelist, NULL);
    }

    filelist_menu_show_gui (background, show_menu_media);
}


void
show_menu_filelist (void *p)
{
    show_last_filelist_menu_cb (oxine);
}


bool
filelist_menu_is_current_menu (void)
{
    bool result = false;

    result |= is_current_menu (show_menu_music);
    result |= is_current_menu (show_menu_video);
#ifdef HAVE_IMAGE_PLAYBACK
    result |= is_current_menu (show_menu_image);
#endif
    result |= is_current_menu (show_menu_media);

    return result;
}


void
free_menu_filelist (void)
{
    /* We cancel any thumbnail job that might be running. */
    cancel_job (thumbnail_job);

    if (next_filelist) {
        l_list_free (next_filelist, NULL);
        next_filelist = NULL;
    }
    if (prev_filelist) {
        l_list_free (prev_filelist, NULL);
        prev_filelist = NULL;
    }

    filelist_ref_set (&favorites_filelist, NULL);
    filelist_ref_set (&extras_filelist, NULL);
    filelist_ref_set (&current_filelist, NULL);
    filelist_ref_set (&toplevel_filelist, NULL);
}
