
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: odk_osd.h 2440 2007-07-05 08:28:13Z mschwerin $
 *
 */

#ifndef HAVE_ODK_OSD_H
#define HAVE_ODK_OSD_H

/*
 * ***************************************************************************
 * Description:     alignment values for text
 * ***************************************************************************
 */
#define ODK_ALIGN_LEFT          0x001
#define ODK_ALIGN_CENTER        0x002
#define ODK_ALIGN_RIGHT         0x004
#define ODK_ALIGN_TOP           0x008
#define ODK_ALIGN_VCENTER       0x010
#define ODK_ALIGN_BOTTOM        0x020


/**
 * Available vector images.
 */
typedef enum {
    OSD_VECTOR_POWEROFF,
    OSD_VECTOR_HOME,
    OSD_VECTOR_FOLDER,
    OSD_VECTOR_PLAYLIST,
    OSD_VECTOR_PLUS,
    OSD_VECTOR_MINUS,
    OSD_VECTOR_ARROW_UP,
    OSD_VECTOR_ARROW_DOWN,
    OSD_VECTOR_ARROW_LEFT,
    OSD_VECTOR_ARROW_RIGHT,
    OSD_VECTOR_ARROW_UP_SIMPLE,
    OSD_VECTOR_ARROW_DOWN_SIMPLE,
    OSD_VECTOR_ARROW_UP_DOUBLE,
    OSD_VECTOR_ARROW_DOWN_DOUBLE,
    OSD_VECTOR_VOLUME,
    OSD_VECTOR_VOLUME_MUTE,
    OSD_VECTOR_CHECK,
} odk_osd_vector_type_t;

typedef struct odk_osd_image_s odk_osd_image_t;

void odk_osd_set_resolution (odk_t * odk, int w, int h);
int odk_osd_get_width (odk_t * odk);
int odk_osd_get_height (odk_t * odk);

void odk_osd_swap (odk_t * odk);
void odk_osd_show (odk_t * odk);
void odk_osd_hide (odk_t * odk);
void odk_osd_hide_images (odk_t * odk);
void odk_osd_clear (odk_t * odk);


/// Returns TRUE if unscaled OSD is used.
bool odk_osd_use_unscaled_osd (odk_t * odk);

/**
 * Draws a point.
 */
void odk_draw_point (odk_t * odk, int x, int y, int color);


/**
 * Draws a line.
 */
void odk_draw_line (odk_t * odk, int x1, int y1, int x2, int y2,
                    int width, int color);


/**
 * Draws a triangle.
 */
void odk_draw_triangle (odk_t * odk, int x1, int y1, int x2, int y2,
                        int x3, int y3, int color, bool filled);


/**
 * Draws a circle.
 */
void odk_draw_circ (odk_t * odk, int x, int y, int r, int color, bool filled);


/**
 * Draws a rectangle.
 */
void odk_draw_rect (odk_t * odk, int x, int y, int w, int h, int r,
                    int color, bool filled);


/**
 * Draws a text.
 */
void odk_draw_text (odk_t * odk, int x, int y, const char *text,
                    int alignment, int color_base);


/**
 * Draws a vector graphic.
 */
void odk_draw_vector (odk_t * odk, int x, int y, int w, int h,
                      odk_osd_vector_type_t type, int color);


/**
 * Returns width and height of the text.
 */
void odk_get_text_size (odk_t * odk, const char *text, int *w, int *h);


/**
 * Sets the font and font size to use.
 */
void odk_osd_set_font (odk_t * odk, const char *font, int font_size);



/**
 * Allocates 11 colors in the palette and returns the index of the first
 * color. A xine text palette always consists of 11 colors of which color 0
 * is transparent, color 1 is the background and color 10 is the foreground.
 * The colors between bg_color and fg_color are interpolated.
 *
 * @param odk                   The ODK object.
 * @param fg_color              Foreground color as an RGB color value.
 * @param fg_transparency       Foreground transparency value.
 * @param bg_color              Background color as an RGB color value.
 * @param bg_transparency       Background transparency value.
 * @param bo_color              Border color as an RGB color value.
 * @param bo_transparency       Border transparency value.
 */
int odk_osd_alloc_text_palette (odk_t * odk,
                                uint32_t fg_color, uint8_t fg_transparency,
                                uint32_t bg_color, uint8_t bg_transparency,
                                uint32_t bo_color, uint8_t bo_transparency);


/**
 * Returns the index of a color in the current palette.
 *
 * @param odk                   The ODK object.
 * @param color                 An RGB color value.
 * @param transparency          A transparency value.
 */
int odk_osd_get_color (odk_t * odk, uint32_t color, uint8_t transparency);


/**
 * Returns TRUE if the OSD is visible.
 */
bool odk_is_osd_visible (odk_t * odk);


#ifdef HAVE_OSD_IMAGE
/**
 * Draws the image into the standard OSD. 
 *
 * @param odk                   The ODK object.
 * @param mrl                   The MRL of the image to load.
 * @param x                     The x-coordinate of the image.
 * @param y                     The y-coordinate of the image.
 * @param w                     The output width of the image. The image is
 *                              scaled depending on the given width and
 *                              height so as not to exceed these values.
 * @param h                     The output height of the image. The image is
 *                              scaled depending on the given width and
 *                              height so as not to exceed these values.
 * @param alignment             The alignment of the image.
 * @param border_draw           If TRUE a border of one pixel will be drawn
 *                              around the image.
 * @param border_color          The color of the border as an RGB color value.
 */
odk_osd_image_t *odk_osd_draw_image (odk_t * odk, const char *mrl,
                                     int x, int y, int w, int h,
                                     bool border_draw, uint32_t border_color,
                                     int alignment);


/**
 * Draws the image into a separate OSD and shows that.
 *
 * @param odk                   The ODK object.
 * @param mrl                   The MRL of the image to load.
 * @param x                     The x-coordinate of the image.
 * @param y                     The y-coordinate of the image.
 * @param w                     The output width of the image. The image is
 *                              scaled depending on the given width and
 *                              height so as not to exceed these values.
 * @param h                     The output height of the image. The image is
 *                              scaled depending on the given width and
 *                              height so as not to exceed these values.
 * @param alignment             The alignment of the image.
 * @param border_draw           If TRUE a border of one pixel will be drawn
 *                              around the image.
 * @param border_color          The color of the border as an RGB color value.
 */
odk_osd_image_t *odk_osd_show_image (odk_t * odk, const char *mrl,
                                     int x, int y, int w, int h,
                                     bool border_draw, uint32_t border_color,
                                     int alignment);


/**
 * Preloads an image into the image cache.
 *
 * @param odk                   The ODK object.
 * @param mrl                   The MRL of the image to load.
 * @param x                     The x-coordinate of the image.
 * @param y                     The y-coordinate of the image.
 * @param w                     The output width of the image. The image is
 *                              scaled depending on the given width and
 *                              height so as not to exceed these values.
 * @param h                     The output height of the image. The image is
 *                              scaled depending on the given width and
 *                              height so as not to exceed these values.
 * @param alignment             The alignment of the image.
 * @param border_draw           If TRUE a border of one pixel will be drawn
 *                              around the image.
 * @param border_color          The color of the border as an RGB color value.
 */
odk_osd_image_t *odk_osd_preload_image (odk_t * odk, const char *mrl,
                                        int x, int y, int w, int h,
                                        bool border_draw, uint32_t border_color,
                                        int alignment);


/**
 * This hides the image. It is important to note, that pointers to the
 * passed image may become invalid after hiding the image, as non-visible
 * entries in the internal image cache may be freed.
 */
void odk_osd_hide_image (odk_osd_image_t * image);


/**
 * Returns TRUE if the passed image is visible.
 */
bool odk_osd_is_image_visible (odk_osd_image_t * image);
#endif /* HAVE_OSD_IMAGE */

#endif /* HAVE_ODK_OSD_H */
