
/*
 * Copyright (C) 2002-2003 Stefan Holst
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: otk_button.c 2412 2007-07-03 08:00:19Z mschwerin $
 */
#include "config.h"

#include "heap.h"
#include "i18n.h"
#include "list.h"
#include "logger.h"
#include "odk.h"
#include "otk.h"


typedef struct {
    otk_widget_t widget;

    /// The text for a text only button.
    char *text;

    /// The type of the vector graphic.
    odk_osd_vector_type_t type;
    /// The width of the vector graphic.
    int vec_w;
    /// The height of the vector graphic.
    int vec_h;

    /// Callback to call on activate
    otk_cb_t activate_cb;
    /// Data to pass to activate callback.
    void *activate_cb_data;

    /// Callback to update button.
    otk_update_cb_t update_cb;
    /// Data to pass to update callback.
    void *update_cb_data;
} otk_button_t;


void
otk_button_set_update_cb (otk_widget_t * this, otk_update_cb_t update_cb,
                          void *update_cb_data)
{
    otk_button_t *button = (otk_button_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_BUTTON))
        return;

    button->update_cb = update_cb;
    button->update_cb_data = update_cb_data;
}


static void
button_destroy (otk_widget_t * this)
{
    otk_button_t *button = (otk_button_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_BUTTON))
        return;

    ho_free (button->text);
    ho_free (button);
}


static void
button_draw (otk_widget_t * this)
{
    otk_button_t *button = (otk_button_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_BUTTON))
        return;

    int palette = 0;
    if (!this->is_enabled) {
        palette = otk_get_palette (this->otk, OTK_PALETTE_BUTTON_DISABLED);
    }
    else if (this->is_focused) {
        palette = otk_get_palette (this->otk, OTK_PALETTE_BUTTON_FOCUSED);
    }
    else {
        palette = otk_get_palette (this->otk, OTK_PALETTE_BUTTON);
    }
    int background_color = palette + OSD_TEXT_PALETTE_BACKGROUND;
    int foreground_color = palette + OSD_TEXT_PALETTE_FOREGROUND;

    odk_draw_rect (this->odk, this->x, this->y, this->w, this->h, 5,
                   background_color, true);

    if (button->text) {
        odk_osd_set_font (this->odk, this->font, this->fontsize);
        char *txt = otk_trunc_text_to_width (this->otk, button->text,
                                             button->widget.w - 10);

        int x = this->x + this->w / 2;
        if (this->alignment & OTK_ALIGN_RIGHT) {
            x = this->x + this->w - 5;
        }
        else if (this->alignment & OTK_ALIGN_LEFT) {
            x = this->x + 5;
        }

        /* We always align the text in vertical direction. */
        this->alignment &= ~OTK_ALIGN_TOP;
        this->alignment &= ~OTK_ALIGN_BOTTOM;
        this->alignment |= OTK_ALIGN_VCENTER;

        odk_draw_text (this->odk, x, this->y + this->h / 2,
                       txt, this->alignment, palette);

        ho_free (txt);
    }

    else {
        int x = this->x + (this->w - button->vec_w) / 2;
        int y = this->y + (this->h - button->vec_h) / 2;
        int w = button->vec_w;
        int h = button->vec_h;
        odk_draw_vector (this->odk, x, y, w, h,
                         button->type, foreground_color);
    }

    this->need_repaint = false;
}


static void
button_key_handler (otk_widget_t * this, oxine_event_t * ev)
{
    otk_button_t *button = (otk_button_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_BUTTON))
        return;

    oxine_key_id_t saved_key = ev->source.key;
    ev->source.key = OXINE_KEY_NULL;

    switch (saved_key) {
    case OXINE_KEY_ACTIVATE:
        if (button->activate_cb) {
            button->activate_cb (button->activate_cb_data);
        }
        break;
    default:
        /* If we did not use this event we restore the event. */
        ev->source.key = saved_key;
        break;
    }
}


static void
button_button_handler (otk_widget_t * this, oxine_event_t * ev)
{
    otk_button_t *button = (otk_button_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_BUTTON))
        return;

    oxine_button_id_t saved_button = ev->source.button;
    ev->source.button = OXINE_MOUSE_BUTTON_NULL;

    switch (saved_button) {
    case OXINE_MOUSE_BUTTON_LEFT:
        if (button->activate_cb) {
            button->activate_cb (button->activate_cb_data);
        }
        break;
    default:
        /* If we did not use this event we restore the event. */
        ev->source.button = saved_button;
        break;
    }
}


static void
button_update (otk_widget_t * this)
{
    otk_button_t *button = (otk_button_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_BUTTON))
        return;

    if (button->update_cb)
        button->update_cb (button->update_cb_data, this);
}


static otk_button_t *
button_new_basic (otk_t * otk, int x, int y, int w, int h,
                  otk_cb_t cb, void *cb_data)
{
    otk_button_t *button = ho_new (otk_button_t);

    otk_widget_constructor ((otk_widget_t *) button, otk,
                            OTK_WIDGET_BUTTON, x, y, w, h);

    button->widget.draw = button_draw;
    button->widget.destroy = button_destroy;
    button->widget.update = button_update;

    button->widget.key_handler = button_key_handler;
    button->widget.button_handler = button_button_handler;

    button->activate_cb = cb;
    button->activate_cb_data = cb_data;

    return button;
}


otk_widget_t *
otk_text_button_new (otk_t * otk, int x, int y, int w, int h,
                     const char *text, otk_cb_t cb, void *cb_data)
{
    otk_button_t *button = button_new_basic (otk, x, y, w, h, cb, cb_data);

    button->text = text ? ho_strdup (text) : NULL;

    return (otk_widget_t *) button;
}


otk_widget_t *
otk_vector_button_new (otk_t * otk, int x, int y, int w, int h,
                       odk_osd_vector_type_t type, int vec_w, int vec_h,
                       otk_cb_t cb, void *cb_data)
{
    otk_button_t *button = button_new_basic (otk, x, y, w, h, cb, cb_data);

    button->type = type;
    button->vec_w = vec_w;
    button->vec_h = vec_h;

    return (otk_widget_t *) button;
}


void
otk_button_set_text (otk_widget_t * this, const char *text)
{
    otk_button_t *button = (otk_button_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_BUTTON))
        return;

    ho_free (button->text);
    button->text = text ? ho_strdup (text) : NULL;

    this->need_repaint = true;
}


void
otk_button_set_vector (otk_widget_t * this, odk_osd_vector_type_t type)
{
    otk_button_t *button = (otk_button_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_BUTTON))
        return;

    button->type = type;

    this->need_repaint = true;
}
