
/*
 * Copyright (C) 2002-2003 Stefan Holst
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: otk_slider.c 2571 2007-07-22 11:13:07Z mschwerin $
 */
#include "config.h"

#include <math.h>
#include <stdio.h>

#include "heap.h"
#include "i18n.h"
#include "logger.h"
#include "odk.h"
#include "otk.h"

typedef struct {
    otk_widget_t widget;

    /// Horizontal or vertical?
    otk_slider_orientation_t direction;
    /// A backup of the last value displayed.
    int last_value;

    /// How to draw the value
    char *value_format;
    /// How to scale the value
    double value_scale;

    int marker_w;
    int marker_h;

    bool simple_marker;

    int def_value;
    int min_value;
    int max_value;
    int min_step;

    /// This is the callback that is called whenever the slider is clicked.
    void *set_value_cb_data;
    otk_int_set_cb_t set_value_cb;

    /// This is the callback that is used to retrieve the current value.
    void *get_value_cb_data;
    otk_int_get_cb_t get_value_cb;
} otk_slider_t;


static void
slider_destroy (otk_widget_t * this)
{
    otk_slider_t *slider = (otk_slider_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_SLIDER))
        return;

    if (slider->value_format) {
        ho_free (slider->value_format);
    }

    ho_free (slider);
}


static void
slider_draw (otk_widget_t * this)
{
    otk_slider_t *slider = (otk_slider_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_SLIDER))
        return;

    int value = slider->last_value;
    if (slider->get_value_cb) {
        value = slider->get_value_cb (slider->get_value_cb_data);
    }

    /* We may get wrong values here sometimes. This is why we always save the
     * last valid value to reuse in such a case. */
    if (value < slider->min_value) {
        debug ("value is outside of valid range (%d-%d): %d",
               slider->min_value, slider->max_value, value);
        value = slider->last_value;
    }
    else if (value > slider->max_value) {
        debug ("value is outside of valid range (%d-%d): %d",
               slider->min_value, slider->max_value, value);
        value = slider->last_value;
    }
    slider->last_value = value;

    double mval = slider->max_value - slider->min_value;
    double cval = value - slider->min_value;
    int pval = (mval != 0) ? round (cval / (mval / 100.0)) : 0;

    /* Just to make sure we don't try to draw anything outside the displayable
     * range. */
    if (pval < 0) {
        pval = 0;
    }
    else if (pval > 100) {
        pval = 100;
    }

    int pal0 = 0;
    int pal1 = 0;
    if (this->is_focused) {
        pal0 = otk_get_palette (this->otk, OTK_PALETTE_SLIDER_FOCUSED);
        pal1 = otk_get_palette (this->otk, OTK_PALETTE_SLIDER);
    }
    else {
        pal0 = otk_get_palette (this->otk, OTK_PALETTE_SLIDER);
        pal1 = otk_get_palette (this->otk, OTK_PALETTE_SLIDER_FOCUSED);
    }
    int foreground_color = pal0 + OSD_TEXT_PALETTE_FOREGROUND;
    int background_color0 = pal0 + OSD_TEXT_PALETTE_BACKGROUND;
    int background_color1 = pal1 + OSD_TEXT_PALETTE_BACKGROUND;

    odk_draw_rect (this->odk, this->x, this->y, this->w, this->h, 5,
                   background_color0, true);

    switch (slider->direction) {
    case OTK_SLIDER_HORIZONTAL:
        {
            int w = slider->marker_w;
            int h = slider->marker_h;

            int b = (this->h - h) / 2;
            int m = this->x + b + (w / 2) +
                (pval * (this->w - (2 * b) - w)) / 100;

            int y0 = this->y + b;
            int x0 = m - w / 2;

            if (slider->simple_marker) {
                odk_draw_rect (this->odk, x0, y0, w, h, 5,
                               foreground_color, true);
            }
            else {
                int x = this->x + b;
                int xmax = this->x + this->w - b;
                for (; (x + 8) < m; x += 10) {
                    odk_draw_rect (this->odk, x, y0, 8, h, 0,
                                   foreground_color, true);
                }
                for (; (x + 8) < xmax; x += 10) {
                    odk_draw_rect (this->odk, x, y0, 8, h, 0,
                                   background_color1, true);
                }
            }
        }
        break;
    case OTK_SLIDER_VERTICAL:
        {
            int w = slider->marker_w;
            int h = slider->marker_h;

            int b = (this->w - w) / 2;
            int m = this->y + this->h - b - (h / 2) -
                (pval * (this->h - (2 * b) - h)) / 100;

            int x0 = this->x + b;
            int y0 = m - h / 2;

            if (slider->simple_marker) {
                odk_draw_rect (this->odk, x0, y0, w, h, 5,
                               foreground_color, true);
            }
            else {
                int y = this->y + b;
                int ymax = this->y + this->h - b;
                for (; (y + 8) < m; y += 10) {
                    odk_draw_rect (this->odk, x0, y, w, 8, 0,
                                   background_color1, true);
                }
                for (; (y + 8) < ymax; y += 10) {
                    odk_draw_rect (this->odk, x0, y, w, 8, 0,
                                   foreground_color, true);
                }
            }
        }
    }

    if (slider->value_format) {
        odk_osd_set_font (this->odk, this->font, this->fontsize);

        double val = (double) value * slider->value_scale;
        char *str = ho_strdup_printf (slider->value_format, val);
        char *cut = otk_trunc_text_to_width (this->otk, str, this->w);

        int pal2 = 0;
        if (!slider->simple_marker && pval > 50) {
            pal2 = otk_get_palette (this->otk, OTK_PALETTE_SLIDER_INVERSE);
        }
        else {
            pal2 = pal0;
        }

        odk_draw_text (this->odk, this->x + (this->w / 2),
                       this->y + (this->h / 2), cut,
                       ODK_ALIGN_CENTER | ODK_ALIGN_VCENTER, pal2);

        ho_free (cut);
        ho_free (str);
    }

    this->need_repaint = false;
}


static void
slider_update (otk_widget_t * this)
{
    otk_slider_t *slider = (otk_slider_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_SLIDER))
        return;
    if (!slider->get_value_cb)
        return;

    int value = slider->get_value_cb (slider->get_value_cb_data);

    /* As repainting the slider covers up the old slider position, we need not
     * repaint the complete the OSD. Repainting the slider should be enough. */
    if (value != slider->last_value) {
        slider_draw (this);
        otk_draw (slider->widget.otk);
    }
}


static void
slider_minus_cb (void *this)
{
    otk_slider_t *slider = (otk_slider_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_SLIDER))
        return;
    if (!slider->get_value_cb)
        return;
    if (!slider->set_value_cb)
        return;

    int value = slider->get_value_cb (slider->get_value_cb_data);
    value -= slider->min_step;
    if (value < slider->min_value)
        value = slider->min_value;

    slider->set_value_cb (slider->set_value_cb_data, value);

    /* As repainting the slider covers up the old slider position, we need not
     * repaint the complete the OSD. Repainting the slider should be enough. */
    if (value != slider->last_value) {
        slider_draw (this);
        otk_draw (slider->widget.otk);
    }
}


static void
slider_plus_cb (void *this)
{
    otk_slider_t *slider = (otk_slider_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_SLIDER))
        return;
    if (!slider->get_value_cb)
        return;
    if (!slider->set_value_cb)
        return;

    int value = slider->get_value_cb (slider->get_value_cb_data);
    value += slider->min_step;
    if (value > slider->max_value)
        value = slider->max_value;

    slider->set_value_cb (slider->set_value_cb_data, value);

    /* As repainting the slider covers up the old slider position, we need not
     * repaint the complete the OSD. Repainting the slider should be enough. */
    if (value != slider->last_value) {
        slider_draw (this);
        otk_draw (slider->widget.otk);
    }
}


static void
slider_reset_cb (void *this)
{
    otk_slider_t *slider = (otk_slider_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_SLIDER))
        return;
    if (!slider->set_value_cb)
        return;

    slider->set_value_cb (slider->set_value_cb_data, slider->def_value);

    /* As repainting the slider covers up the old slider position, we need not
     * repaint the complete the OSD. Repainting the slider should be enough. */
    if (slider->def_value != slider->last_value) {
        slider_draw (this);
        otk_draw (slider->widget.otk);
    }
}


static void
slider_button_handler (otk_widget_t * this, oxine_event_t * ev)
{
    otk_slider_t *slider = (otk_slider_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_SLIDER))
        return;

    oxine_button_id_t saved_button = ev->source.button;
    ev->source.button = OXINE_MOUSE_BUTTON_NULL;

    switch (saved_button) {
    case OXINE_MOUSE_BUTTON_LEFT:
        if (slider->set_value_cb) {
            double p = 0;
            if (slider->direction == OTK_SLIDER_HORIZONTAL) {
                int b = (this->h - slider->marker_h) / 2;
                int w = (this->w - 2 * b);
                int x = (this->x + b);
                p = ((ev->data.mouse.pos.x - x) * 100) / w;
            }
            else if (slider->direction == OTK_SLIDER_VERTICAL) {
                int b = (this->w - slider->marker_w) / 2;
                int h = (this->h - 2 * b);
                int y = (this->y + b);
                p = ((ev->data.mouse.pos.y - y) * 100) / h;
                p = 100 - p;
            }
            int max = slider->max_value - slider->min_value;
            int value = (p * max / 100) + slider->min_value;
            if (value > slider->max_value)
                value = slider->max_value;
            if (value < slider->min_value)
                value = slider->min_value;
            slider->set_value_cb (slider->set_value_cb_data, value);
        }
        break;
    case OXINE_MOUSE_SCROLLWHEEL_UP:
        slider_plus_cb (slider);
        break;
    case OXINE_MOUSE_SCROLLWHEEL_DOWN:
        slider_minus_cb (slider);
        break;
    default:
        /* If we did not use this event we restore the event. */
        ev->source.button = saved_button;
        break;
    }
}


otk_widget_t *
otk_slider_new (otk_t * otk, int x, int y, int w, int h,
                int marker_w, int marker_h, bool simple_marker,
                otk_slider_orientation_t direction,
                const char *value_format, double value_scale,
                bool with_buttons, bool with_reset_button,
                int def_value, int min_step, int min_value, int max_value,
                otk_int_get_cb_t get_value_cb, void *get_value_cb_data,
                otk_int_set_cb_t set_value_cb, void *set_value_cb_data)
{
    otk_slider_t *slider = ho_new (otk_slider_t);

    otk_widget_constructor ((otk_widget_t *) slider, otk,
                            OTK_WIDGET_SLIDER, x, y, w, h);

    slider->widget.selectable = OTK_SELECTABLE_MOUSE;

    slider->widget.draw = slider_draw;
    slider->widget.destroy = slider_destroy;
    slider->widget.update = slider_update;

    slider->widget.button_handler = slider_button_handler;

    slider->direction = direction;
    if (value_format) {
        slider->value_format = ho_strdup (value_format);
        slider->value_scale = value_scale;
    }
    else {
        slider->value_format = NULL;
        slider->value_scale = 1;
    }

    slider->marker_w = marker_w;
    slider->marker_h = marker_h;
    slider->simple_marker = simple_marker;

    slider->def_value = def_value;
    slider->min_value = min_value;
    slider->max_value = max_value;
    slider->min_step = min_step;

    slider->get_value_cb = get_value_cb;
    slider->get_value_cb_data = get_value_cb_data;

    slider->set_value_cb = set_value_cb;
    slider->set_value_cb_data = set_value_cb_data;

    if (slider->direction == OTK_SLIDER_HORIZONTAL) {
        int x1 = x;
        int x2 = x + w - 30;
        int x3 = x + w - 30;

        if (with_buttons) {
            slider->widget.x += 32;
            slider->widget.w -= 64;
        }
        if (with_reset_button) {
            x2 -= 32;
            slider->widget.w -= 32;
        }

        if (with_buttons) {
            otk_vector_button_new (otk, x1, y, 30, h,
                                   OSD_VECTOR_MINUS, 10, 10,
                                   slider_minus_cb, slider);
            otk_vector_button_new (otk, x2, y, 30, h,
                                   OSD_VECTOR_PLUS, 10, 10,
                                   slider_plus_cb, slider);
        }
        if (with_reset_button) {
            otk_vector_button_new (otk, x3, y, 30, h,
                                   OSD_VECTOR_HOME, 20, 20,
                                   slider_reset_cb, slider);
        }
    }

    else if (slider->direction == OTK_SLIDER_VERTICAL) {
        int y1 = y;
        int y2 = y + h - 30;
        int y3 = y + h - 30;

        if (with_buttons) {
            slider->widget.y += 32;
            slider->widget.h -= 64;
        }
        if (with_reset_button) {
            y2 -= 32;
            slider->widget.h -= 32;
        }

        if (with_buttons) {
            otk_vector_button_new (otk, x, y1, w, 30,
                                   OSD_VECTOR_PLUS, 10, 10,
                                   slider_plus_cb, slider);
            otk_vector_button_new (otk, x, y2, w, 30,
                                   OSD_VECTOR_MINUS, 10, 10,
                                   slider_minus_cb, slider);
        }
        if (with_reset_button) {
            otk_vector_button_new (otk, x, y3, w, 30,
                                   OSD_VECTOR_HOME, 20, 20,
                                   slider_reset_cb, slider);
        }
    }

    return (otk_widget_t *) slider;
}
