
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: utils.c 2606 2007-07-25 09:51:03Z mschwerin $
 *
 */
#include "config.h"

#include <ctype.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>

#include "heap.h"
#include "logger.h"
#include "utils.h"


static void
replace_special_char (char *str, char *src, char dst)
{
    char *d;
    if (!str || !src) {
        return;
    }

    while ((d = strstr (str, src))) {
        int i = 0;
        int l = strlen (d);
        for (i = 1; i < l; i++)
            d[i] = d[i + 1];
        d[0] = dst;
    }
}


char *
replace_special_chars (char *str)
{
    if (!str) {
        return NULL;
    }

    replace_special_char (str, "ä", 'a');
    replace_special_char (str, "Ä", 'a');
    replace_special_char (str, "ö", 'o');
    replace_special_char (str, "Ö", 'o');
    replace_special_char (str, "ü", 'u');
    replace_special_char (str, "Ü", 'u');
    replace_special_char (str, "ß", 's');

    return str;
}


bool
swap_strings (const char *s1, const char *s2)
{
    if (!s1 || !s2) {
        return false;
    }

    char *n1 = ho_strdup (s1);
    char *n2 = ho_strdup (s2);

    n1 = replace_special_chars (n1);
    n2 = replace_special_chars (n2);

    bool swap = (strcasecmp (n1, n2) > 0);

    ho_free (n1);
    ho_free (n2);

    return swap;
}


static inline bool
is_space (char c)
{
    return ((c == ' ') || (c == '\t') || (c == '\r') || (c == '\n'));
}


char *
trim_whitespace (char *str)
{
    if (!str) {
        return NULL;
    }

    char *start = str;
    while (is_space (*start)) {
        ++start;
    }

    char *end = start + strlen (start);
    while (end > start && is_space (end[-1])) {
        --end;
    }
    *end = '\0';

    memmove (str, start, (end - start) + 1);

    return str;
}


char *
conv_lowercase (char *str)
{
    if (!str) {
        return NULL;
    }

    int i = 0;
    int l = strlen (str);
    for (; i < l; i++)
        str[i] = tolower (str[i]);

    return str;
}


bool
starts_with (const char *string, const char *start)
{
    if (!string || !start) {
        return false;
    }

    return (strncasecmp (string, start, strlen (start)) == 0);
}


bool
execute_shell (const char *command, int retval)
{
    int res = 0;

    if (!command || (strlen (command) == 0)) {
        return false;
    }

    debug ("Running '%s'...", command);
    if ((res = system (command)) != retval) {
        error (_("Execution of '%s' failed: %d!"), command, res);
        return false;
    }

    return true;
}


char *
xml_escape (const char *str, recode_t * xr, xml_escape_quote_t quote_type)
{
    if (!str) {
        return NULL;
    }

    char *str_encoded = recode (xr, str);
    if (!str_encoded) {
        return ho_strdup (str);
    }

    char *str_escaped = xml_escape_string (str_encoded, quote_type);
    if (!str_escaped) {
        return str_encoded;
    }

    ho_free (str_encoded);
    return str_escaped;
}
