
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: utils_files.c 2575 2007-07-22 11:45:48Z mschwerin $
 *
 */
#include "config.h"

#include <assert.h>
#include <dirent.h>
#include <errno.h>
#include <fcntl.h>
#include <libgen.h>
#include <mntent.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "download.h"
#include "desktop_entry.h"
#include "heap.h"
#include "i18n.h"
#include "list.h"
#include "logger.h"
#include "utils.h"


bool
has_extension (const char *filename, const char *extension)
{
    const char *ext = get_extension (filename);

    return ext && (strcasecmp (extension, ext) == 0);
}


const char *
get_extension (const char *filename)
{
    const char *ext;

    if (!filename) {
        return NULL;
    }

    ext = strrchr (filename, '.');

    if (!ext || (ext == filename)) {
        return NULL;
    }

    return (ext + 1);
}


static char *
get_cwd (void)
{
    char *ret = NULL;
    size_t size = 256;
    char *directory = ho_malloc (size);

    while (((ret = getcwd (directory, size)) == NULL)
           && (errno == ERANGE)) {
        size += 256;
        directory = ho_realloc (directory, size);
    }

    if (ret != directory) {
        ho_free (directory);
        directory = NULL;
        error (_("Failed to get current working directory: %s"),
               strerror (errno));
    }

    return directory;
}


char *
relative_to_absolute (const char *filename)
{
    l_list_t *list = l_list_new ();

    /* Get the current working directory. */
    char *_directory = get_cwd ();

    /* Make a copy of the filename. */
    char *_filename = ho_strdup (filename);

    /* Allocate enough memory to hold both filename and path. */
    char *_absolute =
        ho_malloc (strlen (_filename) + strlen (_directory) + 2);

    /* We add all parts of our current directory to the list. */
    if (filename[0] != '/') {
        char *saveptr;
        char *d = strtok_r (_directory, "/", &saveptr);
        while (d) {
            l_list_append (list, d);
            d = strtok_r (NULL, "/", &saveptr);
        }
    }

    /* We add all parts of the filename to the list. */
    {
        char *saveptr;
        char *f = strtok_r (_filename, "/", &saveptr);
        while (f) {
            if (strcmp (".", f) == 0) {
                /* Do nothing. */
            }

            else if (strcmp ("..", f) == 0) {
                l_list_remove (list, l_list_last (list));
            }

            else {
                l_list_append (list, f);
            }

            f = strtok_r (NULL, "/", &saveptr);
        }
    }

    /* Add all the parts from the list to one string. */
    char *p = _absolute;
    char *s = (char *) l_list_first (list);
    while (s) {
        *p++ = '/';
        char *q = s;
        while (*q != '\0')
            *p++ = *q++;
        s = (char *) l_list_next (list, s);
    }
    *p = '\0';

    l_list_free (list, NULL);

    char *_result = ho_strdup (_absolute);

    ho_free (_absolute);
    ho_free (_filename);
    ho_free (_directory);

    return _result;
}


char *
resolve_softlink (const char *filename)
{
    struct stat filestat;

    if (!filename) {
        return NULL;
    }

    /* We want to make sure we're handling an absolute path without any ups
     * and downs in between. */
    char *abs_mrl = relative_to_absolute (filename);

    if (lstat (abs_mrl, &filestat) == -1) {
        error (_("Unable to get status for file '%s': %s."),
               abs_mrl, strerror (errno));
        return abs_mrl;
    }

    if (!S_ISLNK (filestat.st_mode)) {
        return abs_mrl;
    }

    char link_target[1024];
    int len = readlink (abs_mrl, link_target, 1024);
    if (len == -1) {
        error (_("Unable to read link '%s' file: %s."),
               abs_mrl, strerror (errno));
        return abs_mrl;
    }

    if (link_target[len - 1] == '/')
        link_target[len - 1] = 0;
    else
        link_target[len] = 0;

    if (link_target[0] != '/') {
        char *dev = ho_strdup (abs_mrl);
        char *link = ho_strdup (link_target);
        snprintf (link_target, 1024, "%s/%s", dirname (dev), link);
        ho_free (dev);
        ho_free (link);
    }
#ifdef DEBUG
    char *abs_target = relative_to_absolute (link_target);
    debug ("found symbolic link: %s -> %s", abs_mrl, abs_target);
    ho_free (abs_target);
#endif
    ho_free (abs_mrl);

    return resolve_softlink (link_target);
}


char *
read_entire_file (const char *mrl, int *size)
{
    char *buffer = NULL;

    if (!mrl) {
        return NULL;
    }

    if (is_downloadable (mrl)) {
#ifdef HAVE_CURL
        download_t *download = download_new (NULL);

        if (download && network_download (mrl, download)) {
            buffer = download->buffer;
            *size = download->size;
        }

        download_free (download, false);
#endif /* HAVE_CURL */
    }
    else {
        if (access (mrl, R_OK) != 0) {
            error (_("Could not open '%s': %s!"), mrl, strerror (errno));
            return NULL;
        }

        struct stat st;
        if (stat (mrl, &st) < 0) {
            error (_("Unable to get status for file '%s': %s."), mrl,
                   strerror (errno));
            return NULL;
        }

        if ((*size = st.st_size) == 0) {
            error (_("File '%s' is empty!"), mrl);
            return NULL;
        }

        int fd;
        if ((fd = open (mrl, O_RDONLY)) == -1) {
            error (_("Could not open '%s': %s!"), mrl, strerror (errno));
            return NULL;
        }

        if ((buffer = (char *) ho_malloc (*size + 1)) == NULL) {
            close (fd);
            return NULL;
        }

        int bytes_read;
        if ((bytes_read = read (fd, buffer, *size)) != *size) {
            error (_("Could not read '%s': %s!"), mrl, strerror (errno));
            *size = bytes_read;
        }

        close (fd);

        buffer[*size] = '\0';
    }

    return buffer;
}


char *
get_basename (const char *mrl)
{
    if (!mrl) {
        return NULL;
    }

    char *mrl_copy = ho_strdup (mrl);
    char *mrl_base = ho_strdup (basename (mrl_copy));

    ho_free (mrl_copy);

    return mrl_base;
}


char *
get_dirname (const char *mrl)
{
    if (!mrl) {
        return NULL;
    }

    char *mrl_copy = ho_strdup (mrl);
    char *mrl_dirn = ho_strdup (dirname (mrl_copy));

    ho_free (mrl_copy);

    return mrl_dirn;
}


char *
get_thumbnail (const char *mrl)
{
    if (!mrl) {
        return NULL;
    }

    char *thumbnail_mrl = NULL;
    char *directory_mrl = NULL;
    if (!is_directory (mrl)) {
        directory_mrl = get_dirname (mrl);
    }
    else {
        directory_mrl = ho_strdup (mrl);
    }

    /* First we try to find an image called title.[jpg|png|gif] or
     * folder.[jpg|png|gif]. */
    {
        DIR *dirp;
        struct dirent *entp;

        if ((dirp = opendir (directory_mrl))) {
            while (!thumbnail_mrl && (entp = readdir (dirp))) {
                if (entp->d_name[0] == '.') {
                    continue;
                }
                if (!starts_with (entp->d_name, "title")
                    && !starts_with (entp->d_name, "folder")) {
                    continue;
                }

                thumbnail_mrl = ho_strdup_printf ("%s/%s",
                                                  directory_mrl,
                                                  entp->d_name);

                if (file_exists (thumbnail_mrl)
                    && is_file_image (thumbnail_mrl)) {
                    /* Yepee, we found it. */
                }
                else {
                    ho_free (thumbnail_mrl);
                }
            }
            closedir (dirp);
        }
    }

    /* Next we try to retrieve the thumbnail from a .directory file. */
    if (!thumbnail_mrl) {
        char *filename = ho_strdup_printf ("%s/.directory", directory_mrl);
        if (file_exists (filename)) {
            char *value = get_localestring_from_desktop_entry (filename,
                                                               "Desktop Entry",
                                                               "Icon");
            thumbnail_mrl = file_exists (value) ? value : NULL;
            ho_free (value);
        }
        ho_free (filename);

        if (file_exists (thumbnail_mrl) && is_file_image (thumbnail_mrl)) {
            /* Yepee, we found it. */
        }
        else {
            ho_free (thumbnail_mrl);
        }
    }

    /* Next we search for any image and use the first we find. */
    if (!thumbnail_mrl) {
        DIR *dirp;
        struct dirent *entp;

        if ((dirp = opendir (directory_mrl))) {
            while (!thumbnail_mrl && (entp = readdir (dirp))) {
                if (entp->d_name[0] == '.') {
                    continue;
                }

                thumbnail_mrl = ho_strdup_printf ("%s/%s",
                                                  directory_mrl,
                                                  entp->d_name);

                if (file_exists (thumbnail_mrl)
                    && is_file_image (thumbnail_mrl)) {
                    /* Yepee, we found it. */
                }
                else {
                    ho_free (thumbnail_mrl);
                }
            }
            closedir (dirp);
        }
    }

    ho_free (directory_mrl);
    return thumbnail_mrl;
}


bool
file_exists (const char *mrl)
{
    if (!mrl) {
        return false;
    }

    char *filename = mrl_get_filename (mrl);
    bool exists = (access (filename, R_OK | F_OK) == 0);
    ho_free (filename);

    return exists;
}


bool
mkdir_recursive (const char *mrl, mode_t mode)
{
    if (!mrl) {
        return false;
    }
    if (file_exists (mrl)) {
        return true;
    }

    if (mkdir (mrl, mode) == -1) {
        char *dirn = get_dirname (mrl);
        if (!mkdir_recursive (dirn, mode)) {
            ho_free (dirn);
            return false;
        }
        ho_free (dirn);

        if (mkdir (mrl, mode) == -1) {
            error (_("Could not create '%s': %s!"), mrl, strerror (errno));
            return false;
        }
    }

    return true;
}


/*
 * This first converts a relative path to an absolute 
 * one and then prepends 'file://' to it.
 */
static char *
filename_to_uri (const char *filename)
{
    if (!filename) {
        return NULL;
    }

    char *absolute = relative_to_absolute (filename);
    char *uri = ho_strdup_printf ("file://%s", absolute);

    ho_free (absolute);

    return uri;
}


typedef enum {
    UNSAFE_ALL = 0x1,                                           /* Escape all unsafe characters   */
    UNSAFE_ALLOW_PLUS = 0x2,                                    /* Allows '+'  */
    UNSAFE_PATH = 0x8,                                          /* Allows '/', '&', '=', ':', '@', '+', '$' and ',' */
    UNSAFE_HOST = 0x10,                                         /* Allows '/' and ':' and '@' */
    UNSAFE_SLASHES = 0x20                                       /* Allows all characters except for '/' and '%' */
} UnsafeCharacterSet;

static const char acceptable[96] = {
    /* A table of the ASCII chars from space (32) to DEL (127) */
    /*      !    "    #    $    %    &    '    (    )    *    +    ,    -    .    / */
    0x00, 0x3F, 0x20, 0x20, 0x28, 0x00, 0x2C, 0x3F,
    0x3F, 0x3F, 0x3F, 0x2A, 0x28, 0x3F, 0x3F, 0x1C,
    /* 0    1    2    3    4    5    6    7    8    9    :    ;    <    =    >    ? */
    0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
    0x3F, 0x3F, 0x38, 0x20, 0x20, 0x2C, 0x20, 0x20,
    /* @    A    B    C    D    E    F    G    H    I    J    K    L    M    N    O */
    0x38, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
    0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
    /* P    Q    R    S    T    U    V    W    X    Y    Z    [    \    ]    ^    _ */
    0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
    0x3F, 0x3F, 0x3F, 0x20, 0x20, 0x20, 0x20, 0x3F,
    /* `    a    b    c    d    e    f    g    h    i    j    k    l    m    n    o */
    0x20, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
    0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
    /* p    q    r    s    t    u    v    w    x    y    z    {    |    }    ~  DEL */
    0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
    0x3F, 0x3F, 0x3F, 0x20, 0x20, 0x20, 0x3F, 0x20
};

static const char hex[16] = "0123456789ABCDEF";

/**
 * @todo The original also escapes the ? as this can be part of a valid http
 *       URL I decided to leave it. I'm not sure if this is a good idea though. 
 */
//#define ACCEPTABLE_URI(a) ((a)>=32 && (a)<128 && (acceptable[(a)-32] & use_mask))
#define ACCEPTABLE_URI(a) ((a) == '?' || ((a)>=32 && (a)<128 && (acceptable[(a)-32] & use_mask)))

char *
filename_escape_to_uri (const char *filename)
{
    if (!filename) {
        return NULL;
    }

    /* We convert a normal filename to an URI, everything else we leave as
     * it is. This could be done a lot better as this only converts an
     * existing file to an URI (but we dont care about nonexistant files). */
    char *_filename;
    if (access (filename, F_OK) == 0)
        _filename = filename_to_uri (filename);
    else
        _filename = ho_strdup (filename);

    /* The following code was mostly taken from glib (g_filename_to_uri). */
    UnsafeCharacterSet use_mask = UNSAFE_PATH;
    int unacceptable = 0;

    const char *p;
    for (p = _filename; *p != '\0'; p++) {
        int c = (unsigned char) *p;
        if (!ACCEPTABLE_URI (c))
            unacceptable++;
    }

    char *result = ho_malloc (strlen (_filename) + 2 * unacceptable + 1);

    char *q = result;
    for (p = _filename; *p != '\0'; p++) {
        int c = (unsigned char) *p;
        if (!ACCEPTABLE_URI (c)) {
            *q++ = '%';
            *q++ = hex[c >> 4];
            *q++ = hex[c & 15];
        }
        else {
            *q++ = *p;
        }
    }
    *q = '\0';

    ho_free (_filename);

#if 0
    debug ("%s -> %s", filename, result);
#endif

    return result;
}

#define UNACCEPTABLE_SHELL(a) ((a == ' ') \
                              || (a == '"') \
                              || (a == '&') \
                              || (a == '(') \
                              || (a == ')') \
                              || (a == '\''))

char *
filename_escape_for_shell (const char *filename)
{
    if (!filename) {
        return NULL;
    }

    int unacceptable = 0;
    const char *p;
    for (p = filename; *p != '\0'; p++) {
        char c = (char) *p;
        if (UNACCEPTABLE_SHELL (c)) {
            unacceptable++;
        }
    }

    char *result = ho_malloc (strlen (filename) + unacceptable + 1);

    char *q = result;
    for (p = filename; *p != '\0'; p++) {
        char c = (char) *p;
        if (UNACCEPTABLE_SHELL (c)) {
            *q++ = '\\';
        }
        *q++ = *p;
    }
    *q = '\0';

#if 0
    debug ("%s -> %s", filename, result);
#endif

    return result;
}


bool
is_playlist_m3u (const char *mrl)
{
    /* This is the simple way to detect mime types. We could also use
     * gnomevfs-info -s <filename> */
    return has_extension (mrl, "m3u");
}


bool
is_playlist_pls (const char *mrl)
{
    /* Mimetype: audio/x-scpls */
    return has_extension (mrl, "pls");
}


bool
is_playlist_oxp (const char *mrl)
{
    return has_extension (mrl, "oxp");
}


bool
is_directory (const char *mrl)
{
    struct stat filestat;
    stat (mrl, &filestat);

    return S_ISDIR (filestat.st_mode) ? true : false;
}


/// Tests for the existance of a subentry of a directory.
static bool
subentry_exists (const char *directory, const char *subentry)
{
    char *mrl = ho_strdup_printf ("%s/%s", directory, subentry);
    bool exists = file_exists (mrl);
    ho_free (mrl);

    return exists;
}


bool
is_directory_dvd (const char *mrl)
{
    return (is_directory (mrl)
            && subentry_exists (mrl, "VIDEO_TS/VIDEO_TS.IFO"));
}


bool
is_directory_vcd (const char *mrl)
{
    return (is_directory (mrl)
            && (subentry_exists (mrl, "VCD/INFO.VCD")
                || subentry_exists (mrl, "SVCD/INFO.VCD")
                || subentry_exists (mrl, "SVCD/INFO.SVD")));
}


bool
is_iso_image (const char *mrl)
{
    return (has_extension (mrl, "iso")
            || has_extension (mrl, "img")
            || has_extension (mrl, "bin")
            || has_extension (mrl, "cue")
            || has_extension (mrl, "toc")
            || has_extension (mrl, "nrg")
            || has_extension (mrl, "mdf")
            || has_extension (mrl, "mds")
            || has_extension (mrl, "ccd"));
}


bool
is_file_hidden (const char *mrl)
{
    if (!mrl) {
        return false;
    }

    char *base = get_basename (mrl);
    bool hidden = starts_with (base, ".");
    ho_free (base);

    return hidden;
}


bool
is_file_image (const char *mrl)
{
    if (!mrl) {
        return false;
    }

    return (has_extension (mrl, "jpg")
            || has_extension (mrl, "jpeg")
            || has_extension (mrl, "png")
            || has_extension (mrl, "gif"));
}


off_t
get_filesize (const char *mrl)
{
    if (!mrl) {
        return -1;
    }

    struct stat st;
    if (stat (mrl, &st) < 0) {
        error (_("Unable to get status for file '%s': %s."),
               mrl, strerror (errno));
        return -1;
    }

    return st.st_size;
}
