/*
 * P3Scan v1.0
 *
 * (C) 2003 by Jack S. Lai <laitcg@cox.net>
 *
 * It's intent is to provide a follow on program to POP3-Virusscan-Proxy 0.4
 * by Folke Ashberg <folke@ashberg.de>.
 *
 * It is based upon his program but provides numerous changes to include
 * scanning pop3 mail for spam, hardening the program, addaption to todays
 * email environment, and many other changes.
 *
 * The initial release of p3scan includes patches made and submitted to the
 * original project but were never incorporated. Please see the README for
 * further information.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <arpa/inet.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdarg.h>
#include <sys/signal.h>
#include <sys/wait.h>
#include <pwd.h>
#include <time.h>
#include <sys/time.h>
#include <syslog.h>
#include <sys/param.h>
#include <ctype.h>
#include <linux/netfilter_ipv4.h>
#include <malloc.h>
#include <getopt.h>
#include <netdb.h>
#include <libgen.h>
#include <errno.h>
#include <dirent.h>
#include <sys/statvfs.h>

#include "p3scan.h"
#include "getline.h"
#include "parsefile.h"
#include "scanner.h"

#include "ripmime/mime.h"
#include "ripmime/ripmime-api.h"          /* ripmime-1.0.3.5 */
//#include "ripmime/MIME_headers_api.h"   /* ripmime-1.0.3.4 */

/* globals */
int numprocs;
struct configuration_t * config;
/* default configuration, anything can be changed at runtime */
#define PORT_NUMBER              8110
#define MAX_CHILDS               10
#define RUNAS_USER               "mail"
#define VIRUS_DIR	               "/var/spool/p3scan"
#define NOTIFY_MAIL_DIR	         "/var/spool/p3scannotify"
#define VIRUS_SCANNER            NULL
#define VIRUS_SCANNER_VIRUSCODE  1
#define PID_FILE                 "/var/run/p3scan/p3scan.pid"
#define SYSLOG_NAME              "p3scan"
#define CONFIGFILE               "/etc/p3scan/p3scan.conf"
#define VIRUS_TEMPLATE           "/etc/p3scan/p3scan.mail"
#define DEBUG                    0
#define QUIET                    0
#define OVERWRITE                0
#define CHECKSPAM                0
#define SPAMCHECK                "/usr/bin/spamc"
#define MINSPACE                 0
#define DELIT                    0
/* TOS:  do not set, or use IPTOS_[LOWDELAY|THROUGHPUT|RELIABILITY|LOWCOST] */
#define SET_TOS                  IPTOS_THROUGHPUT

#undef DEBUG_MEM                 /* print meminfo every log message when debugging */
#undef DEBUG_MESSAGE             /* print message lines */

/* default configuration ends here */

/* globals / protos */

/* filecount from ripmime/mime.c */
extern int filecount;

static int sockfd; /* has to be global, do_sigterm_main() want's to close them */
/* the proxycontext is global for do_sigterm_proxy().
 * no other function should use it! */
static struct proxycontext* global_p;

void do_log(int level, const char *fmt,...)
{
    char puffer[4096];
    va_list az;
#ifdef DEBUG_MEM
    struct mallinfo m=mallinfo();
#endif

   if (!config->debug && level==LOG_DEBUG) return;
   if (config->quiet && level==LOG_NOTICE) return;
   va_start(az,fmt);
   vsnprintf(puffer, 4000, fmt, az);
   if (!config->debug){
      openlog(config->syslogname, LOG_PID|LOG_CONS, LOG_DAEMON);
      syslog(LOG_NOTICE, "%s\n", puffer);
      closelog();
   } else {
      fflush(stdout);
      fprintf(stderr, "%s[%i]: "
#ifdef DEBUG_MEM
      "Mem: %i "
#endif
      "%s\n", config->syslogname, getpid(),
#ifdef DEBUG_MEM
      m.uordblks,
#endif

      puffer);
      fflush(NULL);
   }
   if (level==LOG_EMERG){
      do_log(LOG_NOTICE, "Exiting now...\n");
      fprintf(stderr, "%s\n", puffer);
      exit(1);
   }
   return;
}

void avoid_root(void){
   struct passwd *userInfo;
   if (getuid()!=0) return; /* then it's ok */
   do_log(LOG_DEBUG, "Changing uid (we are root)");
   userInfo = getpwnam(config->runasuser);
   if ((setgid(userInfo->pw_gid)) == -1) do_log(LOG_EMERG, "Can't change to group of user %s (%i.%i)", config->runasuser, userInfo->pw_uid, userInfo->pw_gid);
   if ((setuid(userInfo->pw_uid)) == -1) do_log(LOG_EMERG, "Can't change to user %s (%i.%i)", config->runasuser, userInfo->pw_uid, userInfo->pw_gid);
   do_log(LOG_DEBUG, "Changed UID.GID to %i.%i", userInfo->pw_uid, userInfo->pw_gid);
}

int scan_directory(struct proxycontext *p){
   int ret, ret_all;
   DIR *dp;
   struct dirent *de;
   struct stat  s;
   char * file;
   int maildirlen;
   char * virname;
#define VISIZE 1000
   char *vi=malloc(VISIZE);
   int vipos = 0;

   /* scan directory */
   maildirlen=strlen(p->maildir);
   if (stat (p->maildir, &s) == -1){
      do_log(LOG_EMERG, "%s does not exists", p->maildir);
      return SCANNER_RET_ERR;
   }
   if (!S_ISDIR(s.st_mode)){
      do_log(LOG_EMERG, "oops, %s is not a directory", p->maildir);
      return SCANNER_RET_ERR;
   }
   if ((dp = opendir (p->maildir)) == NULL){
      do_log(LOG_EMERG, "oops, can't open directory %s", p->maildir);
      return SCANNER_RET_ERR;
   }
   ret_all=SCANNER_RET_OK;
   vi[0]='\0';
   while ((de = readdir (dp)) != NULL){
   if (strcmp (de->d_name, ".") == 0) continue;
   if (strcmp (de->d_name, "..") == 0) continue;
   file=malloc(maildirlen + strlen(de->d_name) +2);
   sprintf(file, "%s/%s", p->maildir, de->d_name);
   if (lstat (file, &s) == -1){
      do_log(LOG_EMERG, "Can't stat %s - I won't touch it.", file);
      free(file);
      continue;
   }
   if (!S_ISREG(s.st_mode)){
      do_log(LOG_EMERG, "%s is not a regular file. I won't touch it.", file);
      free(file);
      continue;
   }
   /* build filename */
   do_log(LOG_DEBUG, "Going to scan '%s'", file);

   p->scanthis=file;
   virname=NULL;
   if (config->delit) p->filestatus="Message deleted";
   else p->filestatus="";
   ret=config->scanner->scan(p, &virname);

   if (ret==SCANNER_RET_VIRUS){
      ret_all=SCANNER_RET_VIRUS;
      if (virname && strlen(virname)<VISIZE - vipos - 4){
         strcat(&vi[vipos], virname);
         vipos+=strlen(virname);
         strcat(vi, " & ");
         vipos+=3;
      }
   } else if (ret == SCANNER_RET_ERR && ret_all != SCANNER_RET_VIRUS)
      ret_all = SCANNER_RET_ERR;
      if (virname) free(virname);
         free(file);
      }
      closedir (dp);
      if (vipos>4){
         vi[vipos-3]='\0';
         p->virinfo=vi;
   } else p->virinfo=NULL;
   return ret_all;
}

char *sstrdel(char *s, ...){
   /* Find out how many arguments are present */
   int c = 0;
   va_list ap, hold;

   if (s == NULL) return NULL;
   va_start(ap, s);
   memcpy(&hold, &ap, sizeof(va_list));
   while (va_arg(ap, char*) != NULL)
      c++;
   va_end(ap);
   if (c) {
      /* Assign pointers  */
      char *r = s,*n = s;
      char *p;
      int len, i;
      /* Copy next character to result */
      /* And then check for matches if */
      /* not at end of string          */
      while ((*r = *n) != 0){
         int l = 0;
         /* Substitute for va_start(ap,s) */
         memcpy(&ap, &hold, sizeof(va_list));
         for (i = 0; i < c; i++){
            /* Initialise the pointer and the length    */
            len = strlen(p = va_arg(ap, char*));
            /* Compare ONLY if we haven't found one yet */
            /* or if this one is bigger than the one we */
            /* found AND this arg has a length > 0      */
            if(len > 0 && (l == 0 || len> l) && strncmp(n, p, len) == 0){
               l = len;
            }
         }
         va_end(ap);
         if (l) n += l;
         else  n++, r++;
      }
   }
   return s;
}

char *parsehtml(char * line){
   /* Parse to remove HTML tags. */
#define MAX_BUF_SIZE 65535
   static char clean[MAX_BUF_SIZE], buf[5];
   char *start, *end, *ptr, *tag1, *tag2, *tag3, *tag4;
   int i;

   do_log(LOG_DEBUG,"HTML: In: %s",line);
   ptr=line;
   strcpy (clean, "");
   while (1) {
      /* find the beginning of a tag */
      start=strchr(ptr, '<');
      /* there aren't any more, just append the rest of the string */
      if (!start) {
         strncat(clean, ptr, MAX_BUF_SIZE - (start - line));
         break;
      }
      /* copy up to the start of tag */
      strncat(clean, ptr, (start-ptr));
      /* find the end of a tag */
      end=strchr(start, '>');
      /* no end of tag, just append the rest of the string */
      if (!end){
         strncat(clean, start, MAX_BUF_SIZE - (start - line));
         break;
      }
      /* a couple of tag types will be converted */
      if ((end-start) < 5){
         strcpy (buf, "");
         strncat (buf, start, (end-start)+1);
         for (i=0; buf[i]; i++) buf[i]=toupper(buf[i]);
         /* line break tag */
         if (!strcmp (buf, "<BR>")) strcat (clean, "\n");
         /* paragraph tag */
         else if (!strcmp (buf, "<P>")) strcat (clean, "\n\n");
      }
      ptr = end+1;
   }
   /* These are the tags we are looking for on multiple lines:
      for example: line1 "this is a web bug... <img src=
                   line2 "http://somewebsite.com/bug.gif?email@address.com>"
      These tags will be DELETED! */
   //do_log(LOG_DEBUG,"HTML: clean: %s",clean);
   tag1="SRC";
   tag2="src";
   tag3="HREF";
   tag4="href";
   line=sstrdel(clean, tag1, tag2, tag3, tag4, NULL);
   //strcpy(line,clean); /* This is used in the event multi-line checking is disabled */
   do_log(LOG_DEBUG,"HTML Out: %s",line);
   /* TODO: add code to catch broken lines "hr""ef", etc...
      only when some encoding schemes are used. */
   return line;
}

int scan_mailfile(struct proxycontext *p){
   int ret, viret;
   DIR *dp;
   struct dirent *de;
   int maildirlen;
   char * file;
   int spamfd=-1;
   unsigned long len=0;
   char spmcmd[512];
   char newmsg[512];
#define NEWMSG "newmsg"
#define MOVEIT "/bin/mv"
   FILE * scanner;
   static char  line[4096*16];
   struct statvfs fs;
   int kbfree;
   struct linebuf *filebuf;
   int res, htmlfd, html, toggle;

  /* See if we have enough room to process the message based upon
   what the user determines is enough room in p3scan.conf */
   if ( statvfs( config->virusdir, &fs ) == SCANNER_RET_ERR){
      do_log(LOG_EMERG, "Unable to get available space!");
      return SCANNER_RET_CRIT; // Should never reach here, but keep it clean. :)
   }
   kbfree=(fs.f_bavail * fs.f_frsize / 1024);
   if ( config->freespace != 0 && kbfree < config->freespace ){
      do_log(LOG_CRIT, "Not enough space! Available space: %d", kbfree);
      return SCANNER_RET_CRIT;
   }

   /* This is where we should scan for spam - before demime to
      give SpamAssassin the virgin message */
   do_log(LOG_DEBUG, "Check for spam = %i", config->checkspam);
   //TODO: Move this out. Make it a module.
   if (config->checkspam){
      // Ok, first, create a new message
      len=strlen(config->virusdir)+strlen(NEWMSG);
      snprintf(newmsg, len, "%s%s", config->virusdir,NEWMSG);
      do_log(LOG_DEBUG, "newmsg = %s", newmsg);
      do_log(LOG_DEBUG, "oldmsg = %s", p->mailfile);
      if ((spamfd=open(newmsg,O_WRONLY | O_CREAT | O_TRUNC,  S_IRUSR | S_IWUSR))<0){
         do_log(LOG_ALERT, "Can't create newmsg!");
         return SCANNER_RET_CRIT;
      }
      // Now call spamc
      len=strlen(config->spamcheck)+strlen(" < ")+strlen(p->mailfile)+strlen(" 2>&1 ");
      // Trap spamassassin output to a buffer.
      snprintf(spmcmd, len, "%s < %s 2>&1", config->spamcheck, p->mailfile);
      do_log(LOG_DEBUG, "popen %s", spmcmd);
      if ((scanner=popen(spmcmd, "r"))==NULL){
         do_log(LOG_ALERT, "Can't start spammer '%s' !!!", spmcmd);
         return SCANNER_RET_ERR;
      }
      toggle=1;
      while ((fgets(line, 4095, scanner))!=NULL){
         line[strlen(line)-1]='\0';
         // write the buffer to our new message
         do_log(LOG_DEBUG, "SpammerLine: '%s'", line);
         if ((strncmp(line,".",1 ) == 0 && strlen(line) == 1)){
            do_log(LOG_DEBUG, "Not writing '.'");
         } else if ((strncmp(line,".\r",2) == 0 && strlen(line) == 2)){
            do_log(LOG_DEBUG, "Not writing '.'");
         } else if ((strncmp(line,"X-Spam-Checker-Version",22) == 0 && toggle)){
            writeline(spamfd, WRITELINE_LEADING_N, "X-Virus-Scanner: " PROGNAME " Version " VERSION " by <laitcg@cox.net>/<folke@ashberg.de>");
            writeline(spamfd, WRITELINE_LEADING_N, line);
           toggle=0;
         } else {
            writeline(spamfd, WRITELINE_LEADING_N, line);
         }
      }
      do_log(LOG_DEBUG, "Writing new .");
      writeline(spamfd, WRITELINE_LEADING_N, ".");
      if((spamfd=close(spamfd))<0){
         do_log(LOG_EMERG, "Can't close newmsg Err: %s", spamfd);
         return SCANNER_RET_CRIT;
      }
      ret=pclose(scanner);
      /* Spam report is now in $virusdir/newmsg
         so now replace original msg with newmsg */
      len=strlen(MOVEIT)+1+strlen(newmsg)+1+strlen(p->mailfile)+1;
      do_log(LOG_DEBUG, "MV: %s %s %s=%i",MOVEIT,newmsg,p->mailfile,len);
      snprintf(spmcmd, len, "%s %s %s",MOVEIT,newmsg,p->mailfile);
      if ((scanner=popen(spmcmd, "r"))==NULL){
         do_log(LOG_ALERT, "Can't '%s' !!!", spmcmd);
         return SCANNER_RET_ERR;
      }
      ret=pclose(scanner);
   }
   /* End of spam checking */
   p->virinfo=NULL;
   if (config->demime){
      /* extract MIME Parts into maildir */
      /* we reset the filecount which is defined in ripmime/mime.c */
      /*filecount=0; */
      do_log(LOG_DEBUG, "DeMIMEing to %s", p->maildir);
      viret = mkdir(p->maildir, S_IRWXU);
      if ((viret == -1)&&(errno != EEXIST)){
         do_log(LOG_CRIT, "Cannot create directory '%s' (%s). Can't scan mail.\n",
         p->maildir, strerror(errno));
         return SCANNER_RET_CRIT;
      }
      MIMEH_set_outputdir( p->maildir );
      MIME_init();
      MIME_unpack( p->maildir, p->mailfile, 0);
      MIME_close();
      // TODO: Error checking
      p->scanthis = p->maildir;

      /* SCAN */
      if (config->scanner->dirscan){
         /* scanner wants to scan the directory itself, so call it once */
         /* give directory name to scanner, if he want (basic) */
         do_log(LOG_DEBUG, "Invoking scanner");
         p->virinfo=NULL;
         viret=config->scanner->scan(p, &p->virinfo);
         do_log(LOG_DEBUG, "Scanner returned %i", viret);
      } else {
         /* give all files to scanner
          * also connect all virusinfos to one string */
         viret=scan_directory(p);
      }

      /* unlinking MIME-files */
      do_log(LOG_DEBUG, "Unlinking deMIMEd files", p->maildir);
      maildirlen=strlen(p->maildir);
      if ((dp = opendir (p->maildir)) == NULL){
         do_log(LOG_EMERG, "Oops, can't open directory %s for unlinking files", p->maildir);
      } else {
         while ((de = readdir (dp)) != NULL){
            if (strcmp (de->d_name, ".") == 0) continue;
            if (strcmp (de->d_name, "..") == 0) continue;
            file=malloc(maildirlen + strlen(de->d_name) +2);
            sprintf(file, "%s/%s", p->maildir, de->d_name);
            do_log(LOG_DEBUG, "unlink(%s)", file);
            if ((unlink(file)<0))do_log(LOG_EMERG, "File Error! Could not erase %s",file);
            free(file);
         }
         closedir (dp);
         do_log(LOG_DEBUG, "removing directory %s", p->maildir);
         rmdir(p->maildir);
      }
   } else { /* if config->demime */
      /* no demime */
      p->scanthis = p->mailfile;
      /* invoke configured scanner */
      do_log(LOG_DEBUG, "Invoking scanner");
      p->virinfo=NULL;
      viret=config->scanner->scan(p, &p->virinfo);
      do_log(LOG_DEBUG, "Scanner returned %i", viret);
      // TODO: Fail on unexpected return code.
   }
   if (p->virinfo){
      TRIM(p->virinfo);
   }
   //TODO: Move this out. Make it a module.
   /* Do we want to rename attachments? */
   do_log(LOG_DEBUG, "Rename attachmments = %s", config->renattach);
   if (config->renattach != NULL && !viret ){
   // Only rename non-infected attachments
      len=strlen(config->virusdir)+strlen(NEWMSG);
      snprintf(newmsg, len, "%s%s", config->virusdir,NEWMSG);
      do_log(LOG_DEBUG, "REN newmsg = %s", newmsg);
      do_log(LOG_DEBUG, "REN oldmsg = %s", p->mailfile);
      if ((spamfd=open(newmsg,O_WRONLY | O_CREAT | O_TRUNC,  S_IRUSR | S_IWUSR))<0){
         do_log(LOG_ALERT, "Can't create newmsg!");
         return SCANNER_RET_CRIT;
      }
      len=strlen(config->renattach)+strlen(" < ")+strlen(p->mailfile)+strlen(" 2>&1 ");
      snprintf(spmcmd, len, "%s < %s 2>&1", config->renattach, p->mailfile);
      do_log(LOG_DEBUG, "popen %s", spmcmd);
      if ((scanner=popen(spmcmd, "r"))==NULL){
         do_log(LOG_ALERT, "Can't start renattach '%s' !!!", spmcmd);
         return SCANNER_RET_ERR;
      }
      while ((fgets(line, 4095, scanner))!=NULL){
         line[strlen(line)-1]='\0';
         do_log(LOG_DEBUG, "AttachLine: '%s'", line);
            writeline(spamfd, WRITELINE_LEADING_N, line);
      }
      if((spamfd=close(spamfd))<0){
         do_log(LOG_EMERG, "Can't close newmsg Err: %s", spamfd);
         return SCANNER_RET_CRIT;
      }
      ret=pclose(scanner);
      len=strlen(MOVEIT)+1+strlen(newmsg)+1+strlen(p->mailfile)+1;
      do_log(LOG_DEBUG, "REN-MV %s %s %s=%i",MOVEIT,newmsg,p->mailfile,len);
      snprintf(spmcmd, len, "%s %s %s",MOVEIT,newmsg,p->mailfile);
      if ((scanner=popen(spmcmd, "r"))==NULL){
         do_log(LOG_ALERT, "Can't '%s' !!!", spmcmd);
         return SCANNER_RET_ERR;
      }
      ret=pclose(scanner);
   }
   /* end renattach */
   /* Start HTML parsing */
   if (config->overwrite && !viret){
      // Do not parse infected mail as client will not see it anyway.
      len=strlen(config->virusdir)+strlen(NEWMSG);
      snprintf(newmsg, len, "%s%s", config->virusdir,NEWMSG);
      do_log(LOG_DEBUG, "HTML newmsg = %s", newmsg);
      do_log(LOG_DEBUG, "HTML oldmsg = %s", p->mailfile);
      if ((spamfd=open(newmsg,O_WRONLY | O_CREAT | O_TRUNC,  S_IRUSR | S_IWUSR))<0){
         do_log(LOG_ALERT, "Can't create newmsg!");
         return SCANNER_RET_CRIT;
      }
      len=strlen(p->mailfile);
      if ((htmlfd=open(p->mailfile,O_RDONLY))<0){
         do_log(LOG_ALERT, "Can't open for HTML parsing!");
         return SCANNER_RET_CRIT;
      }
      filebuf=linebuf_init(16384);
      toggle=0; /* past the header */
      html=0;  /* Don't scan non-HTML messages */
      while ( (res=getline(htmlfd, filebuf))>=0){
         if (filebuf->linelen >=0 ){
            len += filebuf->linelen;
            if (toggle){
               if (!html){
                  if ((strncasecmp(filebuf->line,"<html>",6)==0 ||
                       strncasecmp(filebuf->line,"<head>",6)==0 ||
                       strncasecmp(filebuf->line,"<body>",6)==0 ||
                       strncasecmp(filebuf->line,"Content-Type: text/html",23)==0)){
                       html=1;
                       do_log(LOG_DEBUG, "FOUND HTML!");
                       filebuf->line=parsehtml(filebuf->line);
                       }
                       /* dont parse non-html*/
               }
               else filebuf->line=parsehtml(filebuf->line);
            }
            /* The following line has to match the header line
               written above during spam checking or below in proxy()*/
            else if ((strcmp("X-Virus-Scanner: " PROGNAME " Version " VERSION " by <laitcg@cox.net>/<folke@ashberg.de>", filebuf->line)==0)){
               toggle=1;
               //writeline(spamfd, WRITELINE_LEADING_N, filebuf->line);
            }
            if (writeline(spamfd, WRITELINE_LEADING_N, filebuf->line)){
               do_log(LOG_EMERG, "Can't write parsed HTML line!");
               /* we are dead now. Should not reach here. But allow it
               to fall through in case LOG_EMERG is changed in the future. */
               linebuf_uninit(filebuf);
               return 0;
            }
         }
      }
      linebuf_uninit(filebuf);
      close(htmlfd);
      len=strlen(MOVEIT)+1+strlen(newmsg)+1+strlen(p->mailfile)+1;
      do_log(LOG_DEBUG, "HTML-MV: %s %s %s=%i",MOVEIT,newmsg,p->mailfile,len);
      snprintf(spmcmd, len, "%s %s %s",MOVEIT,newmsg,p->mailfile);
      if ((scanner=popen(spmcmd, "r"))==NULL){
         do_log(LOG_ALERT, "Can't '%s' !!!", spmcmd);
         return SCANNER_RET_ERR;
      }
      ret=pclose(scanner);
   }
   /* End HTML parsing */
   if (strlen(NONULL(p->virinfo))<1){
      if (p->virinfo) free(p->virinfo);
      p->virinfo=strdup(MESSAGE_NOVIRINFO);
   }
   ret=viret;
   return ret;
}

unsigned long send_mailfile(char * mailfile, int fd){
   struct stat_t st;
   struct linebuf *filebuf;
   int mailfd;
   int res;
   unsigned long len=0;
   int gotprd;

   if ( (mailfd=open(mailfile, O_RDONLY ))<0){
      do_log(LOG_EMERG, "Can't open mailfile (%s)!\n", mailfile);
      return 0;
   }
   /* getmail-size */
   if (FSTAT(mailfd, &st)){
      do_log(LOG_EMERG, "Can't fstat mail!\n");
      return 0;
   }
   if (writeline_format(fd, WRITELINE_LEADING_RN, "+OK %lu octets", st.st_size)){
      do_log(LOG_EMERG, "can't send mail-response to client");
      return 0;
   }

   filebuf=linebuf_init(16384);
   //TODO: error checking
   gotprd=0;
   while ( (res=getline(mailfd, filebuf))>=0){
      if (filebuf->linelen >=0 ){
         len += filebuf->linelen;
#ifdef DEBUG_MESSAGE
         do_log(LOG_DEBUG, ">%s", filebuf->line);
#endif
         if ((strncmp(filebuf->line,".",1 ) == 0 && strlen(filebuf->line) == 1)) gotprd=1;
         if ((strncmp(filebuf->line,".\r",2) == 0 && strlen(filebuf->line) == 2)) gotprd=1;
         if (writeline(fd, WRITELINE_LEADING_RN, filebuf->line)){
            do_log(LOG_EMERG, "can't send mail to client");
            /* we are dead now. Should not reach here. But allow it
            to fall through in case LOG_EMERG is changed in the future. */
            linebuf_uninit(filebuf);
            return 0;
         }
      }
   }
   if (res!=GETLINE_EOF){
      do_log(LOG_CRIT, "error reading from mailfile %s, error code: %d", mailfile, res);
      linebuf_uninit(filebuf);
      return 0;
   }
   if (!gotprd)writeline(fd, WRITELINE_LEADING_N, ".");
   linebuf_uninit(filebuf);
   close(mailfd);
   return len;
}

void set_defaultparams(struct proxycontext * p){
   char buf[256];
   gethostname(buf, 256);
   paramlist_set(p->params, "%HOSTNAME%", buf);
   getdomainname(buf, 256);
   paramlist_set(p->params, "%DOMAINNAME%", buf);
   paramlist_set(p->params, "%PROGNAME%", PROGNAME);
   paramlist_set(p->params, "%VERSION%", VERSION);
// paramlist_set(p->params, "%VERSION_CVS%", VERSION_CVS);

   paramlist_set(p->params, "%CLIENTIP%", inet_ntoa(p->client_addr.sin_addr));
   sprintf(buf, "%i", ntohs(p->client_addr.sin_port));
   paramlist_set(p->params, "%CLIENTPORT%", buf);
   paramlist_set(p->params, "%SERVERIP%", inet_ntoa(p->server_addr.sin_addr));
   sprintf(buf, "%i", ntohs(p->server_addr.sin_port));
   paramlist_set(p->params, "%SERVERPORT%", buf);

}

void set_maildateparam(struct paramlist * params){
   char buf[256];
   int diff_hour, diff_min;
   time_t now = time(NULL);
   struct tm *tm = localtime(&now);
   struct tm local;
   struct tm *gmt;
   int len;
   memcpy(&local, tm, sizeof(struct tm));
   gmt = gmtime(&now);

   diff_min = 60*(local.tm_hour - gmt->tm_hour) + local.tm_min - gmt->tm_min;
   if (local.tm_year != gmt->tm_year)
      diff_min += (local.tm_year > gmt->tm_year)? 1440 : -1440;
   else if (local.tm_yday != gmt->tm_yday)
      diff_min += (local.tm_yday > gmt->tm_yday)? 1440 : -1440;
      diff_hour = diff_min/60;
      diff_min  = abs(diff_min - diff_hour*60);

     len = strftime(buf, sizeof(buf), "%a, ", &local);
    (void) sprintf(buf + len, "%02d ", local.tm_mday);
    len += strlen(buf + len);
    len += strftime(buf + len, sizeof(buf) - len, "%b %Y %H:%M:%S", &local);
    (void) sprintf(buf + len, " %+03d%02d", diff_hour, diff_min);
    paramlist_set(params, "%MAILDATE%", buf);
}

void set_paramsfrommailheader(char * mailfile, struct paramlist * params){
   struct linebuf *l;
   int fd;
   char * c;
   if ( (fd=open(mailfile, O_RDONLY ))<0) return;
   l=linebuf_init(4096*16);
   while (getline(fd, l)>=0){
      if (l->linelen >0 ){
	      if (!strncasecmp(l->line, "from: ", 6)){
            c=l->line+6;
            TRIM(c);
            paramlist_set(params, "%MAILFROM%", c);
         } else if (!strncasecmp(l->line, "subject: ", 9)) {
            c=l->line+9;
            TRIM(c);
            paramlist_set(params, "%SUBJECT%", c);
         } else if (!strncasecmp(l->line, "To: ", 4)) {
            c=l->line+4;
            TRIM(c);
            paramlist_set(params, "%MAILTO%", c);
         }
      } else if (l->linelen == 0)
         break; /* only the header */
   }
   linebuf_uninit(l);
   close(fd);
}

void unset_paramsfrommailheader(struct paramlist * params){
   paramlist_set(params, "%MAILFROM%", NULL);
   paramlist_set(params, "%SUBJECT%", NULL);
   paramlist_set(params, "%MAILTO%", NULL);
}

int do_virusaction(struct proxycontext * p){
   char *mail;
   char comm[4096];
   unsigned long len;
   mail=malloc(strlen(p->mailfile)+10);
#define  MOVEIT   "/bin/mv"
   /* If mail is infected AND we just want to delete it, just don't move it */
   if (!config->delit){
      snprintf(comm,4096,"%s %s %s",MOVEIT,p->mailfile,config->virusdirbase);
      do_log(LOG_DEBUG,"Moving the infected file %s",comm);
      system(comm);
   }
   sprintf(mail, "%s/%i.mailout", config->notifydir,getpid());
   do_log(LOG_DEBUG,"mail=%s",mail);
   if (parsefile(config->virustemplate, mail, p->params, WRITELINE_LEADING_RN)) {
      do_log(LOG_EMERG, "Can't open mail notification template %s",config->virustemplate);
      unlink(mail);
      return -1;
   }
   do_log(LOG_DEBUG, "sending new mail");
   len=send_mailfile(mail, p->client_fd);
   p->bytecount+=len;
   unlink(mail);
   free(mail);
   if (len>0) return 0;
   return -1;
}

struct proxycontext * context_init(void){
   struct proxycontext * p;
   p=malloc(sizeof(struct proxycontext));
   p->ismail=0;
   p->msgnum=0;
   p->mailcount=0;
   p->bytecount=0;
   p->socksize=sizeof(struct sockaddr_in);
   p->client_fd=-1;
   p->server_fd=-1;
   p->clientbuf=NULL;
   p->serverbuf=NULL;
   p->virinfo=NULL;
   p->scanthis=NULL;
   p->scannerinit=SCANNER_INIT_NO;
   return p;
}

void context_uninit(struct proxycontext * p){
   if (p->client_fd > 0 ) close(p->client_fd);
   if (p->server_fd > 0 ) close(p->server_fd);
   paramlist_uninit(&p->params);
   linebuf_uninit(p->clientbuf);
   linebuf_uninit(p->serverbuf);
   free(p);
}

void do_sigterm_proxy(int signr){
   do_log(LOG_DEBUG, "do_sigterm_proxy, signal %i", signr);
   if (global_p == NULL){
      do_log(LOG_DEBUG, "already uninitialized");
   return;
   }
   if (signr != -1) /* sig -1 is ok */
      do_log(LOG_CRIT, "We cot SIGTERM!");
   if (global_p->client_fd != -1) close(global_p->client_fd);
   if (global_p->server_fd != -1) close(global_p->server_fd);
   if (global_p->scannerinit==SCANNER_INIT_OK && config->scanner->uninit2){
      do_log(LOG_DEBUG, "calling uninit2");
      config->scanner->uninit2(global_p);
      do_log(LOG_DEBUG, "uninit2 done");
   }
   do_log(LOG_DEBUG, "Uninit context");
   context_uninit(global_p);
   global_p=NULL;
   do_log(LOG_DEBUG, "context_uninit done, exiting now");
   if (signr != -1) exit(1);
}

int proxy(struct proxycontext *p){
   fd_set fds_read;
   struct timeval timeout;
   int scanfd=-1;
   int error;
//    time_t epoch;
//    struct tm *tm;
   int clientret, serverret;
   unsigned long len;
   char buf[64];
#ifdef SET_TOS
   int tos;
#endif
   int scannerret, ret;

   do_log(LOG_NOTICE, "Connection from %s:%i", inet_ntoa(p->client_addr.sin_addr), ntohs(p->client_addr.sin_port));

   p->server_addr.sin_family = AF_INET;
   if (getsockopt(p->client_fd, SOL_IP, SO_ORIGINAL_DST, &p->server_addr, &p->socksize)){
      do_log(LOG_CRIT, "No IP-Conntrack-data (getsockopt failed)");
      return 1;
   }
   do_log(LOG_NOTICE, "Real-server adress is %s:%i", inet_ntoa(p->server_addr.sin_addr), ntohs(p->server_addr.sin_port));
   if (
      /* try to avoid loop */
      ((ntohl(p->server_addr.sin_addr.s_addr) == INADDR_LOOPBACK) &&
      p->server_addr.sin_port == config->addr.sin_port ) ||
      /* src.ip == dst.ip */
      (p->server_addr.sin_addr.s_addr == p->client_addr.sin_addr.s_addr)
      ){
         do_log(LOG_CRIT, "Oops, that would loop!");
         return 1;
      }
      /* open socket to 'real-server' */
      if ( (p->server_fd = socket(PF_INET, SOCK_STREAM, 0)) < 0){
         do_log(LOG_CRIT, "Cannot open socket to real-server");
         return 1;
      }
#ifdef SET_TOS
      tos=SET_TOS;
      if (setsockopt(p->client_fd, IPPROTO_IP, IP_TOS, &tos, sizeof(tos)))
         do_log(LOG_WARNING, "Can't set TOS (incoming connection)");
      if (setsockopt(p->server_fd, IPPROTO_IP, IP_TOS, &tos, sizeof(tos)))
         do_log(LOG_WARNING, "Can't set TOS (outgoing connection)");
#endif
      if (connect(p->server_fd, (struct sockaddr *) &p->server_addr, p->socksize)){
         do_log(LOG_CRIT, "Cannot connect to real-server");
         return 1;
      }

      p->clientbuf=linebuf_init(4096*16);
      p->serverbuf=linebuf_init(4096*16);
      p->params=paramlist_init();
      set_defaultparams(p);
      /* releasing sockfd
      * it seems to work, that if the listener (our parent-PID) gots kicked
      * we can work AND another listener can bind to the port
      */
      close(sockfd);
      do_log(LOG_DEBUG, "starting mainloop");
      while ( 1 ){
      /* read from client */
      if ((clientret=getline(p->client_fd, p->clientbuf))<0){
         do_log(LOG_DEBUG, "Closing connection (no more input from client)");
         return 0;
      }
      if (clientret==GETLINE_OK) do_log(LOG_DEBUG, "--> %s", p->clientbuf->line);

      /* read from server */
      if ((serverret=getline(p->server_fd, p->serverbuf))<0){
         do_log(LOG_DEBUG, "Closing connection (no more input from server)");
         return 0;
      }
      if (serverret==GETLINE_OK
#ifndef DEBUG_MESSAGE
         && p->ismail!=2
#endif
      ) do_log(LOG_DEBUG, "<-- %s", p->serverbuf->line);

      if (clientret == GETLINE_NEED_READ && serverret == GETLINE_NEED_READ){
         FD_ZERO(&fds_read);
         FD_SET(p->client_fd, &fds_read);
         FD_SET(p->server_fd, &fds_read);
         timeout.tv_sec = 300;
         timeout.tv_usec = 0;
         if ((ret=select(p->server_fd + 1, &fds_read, NULL, NULL, &timeout))<1){
            /* timeout */
            do_log(LOG_DEBUG, "select returned %i", ret);
            break;
         } else continue ;
      }

      if ( p->clientbuf->linelen>=0 && p->ismail<2 ){
         /* scan command the client sent */
         if (!strncasecmp(p->clientbuf->line,"retr", 4)){
            p->msgnum=atoi(&p->clientbuf->line[5]);
            if (p->msgnum<1){
               /* that's not ok */
               do_log(LOG_WARNING,"RETR msg %i (<1) !!!! ", p->msgnum);
               p->ismail=0;
            } else {
               do_log(LOG_DEBUG,"RETR %i", p->msgnum);
               /* enable message parsing (only if scanner enabled) */
               if (config->scannerenabled) p->ismail=1;
               p->mailcount++;
            }
         } else if (!strncasecmp(p->clientbuf->line,"top", 3)){
            p->msgnum=atoi(&p->clientbuf->line[4]);
            if (p->msgnum<1){
               /* that's not ok */
               do_log(LOG_WARNING,"TOP msg %i (<1) !!!! ", p->msgnum);
               p->ismail=0;
            } else {
               do_log(LOG_DEBUG,"TOP %i", p->msgnum);
               /* enable message parsing (only if scanner enabled) */
               if (config->scannerenabled) p->ismail=1;
               p->mailcount++;
            }
      } else p->ismail=0;
         if (!strncasecmp(p->clientbuf->line,"user", 4)){
            len=p->clientbuf->linelen -5;
            if (len>0){
               memcpy(buf, (char *)(p->clientbuf->line)+5, len<64 ? len : 63 );
               buf[len]='\0';
               TRIM(buf);
               paramlist_set(p->params, "%USERNAME%", buf);
            } else {
               paramlist_set(p->params, "%USERNAME%", "");
            }
            do_log(LOG_NOTICE, "USER '%s'", paramlist_get(p->params, "%USERNAME%"));
         }
         /* write clientbuf to server_fd */
         /* ok, we can write */
         if (writeline(p->server_fd, WRITELINE_LEADING_RN, p->clientbuf->line)){
            do_log(LOG_CRIT, "can't send to server_fd");
            return 1;
         }
         p->clientbuf->linelen=-2;
      }
      if (p->serverbuf->linelen>=0){
         if (p->ismail==1){
            /* scan for answer */
            if (!strncasecmp(p->serverbuf->line,"+ok", 3)){
               do_log(LOG_DEBUG, "ismail=2");
               /* generate unique filename */
               len=strlen(config->virusdir)+14;
               snprintf(p->mailfile, len, "%sp3scan.XXXXXX", config->virusdir);
               do_log(LOG_DEBUG, "mailfile: %s:%i", p->mailfile,len);
               if (( scanfd=mkstemp(p->mailfile)) < 0 ){
                  p->ismail=0;
                  do_log(LOG_EMERG,"Critical error opening file '%s', Program aborted.", p->mailfile);
                  /* Should not reach here as we are dead */
               } else {
                  p->ismail=2;
                  p->header_exists=0;
                  p->serverbuf->linelen=-2; /* dont't send response */
                  do_log(LOG_DEBUG, "file '%s' opened", p->mailfile);
               }
            } else {
               do_log(LOG_DEBUG, "ismail=1, but we haven't got '+ok'");
               p->ismail=0;
            }
         } else if (p->ismail==2){
            /* that means that we have to read the mail into file */
            if (p->serverbuf->linelen == 0 && !p->header_exists){
               if (!config->checkspam) writeline(scanfd, WRITELINE_LEADING_N, "X-" PROGNAME ": Version " VERSION " by <laitcg@cox.net>/<folke@ashberg.de>");
               p->header_exists=1;
            }
            writeline(scanfd, WRITELINE_LEADING_N, p->serverbuf->line);
            //do_log(LOG_DEBUG, "written to file");
            if (p->serverbuf->linelen==1 && p->serverbuf->line[0]=='.') {
               /* mail is complete */
               error=0;
               close(scanfd);
               do_log(LOG_DEBUG, "got '.\\r\\n', mail is complete");
               /* initialize the scanner before scanning the first mail */
               /* but only if scanning is enabled */
               if (config->scannerenabled && p->scannerinit == SCANNER_INIT_NO) {
                  if (config->scanner->init2) {
                     if (config->scanner->init2(p)!=0) {
                        do_log(LOG_EMERG, "Can't initialize scanner!");
                        // Dead now. Configuration error!
                        p->scannerinit=SCANNER_INIT_ERR;
                     }
                     else p->scannerinit=SCANNER_INIT_OK;
                  }
                  else p->scannerinit=SCANNER_INIT_NULL;
               }
               /* Scan the file now */
               scannerret=SCANNER_RET_OK;
               snprintf(p->maildir, 4090, "%s.dir", p->mailfile);
               if (p->scannerinit > 0) {
                  if ((scannerret=scan_mailfile(p))==SCANNER_RET_VIRUS) {
                     /* virus */
                     p->ismail=0;
                     if (p->virinfo) TRIM(p->virinfo);
                     paramlist_set(p->params, "%VIRUSNAME%", NONULL(p->virinfo));
                     do_log(LOG_WARNING, "'%s' contains a virus (%s)!",
                     p->mailfile, paramlist_get(p->params, "%VIRUSNAME%"));
                     paramlist_set(p->params, "%MAILFILE%", p->mailfile);
                     set_maildateparam(p->params);
                     set_paramsfrommailheader(p->mailfile, p->params);
                     paramlist_set(p->params, "%P3SCANID%", p->mailfile);
                     paramlist_set(p->params, "%FILESTATUS%", p->mailfile);
                     if (do_virusaction(p)!=0) {
                        do_log(LOG_CRIT,
                        "Virusaction failed. Sending -ERR "
                        "and closing connection");
                        writeline_format(p->client_fd, WRITELINE_LEADING_RN,
                        "-ERR Message %i contains a virus (%s). ",
                        p->msgnum, paramlist_get(p->params, "%VIRUSNAME%"));
                        return 1;
                     };
                     unset_paramsfrommailheader(p->params);
                     p->clientbuf->linelen=-2;
                     p->serverbuf->linelen=-2;
                     if (p->virinfo) free(p->virinfo);
                     if (config->delit) unlink(p->mailfile);
                  } /* virus */
                  // see if there was a critical error
                  if (scannerret==SCANNER_RET_CRIT){
                     /* exact error already reported so kill the child. This
                        should get the sysadmins attention. */
                     return 1;
                  }
               }
               else
                  scannerret=SCANNER_RET_ERR; /* ! error */

               if (scannerret!=SCANNER_RET_VIRUS) { /* send mail if no virus */
                  if (scannerret==SCANNER_RET_ERR) {
                     do_log(LOG_ALERT, "We can't say if it is a virus! "
                     "So we have to give the client the mail! "
                     "You should check your configuration/system");
                     do_log(LOG_EMERG, "Scanner returned unexpected error code. You should check your configuration/system.");
                     // We are dead now. Don't let virus mails pass
                  }
                  /* no virus  / error / scanning disabled */
                  do_log(LOG_DEBUG, "Scanning done, sending mail now");
                  p->ismail=0;
                  if ((len=send_mailfile(p->mailfile, p->client_fd))<0) {
                     do_log(LOG_CRIT, "Can't send mail! We have to quit now!");
                     return 1;
                  }
                  do_log(LOG_DEBUG, "Sending done");
                  p->bytecount+=len;
                  p->serverbuf->linelen=-2;
                  unlink(p->mailfile); /* we do not unlink virusmails, so only here */
               }
               do_log(LOG_DEBUG, "Mail action complete");
            } /* mail complete */
            p->serverbuf->linelen=-2; /* don't sent to client */
         } else if (p->ismail==3){
            /* catch message has been deleted when we've deleted it */
            p->serverbuf->linelen=-2;
            p->ismail=0;
            do_log(LOG_DEBUG, "set ismail=1");
         }
      } /* server_buf_len >0 */

      /* we are not in mail-reading mode (ismail==0) */
      if ( p->serverbuf->linelen>=0 ){
          /* write server_buf to fd */
         if (writeline(p->client_fd, WRITELINE_LEADING_RN, p->serverbuf->line)){
         do_log(LOG_CRIT, "can't send to client");
         return 1;
      }
      p->serverbuf->linelen=-2;
      p->clientbuf->linelen=-2;
   }
   }
   do_log(LOG_WARNING, "Connection timeout");
// do_log(LOG_DEBUG, "Child beendet");
   do_log(LOG_DEBUG, "Child finished");
   return 0;
}

int do_sigchld_check(pid_t pid, int stat)
{
   int termsig = WTERMSIG(stat);
   char child_dir[1024];
   if (numprocs>0) numprocs--;
   if (termsig){
      do_log(LOG_CRIT, "Attention: child with pid %i died with abnormal termsignal (%i)! "
      "This is probably a bug. Please report to the author. "
      "numprocs is now %i",
      pid, termsig, numprocs );
   } else {
   do_log(LOG_DEBUG, "waitpid: child %i died with status %i, numprocs is now %i",
   pid, WEXITSTATUS(stat), numprocs);
   }

   snprintf	(child_dir,1024,"%s/children/%d/",
   config->virusdirbase,pid);
   do_log(LOG_DEBUG, "removing virusdir %s",child_dir);
   if((rmdir(child_dir)<0))do_log(LOG_DEBUG, "%s is not empty! Could not remove!",child_dir);
   return 1;
}

void do_sigchld(int signr){
   pid_t pid;
   int stat;
   while ((pid=waitpid(-1, &stat, WNOHANG)) > 0) {
      do_sigchld_check(pid, stat);
   }
}

void printversion(void){
   printf(
   "\n"
   PROGNAME " " VERSION "\n"
   "(C) 2003 by Jack S. Lai, <laitcg@cox.net> 12/05/2003\n"
   "         and by Folke Ashberg <folke@ashberg.de>\n"
   "\n"
   );
}

void usage(char * progname){
   int i=0;
   printversion();
   printf(
   "Usage: %s [options]\n"
   "\n"
   "where options are:\n"
   "  -h, --help              Prints this text\n"
   "  -v, --version           Prints version information\n"
   "  -a, --renattach=FILE    Specify location of renattach if wanted\n"
   "  -f, --configfile=FILE   Specify a configfile\n"
   "                          Default is " CONFIGFILE "\n"
   "  -d, --debug             Turn on debugging\n"
   "  -l, --pidfile=FILE      Specify where to write a pid-file\n"
   "  -m, --maxchilds=NUM     Allow not more then NUM childs\n"
   "  -i, --ip=IP             Listen only on IP <IP>. Default: ANY\n"
   "  -o, --overwrite         Overwrite (disable) HTML (must enable checkspam)\n"
   "  -p, --port=PORT         Listen on port <PORT>. Default: %i\n"
   "  -u, --user=[UID|NAME]   Run as user <UID>. Default: " RUNAS_USER "\n"
   "                          Only takes effect when started as superuser\n"
   "  -q, --quiet             Turn off normal reporting\n"
   "  -r, --virusdir=DIR      Save infected mails in <DIR>\n"
   "                          Default: " VIRUS_DIR "\n"
   "  -n, --notifydir=DIR     Create notification mails in <DIR>\n"
   "                          Default: " NOTIFY_MAIL_DIR "\n"
   "                          Also used for temporary storing\n"
   , basename(progname), PORT_NUMBER);
   printf(
   "  -y, --scannertype=T     Define which buildin scanner-frontend to use.\n"
   "  Supported types:\n");
   while (scannerlist[i]){
      printf("%17s: %s\n",
      scannerlist[i]->name, scannerlist[i]->descr);
      i++;
   }
   printf( "\n"
   "  -s, --scanner=FILE      Specify the scanner. Every scannertype handles this\n"
   "                          in an other way. This could be the scanner-\n"
   "                          executable or a FIFO, Socket, ...\n"
   "  -x, --demime            eXtract all MIME-Parts before scanning\n"
   "  -g, --virusregexp=RX    Specify a RegularExpression which describes where to\n"
   "                          get the name of the virus. The first substring is\n"
   "                          used, or if regexp ends with /X the X substring\n"
   "  -c, --viruscode=N[,N]   The code(s) the scanner returns when a virus is found\n"
   "  -t, --template=FILE     Use virus-notification-template <FILE>\n"
   "  -k, --checkspam         Turn on Spam Checking\n"
   "  -z, --spamcheck         Specify path to Spam Checking program executable\n"
   "                          Default /usr/bin/spamc (Mail::SpamAssassin)\n"
   "  -b, --bytesfree=NUM     Number (in KBytes) that should be available before we\n"
   "                          can process messages. If not enough, report it and die.\n"
   "  -j, --justdelete        Just delete infected mail after reporting infection\n"
   "\n"
   "        see " CONFIGFILE " for detailed information\n"
   "\n\n"
   );
}
void parseoptions(int argc, char **argv){
   long i, ii;
   char * rest;
   int error = 0;
   struct servent *port;
   int fp, res;
   struct linebuf *cf;
   char *pargv[MAX_PSEUDO_ARGV];
   int pargc;
   char *line;
   int pidfd;
   struct option long_options[] = {
   { "help",         no_argument,         NULL, 'h' },
   { "version",      no_argument,         NULL, 'v' },
   { "configfile",   required_argument,   NULL, 'f' },
   { "renattach",	   required_argument,   NULL, 'a' },
   { "debug",        no_argument,         NULL, 'd' },
   { "overwrite",    no_argument,         NULL, 'o' },
   { "pidfile",      required_argument,   NULL, 'l' },
   { "virusdir",     required_argument,   NULL, 'r' },
   { "notifydir",    required_argument,   NULL, 'n' },
   { "maxchilds",    required_argument,   NULL, 'm' },
   { "ip",           required_argument,   NULL, 'i' },
   { "port",         required_argument,   NULL, 'p' },
   { "user",         required_argument,   NULL, 'u' },
   { "quiet",        no_argument,         NULL, 'q' },
   { "scannertype",  required_argument,   NULL, 'y' },
   { "scanner",      required_argument,   NULL, 's' },
   { "viruscode",    required_argument,   NULL, 'c' },
   { "template",     required_argument,   NULL, 't' },
   { "demime",       no_argument,         NULL, 'x' },
   { "virusregexp",  required_argument,   NULL, 'g' },
   { "checkspam",    no_argument,         NULL, 'k' },
   { "spamcheck",    required_argument,   NULL, 'z' },
   { "bytesfree",    required_argument,   NULL, 'b' },
   { "justdelete",   no_argument,         NULL, 'j' },
   { NULL,           no_argument,         NULL, 0 }
   };
   char getoptparam[] = "hvf:a:dol:r:n:m:i:p:u:qy:s:c:t:xg:kz:b:j";
   void switchoption(char opt, char * arg, char * optstr, char * where, int state){
   char *next_tok;
   switch (opt){
      case 'h':
      case 'v':
      case 'f':
         /* don't check in second run (is in the first) */
         if (state==CONFIG_STATE_CMD) return;
         /* disallow help/version/configfile in configfile */
         if (state==CONFIG_STATE_FILE){
            fprintf(stderr, "%s '%s' is not allowed in configfile!\n", where, optstr);
            error=1;
            return;
         }
      break;
   default:
   /* only check help/version/configfile for the first cmd run */
   if (state==CONFIG_STATE_CMDPRE) return;
   }

   switch (opt){
      case 'h': /* usage */
         usage(argv[0]);
         exit(0);
         break;
      case 'v': /* version */
         printversion();
         exit(0);
      break;
      case 'f': /* config (file) */
         config->configfile = arg;
         break;
      case 'd': /* debug */
         config->debug=1;
      break;
      case 'l': /* PID File */
         config->pidfile=arg;
         if ((pidfd=open(config->pidfile,O_RDONLY ))>0){
            do_log(LOG_EMERG, "%s exists! Aborting!",config->pidfile);
            // Should not reach here. We are dead.
            pidfd=close(pidfd);
            exit(0);
         }
      break;
      case 'a': /* rename attachments using renattach */
         config->renattach=arg;
      break;
      case 'r': /* virusdir */
         config->virusdirbase=arg;
         config->virusdir=config->virusdirbase;
      break;
      case 'n': /* notifydir */
         config->notifydir=arg;
      break;
      case 'm': /* Max Childs */
         i=strtol(arg, &rest, 10);
         if ((rest && strlen(rest)>0) || i<1 || i>99){
            fprintf(stderr, "%s --maxchilds has to be 1 < val < 100\n", where);
            error=1;
         } else config->maxchilds=(int)i;
      break;
      case 'i': /* IP (to listen on) */
         if (!strcmp(arg, "0.0.0.0")){
            config->addr.sin_addr.s_addr=htonl(INADDR_ANY);
         } else if (!inet_aton(arg, &config->addr.sin_addr)){
            fprintf(stderr, "%s %s isn't a valid IP Adress\n", where, arg);
            error=1;
         }
      break;
      case 'o': /* overwrite (disable) HTML */
         config->overwrite=1;
      break;
      case 'p': /* Port */
         i=strtol(arg, &rest, 10);
         if (rest && strlen(rest)>0){
            if (i>0){ /* 123abc */
               fprintf(stderr, "%s %s isn't a valid port\n", where, arg);
               error=1;
            } else {
               if ((port=getservbyname(arg, "tcp"))!=NULL){
                  config->addr.sin_port=port->s_port;
               } else {
                  fprintf(stderr, "Port lookup for '%s/tcp' failed! "
                  "Check /etc/services\n", arg);
                  error=1;
               }
            }
         } else {
            if (i>0){
               config->addr.sin_port=htons((int)i);
            } else {
               fprintf(stderr, "%s Incorrect portnumber\n", where);
               error=1;
            }
         }
      break;
      case 'q': /* quiet */
         config->quiet=1;
      break;
      case 'u': /* Run as User */
         config->runasuser=arg;
         /* getpwnam will also accept UID's, so we need no converting*/
      break;
      case 's': /* Scanner */
         config->virusscanner=arg;
      break;
      case 't': /* template */
         config->virustemplate=arg;
      break;
      case 'c': /* Virus (exit) code */
         ii = 0;
         next_tok = strtok(arg, " \t,");
         if (next_tok) {
            do {
               if (ii < MAX_VIRUS_CODES) {
                  i=strtol(next_tok, &rest, 10);
                  if ( (rest && strlen(rest)>0) || i<1 || i>99){
                     fprintf(stderr, "%s --viruscode has be a list of numbers (%s)\n", where, rest);
                     error=1;
                  } else config->viruscode[ii]=(int)i;
                  ii++;
               }
            } while ((next_tok = strtok(NULL, " \t,")) || (ii >= MAX_VIRUS_CODES));
         }
         config->viruscode[ii] = -1;
         if (ii == 0) {
            fprintf(stderr, "%s --viruscode has be a list of numbers (%s)\n", where, rest);
            error=1;
         }
      break;
      case 'x': /* demime */
         config->demime = 1;
      break;
      case 'y': /* scannertype */
         i=0;
         while (scannerlist[i]){
            if (!strcasecmp(arg, scannerlist[i]->name)){
               config->scanner=scannerlist[i];
               i=-1;
               break;
            }
            i++;
         }
         if (i!=-1){
            fprintf(stderr, "%s scannertype '%s' is not supported", where, arg);
            error=1;
         }
      break;
      case 'g': /* virusregexp */
         config->virusregexp = arg;
         i=strlen(arg);
         if (arg[i-2]=='/' && isdigit(arg[i-1])){
            arg[i-2]='\0';
            config->virusregexpsub=arg[i-1]-'0';
         }
      break;
      case 'k': /* check for spam */
         config->checkspam=1;
      break;
      case 'z': /* path to spam checking executable */
         config->spamcheck = arg;
      break;
      case 'b': /* bytes free */
         i=strtol(arg, &rest, 10);
         config->freespace=(int)i;
      break;
      case 'j': /* justdelete */
         config->delit=1;
      break;

      default:
      fprintf(stderr, "%s Option '%s' isn't known\n", where, optstr);
      error=1;
      }
   } /* sub function switchoption  }}} */
void parseargs(int c, char **v, char * where, int state){
   int option, option_index = 0;
   opterr=0;
   optind=1;
   while (1){
      option = getopt_long(c, v, getoptparam,
      long_options, &option_index);
      if (option == EOF) break;
      switchoption(option, optarg, v[optind-1], where, state);
   }
   if (state != CONFIG_STATE_CMDPRE && optind < c){
      error=1;
      while (optind < c)
   fprintf(stderr, "%s Unknown option '%s'\n", where, v[optind++]);
   }
   }
   if ((config=malloc(sizeof(struct configuration_t)))==NULL){
      fprintf(stderr, "malloc error\n");
      exit(1);
   }
   /* set defaults for in, char* to NULL */
   config->debug=DEBUG;
   config->quiet=QUIET;
   config->overwrite=OVERWRITE;
   config->renattach=NULL;
   config->addr.sin_port=htons((int)PORT_NUMBER);
   config->maxchilds=MAX_CHILDS;
   config->viruscode[0]=VIRUS_SCANNER_VIRUSCODE;
   config->viruscode[1]=-1;
   config->runasuser=NULL;
   config->virusdir=NULL;
   config->virusdirbase=NULL;
   config->notifydir=NULL;
   config->virustemplate=NULL;
   config->virusscanner=NULL;
   config->demime=0;
   config->pidfile=NULL;
   config->syslogname=NULL;
   config->addr.sin_addr.s_addr=htonl(INADDR_ANY);
   config->scanner=NULL;
   config->virusregexp=NULL;
   config->virusregexpsub=1;
   config->checkspam=CHECKSPAM;
   config->spamcheck=SPAMCHECK;
   config->freespace=MINSPACE;
   config->delit=DELIT;
   /* parse line args, but for the first time only configfile/version/help */
   parseargs(argc, argv, "\t[cmdlineparm]", CONFIG_STATE_CMDPRE);
   /* parse configfile */
   if (!config->configfile) config->configfile=strdup(CONFIGFILE);
   if ((fp=open(config->configfile, O_RDONLY))>0){
      cf=linebuf_init(4096);
      pargc=1;
      pargv[0]="";
      while ((res=getline(fp, cf))>=0 && pargc<MAX_PSEUDO_ARGV-1){
         if (cf->linelen > 2){
            //fprintf(stderr, "CFLine: '%s'\n", cf->line);
            TRIM(cf->line);
            if (cf->line[0]!='#' && cf->line[0]!=';' &&
            !(cf->line[0]=='/' && cf->line[1]=='/') &&
            cf->line[0]!='='){
            /* copy to pseudo argv, change
            * 'x = y' or 'x y' to 'x='
            * This code is the horror, but it seems to work */
               line=malloc(strlen(cf->line)+3);
               line[0]='-'; line[1]='-'; line[2]='\0';
               strcat(line, cf->line);
               pargv[pargc]=line;
               if ((i=strcspn(line, " =\t"))>1){
                  if (i<strlen(line)){
                     line[i]='\0';
                     if ((ii=strspn(line+i+1," =\t"))>=0){
                        rest=line+i+ii+1;
                        if (rest && strlen(rest)>0 ){
                           pargv[pargc][strlen(pargv[pargc])]='=';
                           memcpy(pargv[pargc]+i+1, rest, strlen(rest)+1);
                        }
                     }
                  }
               }
               //printf("argv[%i]='%s'\n", pargc, pargv[pargc]);
               pargc++;
            }
         }
      }
      close(fp);
      linebuf_uninit(cf);
      pargv[pargc]=NULL;
      parseargs(pargc, pargv, "\t[configfile]", CONFIG_STATE_FILE);
   }
   /* now check the rest of commandline args (higher precedence than configfile) */
   parseargs(argc, argv, "\t[cmdlineparm]", CONFIG_STATE_CMD);
   if (error){
      printf(
      "Commandline options/configfile are not ok\n"
      "try --help or RTFM to get some information\n"
      "\n"
      );
      exit(1);
   }

   /* set unset values to default */
   SETIFNULL(config->runasuser, RUNAS_USER);
   SETIFNULL(config->virusdirbase, VIRUS_DIR);
   SETIFNULL(config->virusdir, config->virusdirbase );
   SETIFNULL(config->notifydir, NOTIFY_MAIL_DIR);
   SETIFNULL(config->virustemplate, VIRUS_TEMPLATE);
   SETIFNULL(config->virusscanner, VIRUS_SCANNER);
   SETIFNULL(config->pidfile, PID_FILE);
   SETIFNULL(config->syslogname, SYSLOG_NAME);
   SETIFNULL(config->scanner, scannerlist[0]);
   if (config->overwrite && !config->checkspam){
      do_log(LOG_WARNING,"HTML Overwrite disabled. MUST have checkspam enabled");
      config->overwrite=0;
   }
}

void do_sigterm_main(int signr){
   int ret;

   if (signr != -1 ) do_log(LOG_NOTICE, "signalled, doing cleanup");
   close(sockfd);
   if (config->scannerenabled && config->scanner->uninit1){
      do_log(LOG_DEBUG, "calling uninit1");
      config->scanner->uninit1();
      do_log(LOG_DEBUG, "uninit1 done");
   }
   if ((ret=unlink(config->pidfile)!=0)){
      do_log(LOG_NOTICE, "Unable to remove %s", config->pidfile);
   }
   do_log(LOG_NOTICE, PROGNAME " terminates now");
   exit(0);
}

int main(int argc, char ** argv){
   int connfd, i;
   struct sockaddr_in  addr;
   size_t socksize = sizeof(struct sockaddr_in);
   pid_t pid;
   int stat;
   FILE * fp;
   struct proxycontext * p;
   char * responsemsg;
   int virusdirlen;
   char chownit[100];
#define CHOWNCMD "/bin/chown"
   int len;
   int ret;
   FILE * chowncmd;

   parseoptions(argc, argv);

   do_log(LOG_NOTICE, PROGNAME " Version " VERSION);
   do_log(LOG_NOTICE, "Selected scannertype: %s (%s)",
   config->scanner->name,config->scanner->descr);

   if ( (sockfd = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP)) < 0)
      do_log(LOG_EMERG, "Can't open socket");

   i = 1;
   setsockopt(sockfd, SOL_SOCKET, SO_REUSEADDR, &i, sizeof(i));
   config->addr.sin_family = AF_INET;

   if (bind(sockfd, (struct sockaddr *) &config->addr, sizeof(config->addr)))
      do_log(LOG_EMERG, "Can't bind to socket %s:%i",
      inet_ntoa(config->addr.sin_addr), ntohs(config->addr.sin_port));

   if (listen(sockfd, 5)) do_log(LOG_EMERG, "Can't listen on socket");

   do_log(LOG_NOTICE, "Listen now on %s:%i",
      inet_ntoa(config->addr.sin_addr), ntohs(config->addr.sin_port));
   if (!config->debug){
      /* daemonize */
      do_log(LOG_DEBUG, "Daemonize");
      if ( (pid = fork()) < 0) return(-1);
      else if (pid != 0) exit(0);
      setsid();
      if (!config->debug){
         if ( (fp=fopen(config->pidfile, "w+"))!=NULL){
            fprintf(fp, "%i\n", getpid());
            fclose(fp);
         } else do_log(LOG_CRIT, "Can't write PID to %s", PID_FILE);
      };
      // chown /var/run/p3scan/p3scan.pid mail.mail
      len=strlen(CHOWNCMD)+1+strlen(config->runasuser)+1+strlen(config->runasuser)+1+strlen(config->pidfile)+1;
      do_log(LOG_DEBUG, "%s %s.%s %s=%i",CHOWNCMD, config->runasuser, config->runasuser, config->pidfile, len);
      snprintf(chownit, len, "%s %s.%s %s", CHOWNCMD, config->runasuser, config->runasuser, config->pidfile);
      if ((chowncmd=popen(chownit, "r"))==NULL){
         do_log(LOG_ALERT, "Can't '%s' !!!", chowncmd);
         return SCANNER_RET_ERR;
      }
      ret=pclose(chowncmd);
   }
   avoid_root();
   signal(SIGCHLD, do_sigchld);
   signal(SIGTERM, do_sigterm_main);
   signal(SIGINT, do_sigterm_main);

   if (config->scanner->init1){
      if (config->scanner->init1()!=0){
         do_log(LOG_EMERG, "Scanner init failed! config and restart " PROGNAME);
         config->scannerenabled = 0;
      } else config->scannerenabled = 1;
   } else config->scannerenabled = 1;
   if (config->quiet){
      config->quiet=0;
      do_log(LOG_NOTICE, "%s %s started.", PROGNAME, VERSION);
      config->quiet=1;
   }
   numprocs=0;
   do_log(LOG_DEBUG, "Waiting for connections.....");
   while ( (connfd = accept(sockfd, (struct sockaddr *)&addr,
      &socksize)) >= 0) {
      if ( (pid=fork())>0){
         /* parent */
         numprocs++;
         do_log(LOG_DEBUG, "Forked, pid=%i, numprocs=%i", pid, numprocs);

         close (connfd);

         /* wir brauchen die nicht, der childprocess kmmert sich drum
         we don't need "them" (connfd?), child process takes care of that */
         if (numprocs>=config->maxchilds){
            do_log(LOG_WARNING, "MAX_CHILDS (%i) reached!", config->maxchilds);
            while (1){
               pid=waitpid(-1, &stat, 0); /* blocking */
               if (do_sigchld_check(pid, stat)) break;
            }
         }
      } else {
         /* child */

         virusdirlen=strlen(config->virusdirbase)+20;
         config->virusdir=malloc(virusdirlen);
         sprintf	(config->virusdir,"%s/children/%d/",
         config->virusdirbase,getpid());
         do_log(LOG_DEBUG, "setting the virusdir to %s", config->virusdir);
         if((mkdir (config->virusdir, S_IRWXU)<0))do_log(LOG_EMERG,"Could not create virusdir %s",config->virusdir);
         signal(SIGCHLD, NULL); /* unset signal handler for child */
         do_log(LOG_DEBUG, "Initialize Context");
         p=context_init();
         p->client_fd=connfd;
         p->client_addr=addr;
         global_p=p;
         signal(SIGTERM, do_sigterm_proxy);
         signal(SIGINT,  do_sigterm_proxy);
         do_log(LOG_DEBUG, "starting proxy");
         if (proxy(p)){
            /* error, but a message has already be sent */
            responsemsg=strdup("Critial abort");
         } else {
            responsemsg=strdup("Clean Exit");
         };
         if (config->scannerenabled){
            do_log(LOG_NOTICE, "Session done (%s). Mails: %i Bytes: %lu",
            responsemsg, p->mailcount, p->bytecount);
         } else {
            do_log(LOG_NOTICE, "Session done (%s). Mails: %i",
            responsemsg, p->mailcount);
         }
         free(responsemsg);
         do_sigterm_proxy(-1);
         return 0;
      }
   }
   do_log(LOG_NOTICE, "accept error");
   do_sigterm_main(-1);
   return 0;
}
