#include <QMainWindow>
#include <QTextBrowser>
#include <QTextOStream>

#include <helpers.h>
#include <iprovider.h>

#include "packagedescriptionplugin.h"

#include "ipackagedb.h"
#include "iaptmediator.h"

#include "packagenotfoundexception.h"

namespace NPlugin
{

const QString PackageDescriptionPlugin::PLUGIN_NAME = "PackageDescriptionPlugin";

const QString PackageDescriptionPlugin::_emptyString;


/**
 * Constructors/Destructors
 */
PackageDescriptionPlugin::PackageDescriptionPlugin(NApt::IPackageDB* pPackageDB, IAptMediator* pMediator) :
	_pPackageDB(pPackageDB),
	_pMediator(pMediator)
{
	_pDescriptionView = 0;
	_pProvider = 0;
}
 
PackageDescriptionPlugin::~PackageDescriptionPlugin()
{
	delete _pDescriptionView;
}
 
/////////////////////////////////////////////////////
// Plugin Interface
/////////////////////////////////////////////////////

void PackageDescriptionPlugin::init(IProvider* pProvider)
{
	_pProvider = pProvider;
	QMainWindow* pWindow = pProvider->mainWindow();
	_pDescriptionView = new QTextBrowser(pWindow);
	_pDescriptionView->setObjectName("DescriptionView");
}


/////////////////////////////////////////////////////
// InformationPlugin Interface
/////////////////////////////////////////////////////

QWidget* PackageDescriptionPlugin::informationWidget() const	
{ 
	return _pDescriptionView; 
}

QString PackageDescriptionPlugin::informationWidgetTitle() const 
{
	return tr("Description"); 
}

namespace
{

/** Function object. */
class HTMLify
{
	typedef list< pair<QChar, QString> > Repl;
	Repl _repl;
public:
	HTMLify()
	{
		_repl.push_back(make_pair(QChar('<'),QString("&lt;")));
		_repl.push_back(make_pair(QChar('>'),QString("&gt;")));
		_repl.push_back(make_pair(QChar('\n'),QString("<br>")));
	}
	void operator()(QString& s)	
	{
		for (Repl::iterator it=_repl.begin(); it!=_repl.end(); ++it)
			s.replace(it->first, it->second);
	}
};

void PackageDescriptionPlugin::updateInformationWidget(const string& package)
{
	pair<bool, NApt::Package> packageResult = getPackageInformation( package );
	QString text="";
	if (packageResult.first)	// the package information could be retrieved
	{
		NApt::Package& pkg = packageResult.second;	// make it more readable
	//		QString name("<h3>" + pkg.package+"</h3>");
	//		_pPackageName->setText(name);
		if (!pkg.description.isEmpty())
		{
			HTMLify htmlify;
			/// @todo only htmlify the packageDescription here
			pkg.processEntries(htmlify);
			QString description = pkg.description;
			QStringList patterns = _pMediator->searchPatterns();
			for (QStringList::const_iterator it = patterns.begin(); it != patterns.end(); ++it )
			{
				int index = description.find(*it, 0, false);
				while ( index != -1 )
				{
					description.insert(index + (*it).length(), "</font>");	// insert the last part first
					description.insert(index, "<font color=\"#ff0000\">");
					// point behind the inserted string
					index += (*it).length() + 29;	// 29 ==  strlen("<font color=\"#ffff00\"></font>")
					index = description.find(*it, index, false);
				}
			}
			text = "<b>Description</b>: " + description;
		}
	}
	else	// the package information could not be retrieved
	{
		text = tr("<h3>Package not found</h3>"
			"<p>Could not find a valid description for the package <b>") + toQString(package) +
			tr("</b> in the database.<br>"
			"This could either mean that you have selected a virtual package or that from an unknown "
			"Reason the package description could not be found. It is possible that your debtags and "
			"apt database are out of sync. Try running <tt>debtags update</tt> and <tt>apt-get update</tt> "
			"as root.</p>");
//			"The original error message was:<br>" + QString(e.desc().c_str())
//		);
	}
	_pDescriptionView->setText(text);
}


void PackageDescriptionPlugin::clearInformationWidget()
{
	_pDescriptionView->clear();
}

QString PackageDescriptionPlugin::informationText(const string& package)
{
	pair<bool, NApt::Package> packageResult = getPackageInformation( package );
	QString details;
	if (packageResult.first)	// the package information could be retrieved
	{
		NApt::Package& pkg = packageResult.second;	// make it more readable
		HTMLify htmlify;
		pkg.processEntries(htmlify);
		QTextOStream os(&details);
		{	// fill the details string
			if (!pkg.installedVersion.isNull())
				os << "<b>Installed Version</b>: " << pkg.installedVersion << "<br>";
				
			if (!pkg.version.isNull())
				os << "<b>Available Version</b>: " << pkg.version << "<br>";
			if (!pkg.essential.isNull())
				os << "<b>Essential</b>: " << pkg.essential<< "<br>";
			if (!pkg.priority.isNull())
				os << "<b>Priority</b>: " << pkg.priority << "<br>";
			if (!pkg.section.isNull())
				os << "<b>Section</b>: " << pkg.section << "<br>";
			if (!pkg.installedSize.isNull())
				os << "<b>Installed Size</b>: " << pkg.installedSize << "<br>";
			if (!pkg.maintainer.isNull())
				os << "<b>Maintainer</b>: " << pkg.maintainer << "<br>";
			if (!pkg.architecture.isNull())
				os << "<b>Architecture</b>: " << pkg.architecture << "<br>";
			if (!pkg.source.isNull())
				os << "<b>Source</b>: " << pkg.source << "<br>";
			if (!pkg.replaces.isNull())
			{
				NApt::Package::BorderList borderList = pkg.getPackageList(pkg.replaces);
				os << "<b>Replaces</b>: " << createLinks(borderList, pkg.replaces) << "<br>";
			}
			if (!pkg.provides.isNull())
			{
				NApt::Package::BorderList borderList = pkg.getPackageList(pkg.provides);
				os << "<b>Provides</b>: " << createLinks(borderList, pkg.provides) << "<br>";
			}
			if (!pkg.preDepends.isNull())
			{
				NApt::Package::BorderList borderList = pkg.getPackageList(pkg.preDepends);
				os << "<b>Pre-Depends</b>: " << createLinks(borderList, pkg.preDepends) << "<br>";
			}
			if (!pkg.depends.isNull())
			{
				NApt::Package::BorderList borderList = pkg.getPackageList(pkg.depends);
				os << "<b>Depends</b>: " << createLinks(borderList, pkg.depends) << "<br>";
			}
			if (!pkg.recommends.isNull())
			{
				NApt::Package::BorderList borderList = pkg.getPackageList(pkg.recommends);
				os << "<b>Recommends</b>: " << createLinks(borderList, pkg.recommends) << "<br>";
			}
			if (!pkg.suggests.isNull())
			{
				NApt::Package::BorderList borderList = pkg.getPackageList(pkg.suggests);
				os << "<b>Suggests</b>: " << createLinks(borderList, pkg.suggests) << "<br>";
			}
			if (!pkg.conflicts.isNull())
			{
				NApt::Package::BorderList borderList = pkg.getPackageList(pkg.conflicts);
				os << "<b>Conflicts</b>: " << createLinks(borderList, pkg.conflicts) << "<br>";
			}
			if (!pkg.filename.isNull())
				os << "<b>Filename</b>: " << pkg.filename << "<br>";
			if (!pkg.size.isNull())
				os << "<b>Size</b>: " << pkg.size << "<br>";
			if (!pkg.md5sum.isNull())
				os << "<b>MD5sum</b>: " << pkg.md5sum << "<br>";
			if (!pkg.conffiles.isNull())
				os << "<b>Conffiles</b>: " << pkg.conffiles << "<br>";
		}
	}
	else
	{
		throw PackageNotFoundException(package);
	}
	return details;
}

}	// unnamed namespace


/////////////////////////////////////////////////////
// ShortInformationPlugin Interface
/////////////////////////////////////////////////////

const QString PackageDescriptionPlugin::shortInformationText(const string& package)
{
	try 
	{
		return _pPackageDB->getShortDescription(package);
	}
	catch (const PackageNotFoundException& e)
	{
		return _emptyString;
	}
}


/////////////////////////////////////////////////////
// Helper Methods
/////////////////////////////////////////////////////

pair<bool, NApt::Package> PackageDescriptionPlugin::getPackageInformation(const string& package)
{
 	try
	{
		NApt::Package pkg = _pPackageDB->getPackageRecord( package );
		return make_pair(true, pkg);
	}
	catch(PackageNotFoundException e)
	{
		return make_pair(false, NApt::Package());
	}
}

QString PackageDescriptionPlugin::createLinks( NApt::Package::BorderList packages, const QString & s)
{
	typedef NApt::Package::BorderList BL;
	QString result = s;
	// iterate from behind to not destroy the order
	for (BL::reverse_iterator it = packages.rbegin(); it!=packages.rend(); ++it)
	{
		QString package = result.mid(it->first, it->second - it->first);
		const set<string>& allPackages = _pProvider->packages();
		if (allPackages.find(toString(package)) != allPackages.end())
		{
			result.insert(it->second,"</a>");	// insert behind the package name
			result.insert(it->first,"<a HREF=\""+package+"\">");	// insert before the package name
		}
	}
	return result;
}


}	// namespace NPlugin

