#include <sys/stat.h>
#ifndef WIN32
#include <unistd.h>
#endif
#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include <gdk/gdkkeysyms.h>
#include <glade/glade.h>
#include <string.h>

#include "mudclient.h"

#include "Connection.h"
#include "Message.h"

struct auto_data {
  int countdown;
  gint timeout; // Need to remove me.
  GladeXML * xml; // The XML holding this autoreconnect window's widgets.
  Connection * conn; // The connection this autoreconnect is from.
};

int auto_reconnect_timeout(gpointer d) {
  struct auto_data * data = (struct auto_data *)d;
  char buf[1024];
  
  if (data->countdown == 0) {
    g_source_remove(data->timeout);
    data->conn->reconnect();
    GtkWidget * window = glade_xml_get_widget(data->xml, "autoconnect_window");
    gtk_widget_destroy(window);
    g_object_unref(data->xml);
    free(data);
    return 0;
  }
  
  sprintf(buf, _("Reconnecting to %.900s:%d in %d second%s"), data->conn->getMUD()->getHostname(), data->conn->getMUD()->getPort(), data->countdown / 10 + 1, data->countdown <= 10 ? "" : "s");


  GtkWidget * frame_label = glade_xml_get_widget(data->xml, "frame_label");
  GtkWidget * window = glade_xml_get_widget(data->xml, "autoconnect_window");
  GtkWidget * progressbar = glade_xml_get_widget(data->xml, "progressbar");

  gtk_label_set_text(GTK_LABEL(frame_label), buf);
  gtk_window_set_title(GTK_WINDOW(window), buf);

  gtk_progress_bar_set_text(GTK_PROGRESS_BAR(progressbar), buf);
  gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(progressbar),  1.0 - ((gdouble) data->countdown) / ((gdouble) 100));

  data->countdown--;
  return 1;
}

void on_autoreconnect_disconnect_button_clicked(GtkButton * button, void * d) {
  
  struct auto_data *data = (struct auto_data *)d;
  g_source_remove(data->timeout);
  delete data->conn; // This should remove the item from the MainWindow.
  data->conn = NULL;

  GtkWidget * window = glade_xml_get_widget(data->xml, "autoconnect_window");
  gtk_widget_destroy(window);

  g_object_unref(data->xml);
  free(data);
}

void on_autoreconnect_reconnect_button_clicked(GtkButton * button, void * d) {

  struct auto_data *data = (struct auto_data *)d;
  g_source_remove(data->timeout);

  GtkWidget * window = glade_xml_get_widget(data->xml, "autoconnect_window");
  gtk_widget_destroy(window);
  g_object_unref(data->xml);
  data->conn->reconnect();
  free(data);
}

extern "C" void on_autoreconnect_delete_event(GtkWidget * window, GdkEvent * event, gpointer data) {
	on_autoreconnect_disconnect_button_clicked(NULL, data);
}

void create_autoconnect_dialog(struct auto_data * data) {

  char buf[1024];

  snprintf(buf, 1024, "%s/share/papaya/autoreconnect.glade", getPrefix());

  data->xml = glade_xml_new(buf, NULL, NULL);
  g_object_ref(data->xml);
  // Connect up the signals from the buttons.


  sprintf(buf, _("Reconnecting to %.900s:%d in %d second%s."), data->conn->getMUD()->getHostname(), data->conn->getMUD()->getPort(), data->countdown / 10 + 1, data->countdown <= 10 ? "" : "s");

  GtkWidget * frame_label = glade_xml_get_widget(data->xml, "frame_label");
  GtkWidget * window = glade_xml_get_widget(data->xml, "autoconnect_window");
  GtkWidget * progressbar = glade_xml_get_widget(data->xml, "progressbar");

  GtkWidget * button = glade_xml_get_widget(data->xml, "cancel_button");
  gtk_widget_grab_focus(GTK_WIDGET (button));
 

  gtk_label_set_text(GTK_LABEL(frame_label), buf);
  gtk_window_set_title(GTK_WINDOW(window), buf);

  gtk_progress_bar_set_text(GTK_PROGRESS_BAR(progressbar), buf);
  gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(progressbar),  1.0 - ((gdouble) data->countdown) / ((gdouble) 100));


  int written, raw_read, read, connected;
  data->conn->getSocket()->getStats(&written, &raw_read, &read, &connected);

  int hours, minutes, seconds;
  seconds = connected % 60;
  minutes = connected / 60;
  hours = connected / 3600;
  
  int write_rate, read_rate;

  if (connected == 0) {
    write_rate = 0;
    read_rate = 0;
  }
  else {
    write_rate = written  / connected;
    read_rate = raw_read / connected;
  }

  snprintf(buf, 1024, _("Total data sent: %d bytes\n"
			"Raw data received: %d bytes\n"
			"Compressed data received: %d bytes\n"
			"Time connected: %.2dh %.2dm %.2ds\n"
			"Average data sent: %d bytes/sec\n"
			"Average raw data received: %d bytes/sec\n"),
	   
	   written, raw_read, read, hours, minutes, seconds,
	   write_rate, read_rate);

  GtkTextBuffer * buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(glade_xml_get_widget(data->xml, "textview")));
  GtkTextIter start, end;
  gtk_text_buffer_get_bounds(buffer, &start, &end);
  gtk_text_buffer_delete(buffer, &start, &end);
  gtk_text_buffer_get_bounds(buffer, &start, &end);
  gtk_text_buffer_insert(buffer, &end, buf, strlen(buf));

  glade_xml_signal_connect_data(data->xml, "on_connect_button_clicked", GTK_SIGNAL_FUNC(on_autoreconnect_reconnect_button_clicked), data);
  glade_xml_signal_connect_data(data->xml, "on_cancel_button_clicked", GTK_SIGNAL_FUNC(on_autoreconnect_disconnect_button_clicked), data);
  glade_xml_signal_connect_data(data->xml, "on_autoreconnect_delete_event", GTK_SIGNAL_FUNC(on_autoreconnect_delete_event), data);

/*
  g_signal_connect_data(G_OBJECT(window), "delete_event",
	  GTK_SIGNAL_FUNC(on_autoreconnect_delete_event),
	  data, NULL,
	  (GConnectFlags)0);
*/
}

struct auto_dialog {
  GtkWidget * dialog;
  Connection * conn;
};

void on_autoreconnect_http_message_button_clicked(GtkButton * button, gpointer data) {

  struct auto_dialog * d = (struct auto_dialog *)data;

  gtk_widget_hide(d->dialog);
  gtk_widget_destroy(d->dialog);

  // Attempt to restart the HTTP connection.
  d->conn->getSocket()->restart(d->conn->getMUD()->getHostname(), d->conn->getMUD()->getPort());
  free(d);
}

void on_autoreconnect_message_button_clicked(GtkButton * button, gpointer data) {

  struct auto_dialog * d = (struct auto_dialog *)data;

  gtk_widget_hide(d->dialog);
  gtk_widget_destroy(d->dialog);

  delete d->conn;
  free(d);
}

void autoreconnect_message(char * title, char * text, Connection * conn) {

  struct auto_dialog * auto_d = (struct auto_dialog *)malloc(sizeof(struct auto_dialog));
  auto_d->conn = conn;

  auto_d->dialog = gtk_dialog_new();

  gtk_window_set_title(GTK_WINDOW(auto_d->dialog), title);
  gtk_window_set_resizable(GTK_WINDOW(auto_d->dialog), FALSE);

  // Set up the message and icon.

  GtkWidget * vbox = GTK_DIALOG(auto_d->dialog)->vbox;
  gtk_widget_show(vbox);

  GtkWidget * hbox = gtk_hbox_new(FALSE, 0);
  gtk_widget_show(hbox);

  GtkStockItem item;

  char * icon;
  if (!strcmp(title, "Error"))
    icon = GTK_STOCK_DIALOG_ERROR;
  else if (!strcmp(title, "Warning"))
    icon = GTK_STOCK_DIALOG_WARNING;
  else
    icon = GTK_STOCK_DIALOG_INFO;

  gtk_stock_lookup(icon, &item);
  GtkWidget * image = gtk_image_new_from_stock(icon, GTK_ICON_SIZE_DIALOG);

  gtk_widget_show(image);

  GtkWidget * label = gtk_label_new(text);
  gtk_widget_show(label);

  gtk_box_pack_start(GTK_BOX(hbox), image, FALSE, FALSE, 0);
  gtk_box_pack_end(GTK_BOX(hbox), label, FALSE, FALSE, 0);

  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

  // Set up the button area
  GtkWidget * action_area = GTK_DIALOG(auto_d->dialog)->action_area;
  gtk_widget_show(action_area);
  gtk_widget_show (action_area);
  gtk_container_set_border_width (GTK_CONTAINER (action_area), 10);
  
  GtkWidget * hbuttonbox1 = gtk_hbutton_box_new ();
  gtk_widget_ref (hbuttonbox1);
  gtk_widget_show (hbuttonbox1);
  gtk_box_pack_start (GTK_BOX (action_area), hbuttonbox1, TRUE, TRUE, 0);

  GtkWidget * button;

  /* Only display this widget if the connection is set up for HTTP connect. */

  bool use_turf_httpd = conn->queryPreferences()->getPreferenceBoolean("ProxyTurfHTTPd");
  if (use_turf_httpd) {
    button = gtk_button_new_with_label (_("Attempt HTTP Reconnect"));
    gtk_widget_ref(button);
    gtk_widget_show (button);
    gtk_container_add (GTK_CONTAINER (hbuttonbox1), button);

    g_signal_connect_data (button, "clicked",
			   GTK_SIGNAL_FUNC (on_autoreconnect_http_message_button_clicked),
			   (gpointer)auto_d,
			   NULL,
			   (GConnectFlags)0);
  }

  button = gtk_button_new_with_label (_("Close Window"));
  gtk_widget_ref (button);
  gtk_widget_show (button);
  gtk_container_add (GTK_CONTAINER (hbuttonbox1), button);
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  g_signal_connect_data(button, "clicked",
			GTK_SIGNAL_FUNC(on_autoreconnect_message_button_clicked),
			auto_d,
			NULL,
			(GConnectFlags)0);

  gtk_widget_grab_focus(GTK_WIDGET (button));
  gtk_widget_show(auto_d->dialog);

}

void auto_reconnect_init(Connection * c) {
  if (c->queryPreferences()->getPreferenceBoolean("ConnectionAutoReconnect")) {
    struct auto_data * data = (struct auto_data *)malloc(sizeof(struct auto_data));
    memset(data, 0, sizeof(struct auto_data));
    data->conn = c;
    data->countdown = 100;
    create_autoconnect_dialog(data);
    data->timeout = g_timeout_add(100, auto_reconnect_timeout, data);
  } else {
    char buf[1024];

    int written;
    int raw_read;
    int read;
    int connected;

    c->getSocket()->getStats(&written, &raw_read, &read, &connected);

  int hours, minutes, seconds;
  hours = connected / 3600;
  minutes = connected / 60;
  seconds = connected % 60;
  int write_rate, read_rate;

  if (connected == 0) {
    write_rate = 0;
    read_rate = 0;
  }
  else {
    write_rate = written  / connected;
    read_rate = raw_read / connected;
  }

  snprintf(buf, 1024, _("Total data sent: %d bytes\n"
			"Raw data received: %d bytes\n"
			"Compressed data received: %d bytes\n"
			"Time connected: %.2dh %.2dm %.2ds\n"
			"Average data sent: %d bytes/sec\n"
			"Average raw data received: %d bytes/sec\n"),
	   written, raw_read, read, hours, minutes, seconds,
	   write_rate, read_rate);
  
  autoreconnect_message(_("Connection closed."), buf, c);

    // @@ We may need to delete c here.
  }
}



