package App::Parcimonie;

use warnings;
use strict;

our $VERSION = '0.7'; # VERSION

use App::Parcimonie::GnuPG::Interface;
use Carp;
use Clone qw{clone};
use Config::General qw/ParseConfig/;
use Encode;
use English qw/-no_match_vars/;
use File::HomeDir;
use List::MoreUtils qw/uniq/;
use List::Util qw/shuffle/;
use Path::Class;
use Try::Tiny;

use Exporter;
our @ISA = qw(Exporter);

=head1 NAME

App::Parcimonie - tools for the parcimonie program

=head1 SYNOPSIS

See bin/parcimonie :)

=head1 EXPORT

=cut

our @EXPORT = qw/pickRandomItems gpgPublicKeys gpgRecvKeys
                 checkGpgHasDefinedKeyserver averageLapseTime/;

my $codeset;
try {
    require I18N::Langinfo;
    I18N::Langinfo->import(qw(langinfo CODESET));
    $codeset = langinfo(CODESET());
} catch {
    croak "No default character code set configured.\nPlease fix your locale settings.";
};


=head1 SUBROUTINES

=head2 pickRandomItems(N, @list)

Returns a list of N unique items picked at random from the input list.

=cut

sub pickRandomItems {
    my $n = shift;
    my @randomized_pool = shuffle(uniq(@_));

    defined $n
        or croak "pickRandomItems must be passed at least one argument";
    $n =~ m/[0-9]+/
        or croak "pickRandomItems must be passed an integer";
    $n <= @randomized_pool
        or croak "pickRandomItems can't return more unique items than you feed it";

    return @randomized_pool[0..$n - 1];
}

=head2 gpgPublicKeys()

Returns the list of key fingerprints found in the public keyring.

Args:
  - $gnupg_options: reference to a hash passed to ->hash_init

=cut

sub gpgPublicKeys {
    my $gnupg_options = shift;
    my %gi_args       = @_;

    my $gnupg = App::Parcimonie::GnuPG::Interface->new(%gi_args);
    $gnupg->options->hash_init(%{ clone($gnupg_options) });
    $gnupg->get_public_keys_hex();
}

=head2 gpgRecvKeys( @keyids )

Imports the keys with the given key IDs from a keyserver.

Args:
  - $keyids: reference to an array of key IDs
  - $gnupg_options: reference to a hash passed to ->hash_init

=cut

sub gpgRecvKeys {
    my $keyids        = shift;
    my $gnupg_options = shift;
    my %gi_args       = @_;

    my $gnupg = App::Parcimonie::GnuPG::Interface->new(%gi_args);
    $gnupg->options->hash_init(%{ clone($gnupg_options) });
    my $err_h = IO::Handle->new();
    my $handles = GnuPG::Handles->new(stderr => $err_h);
    my $pid = $gnupg->recv_keys( handles => $handles, command_args => $keyids );

    my @raw_stderr = <$err_h>;  # reading the output
    close $err_h;

    waitpid $pid, 0;  # clean up the finished GnuPG process

    my $std_err = decode($codeset, join('', @raw_stderr));

    # Filter out lines such as:
    # 11:21:19 libtorsocks(27234): The symbol res_query() was not found...
    $std_err =~ s{
                     ^                  # anchor at a line beginning
                     [[:digit:]]{2}
                     [:]
                     [[:digit:]]{2}     # time such as 11:21:19
                     [:]
                     [[:digit:]]{2}
                     [[:space:]]+
                     libtorsocks
                     [(]
                     [[:digit:]]+       # PID surrounded by parenthesis
                     [)]
                     [:]
                     [[:space:]]+
                     [^\n]*             # anything but a line break
                     $
                     [\n]?
             }{}xmsg;

    no warnings;
    unless ($CHILD_ERROR == 0) {
        use warnings;
        croak $std_err;
    }
    use warnings;

    return $std_err;
}

=head2 checkGpgHasDefinedKeyserver

Throws an exception if no keyserver is defined in GnuPG configuration.

=cut

sub checkGpgHasDefinedKeyserver {
    my $arg_ref = shift;
    my $gnupg_homedir = $arg_ref->{homedir};

    my $gpgconf;
    if (defined $gnupg_homedir) {
        $gpgconf = file($gnupg_homedir, 'gpg.conf');
    }
    else {
        $gpgconf = file(File::HomeDir->my_home, '.gnupg', 'gpg.conf');
    }

    -f $gpgconf
        or croak "No GnuPG configuration file was found in '$gpgconf'";
    -r $gpgconf
        or croak "The GnuPG configuration file ($gpgconf) is not readable";

    my %gpgconf = ParseConfig($gpgconf);

    defined $gpgconf{keyserver} && length $gpgconf{keyserver}
        or croak
            "No keyserver is defined in GnuPG configuration ($gpgconf).\n" .
            "See 'man parcimonie' to learn how to correct that.";

    return;
}

=head2 averageLapseTime

Argument: number of public keys in the keyring.
Returns the desirable average lapse time (in seconds) between two key
fetches.

=cut

sub averageLapseTime {
    604800 / shift;     # 1 week = 604800 seconds
}

1; # End of App::Parcimonie
