%module parted

%{
#include <parted/parted.h>
%}


// %section "Original copyright notice"

/*
    libparted - a library for manipulating disk partitions
    Copyright (C) 1999, 2000 Free Software Foundation, Inc.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

// %section "parted.h"

#include <stdlib.h>

// FIXME: wrap "architecture" stuff ?

%name(get_version) extern const char* ped_get_version ();


// %section "device.h"

// FIXME: wrap PED_SECTOR_SIZE ?

typedef long long PedSector;

typedef enum {
	PED_DEVICE_UNKNOWN	= 0,
	PED_DEVICE_SCSI		= 1,
	PED_DEVICE_IDE		= 2,
	PED_DEVICE_DAC960	= 3,
	PED_DEVICE_CPQARRAY	= 4,
	PED_DEVICE_FILE		= 5,
	PED_DEVICE_ATARAID	= 6,
	PED_DEVICE_I2O		= 7
} PedDeviceType;

/* A hard disk device - eg. /dev/hda, not /dev/hda3 */
%rename(Device) PedDevice;
typedef struct {
/* 	PedDevice*	next; */

	char*		model;			/* description of hardware */
	char*		path;			/* device /dev entry */

	PedDeviceType	type;			/* SCSI, IDE, etc. */
	int		sector_size;
	PedSector	length;

  // FIXME: maybe there are useful fields here, maybe some have to be
  // wrapped read-only.  By security I don't wrap them, we'll see as
  // time goes by.

/* 	int		open_count; */
	int		read_only;
/* 	int		external_mode; */
/* 	int		dirty; */
/* 	int		boot_dirty; */

/* 	int		heads, sectors, cylinders; */
/* 	int		geom_known; */
/* 	int		geom_already_guessed; */
/* 	short		host, did; */

/* 	void*		arch_specific; */

} PedDevice;

/* NOT mapping ArchOps */

%name(device_probe_all)
extern void ped_device_probe_all ();
%name(device_free_all)
extern void ped_device_free_all ();

/* FIXME: implement devices_get instead */
%{
PedDevice* ped_device_get_first () {
  return ped_device_get_next (NULL);
}
 %}
/* this must be declared after PedDevice itself */
%name(device_get_first)
extern PedDevice* ped_device_get_first ();


%name(device_get)
extern PedDevice* ped_device_get (const char* name);

%extend PedDevice {
  PedDevice* get_next () { return ped_device_get_next(self); }
  int is_busy() { return ped_device_is_busy (self); }
  int open() { return ped_device_open (self); }
  int close() { return ped_device_close (self); }
  void destroy() { ped_device_destroy (self); }

  int begin_external_access() { return ped_device_begin_external_access (self); }
  int end_external_access() { return ped_device_end_external_access (self); }
  int read(void* buffer, PedSector start, PedSector count) { 
    return ped_device_read (self, buffer, start, count);
  }
  int write (const void* buffer, PedSector start, PedSector count) {
    return ped_device_write (self, buffer, start, count);
  }
  int sync() { return ped_device_sync (self); }
  PedSector check (void* buffer, PedSector start, PedSector count) {
    return ped_device_check (self, buffer, start, count);
  }
}


// %section "geom.h"

%rename(Geometry) PedGeometry;
typedef struct {
	PedDevice*		dev;
	PedSector		start;
	PedSector		length;
	PedSector		end;
} PedGeometry;

%extend PedGeometry {
  int init(const PedDevice* dev, PedSector start, PedSector length) {
    return ped_geometry_init (self, dev, start, length);
  }
}

%extend PedDevice {
  PedGeometry* geometry_new(PedSector start, PedSector length) {
    return ped_geometry_new (self, start, length);
  }
}

%extend PedGeometry {
  PedGeometry* duplicate() { return ped_geometry_duplicate (self); }
  PedGeometry* intersect(const PedGeometry* b) { 
    return ped_geometry_intersect (self, b);
  }
  void destroy() { ped_geometry_destroy (self); }
  int set(PedSector start, PedSector length) {
    return ped_geometry_set (self, start, length);
  }
  int set_start(PedSector start) { return ped_geometry_set_start (self, start); }
  int set_end(PedSector end) { return ped_geometry_set_end (self, end); }

  int test_overlap(const PedGeometry* b) {
    return ped_geometry_test_overlap (self, b);
  }
  int test_inside(const PedGeometry* b) {
    return ped_geometry_test_inside (self, b);
  }
  int test_equal(const PedGeometry* b) {
    return ped_geometry_test_equal (self, b);
  }
  int test_sector_inside(PedSector sect) {
    return ped_geometry_test_sector_inside (self, sect);
  }

  int read(void* buffer, PedSector offset, PedSector count) {
    return ped_geometry_read (self, buffer, offset, count);
  }
  int write(const void* buffer, PedSector offset, PedSector count) {
    return ped_geometry_write (self, buffer, offset, count);
  }
  PedSector check(void* buffer,
		  PedSector buffer_size, PedSector offset,
		  PedSector granularity, PedSector count,
		  PedTimer* timer) {
    return ped_geometry_check (self, buffer, buffer_size, offset,
			       granularity, count, timer);
  }
  int sync() { return ped_geometry_sync (self); }

/* returns -1 if "sector" is not within dest's space. */
  PedSector map(const PedGeometry* src, PedSector sector) {
    return ped_geometry_map (self, src, sector);
  }
}


// %section "disk.h"

typedef enum {
	PED_PARTITION_LOGICAL		= 0x01,
	PED_PARTITION_EXTENDED		= 0x02,
	PED_PARTITION_FREESPACE		= 0x04,
	PED_PARTITION_METADATA		= 0x08
} PedPartitionType;

typedef enum {
	PED_PARTITION_BOOT=1,
	PED_PARTITION_ROOT=2,
	PED_PARTITION_SWAP=3,
	PED_PARTITION_HIDDEN=4,
	PED_PARTITION_RAID=5,
	PED_PARTITION_LVM=6,
	PED_PARTITION_LBA=7,
	PED_PARTITION_HPSERVICE=8
} PedPartitionFlag;
#define PED_PARTITION_FIRST_FLAG	PED_PARTITION_BOOT
#define PED_PARTITION_LAST_FLAG		PED_PARTITION_HPSERVICE

typedef enum {
	PED_DISK_TYPE_EXTENDED=1,	/* supports extended partitions */
	PED_DISK_TYPE_PARTITION_NAME=2	/* supports partition names */
} PedDiskTypeFeature;
#define PED_DISK_TYPE_FIRST_FEATURE	PED_DISK_TYPE_EXTENDED
#define PED_DISK_TYPE_LAST_FEATURE	PED_DISK_TYPE_PARTITION_NAME

%rename(Partition) PedPartition;
typedef struct {
	PedPartition*		prev;
	PedPartition*		next;

	PedDisk*		disk;
	PedGeometry		geom;
	int			num;

	PedPartitionType	type;
	const PedFileSystemType* fs_type;
	PedPartition*		part_list; /* for extended partitions */

	void*			disk_specific;
} PedPartition;

%rename(Disk) PedDisk;
typedef struct {
	PedDevice*		dev;
	const PedDiskType*	type;
	PedPartition*		part_list;

	void*			disk_specific;

/* office use only ;-) */
	int			needs_clobber;	/* clobber before write? */
	int			update_mode;	/* mode without free/metadata
						   partitions, for easier
						   update */
} PedDisk;

/* NOT mapping PedDiskOps */

%rename(DiskType) PedDiskType;
typedef struct {
	const char*		name;
	PedDiskTypeFeature	features;   /* bitmap of supported features */
} PedDiskType;

/* NOT mapping PedDiskArchOps */

%extend PedDiskType {
  void register() { ped_register_disk_type (self); }
  void unregister() { ped_unregister_disk_type (self); }
  PedDiskType* get_next() { return ped_disk_type_get_next (self); }
}

%name(disk_type_get) PedDiskType* ped_disk_type_get (const char* name);

%extend PedDiskType {
  int check_feature(PedDiskTypeFeature feature) {
    return ped_disk_type_check_feature (self, feature);
  }
}

%extend PedDevice {
  PedDiskType* disk_probe() { return ped_disk_probe (self); }
  int disk_clobber () { return ped_disk_clobber (self); }
  int disk_clobber_exclude (const PedDiskType* exclude) {
    return ped_disk_clobber_exclude (self, exclude);
  }
  PedDisk* disk_new () { return ped_disk_new (self); }
  PedDisk* disk_new_fresh (const PedDiskType* disk_type) {
    return ped_disk_new_fresh (self, disk_type);
  }
}

%extend PedDisk {
  PedDisk* duplicate() { return ped_disk_duplicate (self); }
  void destroy() { ped_disk_destroy (self); }
  int commit() { return ped_disk_commit (self); }
  int commit_to_dev() { return  ped_disk_commit_to_dev (self); }
  int commit_to_os() { return  ped_disk_commit_to_os (self); }
  int check() { return  ped_disk_check (self); }
  void print() { ped_disk_print (self); }
}

#if 0
%extend PedDevice {
  PedDisk* disk_open() { return ped_disk_open (self); }
  PedDisk* disk_create (PedDiskType* type) {
    return ped_disk_create (self, type);
  }
}
#endif

%extend PedDisk {
#if 0
  int clobber() { return ped_disk_clobber (self); }
  int close() { return ped_disk_close (self); }
  int read() { return ped_disk_read (self); }
  int write() { return ped_disk_write (self); }
  int is_busy() { return ped_disk_is_busy (self); }
#endif

  int get_primary_partition_count() {
    return ped_disk_get_primary_partition_count (self);
  }
  int get_last_partition_num() {
    return ped_disk_get_last_partition_num (self);
  }
  int get_max_primary_partition_count() {
    return ped_disk_get_max_primary_partition_count (self);
  }
}

#if 0
%extend PedDevice {
  PedDisk* disk_alloc(PedDiskType* type) {
    return ped_disk_alloc (self, type);
  }
}
%extend PedDisk {
  void free() { ped_disk_free (self); }
}
#endif

%extend PedDisk {
#if 0
  PedPartition* partition_alloc(PedPartitionType type,
				const PedFileSystemType* fs_type,
				PedSector start, PedSector end) {
    return ped_partition_alloc (self, type, fs_type, start, end);
  }
#endif
  PedPartition* partition_new(PedPartitionType type,
			      const PedFileSystemType* fs_type,
			      PedSector start, PedSector end) {
    return ped_partition_new (self, type, fs_type, start, end);
  }
}

%extend PedPartition {
  void destroy() { ped_partition_destroy (self); }
  int is_active() { return ped_partition_is_active (self); }
  int set_flag(PedPartitionFlag flag, int state) {
    return ped_partition_set_flag (self, flag, state);
  }
  int get_flag(PedPartitionFlag flag) { return ped_partition_get_flag (self, flag); }
  int is_flag_available(PedPartitionFlag flag) {
    return ped_partition_is_flag_available (self, flag);
  }
  int set_system(const PedFileSystemType* fs_type) {
    return ped_partition_set_system (self, fs_type);
  }
  int set_name(const char* name) { return ped_partition_set_name (self, name); }
  const char* get_name() { return ped_partition_get_name (self); }
  int is_busy() { return ped_partition_is_busy (self); }
}

%name(partition_type_get_name) const char* ped_partition_type_get_name (PedPartitionType part_type);
%name(partition_flag_get_name) const char* ped_partition_flag_get_name (PedPartitionFlag flag);
%name(partition_flag_get_by_name) PedPartitionFlag ped_partition_flag_get_by_name (const char* name);
%name(partition_flag_next) PedPartitionFlag ped_partition_flag_next (PedPartitionFlag flag);

%extend PedDisk {
  int add_partition(PedPartition* part, const PedConstraint* constraint) {
    return ped_disk_add_partition (self, part, constraint);
  }
  int remove_partition(PedPartition* part) {
    return ped_disk_remove_partition (self, part);
  }
  int delete_partition(PedPartition* part) {
    return ped_disk_delete_partition (self, part);
  }
  int delete_all() { return ped_disk_delete_all (self); }
}

/* FIXME: why is the PedDisk necessary ?
extern int ped_disk_set_partition_geom (PedDisk* disk, PedPartition* part,
					const PedConstraint* constraint,
					PedSector start, PedSector end);
extern int ped_disk_maximize_partition (PedDisk* disk, PedPartition* part,
					const PedConstraint* constraint);
extern PedGeometry* ped_disk_get_max_partition_geometry (PedDisk* disk,
		PedPartition* part, const PedConstraint* constraint);
extern PedPartition* ped_disk_next_partition (const PedDisk* disk,
					      const PedPartition* part);
*/

%extend PedDisk {
  int minimize_extended_partition() {
    return ped_disk_minimize_extended_partition (self);
  }
  PedPartition* get_partition(int num) { return ped_disk_get_partition (self, num); }
  PedPartition* get_partition_by_sector(PedSector sect) {
    return ped_disk_get_partition_by_sector (self, sect);
  }
  PedPartition* extended_partition() { return ped_disk_extended_partition (self); }
}

// %section "filesys.h"

/* NOT mapping PedFileSystemOps */

%rename(FileSystemType) PedFileSystemType;
typedef struct {
	const char* const		name;
} PedFileSystemType;

%rename(FileSystem) PedFileSystem;
typedef struct {
	PedFileSystemType*		type;
	PedGeometry*			geom;
	int			checked;

/* 	void*				type_specific; */
} PedFileSystem;

%extend PedFileSystemType {
  void register() { ped_file_system_type_register (self); }
  void unregister() { ped_file_system_type_unregister (self); }
}

%name(file_system_type_get)
     PedFileSystemType* ped_file_system_type_get (const char* name);

%extend PedFileSystemType {
  PedFileSystemType* get_next() { return ped_file_system_type_get_next (self); }
}

%extend PedGeometry {
  PedFileSystemType* file_system_probe() { return ped_file_system_probe (self); }
}

%extend PedFileSystemType {
  PedGeometry* file_system_probe_specific(PedGeometry* geom) {
    return ped_file_system_probe_specific (self, geom);
  }
}

%extend PedGeometry {
  int file_system_clobber() { return ped_file_system_clobber (self); }

  PedFileSystem* file_system_open() { return ped_file_system_open (self); }
  PedFileSystem* file_system_create(PedFileSystemType* type,
				    PedTimer* timer) {
    return ped_file_system_create (self, type, timer);
  }
}

%extend PedFileSystem {
  int close() { return ped_file_system_close (self); }
  int check(PedTimer* timer) { return ped_file_system_check (self, timer); }
  PedFileSystem* copy(PedGeometry* geom, PedTimer * timer) {
    return ped_file_system_copy (self, geom, timer);
  }
  int resize(PedGeometry* geom, PedTimer * timer) {
    return ped_file_system_resize (self, geom, timer);
  }
}

%extend PedFileSystemType {
  PedConstraint* file_system_get_create_constraint(const PedDevice* dev) {
    return ped_file_system_get_create_constraint (self, dev);
  }
}

%extend PedFileSystem {
  PedConstraint* get_resize_constraint() {
    return ped_file_system_get_resize_constraint (self);
  }
  PedConstraint* get_copy_constraint(const PedDevice* dev) {
    return ped_file_system_get_copy_constraint (self, dev);
  }
}


// %section "natmat.h"

%rename(Alignment) PedAlignment;
typedef struct {
	PedSector	offset;
	PedSector	grain_size;
} PedAlignment;

%name(round_up_to) 
     PedSector ped_round_up_to (PedSector sector, PedSector grain_size);
%name(round_down_to)
     PedSector ped_round_down_to (PedSector sector, PedSector grain_size);
%name(round_to_nearest)
     PedSector ped_round_to_nearest (PedSector sector, PedSector grain_size);
%name(greatest_common_divisor)
     PedSector ped_greatest_common_divisor (PedSector a, PedSector b);


%name(alignment_new)
     PedAlignment* ped_alignment_new (PedSector offset, PedSector grain_size);

%extend PedAlignment {
  int init(PedSector offset, PedSector grain_size) {
    return ped_alignment_init (self, offset, grain_size);
  }

  /* ped_alignment_new not mapped ?  or automatically ?  Hm... */

  void destroy() { ped_alignment_destroy (self); }
  PedAlignment* duplicate() { return ped_alignment_duplicate (self); }
  PedAlignment* intersect(const PedAlignment* b) {
    return ped_alignment_intersect (self, b);
  }

  PedSector align_up(const PedGeometry* geom, PedSector sector) {
    return ped_alignment_align_up (self, geom, sector);
  }
  PedSector align_down(const PedGeometry* geom, PedSector sector) {
    return ped_alignment_align_down (self, geom, sector);
  }
  PedSector align_nearest(const PedGeometry* geom, PedSector sector) {
    return ped_alignment_align_nearest (self, geom, sector);
  }

  int is_aligned(const PedGeometry* geom, PedSector sector) {
    return ped_alignment_is_aligned (self, geom, sector);
  }
}

extern const PedAlignment* ped_alignment_any;
extern const PedAlignment* ped_alignment_none;

/* FIXME: consider how to wrap this: */
#if 0
%name(div_round_up)
static inline PedSector
ped_div_round_up (PedSector numerator, PedSector divisor)
{
	return (numerator + divisor - 1) / divisor;
}
#endif


// %section "constraint.h"

%rename(Constraint) PedConstraint;
typedef struct {
	PedAlignment*	start_align;
	PedAlignment*	end_align;
	PedGeometry*	start_range;
	PedGeometry*	end_range;
	PedSector	min_size;
	PedSector	max_size;
} PedConstraint;

%extend PedConstraint {
  int init(const PedAlignment* start_align, const PedAlignment* end_align,
	   const PedGeometry* start_range, const PedGeometry* end_range,
	   PedSector min_size, PedSector max_size) {
    return ped_constraint_init (self, start_align, end_align,
				start_range, end_range,
				min_size, max_size);
  }
}

/* FIXME: make constructors instead ? */
%name(constraint_new) 
     PedConstraint* ped_constraint_new (const PedAlignment* start_align,
					const PedAlignment* end_align,
					const PedGeometry* start_range,
					const PedGeometry* end_range,
					PedSector min_size,
					PedSector max_size);

%name(constraint_new_from_min_max) 
     PedConstraint* ped_constraint_new_from_min_max (const PedGeometry* min,
						     const PedGeometry* max);
%name(constraint_new_from_min) 
     PedConstraint* ped_constraint_new_from_min (const PedGeometry* min);
%name(constraint_new_from_max) 
     PedConstraint* ped_constraint_new_from_max (const PedGeometry* max);

%extend PedConstraint {
  PedConstraint* duplicate() { return ped_constraint_duplicate (self); }
  void done() { ped_constraint_done (self); }
  void destroy() { ped_constraint_destroy (self); }

  PedConstraint* intersect(const PedConstraint* b) {
    return ped_constraint_intersect (self, b);
  }

  PedGeometry* solve_max() { return ped_constraint_solve_max (self); }

  PedGeometry* solve_nearest(const PedGeometry* geom) {
    return ped_constraint_solve_nearest (self, geom);
  }

  int is_solution(const PedGeometry* geom) {
    return ped_constraint_is_solution (self, geom);
  }
}

%extend PedDevice {
  PedConstraint* constraint_any() { return ped_constraint_any (self); }
}

%extend PedGeometry {
  PedConstraint* constraint_exact() { return ped_constraint_exact (self); }
}

/* FIXME: dropped in the lib but not in headers ?
extern const PedConstraint*	ped_constraint_none;
*/

// section "timer.h"

typedef void PedTimerHandler (PedTimer* timer, void* context);

%rename(Timer) PedTimer;
typedef struct {
        float                   frac;           /* fraction of operation done */
        time_t                  start;          /* time of start of op */
        time_t                  now;            /* time of last update (now!) */
        time_t                  predicted_end;  /* expected finish time */
        const char*             state_name;     /* eg: "copying data" */
        PedTimerHandler*        handler;        /* who to notify on updates */
        void*                   context;        /* context to pass to handler */
} PedTimer;

/* FIXME: make this a constructor instead */
%name(timer_new)
PedTimer* ped_timer_new (PedTimerHandler* handler, void* context);

%extend PedTimer {
  void destroy() { ped_timer_destroy (self); }

/* a nested timer automatically notifies it's parent.  You should only
 * create one when you are going to use it (not before)
 */
 PedTimer* new_nested(float nest_frac) {
   return ped_timer_new_nested (self, nest_frac);
 }
 void destroy_nested() { ped_timer_destroy_nested (self); }

 void touch() { ped_timer_touch (self); }
 void reset() { ped_timer_reset (self); }
 void update(float new_frac) { ped_timer_update (self, new_frac); }
 void set_state_name(const char* state_name) {
   ped_timer_set_state_name (self, state_name);
 }
}



#ifdef REALLY_USE_EXCEPTIONS
/* not swigified, based on *old* parted */

// %section "exceptions"

enum _PedExceptionType {
	PED_EXCEPTION_INFORMATION=1,
	PED_EXCEPTION_WARNING=2,
	PED_EXCEPTION_ERROR=3,
	PED_EXCEPTION_FATAL=4,
	PED_EXCEPTION_BUG=5,
	PED_EXCEPTION_NO_FEATURE=6,
};
typedef enum _PedExceptionType PedExceptionType;

enum _PedExceptionOption {
	PED_EXCEPTION_UNHANDLED=0,
	PED_EXCEPTION_FIX=1,
	PED_EXCEPTION_YES=2,
	PED_EXCEPTION_NO=4,
	PED_EXCEPTION_OK=8,
	PED_EXCEPTION_RETRY=16,
	PED_EXCEPTION_IGNORE=32,
	PED_EXCEPTION_CANCEL=64,
};
typedef enum _PedExceptionOption PedExceptionOption;
#define PED_EXCEPTION_OK_CANCEL	    (PED_EXCEPTION_OK + PED_EXCEPTION_CANCEL)
#define PED_EXCEPTION_YES_NO	    (PED_EXCEPTION_YES + PED_EXCEPTION_NO)
#define PED_EXCEPTION_YES_NO_CANCEL (PED_EXCEPTION_YES_NO \
						+ PED_EXCEPTION_CANCEL)
#define PED_EXCEPTION_IGNORE_CANCEL (PED_EXCEPTION_IGNORE		\
				     + PED_EXCEPTION_CANCEL)
#define PED_EXCEPTION_RETRY_CANCEL  (PED_EXCEPTION_RETRY + PED_EXCEPTION_CANCEL)
#define PED_EXCEPTION_RETRY_IGNORE_CANCEL (PED_EXCEPTION_RETRY		\
					   + PED_EXCEPTION_IGNORE_CANCEL)
#define PED_EXCEPTION_OPTION_FIRST PED_EXCEPTION_FIX
#define PED_EXCEPTION_OPTION_LAST PED_EXCEPTION_CANCEL

%rename(Exception) PedException;
typedef struct {
	char*			message;
	PedExceptionType	type;
	PedExceptionOption	options;	/* or'ed list of options that
						   the exception handler can
						   return */
} PedException;

/* FIXME: handler support ? */
/* typedef PedExceptionOption (PedExceptionHandler) (PedException* ex); */

extern int ped_exception;	/* set to true if there's an exception */

extern char* ped_exception_get_type_string (PedExceptionType ex_type);
extern char* ped_exception_get_option_string (PedExceptionOption ex_opt);

extern void ped_exception_set_handler (PedExceptionHandler* handler);

extern PedExceptionOption	ped_exception_throw (PedExceptionType ex_type,
						     PedExceptionOption ex_opt,
						     const char* message,
						     ...);

/* rethrows an exception - i.e. calls the exception handler, (or returns a
   code to return to pass up higher) */
extern PedExceptionOption	ped_exception_rethrow ();

/* frees an exception, indicating that the exception has been handled.
   Calling an exception handler counts. */
extern void			ped_exception_catch ();

/* indicate that exceptions should not go to the exception handler, but passed
   up to the calling function(s) */
extern void			ped_exception_fetch_all ();

/* indicate that exceptions should invoke the exception handler */
extern void			ped_exception_leave_all ();

#endif /* REALLY_USE_EXCEPTIONS */
