/*****************************************************************************
 *                                                                           *
 * Program:   paul                                                           *
 *            (P)rogramm zur (A)uswertung und (U)mformung von                *
 *            (L)aserbildern                                                 *
 * Modul:     insert.c                                                       *
 *            Insert image into other image                                  *
 * Author:    Andreas Tille                                                  *
 * Date:      25.10.1998                                                     *
 * Copyright: Andreas Tille, 1999; GNU Public License                        *
 *                                                                           *
 *****************************************************************************/

#include <fcntl.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/param.h>
#include <sys/stat.h>
#include "paul.h"

static int HtmlReadErr(const char *field, char *name, char *to_free, char *bname)
/* Clean after an error while parsing HTML file
 * --- Parameter: ---
 * char *field   : invalid field name 
 * char *name    : name of invalid HTML file
 * char *to_free : buffer to free
 * char *bname   : backup to put back into name
 * --- Return: ---
 * int   HtmlReadErr(): any value < 0 (-3)
 */
{
  g_warning(_("Invalid or missing %s field in %s"), field, name);
  FREE(to_free);
  rename(bname, name);
  return -3;
}


int InsertSingleImage(PICTURE *target, PAUL *p)
/* insert image into target image
 * --- Parameter: ---
 * PICTURE *target              : image to draw in
 * PAUL    *p                   : PAUL structur with necessary informations
 *                                p->op: operation image to insert
 *                                p->opt->xi: x-coordinate to insert
 *                                p->opt->yi: y-coordinate to insert
 *                                p->opt->html: HTML file to output or NULL if no output
 * --- Return: ---
 * int      InsertSingleImage() : RET_ERR or RET_OK
 */
{
  register unsigned char *tp, *ip, *fap;
  register int            storepix, x, y;
  unsigned char          *trp, *irp, *frp, *ftp, *fip, *buf;
  PICTURE                *ins;
   
  g_return_val_if_fail ( IS_PICTURE(target), RET_ERR);
  g_return_val_if_fail ( IS_PAUL(p), RET_ERR);
  g_return_val_if_fail ( IS_PICTURE(p->op), RET_ERR);
  
  x   = p->opt->xi;
  y   = p->opt->yi;
  ins = p->op;
  if ( x < 0 || x >= target->W || y < 0 || y >= target->H ||
       target->storepix != ins->storepix ) return RET_ERR;

  storepix = target->storepix;
  for ( trp = target->DATA + (y*target->W + x)*storepix, irp = ins->DATA, 
        ftp = target->DATA + target->size*storepix, fip = ins->DATA + ins->size*storepix, 
        frp = target->DATA + ((y+1)*target->W)*storepix; 
        trp < ftp && irp < fip; 
        trp += target->W * storepix, irp += ins->W * storepix,
        frp += target->W * storepix )
    for ( tp = trp, ip = irp, fap = MIN(frp, tp + ins->W*storepix); tp < fap; tp++, ip++ )
      *tp = *ip;

  if ( HtmlOutput(p->opt->f) ) {
    FILE        *fp;
    struct stat  stat_buf;
    int          fh;
    char         *buf;
    const  char  *kw = "keywords", *as = "<AREA SHAPE", *content = "content=";
      
    if ( !target->ext ) target->ext = "png";
    if ( !ins->ext    ) ins->ext    = "html";

    if ( !(p->opt->html) ) p->opt->html = g_strdup("paul.html");
    if ( !(p->opt->link) ) p->opt->link = g_strdup_printf("%s.%s", ins->file, ins->ext ? ins->ext : "html");
    
    fh = stat(p->opt->html, &stat_buf);
    g_return_val_if_fail ( !S_ISDIR(stat_buf.st_mode), RET_ERR ) ;
    
    if ( fh ) {
      g_return_val_if_fail ( (fp = fopen(p->opt->html, "w")), RET_ERR );
      fputs("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.0//EN\">\n", fp);
      fprintf(fp, "<HTML>\n<HEAD>\n<TITLE>%s</TITLE>\n", target->file);
      if ( (buf = GetCharSpec(target->spec, ChunkNameAuthor)) )
        fprintf(fp, "<meta name=\"author\" content=\"%s\">\n", buf);
      fprintf(fp, "<meta name=\"%s\" %s\"%s %s %s\">\n</HEAD>\n<BODY>\n", 
              kw, content, exename, target->file, ins->file);
      fprintf(fp, "<IMG SRC=\"%s.%s\" ALT=\"%s\" BORDER=\"0\" WIDTH=\"%i\" HEIGHT=\"%i\" usemap=\"#%s_map\">\n",
              target->file, target->ext, target->file, target->W, target->H, target->file);
      fprintf(fp, "\n<MAP NAME=\"%s_map\">\n", target->file);
      fprintf(fp, "%s=\"rect\" ALT=\"%s\" COORDS=\"%i,%i,%i,%i\" HREF=\"%s\">\n", as, ins->file, 
              x, y, MIN(x+ins->W, target->W), MIN(y+ins->H, target->H), p->opt->link);
      fputs("</MAP>\n</BODY>\n</HTML>", fp);
    } else {
      char  bname[MAXPATHLEN], *buf, *ap, *ep;
      int   i;
	 
      Backup(p->opt->html);
      strcat(strcpy(bname, p->opt->html), "~");
      g_return_val_if_fail ( (fh  = open(bname, O_RDONLY)) > -1, RET_ERR );
      buf = g_new0(char, stat_buf.st_size+1);
      if ( stat_buf.st_size != ( i = read(fh, buf, stat_buf.st_size)) ) {
        close(fh);
	FREE(buf);
        g_return_val_if_fail ( stat_buf.st_size == i, -2 );
      }
      close(fh);
      g_return_val_if_fail ( (fp = fopen(p->opt->html, "w")), RET_ERR );
      ap = buf;
      if ( !(ep = strstr(ap, kw)) )  return HtmlReadErr(kw, p->opt->html, buf, bname);
      if ( !(ep = strstr(ep, content)) ) return HtmlReadErr(kw, p->opt->html, buf, bname);
      if ( !(ep = strchr(ep, '"')) ) return HtmlReadErr(kw, p->opt->html, buf, bname);
      if ( !(ep = strchr(++ep, '"')) ) return HtmlReadErr(kw, p->opt->html, buf, bname);
      *ep = 0;
      fputs(ap, fp);
      fprintf(fp, " %s\"", ins->file);
      ap = ep + 1;
      if ( !(ep = strstr(ap, as)) ) return HtmlReadErr(as, p->opt->html, buf, bname);
      *ep = 0;
      fputs(ap, fp);
      *(ap = ep) = *as;
      fprintf(fp, "%s=\"rect\" ALT=\"%s\" COORDS=\"%i,%i,%i,%i\" HREF=\"%s\">\n", as, ins->file, 
              x, y, MIN(x+ins->W, target->W), MIN(y+ins->H, target->H), p->opt->link);
      fputs(ap, fp);
      FREE(buf);
    }
    fclose(fp);
  }

  buf = g_strdup_printf("%s inserted at {%i,%i}{%i,%i}", ins->file, x, y, 
                        MIN(x+ins->W, target->W), MIN(y+ins->H, target->H));
              /* there is no special insert flag. vvvvvv */
  ImgChunksUpdate(target, TypInsert, buf, APPINSERT, SAFEFLAG);
  FREE(buf);
   
  return RET_OK;
}

int InsertImage(PAUL *p)
/* insert operation image into each image in image list
 * --- Parameter: ---
 * PAUL *p             : image structure
 * --- Return: ---
 * int   InsertImage() : RET_ERR or RET_OK
 */
{
  PICTURE *bild;
  GList   *pl;

  g_return_val_if_fail ( IS_PAUL(p), RET_ERR ) ;
  g_return_val_if_fail ( NBILDER(p->piclist), RET_ERR ) ;
  g_return_val_if_fail ( IS_PICTURE(p->op), RET_ERR );

  for ( bild = BILD(pl = p->piclist); pl; bild = BILD(pl = pl->next) ) {
    if ( HtmlOutput(p->opt->f) ) {
      if ( !(p->opt->html) ) p->opt->html = g_strdup_printf("%s.html", bild->file);
    } else
      p->opt->html = NULL;
    
    InsertSingleImage(bild, p);
  }

  return RET_OK;
}












