/*  Pdlzip - Data compressor based on the LZMA algorithm
    Copyright (C) 2010, 2011 Antonio Diaz Diaz.

    This program is free software: you have unlimited permission
    to copy, distribute and modify it.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
*/

#ifndef __cplusplus
enum Bool { false = 0, true = 1 };
typedef enum Bool bool;
#endif

#ifndef max
  #define max(x,y) ((x) >= (y) ? (x) : (y))
#endif
#ifndef min
  #define min(x,y) ((x) <= (y) ? (x) : (y))
#endif

enum {
  min_dictionary_bits = 12,
  min_dictionary_size = 1 << min_dictionary_bits,
  max_dictionary_bits = 26,
  max_dictionary_size = 1 << max_dictionary_bits,
  literal_context_bits = 3,
  pos_state_bits = 2,

  len_low_bits = 3,
  len_mid_bits = 3,
  len_high_bits = 8,
  len_low_symbols = 1 << len_low_bits,
  len_mid_symbols = 1 << len_mid_bits,
  len_high_symbols = 1 << len_high_bits,
  max_len_symbols = len_low_symbols + len_mid_symbols + len_high_symbols,

  min_match_len = 2,					/* must be 2 */
  max_match_len = min_match_len + max_len_symbols - 1,	/* 273 */
  min_match_len_limit = 5 };

typedef uint32_t CRC32[256];	/* Table of CRCs of all 8-bit messages. */

extern CRC32 crc32;

static inline void CRC32_init()
  {
  unsigned int n;
  for( n = 0; n < 256; ++n )
    {
    unsigned int c = n;
    int k;
    for( k = 0; k < 8; ++k )
      { if( c & 1 ) c = 0xEDB88320U ^ ( c >> 1 ); else c >>= 1; }
    crc32[n] = c;
    }
  }

static inline void CRC32_update_byte( uint32_t * crc, const uint8_t byte )
  { *crc = crc32[(*crc^byte)&0xFF] ^ ( *crc >> 8 ); }
static inline void CRC32_update_buf( uint32_t * crc, const uint8_t * const buffer,
                                     const int size )
  {
  int i;
  for( i = 0; i < size; ++i )
    *crc = crc32[(*crc^buffer[i])&0xFF] ^ ( *crc >> 8 );
  }


static inline int real_bits( const int value )
  {
  int bits = 0, i, mask;
  for( i = 1, mask = 1; mask > 0; ++i, mask <<= 1 )
    if( value & mask ) bits = i;
  return bits;
  }


static const uint8_t magic_string[4] = { 'L', 'Z', 'I', 'P' };

typedef uint8_t File_header[6];		/* 0-3 magic bytes */
					/*   4 version */
					/*   5 coded_dict_size */
enum { Fh_size = 6 };

static inline void Fh_set_magic( File_header data )
  {
  memcpy( data, magic_string, 4 );
  data[4] = 1;
  }

static inline bool Fh_verify_magic( const File_header data )
  {
  return ( memcmp( data, magic_string, 4 ) == 0 );
  }

static inline uint8_t Fh_version( const File_header data )
  { return data[4]; }

static inline bool Fh_verify_version( const File_header data )
  { return ( data[4] <= 1 ); }

static inline int Fh_get_dictionary_size( const File_header data )
  {
  int sz = ( 1 << ( data[5] & 0x1F ) );
  if( sz > min_dictionary_size && sz <= max_dictionary_size )
    sz -= ( sz / 16 ) * ( ( data[5] >> 5 ) & 0x07 );
  return sz;
  }

static inline bool Fh_set_dictionary_size( File_header data, const int sz )
  {
  if( sz >= min_dictionary_size && sz <= max_dictionary_size )
    {
    data[5] = real_bits( sz - 1 );
    if( sz > min_dictionary_size )
      {
      const int base_size = 1 << data[5];
      const int wedge = base_size / 16;
      int i;
      for( i = 7; i >= 1; --i )
        if( base_size - ( i * wedge ) >= sz )
          { data[5] |= ( i << 5 ); break; }
      }
    return true;
    }
  return false;
  }


typedef uint8_t File_trailer[20];
			/*  0-3  CRC32 of the uncompressed data */
			/*  4-11 size of the uncompressed data */
			/* 12-19 member size including header and trailer */

enum { Ft_size = 20 };

static inline int Ft_versioned_size( const int version )
  { return ( ( version >= 1 ) ? 20 : 12 ); }

static inline uint32_t Ft_get_data_crc( const File_trailer data )
  {
  uint32_t tmp = 0;
  int i;
  for( i = 3; i >= 0; --i ) { tmp <<= 8; tmp += data[i]; }
  return tmp;
  }

static inline void Ft_set_data_crc( File_trailer data, uint32_t crc )
  {
  int i;
  for( i = 0; i <= 3; ++i ) { data[i] = (uint8_t)crc; crc >>= 8; }
  }

static inline long long Ft_get_data_size( const File_trailer data )
  {
  long long tmp = 0;
  int i;
  for( i = 11; i >= 4; --i ) { tmp <<= 8; tmp += data[i]; }
  return tmp;
  }

static inline void Ft_set_data_size( File_trailer data, long long sz )
  {
  int i;
  for( i = 4; i <= 11; ++i ) { data[i] = (uint8_t)sz; sz >>= 8; }
  }

static inline long long Ft_get_member_size( const File_trailer data )
  {
  long long tmp = 0;
  int i;
  for( i = 19; i >= 12; --i ) { tmp <<= 8; tmp += data[i]; }
  return tmp;
  }

static inline void Ft_set_member_size( File_trailer data, long long sz )
  {
  int i;
  for( i = 12; i <= 19; ++i ) { data[i] = (uint8_t)sz; sz >>= 8; }
  }


/* defined in main.c */
extern int verbosity;

void show_error( const char * const msg, const int errcode, const bool help );
void internal_error( const char * const msg );
