/*
 * Permafrost - Physical modelling framework
 *
 * Copyright (C) 2009, 2010 Stefano D'Angelo <zanga.mail@gmail.com>
 *
 * See the COPYING file for license conditions.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <float.h>

#include "src/types.h"
#include "src/util.h"
#include "src/list.h"
#include "src/expr.h"
#include "src/compile.h"

char *compile_output_dir = NULL;
char compile_gen_code = 1;
char compile_gen_descriptor = 1;
char compile_gen_makefile = 1;
char compile_gen_manifest = 1;
char compile_gen_plugin_ttl = 1;
char compile_gen_extra_ttl = 1;
char *compile_uri_prefix = "http://www.example.com/";
char *compile_license_uri = "http://usefulinc.com/doap/licenses/unknown";

static const char *
get_filename(const char *filename, const char *suffix)
{
	char *buf;

	if ((compile_output_dir == NULL) && (suffix == NULL))
		return filename;

	if ((compile_output_dir != NULL) && (suffix != NULL))
	  {
		buf = xmalloc(strlen(compile_output_dir) + strlen(filename)
			      + strlen(suffix) + 2);
		sprintf(buf, "%s/%s%s", compile_output_dir, filename, suffix);
	  }
	else if ((compile_output_dir != NULL) && (suffix == NULL))
	  {
		buf = xmalloc(strlen(compile_output_dir) + strlen(filename)
			      + 2);
		sprintf(buf, "%s/%s", compile_output_dir, filename);
	  }
	else /* ((compile_output_dir == NULL) && (suffix != NULL)) */
	  {
		buf = xmalloc(strlen(filename) + strlen(suffix) + 1);
		sprintf(buf, "%s%s", filename, suffix);
	  }

	return buf;
}

/* FIXME: possibility of false positives (file exists, but couldn't open) */
static char
file_exists(const char *filename, const char *suffix)
{
	FILE *fp;
	const char *f;

	f = get_filename(filename, suffix);

	fp = fopen(f, "r");
	if (fp != NULL)
		fclose(fp);

	if (f != filename)
		free((void *)f);

	return fp != NULL;
}

static FILE *
xfopenw(const char *filename, const char *suffix)
{
	FILE *fp;
	const char *f;

	f = get_filename(filename, suffix);

	fp = fopen(f, "w");
	if (fp == NULL)
	  {
		fprintf(stderr, "error: could not create file `%s': %s\n", f,
			strerror(errno));
		exit(EXIT_FAILURE);
	  }

	if (f != filename)
		free((void *)f);

	return fp;
}

static void
manifest_print_plugin(void *data, void *context)
{
	struct scheduled_system *ss;
	FILE *fp;

	ss = (struct scheduled_system *)data;
	fp = (FILE *)context;

	fprintf(fp, "\n");
	fprintf(fp, "<%s%s> a lv2:Plugin ;\n", compile_uri_prefix,
		ss->system->id);
	fprintf(fp, "\tlv2:binary  <plugin.so> ;\n");
	fprintf(fp, "\trdfs:seeAlso <%s.ttl> ", ss->system->id);
	fprintf(fp, ";\n\trdfs:seeAlso <%s-extra.ttl> ", ss->system->id);
	fprintf(fp, ".\n");
}

struct ttl_print_io_port_context
  {
	struct scheduled_system	*ss;
	FILE			*fp;
  };

static void
ttl_print_io_port_uri(void *data, void *context)
{
	struct scheduled_expr *sexpr;
	struct ttl_print_io_port_context *ctx;

	sexpr = (struct scheduled_expr *)data;
	ctx = (struct ttl_print_io_port_context *)context;

	if (sexpr->component != NULL)
		return;

	fprintf(ctx->fp, ";\n\tlv2:port <%s%s/ports/%s> ", compile_uri_prefix,
		ctx->ss->system->id, sexpr->port->id);
}

static void
ttl_print_io_port(void *data, void *context)
{
	struct scheduled_expr *sexpr;
	struct ttl_print_io_port_context *ctx;

	sexpr = (struct scheduled_expr *)data;
	ctx = (struct ttl_print_io_port_context *)context;

	if (sexpr->component != NULL)
		return;

	fprintf(ctx->fp, "\n");
	fprintf(ctx->fp, "<%s%s/ports/%s> a lv2:Port ;\n", compile_uri_prefix,
		ctx->ss->system->id, sexpr->port->id);
	fprintf(ctx->fp, "\ta lv2:%s ;\n",
		(sexpr->port->sync == port_sync_sync)
		? "AudioPort" : "ControlPort");
	fprintf(ctx->fp, "\ta lv2:%s ;\n",
		sexpr->is_output ? "OutputPort" : "InputPort");
	fprintf(ctx->fp, "\tlv2:index %lu ;\n", sexpr->index);
	fprintf(ctx->fp, "\tlv2:symbol \"%s\" .\n", sexpr->port->id);
}

static void
ttl_compile(void *data, void *context)
{
	struct scheduled_system *ss;
	struct ttl_print_io_port_context t_ctx;
	FILE *fp;

	ss = (struct scheduled_system *)data;

	fp = xfopenw(ss->system->id, ".ttl");

	fprintf(fp, "@prefix lv2: <http://lv2plug.in/ns/lv2core#> .\n");
	fprintf(fp, "\n");
	fprintf(fp, "<%s%s> a lv2:Plugin ", compile_uri_prefix,
		ss->system->id);

	t_ctx.ss = ss;
	t_ctx.fp = fp;
	list_for_each_rev(ss->exprs, ttl_print_io_port_uri, &t_ctx);

	fprintf(fp, ".\n");

	list_for_each_rev(ss->exprs, ttl_print_io_port, &t_ctx);

	fclose(fp);
}

static void
ttl_extra_print_io_port(void *data, void *context)
{
	struct scheduled_expr *sexpr;
	struct ttl_print_io_port_context *ctx;

	sexpr = (struct scheduled_expr *)data;
	ctx = (struct ttl_print_io_port_context *)context;

	if (sexpr->component != NULL)
		return;

	fprintf(ctx->fp, "\n");
	fprintf(ctx->fp, "<%s%s/ports/%s> lv2:name \"%s\" .\n",
		compile_uri_prefix, ctx->ss->system->id, sexpr->port->id,
		sexpr->port->id);
}

static void
ttl_extra_compile(void *data, void *context)
{
	struct scheduled_system *ss;
	struct ttl_print_io_port_context t_ctx;
	FILE *fp;

	ss = (struct scheduled_system *)data;

	if (file_exists(ss->system->id, "-extra.ttl"))
		return;

	fp = xfopenw(ss->system->id, "-extra.ttl");

	fprintf(fp, "@prefix lv2:  <http://lv2plug.in/ns/lv2core#> .\n");
	fprintf(fp, "@prefix doap: <http://usefulinc.com/ns/doap#> .\n");
	fprintf(fp, "\n");
	fprintf(fp, "<%s%s>\n", compile_uri_prefix, ss->system->id);
	fprintf(fp, "\tdoap:name \"%s\" ;\n", ss->system->id);
	fprintf(fp, "\tdoap:license <%s> .", compile_license_uri);
	fprintf(fp, "\n");

	t_ctx.ss = ss;
	t_ctx.fp = fp;
	list_for_each_rev(ss->exprs, ttl_extra_print_io_port, &t_ctx);

	fclose(fp);
}

static void
c_print_include(void *data, void *context)
{
	struct scheduled_system *ss;
	FILE *fp;

	ss = (struct scheduled_system *)data;
	fp = (FILE *)context;

	fprintf(fp, "#include \"%s.h\"\n", ss->system->id);
}

static void
c_print_descriptor(void *data, void *context)
{
	struct scheduled_system *ss;
	FILE *fp;

	ss = (struct scheduled_system *)data;
	fp = (FILE *)context;

	fprintf(fp, "\n");
	fprintf(fp, "static const LV2_Descriptor descriptor_%s =\n",
		ss->system->id);
	fprintf(fp, "  {\n");
	fprintf(fp, "\t/* .URI\t\t\t= */ \"%s%s\",\n", compile_uri_prefix,
		ss->system->id);
	fprintf(fp, "\t/* .instantiate\t\t= */ instantiate_%s,\n",
		ss->system->id);
	fprintf(fp, "\t/* .connect_port\t= */ connect_port_%s,\n",
		ss->system->id);
	fprintf(fp, "\t/* .activate\t\t= */ activate_%s,\n", ss->system->id);
	fprintf(fp, "\t/* .run\t\t\t= */ run_%s,\n", ss->system->id);
	fprintf(fp, "\t/* .deactivate\t\t= */ deactivate_%s,\n",
		ss->system->id);
	fprintf(fp, "\t/* .cleanup\t\t= */ cleanup_%s,\n", ss->system->id);
	fprintf(fp, "\t/* .extension_data\t= */ NULL\n");
	fprintf(fp, "  };\n");
}

static void
c_print_descriptor_name(void *data, void *context)
{
	struct scheduled_system *ss;
	FILE *fp;

	ss = (struct scheduled_system *)data;
	fp = (FILE *)context;

	fprintf(fp, "\t&descriptor_%s,\n", ss->system->id);
}

static void
h_compile(void *data, void *context)
{
	struct scheduled_system *ss;
	FILE *fp;

	ss = (struct scheduled_system *)data;

	fp = xfopenw(ss->system->id, ".h");

	fprintf(fp, "#ifndef _%s_H_\n", ss->system->id);
	fprintf(fp, "#define _%s_H_\n", ss->system->id);

	fprintf(fp, "\n");
	fprintf(fp, "#include <lv2.h>\n");

	fprintf(fp, "\n");
	fprintf(fp, "LV2_Handle\n");
	fprintf(fp, "instantiate_%s(const LV2_Descriptor *descriptor,\n",
		ss->system->id);
	fprintf(fp, "\tdouble sample_rate, const char *bundle_path,\n");
	fprintf(fp, "\tconst LV2_Feature * const *features);\n");

	fprintf(fp, "\n");
	fprintf(fp, "void\n");
	fprintf(fp, "connect_port_%s(LV2_Handle instance, uint32_t port,\n",
		ss->system->id);
	fprintf(fp, "\tvoid *data_location);\n");

	fprintf(fp, "\n");
	fprintf(fp, "void\n");
	fprintf(fp, "activate_%s(LV2_Handle instance);\n", ss->system->id);

	fprintf(fp, "\n");
	fprintf(fp, "void\n");
	fprintf(fp, "run_%s(LV2_Handle instance, uint32_t sample_count);\n",
		ss->system->id);

	fprintf(fp, "\n");
	fprintf(fp, "void\n");
	fprintf(fp, "deactivate_%s(LV2_Handle instance);\n", ss->system->id);

	fprintf(fp, "\n");
	fprintf(fp, "void\n");
	fprintf(fp, "cleanup_%s(LV2_Handle instance);\n", ss->system->id);

	fprintf(fp, "\n");
	fprintf(fp, "#endif /* !_%s_H_ */\n", ss->system->id);

	fclose(fp);
}

static void
c_instance_print_io_port(void *data, void *context)
{
	struct scheduled_expr *sexpr;
	FILE *fp;

	sexpr = (struct scheduled_expr *)data;
	fp = (FILE *)context;

	if (sexpr->component != NULL)
		return;

	fprintf(fp, "\tfloat\t*port_%s;\n", sexpr->port->id);
}

static void
c_instance_print_buf(void *data, void *context)
{
	struct scheduled_expr *sexpr;
	FILE *fp;

	sexpr = (struct scheduled_expr *)data;
	fp = (FILE *)context;

	if (sexpr->db_type == delay_buf_type_none)
		return;

	fprintf(fp, "\n");
	fprintf(fp, "\t/* %s%s%s%s */\n",
		(sexpr->component != NULL) ? sexpr->component->id : "",
		(sexpr->component != NULL) ? "." : "", sexpr->port->id,
		((sexpr->port->type == port_type_w)
		 || (sexpr->port->type == port_type_k)) ? ".out" : "");

	if (sexpr->db_type == delay_buf_type_single)
		fprintf(fp, "\tfloat\t buf_%lu;\n", sexpr->index);
	else if (sexpr->db_type == delay_buf_type_fixed)
	  {
		fprintf(fp, "\tfloat\t buf_%lu[%lu];\n", sexpr->index,
			(size_t)sexpr->delay_min + 1);
		fprintf(fp, "\tsize_t\t buf_%lu_cur;\n", sexpr->index);
	  }
	else
	  {
		fprintf(fp, "\tfloat\t*buf_%lu;\n", sexpr->index);
		fprintf(fp, "\tsize_t\t buf_%lu_cur;\n", sexpr->index);
		fprintf(fp, "\tsize_t\t buf_%lu_len;\n", sexpr->index);
	  }
}

static void
c_instance_print_interm(void *data, void *context)
{
	struct scheduled_expr *sexpr;
	FILE *fp;

	sexpr = (struct scheduled_expr *)data;
	fp = (FILE *)context;

	if (sexpr->component == NULL)
		return;

	if (sexpr->is_df_refd)
	  {
		fprintf(fp, "\n");
		fprintf(fp, "\t/* %s.%s%s */\n", sexpr->component->id,
			sexpr->port->id,
			((sexpr->port->type == port_type_w)
			 || (sexpr->port->type == port_type_k)) ? ".out" : "");
		fprintf(fp, "\tfloat\t val_%lu;\n", sexpr->index);
	  }
}

static void
c_connect_port_print(void *data, void *context)
{
	struct scheduled_expr *sexpr;
	FILE *fp;

	sexpr = (struct scheduled_expr *)data;
	fp = (FILE *)context;

	if (sexpr->component != NULL)
		return;

	fprintf(fp, "\t\tcase %lu:\n", sexpr->index);
	fprintf(fp, "\t\t\tplugin->port_%s = data_location;\n",
		sexpr->port->id);
	fprintf(fp, "\t\t\tbreak;\n");
}

struct c_instantiate_print_delays_context
  {
	size_t	 index;
	FILE	*fp;
  };

static void
c_instantiate_print_delays(void *data, void *context)
{
	struct delay *delay;
	struct c_instantiate_print_delays_context *ctx;

	delay = (struct delay *)data;
	ctx = (struct c_instantiate_print_delays_context *)context;

	fprintf(ctx->fp, "\tvalue = (size_t)ceilf((");
	if (delay->delay_max == NULL)
		expr_c_compile(delay->delay, ctx->fp, 0);
	else
		expr_c_compile(delay->delay_max, ctx->fp, 0);
	fprintf(ctx->fp, ") + 1.0);\n");
	fprintf(ctx->fp, "\tif (value > instance->buf_%lu_len)\n", ctx->index);
	fprintf(ctx->fp, "\t\tinstance->buf_%lu_len = value;\n", ctx->index);
}

static void
c_instantiate_print_alloc(void *data, void *context)
{
	struct scheduled_expr *sexpr;
	FILE *fp;
	struct c_instantiate_print_delays_context ctx;

	sexpr = (struct scheduled_expr *)data;
	fp = (FILE *)context;

	if (sexpr->db_type != delay_buf_type_var)
		return;

	fprintf(fp, "\n");
	fprintf(fp, "\tinstance->buf_%lu_len = (size_t)%.*f;\n", sexpr->index,
		DBL_DIG, sexpr->delay_min + 1.0);
	ctx.fp = fp;
	ctx.index = sexpr->index;
	list_for_each(sexpr->delays, c_instantiate_print_delays, &ctx);
	fprintf(fp, "\n");
	fprintf(fp, "\tinstance->buf_%lu = \n"
		"\t\t(float *)malloc(instance->buf_%lu_len * sizeof(float));\n",
		sexpr->index, sexpr->index);
	fprintf(fp, "\tif (instance->buf_%lu == NULL)\n", sexpr->index);
	fprintf(fp, "\t\tgoto err_%lu;\n", sexpr->index);
}

struct c_instantiate_print_err_context
  {
	FILE	*fp;
	char	 first;
  };

static void
c_instantiate_print_err(void *data, void *context)
{
	struct scheduled_expr *sexpr;
	struct c_instantiate_print_err_context *ctx;

	sexpr = (struct scheduled_expr *)data;
	ctx = (struct c_instantiate_print_err_context *)context;

	if (sexpr->db_type != delay_buf_type_var)
		return;

	if (ctx->first)
		ctx->first = 0;
	else
	  {
		fprintf(ctx->fp, "\tfree(instance->buf_%lu);\n", sexpr->index);
		fprintf(ctx->fp, "\n");
	  }
	fprintf(ctx->fp, "err_%lu:\n", sexpr->index);
}

static void
c_activate_print_buf(void *data, void *context)
{
	struct scheduled_expr *sexpr;
	FILE *fp;

	sexpr = (struct scheduled_expr *)data;
	fp = (FILE *)context;

	if (sexpr->db_type == delay_buf_type_none)
		return;

	fprintf(fp, "\n");

	if (sexpr->db_type == delay_buf_type_single)
		fprintf(fp, "\tplugin->buf_%lu = 0.0;\n", sexpr->index);
	else if (sexpr->db_type == delay_buf_type_fixed)
	  {
		fprintf(fp, "\tmemset(plugin->buf_%lu, 0, %lu * "
			"sizeof(float));\n", sexpr->index,
			(size_t)sexpr->delay_min + 1);
		fprintf(fp, "\tplugin->buf_%lu_cur = 0;\n", sexpr->index);
	  }
	else if (sexpr->db_type == delay_buf_type_var)
	  {
		fprintf(fp, "\tmemset(plugin->buf_%lu, 0, "
			"plugin->buf_%lu_len * sizeof(float));\n", sexpr->index,
			sexpr->index);
		fprintf(fp, "\tplugin->buf_%lu_cur = 0;\n", sexpr->index);
	  }
}

static void
c_activate_print_interm(void *data, void *context)
{
	struct scheduled_expr *sexpr;
	FILE *fp;

	sexpr = (struct scheduled_expr *)data;
	fp = (FILE *)context;

	if (sexpr->component == NULL)
		return;

	if (sexpr->is_df_refd)
		fprintf(fp, "\tplugin->val_%lu = 0.0;\n", sexpr->index);
}

static void
c_run_print_sexpr(void *data, void *context)
{
	struct scheduled_expr *sexpr;
	FILE *fp;

	sexpr = (struct scheduled_expr *)data;
	fp = (FILE *)context;

	if (((sexpr->component == NULL) && !sexpr->is_output)
	    || ((sexpr->component != NULL) && !sexpr->is_df_refd))
		return;

	if (sexpr->component == NULL)
		fprintf(fp, "\t\tplugin->port_%s%s = ", sexpr->port->id,
			(sexpr->port->sync == port_sync_sync) ? "[i]" : "[0]");
	else
		fprintf(fp, "\t\tplugin->val_%lu = ", sexpr->index);
	expr_c_compile(sexpr->expr, fp, 1);
	fprintf(fp, ";\n");
}

static void
c_run_print_buf_update(void *data, void *context)
{
	struct scheduled_expr *sexpr;
	FILE *fp;

	sexpr = (struct scheduled_expr *)data;
	fp = (FILE *)context;

	if (sexpr->db_type == delay_buf_type_none)
		return;
	else if (sexpr->db_type == delay_buf_type_single)
		fprintf(fp, "\t\tplugin->buf_%lu = ", sexpr->index);
	else
		fprintf(fp, "\t\tplugin->buf_%lu[plugin->buf_%lu_cur] = ",
			sexpr->index, sexpr->index);

	if ((sexpr->component == NULL) && !sexpr->is_output)
		fprintf(fp, "plugin->port_%s%s", sexpr->port->id,
			(sexpr->port->sync == port_sync_sync) ? "[i]" : "[0]");
	else
		expr_c_compile(sexpr->expr, fp, 1);
	fprintf(fp, ";\n");
}

static void
c_run_print_cur_update(void *data, void *context)
{
	struct scheduled_expr *sexpr;
	FILE *fp;

	sexpr = (struct scheduled_expr *)data;
	fp = (FILE *)context;

	if ((sexpr->db_type == delay_buf_type_none)
	    || (sexpr->db_type == delay_buf_type_single))
		return;

	fprintf(fp, "\t\tplugin->buf_%lu_cur++;\n", sexpr->index);

	if (sexpr->db_type == delay_buf_type_fixed)
		fprintf(fp, "\t\tplugin->buf_%lu_cur %%= %lu;\n",
			sexpr->index, (size_t)sexpr->delay_min + 1);
	else
		fprintf(fp, "\t\tplugin->buf_%lu_cur %%= "
			"plugin->buf_%lu_len;\n", sexpr->index, sexpr->index);
}

static void
c_cleanup_print_free(void *data, void *context)
{
	struct scheduled_expr *sexpr;
	FILE *fp;

	sexpr = (struct scheduled_expr *)data;
	fp = (FILE *)context;

	if (sexpr->db_type != delay_buf_type_var)
		return;

	fprintf(fp, "\tfree(plugin->buf_%lu);\n", sexpr->index);
}

static void
req_interm(void *data, void *context)
{
	struct scheduled_expr *sexpr;
	char *ret;

	sexpr = (struct scheduled_expr *)data;
	ret = (char *)context;

	if ((sexpr->component != NULL) && !*ret && (sexpr->is_df_refd))
		*ret = 1;
}

static char
system_has_interm(struct scheduled_system *ss)
{
	char ret;

	ret = 0;
	list_for_each(ss->exprs, req_interm, &ret);

	return ret;
}

static void
merge_include(void *data, void *context)
{
	struct ext_func *func;
	list_t includes;

	func = (struct ext_func *)data;
	includes = (list_t)context;

	if (func->include[0] != '\0')
		if (list_find(includes, (int (*)(void *, void *))strcmp,
			      func->include) == NULL)
			list_append(includes, func->include);
}

static void
add_include(void *data, void *context)
{
	struct scheduled_expr *sexpr;
	list_t includes;
	list_t l;

	sexpr = (struct scheduled_expr *)data;
	includes = (list_t)context;

	if ((sexpr->component == NULL) & !sexpr->is_output)
		return;

	l = expr_get_ext_funcs(sexpr->expr);
	list_for_each(l, merge_include, includes);
	list_free(l);
}

static void
c_plugin_print_include(void *data, void *context)
{
	char *inc;
	FILE *fp;

	inc = (char *)data;
	fp = (FILE *)context;

	fprintf(fp, "#include \"%s\"\n", inc);
}

static void
c_compile(void *data, void *context)
{
	struct scheduled_system *ss;
	FILE *fp;
	struct c_instantiate_print_err_context e_ctx;
	char c;
	list_t includes;

	ss = (struct scheduled_system *)data;

	includes = list_new();
	if (ss->has_var_delay || ss->has_fixed_delay)
		list_append(includes, "math.h");
	list_for_each(ss->schedule, add_include, includes);

	fp = xfopenw(ss->system->id, ".c");

	fprintf(fp, "#include <stddef.h>\n");
	fprintf(fp, "#include <stdlib.h>\n");
	fprintf(fp, "#include <string.h>\n");
	if (!list_is_empty(includes))
	  {
		fprintf(fp, "\n");
		list_for_each(includes, c_plugin_print_include, fp);
		
	  }
	fprintf(fp, "\n");
	fprintf(fp, "#include <lv2.h>\n");
	fprintf(fp, "\n");
	fprintf(fp, "#include \"%s.h\"\n", ss->system->id);

	fprintf(fp, "\n");
	fprintf(fp, "struct instance\n");
	fprintf(fp, "  {\n");
	if (ss->uses_sample_rate)
	  {
		fprintf(fp, "\t/* Sample rate */\n");
		fprintf(fp, "\tdouble\t sample_rate;\n");
		fprintf(fp, "\n");
	  }
	fprintf(fp, "\t/* Port connections */\n");
	list_for_each_rev(ss->exprs, c_instance_print_io_port, fp);
	fprintf(fp, "\n");
	fprintf(fp, "\t/* Internal buffers */\n");
	list_for_each(ss->exprs, c_instance_print_buf, fp);
	fprintf(fp, "\n");
	fprintf(fp, "\t/* Intermediate values */\n");
	list_for_each(ss->exprs, c_instance_print_interm, fp);
	fprintf(fp, "  };\n");

	if (ss->has_fixed_delay || ss->has_var_delay)
	  {
		fprintf(fp, "\n");
		fprintf(fp, "static unsigned long\n");
		fprintf(fp, "pmf_delay(float n, unsigned long cur, "
			"unsigned long max)\n");
		fprintf(fp, "{\n");
		fprintf(fp, "\tlong u;\n");
		fprintf(fp, "\n");
		fprintf(fp, "\tu = roundf(n);\n");
		fprintf(fp, "\n");
		fprintf(fp, "\tif (u < 1)\n");
		fprintf(fp, "\t\tu = 1;\n");
		fprintf(fp, "\tif (u > max)\n");
		fprintf(fp, "\t\tu = max;\n");
		fprintf(fp, "\n");
		fprintf(fp, "\tu = ((long)cur - u) %% ((long)max + 1);\n");
		fprintf(fp, "\tif (u < 0)\n");
		fprintf(fp, "\t\tu += max + 1;\n");
		fprintf(fp, "\n");
		fprintf(fp, "\treturn u;\n");
		fprintf(fp, "}\n");
	  }

	fprintf(fp, "\n");
	fprintf(fp, "LV2_Handle\n");
	fprintf(fp, "instantiate_%s(const LV2_Descriptor *descriptor,\n",
		ss->system->id);
	fprintf(fp, "\tdouble sample_rate, const char *bundle_path,\n");
	fprintf(fp, "\tconst LV2_Feature * const *features)\n");
	fprintf(fp, "{\n");
	fprintf(fp, "\tstruct instance *instance;\n");
	if (ss->has_var_delay)
		fprintf(fp, "\tsize_t value;\n");
	fprintf(fp, "\n");
	fprintf(fp, "\tinstance = "
		"(struct instance *)malloc(sizeof(struct instance));\n");
	fprintf(fp, "\tif (instance == NULL)\n");
	fprintf(fp, "\t\treturn NULL;\n");
	if (ss->has_var_delay)
		list_for_each(ss->exprs, c_instantiate_print_alloc ,fp);
	if (ss->uses_sample_rate)
	  {
		fprintf(fp, "\n");
		fprintf(fp, "\tinstance->sample_rate = sample_rate;\n");
	  }
	fprintf(fp, "\n");
	fprintf(fp, "\treturn instance;\n");
	if (ss->has_var_delay)
	  {
		fprintf(fp, "\n");
		e_ctx.fp = fp;
		e_ctx.first = 1;
		list_for_each(ss->exprs, c_instantiate_print_err, &e_ctx);
		fprintf(fp, "\tfree(instance);\n");
		fprintf(fp, "\n");
		fprintf(fp, "\treturn NULL;\n");
	  }
	fprintf(fp, "}\n");

	fprintf(fp, "\n");
	fprintf(fp, "void\n");
	fprintf(fp, "connect_port_%s(LV2_Handle instance, uint32_t port,\n",
		ss->system->id);
	fprintf(fp, "\tvoid *data_location)\n");
	fprintf(fp, "{\n");
	fprintf(fp, "\tstruct instance *plugin;\n");
	fprintf(fp, "\n");
	fprintf(fp, "\tplugin = (struct instance *)instance;\n");
	fprintf(fp, "\n");
	fprintf(fp, "\tswitch (port)\n");
	fprintf(fp, "\t  {\n");
	list_for_each_rev(ss->exprs, c_connect_port_print, fp);
	fprintf(fp, "\t  }\n");
	fprintf(fp, "}\n");

	fprintf(fp, "\n");
	fprintf(fp, "void\n");
	fprintf(fp, "activate_%s(LV2_Handle instance)\n", ss->system->id);
	fprintf(fp, "{\n");
	c = system_has_interm(ss);
	if (ss->has_single_delay || ss->has_fixed_delay || ss->has_var_delay
	    || c)
	  {
		fprintf(fp, "\tstruct instance *plugin;\n");
		fprintf(fp, "\n");
		fprintf(fp, "\tplugin = (struct instance *)instance;\n");
	  }
	if (ss->has_single_delay || ss->has_fixed_delay || ss->has_var_delay)
		list_for_each(ss->exprs, c_activate_print_buf, fp);
	if (c)
	  {
		fprintf(fp, "\n");
		list_for_each(ss->exprs, c_activate_print_interm, fp);
	  }
	fprintf(fp, "}\n");

	fprintf(fp, "\n");
	fprintf(fp, "void\n");
	fprintf(fp, "run_%s(LV2_Handle instance, uint32_t sample_count)\n",
		ss->system->id);
	fprintf(fp, "{\n");
	fprintf(fp, "\tstruct instance *plugin;\n");
	fprintf(fp, "\tuint32_t i;\n");
	fprintf(fp, "\n");
	fprintf(fp, "\tplugin = (struct instance *)instance;\n");
	fprintf(fp, "\n");
	fprintf(fp, "\tfor (i = 0; i < sample_count; i++)\n");
	fprintf(fp, "\t  {\n");
	list_for_each(ss->schedule, c_run_print_sexpr, fp);
	if (ss->has_single_delay || ss->has_fixed_delay || ss->has_var_delay)
	  {
		fprintf(fp, "\n");
		fprintf(fp, "\t\t/* Update buffers */\n");
		list_for_each_rev(ss->schedule, c_run_print_buf_update, fp);
	  }
	if (ss->has_fixed_delay || ss->has_var_delay)
	  {
		fprintf(fp, "\n");
		fprintf(fp, "\t\t/* Update current buffer positions */\n");
		list_for_each_rev(ss->schedule, c_run_print_cur_update, fp);
	  }
	fprintf(fp, "\t  }\n");
	fprintf(fp, "}\n");
	fflush(fp);

	fprintf(fp, "\n");
	fprintf(fp, "void\n");
	fprintf(fp, "deactivate_%s(LV2_Handle instance)\n", ss->system->id);
	fprintf(fp, "{\n");
	fprintf(fp, "}\n");
	fflush(fp);

	fprintf(fp, "\n");
	fprintf(fp, "void\n");
	fprintf(fp, "cleanup_%s(LV2_Handle instance)\n", ss->system->id);
	fprintf(fp, "{\n");
	if (ss->has_var_delay)
	  {
		fprintf(fp, "\tstruct instance *plugin;\n");
		fprintf(fp, "\n");
		fprintf(fp, "\tplugin = (struct instance *)instance;\n");
		fprintf(fp, "\n");
		list_for_each_rev(ss->exprs, c_cleanup_print_free, fp);
		fprintf(fp, "\n");
	  }
	fprintf(fp, "\tfree(instance);\n");
	fprintf(fp, "}\n");

	fclose(fp);
}

static void
mk_print_objs(void *data, void *context)
{
	struct scheduled_system *ss;
	FILE *fp;

	ss = (struct scheduled_system *)data;
	fp = (FILE *)context;

	fprintf(fp, " %s.o", ss->system->id);
}

static void
check_need_libm(void *data, void *context)
{
	struct scheduled_system *ss;
	list_t libs;

	ss = (struct scheduled_system *)data;
	libs = (list_t)context;

	if (ss->has_var_delay || ss->has_fixed_delay)
		if (list_find(libs, (int (*)(void *, void *))strcmp, "m")
		    == NULL)
			list_append(libs, "m");
}

static void
mk_print_libs(void *data, void *context)
{
	char *lib;
	FILE *fp;

	lib = (char *)data;
	fp = (FILE *)context;

	fprintf(fp, " -l%s", lib);
}

static void
merge_libs(void *data, void *context)
{
	struct ext_func *func;
	list_t libs;

	func = (struct ext_func *)data;
	libs = (list_t)context;

	if (func->lib[0] != '\0')
		if (list_find(libs, (int (*)(void *, void *))strcmp, func->lib)
		    == NULL)
			list_append(libs, func->lib);
}

static void
sexpr_add_libs(void *data, void *context)
{
	struct scheduled_expr *sexpr;
	list_t funcs;

	sexpr = (struct scheduled_expr *)data;

	if ((sexpr->component == NULL) && !sexpr->is_output)
		return;

	funcs = expr_get_ext_funcs(sexpr->expr);
	list_for_each(funcs, merge_libs, context);
	list_free(funcs);
}

static void
add_libs(void *data, void *context)
{
	struct scheduled_system *ss;

	ss = (struct scheduled_system *)data;

	list_for_each(ss->schedule, sexpr_add_libs, context);
}

void
compile(list_t systems)
{
	FILE *fp;
	list_t libs;

	if (compile_gen_manifest)
	  {
		fp = xfopenw("manifest.ttl", NULL);

		fprintf(fp, "@prefix lv2:  <http://lv2plug.in/ns/lv2core#> "
			".\n");
		fprintf(fp, "@prefix rdfs: "
			"<http://www.w3.org/2000/01/rdf-schema#> .\n");
		list_for_each(systems, manifest_print_plugin, fp);

		fclose(fp);
	  }

	if (compile_gen_plugin_ttl)
		list_for_each(systems, ttl_compile, NULL);

	if (compile_gen_extra_ttl)
		list_for_each(systems, ttl_extra_compile, NULL);

	if (compile_gen_descriptor)
	  {
		fp = xfopenw("lv2_descriptor.c", NULL);

		fprintf(fp, "#include <stddef.h>\n");
		fprintf(fp, "\n");
		fprintf(fp, "#include <lv2.h>\n");
		fprintf(fp, "\n");
		list_for_each(systems, c_print_include, fp);

		list_for_each(systems, c_print_descriptor, fp);

		fprintf(fp, "\n");
		fprintf(fp, "static const LV2_Descriptor * descriptors[] =\n");
		fprintf(fp, "  {\n");
		list_for_each(systems, c_print_descriptor_name, fp);
		fprintf(fp, "  };\n");

		fprintf(fp, "\n");
		fprintf(fp, "LV2_SYMBOL_EXPORT\n");
		fprintf(fp, "const LV2_Descriptor *lv2_descriptor(uint32_t "
			"index)\n");
		fprintf(fp, "{\n");
		fprintf(fp, "\tif (index >= (sizeof(descriptors) / "
			"sizeof(const LV2_Descriptor *)))\n");
		fprintf(fp, "\t\treturn NULL;\n");
		fprintf(fp, "\n");
		fprintf(fp, "\treturn descriptors[index];\n");
		fprintf(fp, "}\n");

		fclose(fp);
	  }

	if (compile_gen_code)
	  {
		list_for_each(systems, h_compile, NULL);
		list_for_each(systems, c_compile, NULL);
	  }

	if (compile_gen_makefile)
	  {
		libs = list_new();
		list_for_each(systems, check_need_libm, libs);
		list_for_each(systems, add_libs, libs);

		fp = xfopenw("Makefile", NULL);

		fprintf(fp, "CFLAGS = -fPIC\n");
		fprintf(fp, "LIBS =");
		list_for_each(libs, mk_print_libs, fp);
		fprintf(fp, "\n");
		fprintf(fp, "SO_EXT = .so\n");
		list_free(libs);

		fprintf(fp, "\n");
		fprintf(fp, "OBJS = lv2_descriptor.o");
		list_for_each(systems, mk_print_objs, fp);
		fprintf(fp, "\n");

		fprintf(fp, "\n");
		fprintf(fp, "all: plugin.so\n");

		fprintf(fp, "\n");
		fprintf(fp, "plugin$(SO_EXT): $(OBJS)\n");
		fprintf(fp, "\t$(LD) $(OBJS) -o plugin$(SO_EXT) -shared "
			"$(LIBS)");
		fprintf(fp, "\n");

		fprintf(fp, "\n");
		fprintf(fp, "clean:\n");
		fprintf(fp, "\trm -f *.o plugin$(SO_EXT)\n");

		fclose(fp);
	  }
}
