/*
 * inst_hist.c - instruction-based histogram sampling for all PMUs
 *
 * Copyright (c) 2005-2006 Hewlett-Packard Development Company, L.P.
 * Contributed by Stephane Eranian <eranian@hpl.hp.com>
 *
 * This file is part of pfmon, a sample tool to measure performance 
 * of applications on Linux.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307 USA
 */
#include "pfmon.h"
#include <perfmon/perfmon_dfl_smpl.h>

#define	SMPL_MOD_NAME "inst-hist"

typedef struct {
	unsigned long	show_top_num;
	int		per_function;
} inst_hist_options_t;

typedef struct {
		pfmon_hash_key_t addr;
		uint64_t	 count;
} hash_data_t;

typedef struct {
	hash_data_t 	**tab;
	unsigned long 	pos;
	uint64_t	total_count;
	uint64_t	max_count;
} hash_sort_arg_t;

static inst_hist_options_t inst_hist_options;

/*
 * forward declaration
 */
pfmon_smpl_module_t inst_hist_i386_p6_smpl_module;

static int
inst_hist_process_samples(pfmon_sdesc_t *sdesc)
{
	pfm_dfl_smpl_hdr_t *hdr;
	pfm_dfl_smpl_entry_t *ent;
	pfmon_smpl_desc_t *csmpl = &sdesc->csmpl;
	unsigned long entry;
	size_t count, i;
	void *hash_desc, *data;
	hash_data_t *hash_entry;
	pfmon_hash_key_t key;
	int ret;

	hdr        = csmpl->smpl_hdr;
	hash_desc  = csmpl->data;
	ent        = (pfm_dfl_smpl_entry_t *)(hdr+1);
	entry      = options.opt_aggr ? *csmpl->aggr_count : csmpl->entry_count;
	count      = hdr->hdr_count;

	DPRINT(("hdr_count=%"PRIu64" hdr=%p csmpl=%p\n", count, hdr, csmpl));

	for(i=0; i < count; i++) {

		DPRINT(("entry %"PRIu64" PID:%d CPU:%d STAMP:0x%"PRIx64" IIP: %llx\n",
			entry,
			ent->pid,
			ent->cpu,
			ent->tstamp,
			(unsigned long long)ent->ip));

		key = (pfmon_hash_key_t)ent->ip;

		/*
		 * in aggregation mode sample processing is serialized,
		 * therefore we are safe to use a single hash_table here
		 */
		ret = pfmon_hash_find(hash_desc, key, &data);
		if (ret == -1) {
			pfmon_hash_add(hash_desc, key, &data);
			hash_entry = (hash_data_t *)data;
			hash_entry->count = 1;
			hash_entry->addr  = ent->ip;
		} else {
			hash_entry = (hash_data_t *)data;
			hash_entry->count++;
		}

		/*
		 * we only allow entries with no "body"
		 */
		ent++;
		entry++;
	}
	
	/*
	 * when aggregation is used, for are guaranteed sequential access to
	 * this routine by higher level lock
	 */
	if (options.opt_aggr) {
		*csmpl->aggr_count += count;
	} else {
		csmpl->entry_count += count;
	}
	csmpl->last_count = count;
	csmpl->last_ovfl = hdr->hdr_overflows;

	return 0;
}

#define IDX PFMON_OPT_SMPL_BASE
static struct option inst_hist_cmd_options[]={
	{ "smpl-show-top", 1, 0, IDX},
	{ "smpl-show-function", 0, &inst_hist_options.per_function, 1},
	{ NULL, 0, 0, 0}
};
#undef IDX

static void
inst_hist_show_options(void)
{
	printf("\t--smpl-show-top=n\t\tshow only the top n entries in the histogram (default: all entries)\n");
	printf("\t--smpl-show-function\t\tshow per-function histograms (default per address)\n");
}

/*
 * 0  means we understood the option
 * -1 unknown option
 */
static int
inst_hist_parse_options(int code, char *optarg)
{
	char *endptr = NULL;

	switch(code) {
		case  PFMON_OPT_SMPL_BASE:
			if (inst_hist_options.show_top_num) 
				fatal_error("smpl-show-top already defined\n");

			inst_hist_options.show_top_num = strtoul(optarg, &endptr, 0);

			if (*endptr != '\0') 
				fatal_error("invalid value for show-top-num : %s\n", optarg);
			break;

		default:
			return -1;
	}
	return 0;

}


/*
 * module initialization
 */
static int
inst_hist_initialize_module(void)
{
	return pfmon_register_options(inst_hist_cmd_options, sizeof(inst_hist_cmd_options));
}

static int
inst_hist_print_header(pfmon_sdesc_t *sdesc)
{
	FILE *fp = sdesc->csmpl.smpl_fp;

	fprintf(fp, "# description of columns:\n"
		    "#\tcolumn  1: number of samples for this address\n"
	 	    "#\tcolumn  2: relative percentage for this address\n"
		    "#\tcolumn  3: cumulative percentage up to this address\n"
		    "#\tcolumn  4: symbol name or address\n");
	return 0;
}

static int
inst_hist_validate_events(pfmon_event_set_t *set)
{
	unsigned int i;

	/*
	 * must be sampling with one event only (no extra PMDS in payload)
	 */
	if (set->inp.pfp_event_count > 1) {
		warning("the sampling module works with 1 event at a time only\n");
		return -1;
	}
	/*
	 * in system-wide mode, we currently only support kernel level ONLY monitoring
	 * because we do not manage the process pid inside the hash table. This could
	 * cause confusion and lead to interpretation mistakes for the final histogram.
	 */
	if (options.opt_syst_wide) {
		for(i=0; i < set->inp.pfp_event_count; i++) {
			if (set->inp.pfp_events[i].plm != PFM_PLM0) {
				warning("in system-wide mode, the sampling module only works when capturing kernel level ONLY events\n");
				return -1;
			}
		}
	}
	return 0;
}

static int
inst_hist_initialize_session(pfmon_smpl_desc_t *csmpl)
{
	void *hash_desc;
	pfmon_hash_param_t param;

	param.hash_log_size = 12;
	param.max_entries   = ~0;
	param.entry_size    = sizeof(hash_data_t);
#ifdef __ia64__
	param.shifter	    = 4;
#else
	param.shifter	    = 0;
#endif
	param.flags	    = PFMON_HASH_ACCESS_REORDER;

	pfmon_hash_alloc(&param, &hash_desc);

	csmpl->data = hash_desc;
	DPRINT(("initialized session for csmpl=%p data=%p\n", csmpl, csmpl->data));
	return 0;
}

static void
inst_hist_print_data(void *arg, void *data)
{
	hash_data_t *p = (hash_data_t *)data;
	hash_sort_arg_t *sort_arg = (hash_sort_arg_t *)arg;
	hash_data_t **tab = sort_arg->tab;
	unsigned long pos = sort_arg->pos;
	uint64_t count;

	count = p->count;
	tab[pos] = p;

	sort_arg->pos = ++pos;
	sort_arg->total_count += count;

	if (count > sort_arg->max_count) sort_arg->max_count = count;
}

static int
hash_data_sort_cmp(const void *a, const void *b)
{
	hash_data_t **e1 = (hash_data_t **)a;
	hash_data_t **e2 = (hash_data_t **)b;

	return (*e1)->count > (*e2)->count ? 0 : 1;
}

static int
hash_data_sort_byaddr(const void *a, const void *b)
{
	hash_data_t **e1 = (hash_data_t **)a;
	hash_data_t **e2 = (hash_data_t **)b;

	return (*e1)->addr > (*e2)->addr ? 1 : 0;
}

static void
inst_hist_collapse_func(hash_data_t **tab, unsigned long num_entries)
{
	unsigned long i;
	unsigned long start, end;
	hash_data_t *p, *psrc = NULL;
	int ret = -1;
	pfmon_syms_list_t *sym_list;

	sym_list = &options.primary_syms;

	for(i=0; i < num_entries; i++) {
		p = tab[i];
		if (ret == 0 && p->addr >= start && p->addr < end) {
			psrc->count += p->count;
			p->count = 0;
			continue;
		}
		ret = find_sym_byaddr(p->addr, sym_list, PFMON_TEXT_SYMBOL, NULL, NULL, &start, &end);
		if (ret == -1) continue;
		/* resync base address */
		p->addr = start;
		psrc = p;
	}
}

static int
inst_hist_show_results(pfmon_smpl_desc_t *smpl)
{
	uint64_t total_count, cum_count, count, top_num;
	void *hash_desc = smpl->data;
	FILE *fp = smpl->smpl_fp;
	void *sym_hash;
	double d_cum;
	hash_data_t **tab;
	unsigned long addr;
	unsigned long i, num_entries;
	hash_sort_arg_t arg;
	size_t len;
	int need_resolve;
	char buf[32];

	pfmon_hash_num_entries(hash_desc, &num_entries);

	tab = (hash_data_t **)malloc(sizeof(hash_data_t *)*num_entries);
	if (tab == NULL) {
		warning("cannot allocate memory to print %lu samples\n", num_entries);
		return -1;
	}
	sym_hash  = smpl->sym_hash;

	arg.tab = tab;
	arg.pos = 0;
	arg.total_count = 0;
	arg.max_count   = 0;

	pfmon_hash_iterate(smpl->data, inst_hist_print_data, &arg);

	total_count = arg.total_count;
	cum_count   = 0;

	sprintf(buf, "%"PRIu64, arg.max_count);
	len = strlen(buf);
	/* adjust for column heading */
	if (len < 6) len = 6;

	if (inst_hist_options.per_function) {
		qsort(tab, num_entries, sizeof(hash_data_t *), hash_data_sort_byaddr);
		inst_hist_collapse_func(tab, num_entries);
	}

	qsort(tab, num_entries, sizeof(hash_data_t *), hash_data_sort_cmp);

	need_resolve = options.opt_addr2sym;

	top_num = inst_hist_options.show_top_num;
	if (top_num && top_num < num_entries) num_entries = top_num;

	fprintf(fp, "# %*s %7s %7s code address\n", (int)len, "counts", "%self", "%cum");
	len+=2;
	for(i=0; i < num_entries; i++) {

		addr       = tab[i]->addr;
		count      = tab[i]->count;
		if (count == 0) continue; /* can happen in per-function mode */
		cum_count += count;
		d_cum	   = (double)count*100.0     / (double)total_count;

		fprintf(fp, "%*"PRIu64" %6.2f%% %6.2f%% ",
			    (int)len, 
			    count, 
			    d_cum,
			    (double)cum_count*100.0 / (double)total_count);
			    

		if (need_resolve)
			pfmon_print_address(fp, sym_hash, &options.primary_syms, PFMON_TEXT_SYMBOL, addr);
		else 
			fprintf(fp, "%p", (void *)addr);
		fputc('\n', fp);
	}

	free(tab);

	return 0;
}

static int
inst_hist_terminate_session(pfmon_sdesc_t *sdesc)
{
	if ( options.opt_addr2sym 
	  && pfmon_syms_hash_alloc(PFMON_DFL_SYM_HASH_SIZE, PFMON_DFL_SYM_ENTRIES, &sdesc->csmpl.sym_hash)) {
		warning("cannot allocate symbol hash, disabling symbol resolution\n");
		sdesc->csmpl.sym_hash = NULL;
	}
	inst_hist_show_results(&sdesc->csmpl);

	pfmon_hash_free(sdesc->csmpl.data);
	sdesc->csmpl.data = NULL;
	return 0;
}

pfmon_smpl_module_t inst_hist_smpl_module ={
	.name		    = "inst-hist",
	.pmu_mask	    = PFMON_ANY_PMU_MASK,
	.description	    = "IP-based histogram",
	.process_samples    = inst_hist_process_samples,
	.show_options       = inst_hist_show_options,
	.parse_options      = inst_hist_parse_options,
	.initialize_module  = inst_hist_initialize_module,
	.initialize_session = inst_hist_initialize_session,
	.terminate_session  = inst_hist_terminate_session,
	.print_header       = inst_hist_print_header,
	.validate_events    = inst_hist_validate_events,
	.flags		    = PFMON_SMPL_MOD_FL_DEF_SYM,
	.uuid		    = PFM_DFL_SMPL_UUID,
};
