/*
 * detailed_amd64_smpl.c - detailed sampling module for the AMD64 PMU family
 *
 * Copyright (C) 2005-2006 Hewlett-Packard Co
 * Contributed by Stephane Eranian <eranian@hpl.hp.com>
 *
 * This file is part of pfmon, a sample tool to measure performance 
 * of applications on Linux.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307 USA
 */
#include "pfmon.h"

#include "pfmon_amd64.h"
#include <perfmon/pfmlib_amd64.h>
#include <perfmon/perfmon_dfl_smpl.h>

#define	SMPL_MOD_NAME "detailed-amd64"

/*
 * forward declaration
 */
pfmon_smpl_module_t detailed_amd64_smpl_module;

static int
print_amd64_reg(pfmon_event_set_t *set, pfmon_smpl_desc_t *csmpl, int rnum, uint64_t rval)
{
	pfm_amd64_ctr_reg_t reg;
	FILE *fp = csmpl->smpl_fp;
	int ret = 0;

	reg.val = rval;

	switch(rnum) {
		case 0:
		case 1:
		case 2:
		case 3:
			ret = fprintf(fp, "\tPMD%-3d:0x%016llx\n",
				      rnum,
				      reg.val);
			break;
		default:
			warning("unknown PMD%d", rnum);
			ret = -1;
	}
	return ret;
}

static int
detailed_amd64_process_samples(pfmon_sdesc_t *sdesc)
{
	pfm_dfl_smpl_hdr_t *hdr;
	pfm_dfl_smpl_entry_t *ent;
	pfmon_smpl_desc_t *csmpl;
	pfmon_event_set_t *active_set;
	FILE *fp;
	uint64_t *msk;
	uint64_t entry, i, count;
	void *hash_desc;
	pfm_amd64_ctr_reg_t *reg;
	unsigned short ovfl_pmd;
	unsigned int j, n;
	int ret;

	csmpl      = &sdesc->csmpl;
	hdr        = csmpl->smpl_hdr;
	fp         = csmpl->smpl_fp;
	hash_desc  = csmpl->sym_hash;
	ent        = (pfm_dfl_smpl_entry_t *)(hdr+1);
	entry      = options.opt_aggr ? *csmpl->aggr_count : csmpl->entry_count;
	count      = hdr->hdr_count;
	active_set = sdesc->sets; /* only one set when sampling */

	DPRINT(("hdr_count=%lu hdr=%p active_set=%u\n", count, hdr, active_set->id));

	for(i=0; i < count; i++) {

		ovfl_pmd = ent->ovfl_pmd;
		ret = fprintf(fp, "entry %"PRIu64" PID:%d TID:%d CPU:%d STAMP:0x%"PRIx64" OVFL:%d LAST_VAL:%"PRIu64" SET:%u IIP:",
			entry,
			ent->tgid,
			ent->pid,
			ent->cpu,
			ent->tstamp,
			ovfl_pmd, 
			-ent->last_reset_val,
			ent->set);

		pfmon_print_address(fp, hash_desc, &options.primary_syms, PFMON_TEXT_SYMBOL, ent->ip);
		fputc('\n', fp);

		reg = (pfm_amd64_ctr_reg_t*)(ent+1);

		msk = active_set->rev_smpl_pmds[ovfl_pmd].smpl_pmds;
		n   = active_set->rev_smpl_pmds[ovfl_pmd].num_smpl_pmds;

		for(j=0; n; j++) {	
			if (pfmon_bv_isset(msk, j)) {
				ret = print_amd64_reg(active_set, csmpl, j, reg->val);
				reg++;
				n--;
			}
		}
		/* fprintf() error detection */
		if (ret == -1) goto error;

		/*
		 * entries are contiguously stored
		 */
		ent  = (pfm_dfl_smpl_entry_t *)reg;	
		entry++;
	}
	
	/*
	 * when aggregation is used, for are guaranteed sequential access to
	 * this routine by higher level lock
	 */
	if (options.opt_aggr) {
		*csmpl->aggr_count += count;
	} else {
		csmpl->entry_count += count;
	}
	csmpl->last_count = count;
	csmpl->last_ovfl = hdr->hdr_overflows;

	return 0;
error:
	warning("cannot write to sampling file: %s\n", strerror(errno));
	/* not reached */
	return -1;
}

static int
detailed_amd64_print_header(pfmon_sdesc_t *sdesc)
{
	return 0;
}

pfmon_smpl_module_t detailed_amd64_smpl_module={
	.name		    = SMPL_MOD_NAME,
	.pmu_mask	    = PFMON_PMU_MASK(PFMLIB_AMD64_PMU),
	.description	    = "detailed AMD64",
	.process_samples    = detailed_amd64_process_samples,
	.print_header       = detailed_amd64_print_header,
	.uuid		    = PFM_DFL_SMPL_UUID
};
