#!/usr/bin/python
# -*- coding: utf-8 -*-

# Phatch - Photo Batch Processor
# Copyright (C) 2009 Juho Vepsäläinen, www.stani.be
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see http://www.gnu.org/licenses/
#
# Phatch recommends SPE (http://pythonide.stani.be) for editing python files.
#
# Follows PEP8
from optparse import OptionParser
from preview import Preview, PreviewGenerator
from test_actions import system_path


class BlenderPreviewGenerator(PreviewGenerator):
    action = 'blender'

    def __init__(self, preview_object):
        if preview_object:
            self.preview_objects = (preview_object, )
        else:
            # FIXME: should introspect this properly from existing objects!
            self.preview_objects = ('Book', 'Box', 'Can', 'Cd', 'Lcd',
                'Sphere')

    def generate(self):
        for preview_object in self.preview_objects:
            options = self.options
            options['Object'] = [preview_object]
            options['Render Width'] = ['128px']
            options['Render Height'] = ['128px']

            output_path = self.get_output_path(preview_object)

            blender_preview = self.previewer(self.action, options, output_path)
            blender_preview.execute()


class ObjectPreview(Preview):

    def get_preview_filename(self, name, choice):
        return choice['Object'].lower()


class ObjectPreviewGenerator(BlenderPreviewGenerator):
    previewer = ObjectPreview
    option_name = 'object'

    def get_output_path(self, preview_object):
        return system_path(self.base_path + self.option_name)


class RotationPreview(Preview):

    def get_preview_filename(self, name, choice):
        hori_rot = choice['Horizontal Rotation']
        vert_rot = choice['Vertical Rotation']

        return 'hori_' + str(hori_rot) + '_vert_' + str(vert_rot)


class RotationPreviewGenerator(BlenderPreviewGenerator):
    previewer = RotationPreview
    option_name = 'rotation'
    options = {
        'Horizontal Rotation': [-60, -30, 0, 30, 60],
        'Vertical Rotation': [0, 30, 60],
    }

    def get_output_path(self, preview_object):
        return system_path(self.base_path + self.option_name + '/' +
                preview_object.lower())


class BlenderPreviewApp(OptionParser):

    def __init__(self):
        self.option_parser = OptionParser()

        self.option_parser.add_option('-b', '--object', default='',
            help='Object for which to render previews. If no object is '\
            'provided it generates previews for all objects automatically.')
        self.option_parser.add_option('-o', '--objectpreview',
            action='store_false', help="Don't generate object previews.")
        self.option_parser.add_option('-r', '--rotationpreview',
            action='store_false', help="Don't generate rotation previews.")

    def run(self):
        options, args = self.option_parser.parse_args()
        blender_ob = options.object

        if options.objectpreview is None:
            object_preview_generator = ObjectPreviewGenerator(blender_ob)
            object_preview_generator.generate()

        if options.rotationpreview is None:
            rotation_preview_generator = RotationPreviewGenerator(blender_ob)
            rotation_preview_generator.generate()


if __name__ == '__main__':
    blender_preview = BlenderPreviewApp()
    blender_preview.run()
