/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment mapping matrix example
 *
 * Copyright © 2006, 2007, 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Loïc Molinari <loic@fluendo.com>
 */

#include <pgm/pgm.h>

static PgmViewport *viewport = NULL;

/* Update fps text */
static gboolean
update_fps_text_cb (gpointer data)
{
  PgmDrawable *txt = PGM_DRAWABLE (data);
  gchar *fps_string;
  guint fps;

  pgm_viewport_get_frame_rate (viewport, &fps);

  fps_string = g_strdup_printf ("%d FPS", fps);
  pgm_text_set_markup (PGM_TEXT (txt), fps_string);
  g_free (fps_string);

  return TRUE;
}

/* Transform the mapping matrix of the image */
static gboolean
update_pass_cb (PgmViewport *viewport,
                gpointer data)
{
  PgmImage *image = PGM_IMAGE (data);
  static PgmMat4x4 *mapping = NULL;

  /* Get the current mapping matrix at the first call */
  if (G_UNLIKELY (!mapping))
    pgm_image_get_mapping_matrix (PGM_IMAGE (image), &mapping);

  /* Rotate the matrix about 2° around z axis at the center of the image */
  pgm_mat4x4_translate_from_scalars (mapping, 0.5f, 0.5f, 0.0f);
  pgm_mat4x4_rotate_x (mapping, (gfloat) G_PI / 180.0f);
  pgm_mat4x4_rotate_y (mapping, (gfloat) G_PI / 180.0f);
  pgm_mat4x4_rotate_z (mapping, (gfloat) G_PI / 180.0f);
  pgm_mat4x4_translate_from_scalars (mapping, -0.5f, -0.5f, 0.0f);

  /* Set the new mapping matrix */
  pgm_image_set_mapping_matrix (PGM_IMAGE (image), mapping);

  return TRUE;
}

/* key-press handler */
static void
keypress_event_cb (PgmViewport *viewport,
                   PgmEventKey *event,
                   gpointer data)
{
  static gboolean decorated = TRUE;

  if (event->type == PGM_KEY_PRESS)
    {
      switch (event->keyval)
        {
          /* Add/remove window decorations */
        case PGM_d:
          decorated = !decorated;
          pgm_viewport_set_decorated (viewport, decorated);
          break;

          /* Handle quit */
        case PGM_q:
        case PGM_Escape:
          pgm_main_quit ();
          break;

        default:
          break;
        }
    }
}

/* Entry point */
int
main (int argc,
      char *argv[])
{
  const gchar *img_path = "examples/pictures/meiko.jpg";
  PgmDrawable *img, *txt;
  PgmCanvas *canvas;

  /* Init */
  pgm_init (&argc, &argv);

  /* OpenGL viewport creation */
  pgm_viewport_factory_make ("opengl", &viewport);
  if (!viewport)
    {
      g_print ("Cannot create the 'opengl' viewport\n");
      return -1;
    }
  pgm_viewport_set_size (viewport, 600, 600);
  pgm_viewport_set_title (viewport, "Mapping matrix");

  /* Image definition */
  img = pgm_image_new_from_file (img_path, 0);
  pgm_drawable_set_size (img, 400, 400);
  pgm_drawable_set_position (img, 100, 100, 0);
  pgm_drawable_set_bg_color (img, 0, 0, 255, 0);
  pgm_image_set_wrapping (PGM_IMAGE (img), PGM_IMAGE_TRANSPARENT,
                          PGM_IMAGE_TRANSPARENT);
  pgm_drawable_show (img);

  /* Text definition */
  txt = pgm_text_new ("0 FPS");
  pgm_text_set_weight (PGM_TEXT (txt), PGM_TEXT_WEIGHT_BOLD);
  pgm_drawable_set_size (txt, 300, 20);
  pgm_drawable_set_position (txt, 10, 0, 0);
  pgm_drawable_set_bg_color (txt, 0, 0, 0, 0);
  pgm_drawable_show (txt);

  /* Canvas handling */
  canvas = pgm_canvas_new ();
  pgm_canvas_set_size (canvas, 600, 600);
  pgm_viewport_set_canvas (viewport, canvas);
  pgm_canvas_add_many (canvas, PGM_DRAWABLE_MIDDLE, img, txt, NULL);

  /* Main loop */
  g_signal_connect (G_OBJECT (viewport), "key-press-event",
                    G_CALLBACK (keypress_event_cb), NULL);
  g_signal_connect (G_OBJECT (viewport), "delete-event",
                    G_CALLBACK (pgm_main_quit), NULL);
  g_signal_connect (G_OBJECT (viewport), "update-pass",
                    G_CALLBACK (update_pass_cb), img);
  pgm_viewport_show (viewport);

  g_timeout_add (1000, update_fps_text_cb, txt);

  pgm_main ();

  /* Deinit */
  gst_object_unref (canvas);
  gst_object_unref (viewport);
  pgm_deinit ();

  return 0;
}
