/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment OpenGL plugin
 *
 * Copyright © 2006, 2007, 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Loïc Molinari <loic@fluendo.com>
 */

/*
 * The plugin duplicates all "generic" Pigment drawables in GL drawables. The
 * GL drawables are stored in layer (lists) inside the GL viewport object. The
 * generic layers are duplicated when the user calls pgm_viewport_set_canvas().
 * The duplicated GL drawables are deleted when the canvas is disposed, when
 * another canvas is bound or when NULL is bound. The GL Viewport is connected
 * to the Canvas 'added' and 'removed' signals so that it can update the GL
 * drawable layers to reflect the "generic" one.
 *
 * The link between the "generic" and the GL drawable is done through a
 * dedicated hash-table using the "generic" drawable address as the key and the
 * GL drawable address as the value. When a "generic" drawable property changes,
 * it emits the 'changed' signal to which the Gl Viewport is connected. The
 * handler stores the change as a task in an update queue which is flushed
 * during the automatic update done in the rendering thread in PgmContext.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include "pgmglviewport.h"
#include "pgmglimage.h"
#include "pgmgltext.h"

GST_DEBUG_CATEGORY_STATIC (pgm_gl_viewport_debug);
#define GST_CAT_DEFAULT pgm_gl_viewport_debug

/* Task types */
typedef enum {
  TASK_CHANGE,
  TASK_REORDER,
  TASK_ADD,
  TASK_REMOVE,
  TASK_REGENERATE,
  LAST_TASK
} TaskType;

/* Change task */
typedef struct {
  TaskType             type;
  PgmGlDrawable       *gldrawable;
  PgmDrawableProperty  property;
} TaskChange;

/* Reorder task */
typedef struct {
  TaskType          type;
  PgmGlDrawable    *drawable;
  PgmDrawableLayer  layer;
  guint             order;
} TaskReorder;

/* Add task */
typedef struct {
  TaskType          type;
  PgmDrawable      *drawable;
  PgmDrawableLayer  layer;
  guint             order;
} TaskAdd;

/* Remove task */
typedef struct {
  TaskType          type;
  PgmDrawable      *drawable;
  PgmDrawableLayer  layer;
} TaskRemove;

/* Any task */
typedef struct {
  TaskType type;
} TaskAny;

/* Task */
typedef union {
  TaskType    type;
  TaskAny     any;
  TaskChange  change;
  TaskReorder reorder;
  TaskAdd     add;
  TaskRemove  remove;
} Task;

/* Task function type definition */
typedef void (*TaskFunc) (PgmGlViewport *glviewport, Task *task);

/* Task function prototypes */
static void do_task_change     (PgmGlViewport *glviewport, Task *task);
static void do_task_reorder    (PgmGlViewport *glviewport, Task *task);
static void do_task_add        (PgmGlViewport *glviewport, Task *task);
static void do_task_remove     (PgmGlViewport *glviewport, Task *task);
static void do_task_regenerate (PgmGlViewport *glviewport, Task *task);

/* Callback prototypes */
static void changed_cb (PgmDrawable *drawable,
                        PgmDrawableProperty property,
                        gpointer data);

/* Task function array */
static TaskFunc task_func[LAST_TASK] = {
  do_task_change,
  do_task_reorder,
  do_task_add,
  do_task_remove,
  do_task_regenerate
};

static PgmViewportClass *parent_class = NULL;

/* Private functions */

/* Create a change task */
static Task*
task_change_new (PgmGlDrawable *gldrawable,
                 PgmDrawableProperty property)
{
  Task *task;

  task = g_slice_new (Task);
  task->change.type = TASK_CHANGE;
  task->change.gldrawable = gst_object_ref (gldrawable);
  task->change.property = property;

  return task;
}

/* Free a change task */
static void
task_change_free (Task *task)
{
  g_return_if_fail (task != NULL);

  gst_object_unref (task->change.gldrawable);

  g_slice_free (Task, task);
}

/* Create a reorder task */
static Task*
task_reorder_new (PgmDrawable *drawable,
                  PgmDrawableLayer layer,
                  guint order)
{
  Task *task;

  task = g_slice_new (Task);
  task->reorder.type = TASK_REORDER;
  task->reorder.drawable = gst_object_ref (drawable);
  task->reorder.layer = layer;
  task->reorder.order = order;

  return task;
}

/* Free a reorder task */
static void
task_reorder_free (Task *task)
{
  g_return_if_fail (task != NULL);

  gst_object_unref (task->reorder.drawable);

  g_slice_free (Task, task);
}

/* Create a add task */
static Task*
task_add_new (PgmDrawable *drawable,
              PgmDrawableLayer layer,
              guint order)
{
  Task *task;

  task = g_slice_new (Task);
  task->add.type = TASK_ADD;
  task->add.drawable = gst_object_ref (drawable);
  task->add.layer = layer;
  task->add.order = order;

  return task;
}

/* Free a add task */
static void
task_add_free (Task *task)
{
  g_return_if_fail (task != NULL);

  gst_object_unref (task->add.drawable);

  g_slice_free (Task, task);
}

/* Create a remove task */
static Task*
task_remove_new (PgmDrawable *drawable,
                 PgmDrawableLayer layer)
{
  Task *task;

  task = g_slice_new (Task);
  task->remove.type = TASK_REMOVE;
  task->remove.drawable = gst_object_ref (drawable);
  task->remove.layer = layer;

  return task;
}

/* Free a remove task */
static void
task_remove_free (Task *task)
{
  g_return_if_fail (task != NULL);

  gst_object_unref (task->remove.drawable);

  g_slice_free (Task, task);
}

/* Create a new generic task */
static Task*
task_any_new (TaskType type)
{
  Task *task;

  task = g_slice_new (Task);
  task->any.type = type;

  return task;
}

/* Create a new generic task */
static void
task_any_free (Task *task)
{
  g_slice_free (Task, task);
}

/* Free a generic task guessing the type */
static void
task_free (Task *task)
{
  switch (task->type)
    {
    case TASK_CHANGE:
      task_change_free (task);
      break;
    case TASK_REORDER:
      task_reorder_free (task);
      break;
    case TASK_ADD:
      task_add_free (task);
      break;
    case TASK_REMOVE:
      task_remove_free (task);
      break;
    default:
      task_any_free (task);
      break;
    }
}

/* Create a GL drawable from a generic drawable */
static PgmGlDrawable*
gl_drawable_new (PgmGlViewport *glviewport,
                 PgmDrawable *drawable)
{
  PgmGlDrawable *gldrawable = NULL;

  /* Create the GlDrawable depending on the type */
  if (PGM_IS_IMAGE (drawable))
    {
      gldrawable = pgm_gl_image_new (drawable, glviewport);
      GST_DEBUG_OBJECT (glviewport, "created %s", GST_OBJECT_NAME (gldrawable));
    }
  else if (PGM_IS_TEXT (drawable))
    {
      gldrawable = pgm_gl_text_new (drawable, glviewport);
      GST_DEBUG_OBJECT (glviewport, "created %s", GST_OBJECT_NAME (gldrawable));
    }
  else
    {
      gldrawable = NULL;
      GST_WARNING_OBJECT (glviewport, "cannot create object from this type");
    }

  /* Make it easily retrievable */
  if (gldrawable)
    {
      GST_OBJECT_LOCK (glviewport);
      g_hash_table_insert (glviewport->drawable_hash, drawable, gldrawable);
      GST_OBJECT_UNLOCK (glviewport);
    }

  return gldrawable;
}

/* Free a GL drawable */
static void
gl_drawable_free (PgmGlViewport *glviewport,
                  PgmGlDrawable *gldrawable)
{
  GSList *walk, *tmp;
  Task *task;

  /* Disconnect from the 'changed' signal and remove it from the hash */
  GST_OBJECT_LOCK (glviewport);
  GST_OBJECT_LOCK (gldrawable->drawable);

  g_signal_handler_disconnect (gldrawable->drawable, gldrawable->change_handler);
  g_hash_table_remove (glviewport->drawable_hash, gldrawable->drawable);

  GST_OBJECT_UNLOCK (gldrawable->drawable);
  GST_OBJECT_UNLOCK (glviewport);

  /* Remove the gldrawable from the update queue */
  g_mutex_lock (glviewport->update_lock);
  walk = glviewport->update_queue;
  while (walk)
    {
      tmp = walk->next;
      task = walk->data;
      /* Remove it only if it's a task dealing with gldrawable */
      if (task->type == TASK_CHANGE)
        {
          if (task->change.gldrawable == gldrawable)
            {
              glviewport->update_queue =
                g_slist_delete_link (glviewport->update_queue, walk);
              task_change_free (task);
            }
        }
      walk = tmp;
    }
  g_mutex_unlock (glviewport->update_lock);

  /* Then delete it */
  GST_DEBUG_OBJECT (glviewport, "unreferencing %s", GST_OBJECT_NAME (gldrawable));
  gst_object_unref (gldrawable);
}

/* Complete a change task and free it */
static void
do_task_change (PgmGlViewport *glviewport,
                Task *task)
{
  PgmGlViewportClass *klass = PGM_GL_VIEWPORT_GET_CLASS (glviewport);

  klass->changed_func[task->change.property] (task->change.gldrawable);

  task_change_free (task);
}

/* Complete a reorder task and free it */
static void
do_task_reorder (PgmGlViewport *glviewport,
                 Task *task)
{
  PgmGlDrawable *gldrawable;

  GST_OBJECT_LOCK (glviewport);
  gldrawable = g_hash_table_lookup (glviewport->drawable_hash,
                                    task->reorder.drawable);
  GST_OBJECT_UNLOCK (glviewport);

  if (!gldrawable)
    goto removed;

  g_mutex_lock (glviewport->layer_lock);

  switch (task->reorder.layer)
    {
    case PGM_DRAWABLE_NEAR:
      glviewport->near_layer = g_list_remove (glviewport->near_layer,
                                              gldrawable);
      glviewport->near_layer = g_list_insert (glviewport->near_layer,
                                              gldrawable,
                                              task->reorder.order);
      break;

    case PGM_DRAWABLE_MIDDLE:
      glviewport->middle_layer = g_list_remove (glviewport->middle_layer,
                                                gldrawable);
      glviewport->middle_layer = g_list_insert (glviewport->middle_layer,
                                                gldrawable,
                                                task->reorder.order);
      break;

    case PGM_DRAWABLE_FAR:
      glviewport->far_layer = g_list_remove (glviewport->far_layer,
                                             gldrawable);
      glviewport->far_layer = g_list_insert (glviewport->far_layer,
                                             gldrawable,
                                             task->reorder.order);
      break;

    default:
      break;
    }

  g_mutex_unlock (glviewport->layer_lock);

 removed:
  task_reorder_free (task);
}

/* Complete a add task and free it */
static void
do_task_add (PgmGlViewport *glviewport,
             Task *task)
{
  PgmGlDrawable *gldrawable;

  gldrawable = gl_drawable_new (glviewport, task->add.drawable);

  g_mutex_lock (glviewport->layer_lock);
  switch (task->add.layer)
    {
    case PGM_DRAWABLE_NEAR:
      glviewport->near_layer = g_list_insert (glviewport->near_layer,
                                              gldrawable,
                                              task->add.order);
      break;

    case PGM_DRAWABLE_MIDDLE:
      glviewport->middle_layer = g_list_insert (glviewport->middle_layer,
                                                gldrawable,
                                                task->add.order);
      break;

    case PGM_DRAWABLE_FAR:
      glviewport->far_layer = g_list_insert (glviewport->far_layer,
                                             gldrawable,
                                             task->add.order);
      break;

    default:
      break;
    }
  g_mutex_unlock (glviewport->layer_lock);

  task_add_free (task);
}

/* Complete a remove task and free it */
static void
do_task_remove (PgmGlViewport *glviewport,
                Task *task)
{
  PgmGlDrawable *gldrawable;

  /* Get the GlDrawable */
  GST_OBJECT_LOCK (glviewport);
  gldrawable = g_hash_table_lookup (glviewport->drawable_hash,
                                    task->remove.drawable);
  GST_OBJECT_UNLOCK (glviewport);

  /* Remove it from the list */
  g_mutex_lock (glviewport->layer_lock);
  switch (task->remove.layer)
    {
    case PGM_DRAWABLE_NEAR:
      glviewport->near_layer = g_list_remove (glviewport->near_layer,
                                              gldrawable);
      break;

    case PGM_DRAWABLE_MIDDLE:
      glviewport->middle_layer = g_list_remove (glviewport->middle_layer,
                                                gldrawable);
      break;

    case PGM_DRAWABLE_FAR:
      glviewport->far_layer = g_list_remove (glviewport->far_layer,
                                             gldrawable);
      break;

    default:
      break;
    }
  g_mutex_unlock (glviewport->layer_lock);

  /* And delete it */
  gl_drawable_free (glviewport, gldrawable);

  task_remove_free (task);
}

/* Complete a regenerate task and free it */
static void
do_task_regenerate (PgmGlViewport *glviewport,
                    Task *task)
{
  g_mutex_lock (glviewport->layer_lock);

  g_list_foreach (glviewport->near_layer,
                  (GFunc) pgm_gl_drawable_regenerate, NULL);
  g_list_foreach (glviewport->middle_layer,
                  (GFunc) pgm_gl_drawable_regenerate, NULL);
  g_list_foreach (glviewport->far_layer,
                  (GFunc) pgm_gl_drawable_regenerate, NULL);

  g_mutex_unlock (glviewport->layer_lock);

  task_any_free (task);
}

/* Drawable 'changed' handler */
static void
changed_cb (PgmDrawable *drawable,
            PgmDrawableProperty property,
            gpointer data)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (data);
  PgmGlViewport *glviewport = gldrawable->glviewport;

  GST_LOG_OBJECT (glviewport, "drawable_changed_cb");

  if (glviewport)
    {
      Task *task;

      /* Add the change task to the update queue */
      g_mutex_lock (glviewport->update_lock);
      task = task_change_new (gldrawable, property);
      glviewport->update_queue = g_slist_prepend (glviewport->update_queue,
                                                  task);
      g_mutex_unlock (glviewport->update_lock);

      /* Request an update */
      pgm_context_update (glviewport->context);
    }
}

/* Canvas 'drawable-added' handler */
static void
drawable_added_cb (PgmCanvas *canvas,
                   PgmDrawable *drawable,
                   PgmDrawableLayer layer,
                   gint order,
                   gpointer data)
{
  PgmGlViewport *glviewport = PGM_GL_VIEWPORT (data);
  Task *task;

  GST_DEBUG_OBJECT (glviewport, "drawable_added_cb");

  /* Add the add task in the update queue */
  task = task_add_new (drawable, layer, order);
  g_mutex_lock (glviewport->update_lock);
  glviewport->update_queue = g_slist_prepend (glviewport->update_queue, task);
  g_mutex_unlock (glviewport->update_lock);

  /* Request an update */
  pgm_context_update (glviewport->context);
}

/* Canvas 'drawable-removed' handler */
static void
drawable_removed_cb (PgmCanvas *canvas,
                     PgmDrawable *drawable,
                     PgmDrawableLayer layer,
                     gpointer data)
{
  PgmGlViewport *glviewport = PGM_GL_VIEWPORT (data);
  Task *task;

  GST_DEBUG_OBJECT (glviewport, "drawable_removed_cb");

  /* Add the remove task in the update queue */
  task = task_remove_new (drawable, layer);
  g_mutex_lock (glviewport->update_lock);
  glviewport->update_queue = g_slist_prepend (glviewport->update_queue, task);
  g_mutex_unlock (glviewport->update_lock);

  /* Request an update */
  pgm_context_update (glviewport->context);
}

/* Canvas 'drawable-reordered' handler */
static void
drawable_reordered_cb (PgmCanvas *canvas,
                       PgmDrawable *drawable,
                       PgmDrawableLayer layer,
                       gint order,
                       gpointer data)
{
  PgmGlViewport *glviewport = PGM_GL_VIEWPORT (data);
  Task *task;

  GST_DEBUG_OBJECT (glviewport, "drawable_reordered_cb");

  /* Add the reorder task in the update queue */
  task = task_reorder_new (drawable, layer, order);
  g_mutex_lock (glviewport->update_lock);
  glviewport->update_queue = g_slist_prepend (glviewport->update_queue, task);
  g_mutex_unlock (glviewport->update_lock);

  /* Request an update */
  pgm_context_update (glviewport->context);
}

/* Canvas 'regenerated' handler */
static void
regenerated_cb (PgmCanvas *canvas,
                gpointer data)
{
  PgmGlViewport *glviewport = PGM_GL_VIEWPORT (data);
  Task *task;

  GST_DEBUG_OBJECT (glviewport, "drawable_regenerated_cb");

  /* Add the regenerate task in the update queue */
  task = task_any_new (TASK_REGENERATE);
  g_mutex_lock (glviewport->update_lock);
  glviewport->update_queue = g_slist_prepend (glviewport->update_queue, task);
  g_mutex_unlock (glviewport->update_lock);

  /* Request an update */
  pgm_context_update (glviewport->context);
}

/* Delete all GL drawable from the given GL viewport layer list and free it */
static void
delete_all_gl_drawable_from_layer (PgmGlViewport *glviewport,
                                   GList **layer)
{
  GList *walk = *layer;
  PgmGlDrawable *gldrawable;

  while (walk)
    {
      gldrawable = (PgmGlDrawable*) walk->data;
      gl_drawable_free (glviewport, gldrawable);
      walk = walk->next;
    }

  GST_OBJECT_LOCK (glviewport);
  g_list_free (*layer);
  *layer = NULL;
  GST_OBJECT_UNLOCK (glviewport);
}

/* Synchronize the drawables from the given canvas creating and adding the
 * the corresponding GL drawables in the layers */
static void
sync_gl_drawable_from_canvas (PgmGlViewport *glviewport,
                              PgmCanvas *canvas)
{
  PgmGlDrawable *gldrawable;
  GList *walk;

  /* Sync near layer */
  walk = canvas->near_layer;
  while (walk)
    {
      gldrawable = gl_drawable_new (glviewport, (PgmDrawable*) walk->data);
      g_mutex_lock (glviewport->layer_lock);
      glviewport->near_layer = g_list_append (glviewport->near_layer,
                                              gldrawable);
      g_mutex_unlock (glviewport->layer_lock);
      walk = walk->next;
    }

  /* Sync middle layer */
  walk = canvas->middle_layer;
  while (walk)
    {
      gldrawable = gl_drawable_new (glviewport, (PgmDrawable*) walk->data);
      g_mutex_lock (glviewport->layer_lock);
      glviewport->middle_layer = g_list_append (glviewport->middle_layer,
                                                gldrawable);
      g_mutex_unlock (glviewport->layer_lock);
      walk = walk->next;
    }

  /* Sync far layer */
  walk = canvas->far_layer;
  while (walk)
    {
      gldrawable = gl_drawable_new (glviewport, (PgmDrawable*) walk->data);
      g_mutex_lock (glviewport->layer_lock);
      glviewport->far_layer = g_list_append (glviewport->far_layer,
                                             gldrawable);
      g_mutex_unlock (glviewport->layer_lock);
      walk = walk->next;
    }
}

/* PgmViewport methods */

static PgmError
pgm_gl_viewport_set_title (PgmViewport *viewport,
                           const gchar *title)
{
  PgmGlViewport *glviewport = PGM_GL_VIEWPORT (viewport);
  PgmContextTask *task;

  GST_DEBUG_OBJECT (glviewport, "set_title");

  task = pgm_context_task_new (PGM_CONTEXT_TITLE, NULL);
  pgm_context_push_immediate_task (glviewport->context, task);

  return PGM_ERROR_OK;
}

static PgmError
pgm_gl_viewport_show (PgmViewport *viewport)
{
  PgmGlViewport *glviewport = PGM_GL_VIEWPORT (viewport);
  PgmContextTask *task;

  GST_DEBUG_OBJECT (glviewport, "show");

  task = pgm_context_task_new (PGM_CONTEXT_VISIBILITY, NULL);
  pgm_context_push_immediate_task (glviewport->context, task);

  return PGM_ERROR_OK;
}

static PgmError
pgm_gl_viewport_hide (PgmViewport *viewport)
{
  PgmGlViewport *glviewport = PGM_GL_VIEWPORT (viewport);
  PgmContextTask *task;

  GST_DEBUG_OBJECT (glviewport, "hide");

  task = pgm_context_task_new (PGM_CONTEXT_VISIBILITY, NULL);
  pgm_context_push_immediate_task (glviewport->context, task);

  return PGM_ERROR_OK;
}

static PgmError
pgm_gl_viewport_set_decorated (PgmViewport *viewport,
                               gboolean decorated)
{
  PgmGlViewport *glviewport = PGM_GL_VIEWPORT (viewport);
  PgmContextTask *task;

  GST_DEBUG_OBJECT (glviewport, "set_decorated");

  task = pgm_context_task_new (PGM_CONTEXT_DECORATION, NULL);
  pgm_context_push_immediate_task (glviewport->context, task);

  return PGM_ERROR_OK;
}

static PgmError
pgm_gl_viewport_set_cursor (PgmViewport *viewport,
                            PgmViewportCursor cursor)
{
  PgmGlViewport *glviewport = PGM_GL_VIEWPORT (viewport);
  PgmContextTask *task;

  GST_DEBUG_OBJECT (glviewport, "set_cursor");

  task = pgm_context_task_new (PGM_CONTEXT_CURSOR, NULL);
  pgm_context_push_immediate_task (glviewport->context, task);

  return PGM_ERROR_OK;
}

static PgmError
pgm_gl_viewport_set_icon (PgmViewport *viewport,
                          GdkPixbuf *icon)
{
  PgmGlViewport *glviewport = PGM_GL_VIEWPORT (viewport);
  PgmContextTask *task;

  GST_DEBUG_OBJECT (glviewport, "set_icon");

  task = pgm_context_task_new (PGM_CONTEXT_ICON, NULL);
  pgm_context_push_immediate_task (glviewport->context, task);

  return PGM_ERROR_OK;
}

static PgmError
pgm_gl_viewport_set_drag_status (PgmViewport *viewport,
                                 gboolean accept)
{
  PgmGlViewport *glviewport = PGM_GL_VIEWPORT (viewport);
  PgmContextTask *task;

  GST_DEBUG_OBJECT (glviewport, "set_drag_status");

  glviewport->drag_status = accept;

  task = pgm_context_task_new (PGM_CONTEXT_DRAG_STATUS, NULL);
  pgm_context_push_immediate_task (glviewport->context, task);

  return PGM_ERROR_OK;
}

static PgmError
pgm_gl_viewport_set_size (PgmViewport *viewport,
                          gint width,
                          gint height)
{
  PgmGlViewport *glviewport = PGM_GL_VIEWPORT (viewport);
  PgmContextTask *task;

  GST_DEBUG_OBJECT (glviewport, "set_size");

  task = pgm_context_task_new (PGM_CONTEXT_SIZE, NULL);
  pgm_context_push_immediate_task (glviewport->context, task);

  return PGM_ERROR_OK;
}

static PgmError
pgm_gl_viewport_set_alpha_blending (PgmViewport *viewport,
                                    gboolean alpha_blending)
{
  PgmGlViewport *glviewport = PGM_GL_VIEWPORT (viewport);
  PgmContextTask *task;

  GST_DEBUG_OBJECT (glviewport, "set_alpha_blending");

  task = pgm_context_task_new (PGM_CONTEXT_ALPHA_BLENDING, NULL);
  pgm_context_push_immediate_task (glviewport->context, task);

  return PGM_ERROR_OK;
}

static PgmError
pgm_gl_viewport_set_opacity (PgmViewport *viewport,
                             guchar opacity)
{
  PgmGlViewport *glviewport = PGM_GL_VIEWPORT (viewport);
  PgmContextTask *task;

  GST_DEBUG_OBJECT (glviewport, "set_opacity");

  task = pgm_context_task_new (PGM_CONTEXT_OPACITY, NULL);
  pgm_context_push_immediate_task (glviewport->context, task);

  return PGM_ERROR_OK;
}

static PgmError
pgm_gl_viewport_set_fullscreen (PgmViewport *viewport,
                                gboolean fullscreen)
{
  PgmGlViewport *glviewport = PGM_GL_VIEWPORT (viewport);
  PgmContextTask *task;

  GST_DEBUG_OBJECT (glviewport, "set_fullscreen");

  task = pgm_context_task_new (PGM_CONTEXT_FULLSCREEN, NULL);
  pgm_context_push_immediate_task (glviewport->context, task);

  return PGM_ERROR_OK;
}

static PgmError
pgm_gl_viewport_set_iconified (PgmViewport *viewport,
                               gboolean iconified)
{
  PgmGlViewport *glviewport = PGM_GL_VIEWPORT (viewport);
  PgmContextTask *task;

  GST_DEBUG_OBJECT (glviewport, "set_iconified");

  task = pgm_context_task_new (PGM_CONTEXT_ICONIFICATION, NULL);
  pgm_context_push_immediate_task (glviewport->context, task);

  return PGM_ERROR_OK;
}

static PgmError
pgm_gl_viewport_focus (PgmViewport *viewport)
{
  PgmGlViewport *glviewport = PGM_GL_VIEWPORT (viewport);
  PgmContextTask *task;

  GST_DEBUG_OBJECT (glviewport, "present");

  task = pgm_context_task_new (PGM_CONTEXT_FOCUS, NULL);
  pgm_context_push_immediate_task (glviewport->context, task);

  return PGM_ERROR_OK;
}

static PgmError
pgm_gl_viewport_get_screen_resolution (PgmViewport *viewport,
                                       gint *width,
                                       gint *height)
{
  PgmGlViewport *glviewport = PGM_GL_VIEWPORT (viewport);

  GST_LOG_OBJECT (glviewport, "get_screen_resolution");

  pgm_backend_get_screen_resolution (glviewport->context->backend, width,
                                     height);

  return PGM_ERROR_OK;
}

static PgmError
pgm_gl_viewport_get_screen_size_mm (PgmViewport *viewport,
                                    gint *width,
                                    gint *height)
{
  PgmGlViewport *glviewport = PGM_GL_VIEWPORT (viewport);

  GST_LOG_OBJECT (glviewport, "get_screen_size_mm");

  pgm_backend_get_screen_size_mm (glviewport->context->backend, width, height);

  return PGM_ERROR_OK;
}

static PgmError
pgm_gl_viewport_set_canvas (PgmViewport *viewport,
                            PgmCanvas *canvas)
{
  PgmGlViewport *glviewport = PGM_GL_VIEWPORT (viewport);

  GST_DEBUG_OBJECT (glviewport, "set_canvas");

  /* There's a canvas bound, let's clear associated data */
  if (glviewport->canvas)
    {
      GST_OBJECT_LOCK (glviewport);
      GST_OBJECT_LOCK (glviewport->canvas);

      /* Disconnect handlers */
      g_signal_handler_disconnect (glviewport->canvas,
                                   glviewport->add_handler);
      g_signal_handler_disconnect (glviewport->canvas,
                                   glviewport->remove_handler);
      g_signal_handler_disconnect (glviewport->canvas,
                                   glviewport->reorder_handler);
      g_signal_handler_disconnect (glviewport->canvas,
                                   glviewport->regenerated_handler);

      GST_OBJECT_UNLOCK (glviewport->canvas);
      GST_OBJECT_UNLOCK (glviewport);

      /* Delete our hierarchy */
      delete_all_gl_drawable_from_layer (glviewport, &glviewport->near_layer);
      delete_all_gl_drawable_from_layer (glviewport, &glviewport->middle_layer);
      delete_all_gl_drawable_from_layer (glviewport, &glviewport->far_layer);

      GST_OBJECT_LOCK (glviewport);
      glviewport->canvas = NULL;
      GST_OBJECT_UNLOCK (glviewport);
    }

  /* Bind the new canvas if any */
  if (canvas)
    {
      sync_gl_drawable_from_canvas (glviewport, canvas);

      GST_OBJECT_LOCK (glviewport);
      GST_OBJECT_LOCK (canvas);

      /* Connect handlers */
      glviewport->add_handler =
        g_signal_connect (G_OBJECT (canvas), "drawable-added",
                          G_CALLBACK (drawable_added_cb),
                          (gpointer) glviewport);
      glviewport->remove_handler =
        g_signal_connect (G_OBJECT (canvas), "drawable-removed",
                          G_CALLBACK (drawable_removed_cb),
                          (gpointer) glviewport);
      glviewport->reorder_handler =
        g_signal_connect (G_OBJECT (canvas), "drawable-reordered",
                          G_CALLBACK (drawable_reordered_cb),
                          (gpointer) glviewport);
      glviewport->regenerated_handler =
        g_signal_connect (G_OBJECT (canvas), "regenerated",
                          G_CALLBACK (regenerated_cb),
                          (gpointer) glviewport);

      glviewport->canvas = canvas;

      GST_OBJECT_UNLOCK (canvas);
      GST_OBJECT_UNLOCK (glviewport);
    }

  return PGM_ERROR_OK;
}

static PgmError
pgm_gl_viewport_update_projection (PgmViewport *viewport)
{
  PgmGlViewport *glviewport = PGM_GL_VIEWPORT (viewport);
  PgmContextTask *task;

  GST_DEBUG_OBJECT (glviewport, "update_projection");

  GST_OBJECT_LOCK (viewport);

  /* Store the projected size  */
  if (viewport->rotation == PGM_VIEWPORT_ROTATION_NONE
      || viewport->rotation == PGM_VIEWPORT_ROTATION_180)
    {
      glviewport->projected_w = (gfloat) viewport->projected_width;
      glviewport->projected_h = (gfloat) viewport->projected_height;
    }
  else
    {
      glviewport->projected_w = (gfloat) viewport->projected_height;
      glviewport->projected_h = (gfloat) viewport->projected_width;
    }

  /* Store the canvas size */
  if (viewport->canvas)
    {
      GST_OBJECT_LOCK (viewport->canvas);
      glviewport->canvas_w = viewport->canvas->width;
      glviewport->canvas_h = viewport->canvas->height;
      GST_OBJECT_UNLOCK (viewport->canvas);
    }

  GST_OBJECT_UNLOCK (viewport);

  /* Deduct the pixel boundary offsets */
  glviewport->pixel_offset_x = glviewport->canvas_w / glviewport->projected_w;
  glviewport->pixel_offset_y = glviewport->canvas_h / glviewport->projected_h;

  /* And queue the projection update task */
  task = pgm_context_task_new (PGM_CONTEXT_PROJECTION, NULL);
  pgm_context_push_immediate_task (glviewport->context, task);

  return PGM_ERROR_OK;
}

static PgmError
pgm_gl_viewport_get_embedding_id (PgmViewport *viewport,
                                  gulong *embedding_id)
{
  PgmGlViewport *glviewport = PGM_GL_VIEWPORT (viewport);

  GST_LOG_OBJECT (glviewport, "get_embedding_id");

  GST_OBJECT_LOCK (glviewport);
  pgm_backend_get_embedding_id (glviewport->context->backend, embedding_id);
  GST_OBJECT_UNLOCK (glviewport);

  return PGM_ERROR_OK;
}

static PgmError
pgm_gl_viewport_get_pixel_formats (PgmViewport *viewport,
                                   gulong *formats_mask)
{
  PgmGlViewport *glviewport = PGM_GL_VIEWPORT (viewport);
  const gchar *variable;

  GST_LOG_OBJECT (glviewport, "get_pixel_formats");

  GST_OBJECT_LOCK (glviewport);

  /* The classic formats supported by all hardware */
  *formats_mask = PGM_IMAGE_RGB
                | PGM_IMAGE_RGBA
                | PGM_IMAGE_BGR
                | PGM_IMAGE_BGRA;

  /* I420 and YV12 are supported through fragment programs and
   * multi-texturing with a minimum texture units number of three. The
   * PGM_GL_CSP_PROGRAM environment variable allows to disable the use
   * fragment programs to do the color space conversion. */
  if (glviewport->context->feature_mask & PGM_GL_FEAT_PER_PLANE_YCBCR_PROGRAM)
    {
      *formats_mask |= PGM_IMAGE_I420 | PGM_IMAGE_YV12;

      variable = g_getenv ("PGM_GL_CSP_PROGRAM");
      if (variable && variable[0] == '0')
        *formats_mask &= ~(PGM_IMAGE_I420 | PGM_IMAGE_YV12);
    }

  /* FIXME: YUYV and UYVY are not supported */

  GST_OBJECT_UNLOCK (glviewport);

  return PGM_ERROR_OK;
}

static PgmError
pgm_gl_viewport_get_caps_mask (PgmViewport *viewport,
                               gulong *caps_mask)
{
  PgmGlViewport *glviewport = PGM_GL_VIEWPORT (viewport);

  GST_LOG_OBJECT (glviewport, "get_caps_mask");

  GST_OBJECT_LOCK (glviewport);
  *caps_mask = glviewport->capacities;
  GST_OBJECT_UNLOCK (glviewport);

  return PGM_ERROR_OK;
}

static PgmError
pgm_gl_viewport_get_frame_rate (PgmViewport *viewport,
                                guint *frame_rate)
{
  PgmGlViewport *glviewport = PGM_GL_VIEWPORT (viewport);

  GST_LOG_OBJECT (glviewport, "get_frame_rate");

  GST_OBJECT_LOCK (glviewport);
  *frame_rate = glviewport->context->fps;
  GST_OBJECT_UNLOCK (glviewport);

  return PGM_ERROR_OK;
}

static PgmError
pgm_gl_viewport_set_message_filter (PgmViewport *viewport,
                                    GList *filter)
{
  PgmGlViewport *glviewport = PGM_GL_VIEWPORT (viewport);
  PgmContextTask *task;

  GST_LOG_OBJECT (glviewport, "set_message_filter");

  task = pgm_context_task_new (PGM_CONTEXT_MESSAGE_FILTER, NULL);
  pgm_context_push_immediate_task (glviewport->context, task);

  return PGM_ERROR_OK;
}

static PgmError
pgm_gl_viewport_read_pixels (PgmViewport *viewport,
                             guint x,
                             guint y,
                             guint width,
                             guint height,
                             guint8 *pixels)
{
  PgmGlViewport *glviewport = PGM_GL_VIEWPORT (viewport);
  PgmGlViewportPixelRectangle *rectangle = NULL;
  PgmContextTask *task = NULL;

  GST_DEBUG_OBJECT (glviewport, "read_pixels");

  rectangle = g_slice_new (PgmGlViewportPixelRectangle);
  if (!rectangle)
    return PGM_ERROR_X;

  rectangle->x = x;
  rectangle->y = y;
  rectangle->width = width;
  rectangle->height = height;
  rectangle->pixels = pixels;

  task = pgm_context_task_new (PGM_CONTEXT_READ_PIXELS, rectangle);
  pgm_context_push_immediate_task (glviewport->context, task);

  return PGM_ERROR_OK;
}

/* GObject stuff */

PGM_DEFINE_DYNAMIC_TYPE (PgmGlViewport, pgm_gl_viewport, PGM_TYPE_VIEWPORT);

void
pgm_gl_viewport_register (GTypeModule *module)
{
  pgm_gl_viewport_register_type (module);
}

static void
pgm_gl_viewport_dispose (GObject *object)
{
  PgmGlViewport *glviewport = PGM_GL_VIEWPORT (object);
  PgmContextTask *task;

  GST_DEBUG_OBJECT (glviewport, "dispose");

  if (glviewport->canvas)
    {
      /* Disconnect handlers */
      g_signal_handler_disconnect (glviewport->canvas,
                                   glviewport->add_handler);
      g_signal_handler_disconnect (glviewport->canvas,
                                   glviewport->remove_handler);
      g_signal_handler_disconnect (glviewport->canvas,
                                   glviewport->reorder_handler);
      g_signal_handler_disconnect (glviewport->canvas,
                                   glviewport->regenerated_handler);

      /* Delete our hierarchy */
      g_mutex_lock (glviewport->layer_lock);
      delete_all_gl_drawable_from_layer (glviewport, &glviewport->near_layer);
      delete_all_gl_drawable_from_layer (glviewport, &glviewport->middle_layer);
      delete_all_gl_drawable_from_layer (glviewport, &glviewport->far_layer);
      g_mutex_unlock (glviewport->layer_lock);
    }

  g_mutex_lock (glviewport->update_lock);
  g_slist_foreach (glviewport->update_queue, (GFunc) task_free, NULL);
  g_slist_free (glviewport->update_queue);
  glviewport->update_queue = NULL;
  g_mutex_unlock (glviewport->update_lock);

  task = pgm_context_task_new (PGM_CONTEXT_QUIT, NULL);
  pgm_context_push_immediate_task (glviewport->context, task);
  pgm_context_free (glviewport->context);

  g_mutex_free (glviewport->layer_lock);
  g_mutex_free (glviewport->update_lock);

  GST_CALL_PARENT (G_OBJECT_CLASS, dispose, (object));
}

static void
pgm_gl_viewport_class_init (PgmGlViewportClass *klass)
{
  GObjectClass *gobject_class;
  PgmViewportClass *viewport_class;

  GST_DEBUG_CATEGORY_INIT (pgm_gl_viewport_debug, "pgm_gl_viewport", 0,
                           "OpenGL plugin: PgmGlViewport");

  parent_class = g_type_class_peek_parent (klass);

  gobject_class = G_OBJECT_CLASS (klass);
  viewport_class = PGM_VIEWPORT_CLASS (klass);

  /* GObject virtual table */
  gobject_class->dispose = GST_DEBUG_FUNCPTR (pgm_gl_viewport_dispose);

  /* PgmViewport virtual table */
  viewport_class->set_title = GST_DEBUG_FUNCPTR (pgm_gl_viewport_set_title);
  viewport_class->show = GST_DEBUG_FUNCPTR (pgm_gl_viewport_show);
  viewport_class->hide = GST_DEBUG_FUNCPTR (pgm_gl_viewport_hide);
  viewport_class->set_decorated =
    GST_DEBUG_FUNCPTR (pgm_gl_viewport_set_decorated);
  viewport_class->set_cursor = GST_DEBUG_FUNCPTR (pgm_gl_viewport_set_cursor);
  viewport_class->set_icon = GST_DEBUG_FUNCPTR (pgm_gl_viewport_set_icon);
  viewport_class->set_drag_status =
    GST_DEBUG_FUNCPTR (pgm_gl_viewport_set_drag_status);
  viewport_class->set_size = GST_DEBUG_FUNCPTR (pgm_gl_viewport_set_size);
  viewport_class->set_alpha_blending =
    GST_DEBUG_FUNCPTR (pgm_gl_viewport_set_alpha_blending);
  viewport_class->set_opacity =
    GST_DEBUG_FUNCPTR (pgm_gl_viewport_set_opacity);
  viewport_class->set_fullscreen =
    GST_DEBUG_FUNCPTR (pgm_gl_viewport_set_fullscreen);
  viewport_class->set_iconified =
    GST_DEBUG_FUNCPTR (pgm_gl_viewport_set_iconified);
  viewport_class->focus = GST_DEBUG_FUNCPTR (pgm_gl_viewport_focus);
  viewport_class->get_screen_resolution =
    GST_DEBUG_FUNCPTR (pgm_gl_viewport_get_screen_resolution);
  viewport_class->get_screen_size_mm =
    GST_DEBUG_FUNCPTR (pgm_gl_viewport_get_screen_size_mm);
  viewport_class->set_canvas = GST_DEBUG_FUNCPTR (pgm_gl_viewport_set_canvas);
  viewport_class->update_projection =
    GST_DEBUG_FUNCPTR (pgm_gl_viewport_update_projection);
  viewport_class->get_embedding_id =
    GST_DEBUG_FUNCPTR (pgm_gl_viewport_get_embedding_id);
  viewport_class->get_pixel_formats =
    GST_DEBUG_FUNCPTR (pgm_gl_viewport_get_pixel_formats);
  viewport_class->get_caps_mask =
    GST_DEBUG_FUNCPTR (pgm_gl_viewport_get_caps_mask);
  viewport_class->get_frame_rate =
    GST_DEBUG_FUNCPTR (pgm_gl_viewport_get_frame_rate);
  viewport_class->set_message_filter =
    GST_DEBUG_FUNCPTR (pgm_gl_viewport_set_message_filter);
  viewport_class->read_pixels = GST_DEBUG_FUNCPTR (pgm_gl_viewport_read_pixels);

  /* PgmGlDrawable changed table */
  klass->changed_func[PGM_DRAWABLE_VISIBILITY] =
    GST_DEBUG_FUNCPTR (pgm_gl_drawable_set_visibility);
  klass->changed_func[PGM_DRAWABLE_SIZE] =
    GST_DEBUG_FUNCPTR (pgm_gl_drawable_set_size);
  klass->changed_func[PGM_DRAWABLE_POSITION] =
    GST_DEBUG_FUNCPTR (pgm_gl_drawable_set_position);
  klass->changed_func[PGM_DRAWABLE_TRANSFORMATION_MATRIX] =
    GST_DEBUG_FUNCPTR (pgm_gl_drawable_set_transformation_matrix);
  klass->changed_func[PGM_DRAWABLE_BG_COLOR] =
    GST_DEBUG_FUNCPTR (pgm_gl_drawable_set_bg_color);
  klass->changed_func[PGM_DRAWABLE_FG_COLOR] =
    GST_DEBUG_FUNCPTR (pgm_gl_drawable_set_fg_color);
  klass->changed_func[PGM_DRAWABLE_OPACITY] =
    GST_DEBUG_FUNCPTR (pgm_gl_drawable_set_opacity);
  klass->changed_func[PGM_DRAWABLE_REGENERATE] =
    GST_DEBUG_FUNCPTR (pgm_gl_drawable_regenerate);

  /* PgmGlImage changed table */
  klass->changed_func[PGM_IMAGE_DATA_EMPTY] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_image_clear);
  klass->changed_func[PGM_IMAGE_DATA_FILE] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_image_set_from_file);
  klass->changed_func[PGM_IMAGE_DATA_BUFFER] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_image_set_from_buffer);
  klass->changed_func[PGM_IMAGE_DATA_GST_BUFFER] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_image_set_from_gst_buffer);
  klass->changed_func[PGM_IMAGE_DATA_PIXBUF] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_image_set_from_pixbuf);
  klass->changed_func[PGM_IMAGE_DATA_SYSTEM_BUFFER] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_image_set_from_system_buffer);
  klass->changed_func[PGM_IMAGE_DATA_IMAGE] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_image_set_from_image);
  klass->changed_func[PGM_IMAGE_SYSTEM_BUFFER_CONTENT] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_image_system_buffer_changed);
  klass->changed_func[PGM_IMAGE_MAPPING_MATRIX] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_image_set_mapping_matrix);
  klass->changed_func[PGM_IMAGE_ALIGNMENT] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_image_set_alignment);
  klass->changed_func[PGM_IMAGE_LAYOUT] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_image_set_layout);
  klass->changed_func[PGM_IMAGE_INTERP] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_image_set_interp);
  klass->changed_func[PGM_IMAGE_WRAPPING] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_image_set_wrapping);
  klass->changed_func[PGM_IMAGE_ASPECT_RATIO] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_image_set_aspect_ratio);
  klass->changed_func[PGM_IMAGE_BORDER_WIDTH] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_image_set_border_width);
  klass->changed_func[PGM_IMAGE_BORDER_INNER_COLOR] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_image_set_border_inner_color);
  klass->changed_func[PGM_IMAGE_BORDER_OUTER_COLOR] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_image_set_border_outer_color);

  /* PgmGlText changed table */
  klass->changed_func[PGM_TEXT_LABEL] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_text_set_label);
  klass->changed_func[PGM_TEXT_MARKUP] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_text_set_markup);
  klass->changed_func[PGM_TEXT_FONT_FAMILY] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_text_set_font_family);
  klass->changed_func[PGM_TEXT_HEIGHT] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_text_set_font_height);
  klass->changed_func[PGM_TEXT_ELLIPSIZE] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_text_set_ellipsize);
  klass->changed_func[PGM_TEXT_JUSTIFY] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_text_set_justify);
  klass->changed_func[PGM_TEXT_ALIGNMENT] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_text_set_alignment);
  klass->changed_func[PGM_TEXT_WRAP] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_text_set_wrap);
  klass->changed_func[PGM_TEXT_GRAVITY] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_text_set_gravity);
  klass->changed_func[PGM_TEXT_STRETCH] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_text_set_stretch);
  klass->changed_func[PGM_TEXT_STYLE] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_text_set_style);
  klass->changed_func[PGM_TEXT_VARIANT] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_text_set_variant);
  klass->changed_func[PGM_TEXT_WEIGHT] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_text_set_weight);
  klass->changed_func[PGM_TEXT_LINE_SPACING] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_text_set_line_spacing);
  klass->changed_func[PGM_TEXT_OUTLINE_COLOR] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_text_set_outline_color);
  klass->changed_func[PGM_TEXT_OUTLINE_WIDTH] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_text_set_outline_width);
}

static void
pgm_gl_viewport_class_finalize (PgmGlViewportClass *klass)
{
  return;
}

static void
pgm_gl_viewport_init (PgmGlViewport *glviewport)
{
  GST_DEBUG_OBJECT (glviewport, "init");

  /* Attached canvas */
  glviewport->canvas = NULL;

  /* Empty lists */
  glviewport->far_layer = NULL;
  glviewport->middle_layer = NULL;
  glviewport->near_layer = NULL;
  glviewport->layer_lock = g_mutex_new ();

  /* Update task queue */
  glviewport->update_queue = NULL;
  glviewport->update_lock = g_mutex_new ();

  /* Create the drawable hash using pointers as keys */
  glviewport->drawable_hash = g_hash_table_new (NULL, NULL);

  /* Create context */
  glviewport->context = pgm_context_new (glviewport);

  /* Projection informations */
  glviewport->projected_w = 800;
  glviewport->projected_h = 600;
  glviewport->canvas_w = 4.0f;
  glviewport->canvas_h = 3.0f;
  glviewport->pixel_offset_x = 0.005f;
  glviewport->pixel_offset_y = 0.0005f;

  /* Fill capacities */
  glviewport->capacities = 0;
  if (pgm_backend_is_accelerated (glviewport->context->backend))
    glviewport->capacities |= PGM_VIEWPORT_HARDWARE_ACCELERATION;
  if (pgm_backend_is_embeddable (glviewport->context->backend))
    glviewport->capacities |= PGM_VIEWPORT_APPLICATION_EMBEDDING;
  if (pgm_backend_has_alpha_component (glviewport->context->backend))
    glviewport->capacities |= PGM_VIEWPORT_OPACITY;

#ifdef HAVE_GL_GLX_H
  if (pgm_backend_has_system_buffer (glviewport->context->backend) &&
      glviewport->context->feature_mask & PGM_GL_FEAT_TEXTURE_NON_POWER_OF_TWO)
    glviewport->capacities |= PGM_VIEWPORT_X11_SYSTEM_BUFFER;
#endif /* HAVE_GL_GLX_H */

  /* Drag status */
  glviewport->drag_status = FALSE;
}

/* Public methods */

PgmViewport *
pgm_gl_viewport_new (void)
{
  PgmGlViewport *glviewport;

  glviewport = g_object_new (PGM_TYPE_GL_VIEWPORT, NULL);
  GST_DEBUG_OBJECT (glviewport, "created new glviewport");

  return PGM_VIEWPORT (glviewport);
}

void
pgm_gl_viewport_flush_update_queue (PgmGlViewport *glviewport)
{
  GSList *copy, *walk;
  Task *task;

  /* Get a reverse copy of the update task list, and NULLify it */
  g_mutex_lock (glviewport->update_lock);
  copy = g_slist_reverse (glviewport->update_queue);
  glviewport->update_queue = NULL;
  g_mutex_unlock (glviewport->update_lock);

  /* Flush the update task list */
  walk = copy;
  while (walk)
    {
      task = walk->data;
      task_func[task->type] (glviewport, task);
      walk = walk->next;
    }

  g_slist_free (copy);
  copy = NULL;
}

void
pgm_gl_viewport_update_drawable_projection (PgmGlViewport *glviewport)
{
  g_mutex_lock (glviewport->layer_lock);

  /* Adapt position and size of all the drawables */
  g_list_foreach (glviewport->near_layer,
                  (GFunc) pgm_gl_drawable_update_projection, NULL);
  g_list_foreach (glviewport->middle_layer,
                  (GFunc) pgm_gl_drawable_update_projection, NULL);
  g_list_foreach (glviewport->far_layer,
                  (GFunc) pgm_gl_drawable_update_projection, NULL);

  g_mutex_unlock (glviewport->layer_lock);
}

/* Connects the GL drawable to the "changed" signal. Should be called at the
 * creation of a new GL drawable to be notified of any change. */
void
pgm_gl_viewport_connect_changed_callback (PgmGlViewport *glviewport,
                                          PgmGlDrawable *gldrawable)
{
  PgmDrawable *drawable = gldrawable->drawable;

  GST_OBJECT_LOCK (drawable);
  gldrawable->change_handler = g_signal_connect (drawable, "changed",
                                                 G_CALLBACK (changed_cb),
                                                 (gpointer) gldrawable);
  GST_OBJECT_UNLOCK (drawable);
}
