# -*- mode: python; coding: utf-8 -*-
#
# Pigment Python tools
#
# Copyright © 2006, 2007 Fluendo Embedded S.L.
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the
# Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.

from pypgmtools.graph.group import Group
from pypgmtools.graph.image import Image
from pypgmtools.timing.implicit import *
import pgm
import os


class Slideshow(Group):

    def __init__(self, canvas, layer):
        Group.__init__(self, canvas, layer)

        self._pictures_paths = []
        self._current_index = None
        self._target_index = None
        self._frame_size = 0.1

        self._current = Image()
        self._current.bg_color = (0, 0, 0, 0)
        self._current.visible = True
        self._animated_current = AnimatedObject(self._current)
        self._animated_current.mode = REPLACE

        self._hidden = Image()
        self._hidden.bg_color = (0, 0, 0, 0)
        self._hidden.visible = True
        self._animated_hidden = AnimatedObject(self._hidden)
        self._animated_hidden.mode = REPLACE

        # invisible drawable used to capture mouse events
        self._mouse_zone = Image()
        self._mouse_zone.opacity = 0
        self._mouse_zone.position = (0.0, 0.0, 0.0)
        self._mouse_zone.visible = True

        self.delay = 500

        self.add(self._current)
        self.add(self._hidden)
        self.add(self._mouse_zone)

        self._current_position = (0.0, 0.0, 0.0)
        self.size = self._current.size

        self._moved = False
        self._moved_orig_dx = 0.0
        self._loaded_paths = {self._hidden: "", self._current: ""}


    def delay__set(self, delay):
        self._animated_current.setup_next_animations(transformation = DECELERATE,
                                                     duration = delay,
                                                     end_callback = self._current_ready)

        self._animated_hidden.setup_next_animations(transformation = DECELERATE,
                                                    duration = delay)

    def pictures_paths__set(self, pictures_paths):
        try:
            if len(self._pictures_paths) > 0:
                old_path = self._pictures_paths[self._current_index]
            else:
                old_path = ""
            self._pictures_paths = pictures_paths
            new_path = self._pictures_paths[self._current_index]
            if new_path != old_path:
                self._load_picture_from_index(self._current, self._current_index)
        except:
            pass

        self._pictures_paths = pictures_paths

    def _load_picture_from_index(self, drawable, index):
        try:
            if index < 0 or index >= len(self._pictures_paths):
                raise
            path = self._pictures_paths[index]
            if self._loaded_paths[drawable] != path:
                self._loaded_paths[drawable] = path
                drawable.set_from_fd(os.open(path, os.O_RDONLY), 1024)
        except:
            drawable.clear()
            self._loaded_paths[drawable] = ""

    def set_path_for_index(self, index, picture_path):
        self._pictures_paths[index] = picture_path
        
        if index == self._current_index:
            self._load_picture_from_index(self._current, index)

    def length__get(self):
        return len(self._pictures_paths)

    def current_index__set(self, index):
        index = max(0, index)
        index = min(len(self._pictures_paths)-1, index)

        if len(self._pictures_paths) == 0:
            return

        self._target_index = index

        if not self._animated_current.is_animated() and \
           not self._animated_hidden.is_animated():
            self._update_current_index()

    def current_index__get(self):
        return self._target_index

    def _update_current_index(self):
        if self._current_index == None:
            self._current_index = self._target_index
            self._current.opacity = 0
            self._load_picture_from_index(self._current, self._target_index)
            self._animated_current.opacity = 255
        elif self._target_index - self._current_index < 0:
            width = self._current.width
            self._hidden.x = self._current.x - width - self._frame_size*width

            self._current_index = self._target_index
            self._switch_hidden_current(self._previous_position,
                                        self._next_position)

        elif self._target_index - self._current_index > 0:
            width = self._current.width
            self._hidden.x = self._current.x + width + self._frame_size*width
            
            self._current_index = self._target_index
            self._switch_hidden_current(self._next_position,
                                        self._previous_position)


    def _switch_hidden_current(self, from_position, to_position):
        if not self._moved:
            self._current.position = self._current_position
            self._hidden.position = from_position
        else:
            self._moved = False

        self._current, self._hidden = self._hidden, self._current
        self._animated_current, self._animated_hidden = self._animated_hidden, self._animated_current

        self._load_picture_from_index(self._current, self._current_index)

        self._animated_current.position = self._current_position
        self._animated_hidden.position = to_position
        #self._animated_current.opacity = 255
        #self._animated_hidden.opacity = 1

    def _current_ready(self, controller):
        self._update_current_index()

    def size__set(self, size):
        self._current.size = size
        self._hidden.size = size

        self._previous_position = (self._current_position[0] - \
                                   self._canvas.width - \
                                   self._frame_size*size[0], \
                                   self._current_position[1],
                                   self._current_position[2])
        self._next_position = (self._current_position[0] + \
                               self._canvas.width + \
                               self._frame_size*size[0], \
                               self._current_position[1],
                               self._current_position[2])

        self._mouse_zone.size = size

    def size__get(self):
        return self._current.size

    def width__set(self, width):
        self.size = (width, self.height)

    def width__get(self):
        return self.size[0]
    
    def height__set(self, height):
        self.size = (self.width, height)

    def height__get(self):
        return self.size[1]

    def connect(self, signal, *args):
        self._mouse_zone.connect(signal, *args)

    def move(self, dx):
        # FIXME: this is hacky
        if not self._moved:
            self._moved = True
            self._moved_orig_dx = self._current.x
            self._animated_current.stop_animations()
            self._animated_hidden.stop_animations()

        self._current.x = self._moved_orig_dx + dx
        if self._current.x > 0:
            self._hidden.x = self._previous_position[0] + self._moved_orig_dx + dx
            self._load_picture_from_index(self._hidden, self._current_index - 1)
        elif self._current.x < 0:
            self._hidden.x = self._next_position[0] + self._moved_orig_dx + dx
            self._load_picture_from_index(self._hidden, self._current_index + 1)
        
    def release(self):
        # FIXME: this is hacky
        self._moved = False
        self._animated_current.position = self._current_position
        if abs(self._hidden.x - self._previous_position[0]) < \
           abs(self._hidden.x - self._next_position[0]):
            self._animated_hidden.position = self._previous_position
        else:
            self._animated_hidden.position = self._next_position


def main(args):
    import pgm
    import gobject

    # Terminate the mainloop on a delete event
    def on_delete(viewport, event):
        pgm.main_quit()

    # Handles the key presses
    def on_key_press(viewport, event, slideshow):
        if event.keyval == pgm.keysyms.q or event.keyval == pgm.keysyms.Escape:
            pgm.main_quit()

        elif event.keyval == pgm.keysyms.Right:
            slideshow.current_index += 1

        elif event.keyval == pgm.keysyms.Left:
            slideshow.current_index -= 1

    def on_drag_begin(drawable, x, y, z, button, time, slideshow):
        global start_position
        start_position = (x, y)
        slideshow.move(0.0)
        return True
 
    def on_drag_motion(drawable, x, y, z, button, time, slideshow):
        global start_position
        if start_position != None:
            dx = x - start_position[0]
            slideshow.move(dx)
            return True

        return False
          
    def on_drag_end(drawable, x, y, z, button, time, slideshow):
        global start_position
        if start_position != None:
            dx = start_position[0] - x
            dy = start_position[1] - y

            if dx > 0.3:
                if slideshow.current_index < slideshow.length-1:
                    slideshow.current_index += 1
                else:
                    slideshow.release()
            elif dx < -0.3:
                if slideshow.current_index > 0:
                    slideshow.current_index -= 1
                else:
                    slideshow.release()
            else:
                slideshow.release()
            
            start_position = None
            return True

        return False


    # OpenGL viewport creation
    factory = pgm.ViewportFactory('opengl')
    gl = factory.create()
    gl.title = 'Implicit animation'

    # Canvas creation
    canvas = pgm.Canvas()

    # Bind the canvas to the OpenGL viewport
    gl.set_canvas(canvas)

    # Create and setup the slideshow widget
    #images = ["examples/pictures/fluendo.png" for i in range(5)]
    images = args
    slideshow = Slideshow(canvas, pgm.DRAWABLE_MIDDLE)
    slideshow.pictures_paths = images
    slideshow.current_index = 0
    #slideshow.position = (canvas.width * 0.05, canvas.height * 0.05, 0.0)
    #slideshow.size = (canvas.width * 0.9, canvas.height * 0.9)
    slideshow.size = canvas.size
    slideshow.visible = True

    start_position = None
    slideshow.connect('drag_begin', on_drag_begin, slideshow)
    slideshow.connect('drag_motion', on_drag_motion, slideshow)
    slideshow.connect('drag_end', on_drag_end, slideshow)

    # Let's start the mainloop
    gobject.timeout_add(15, gl.update)
    gl.connect('delete-event', on_delete)
    gl.connect('key-press-event', on_key_press, slideshow)
    gl.show()
    pgm.main()


if __name__ == '__main__':
    import sys
    sys.exit(main(sys.argv[1:]))
