/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment media rendering library
 *
 * Copyright © 2007 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author(s): Loïc Molinari <loic@fluendo.com>
 *            Florian Boucault <florian@fluendo.com>
 */

#ifndef __PGM_LINALG_H__
#define __PGM_LINALG_H__

#include <glib.h>
#include <string.h> /* memcpy */

G_BEGIN_DECLS

typedef struct _PgmVec3   PgmVec3;
typedef struct _PgmVec4   PgmVec4;
typedef struct _PgmMat4x4 PgmMat4x4;

/**
 * PgmVec3:
 * @v: the 3 components vector array.
 *
 * Describes a 3 components vector.
 */
struct _PgmVec3 {
  gfloat v[3];
};

/**
 * PgmVec4:
 * @v: the 4 components vector array.
 *
 * Describes a 4 components vector.
 */
struct _PgmVec4 {
  gfloat v[4];
};

/**
 * PgmMat4x4:
 * @m: the 4x4 matrix array.
 *
 * Describes a 4x4 matrix in row major order.
 */
struct _PgmMat4x4 {
  gfloat m[4][4];
};

/* 3 components vector initialization macros */

/**
 * PGM_VEC3_INIT:
 * @vec: the 3 components vector.
 * @v0: the 1st component.
 * @v1: the 2nd component.
 * @v2: the 3rd component.
 *
 * Initializes @vec 3 components vector.
 */
#define PGM_VEC3_INIT(vec,v0,v1,v2)                                     \
  G_STMT_START {                                                        \
    vec.v[0] = v0;                                                      \
    vec.v[1] = v1;                                                      \
    vec.v[2] = v2;                                                      \
  } G_STMT_END

/**
 * PGM_VEC3_COPY:
 * @dest: the destination 3 components vector. 
 * @src: the source 3 components vector.
 *
 * Copies contents from @src 3 component vector @dest 3 components vector.
 */
#define PGM_VEC3_COPY(dest,src)                                         \
  G_STMT_START {                                                        \
    memcpy ((void *) (dest), (void *) (src), sizeof (PgmVec3));         \
  } G_STMT_END

/* 4 components vector initialization macros */

/**
 * PGM_VEC4_INIT:
 * @vec: the 4 components vector.
 * @v0: the 1st component.
 * @v1: the 2nd component.
 * @v2: the 3rd component.
 * @v3: the 4th component.
 *
 * Initializes @vec 4 components vector.
 */
#define PGM_VEC4_INIT(vec,v0,v1,v2,v3)                                  \
  G_STMT_START {                                                        \
    vec.v[0] = v0;                                                      \
    vec.v[1] = v1;                                                      \
    vec.v[2] = v2;                                                      \
    vec.v[3] = v3;                                                      \
  } G_STMT_END

/**
 * PGM_VEC4_COPY:
 * @dest: the destination 4 components vector.
 * @src: the source 4 components vector.
 *
 * Copies contents from @src 4 component vector @dest 4 components vector.
 */
#define PGM_VEC4_COPY(dest,src)                                         \
  G_STMT_START {                                                        \
    memcpy ((void *) (dest), (void *) (src), sizeof (PgmVec4));         \
  } G_STMT_END

/* 4x4 matrix initialization macros */

/**
 * PGM_MAT4X4_ROW_INIT:
 * @mat: the 4x4 matrix.
 * @row: the row number.
 * @v0: the 1st component of the row.
 * @v1: the 2nd component of the row.
 * @v2: the 3rd component of the row.
 * @v3: the 4th component of the row.
 *
 * Initializes the row @row of the @mat 4x4 matrix.
 */
#define PGM_MAT4X4_ROW_INIT(mat,row,v0,v1,v2,v3)                        \
  G_STMT_START {                                                        \
    mat.m[(row)][0] = (v0);                                             \
    mat.m[(row)][1] = (v1);                                             \
    mat.m[(row)][2] = (v2);                                             \
    mat.m[(row)][3] = (v3);                                             \
  } G_STMT_END

/**
 * PGM_MAT4X4_INIT:
 * @mat: the 4x4 matrix.
 * @v00: the 1st component of the 1st row.
 * @v01: the 2nd component of the 1st row.
 * @v02: the 3rd component of the 1st row.
 * @v03: the 4th component of the 1st row.
 * @v10: the 1st component of the 2nd row.
 * @v11: the 2nd component of the 2nd row.
 * @v12: the 3rd component of the 2nd row.
 * @v13: the 4th component of the 2nd row.
 * @v20: the 1st component of the 3rd row.
 * @v21: the 2nd component of the 3rd row.
 * @v22: the 3rd component of the 3rd row.
 * @v23: the 4th component of the 3rd row.
 * @v30: the 1st component of the 4th row.
 * @v31: the 2nd component of the 4th row.
 * @v32: the 3rd component of the 4th row.
 * @v33: the 4th component of the 4th row.
 *
 * Initializes @mat 4x4 matrix.
 */
#define PGM_MAT4X4_INIT(mat,v00,v01,v02,v03,v10,v11,v12,v13,v20,v21,v22,v23,v30,v31,v32,v33) \
  G_STMT_START {                                                        \
    PGM_MAT4X4_ROW_INIT (mat, 0, v00, v01, v02, v03);                   \
    PGM_MAT4X4_ROW_INIT (mat, 1, v10, v11, v12, v13);                   \
    PGM_MAT4X4_ROW_INIT (mat, 2, v20, v21, v22, v23);                   \
    PGM_MAT4X4_ROW_INIT (mat, 3, v30, v31, v32, v33);                   \
  } G_STMT_END

/**
 * PGM_MAT4X4_IDENTITY:
 * @mat: the 4x4 matrix.
 *
 * Initializes @mat 4x4 matrix to the 4x4 identity matrix.
 */
#define PGM_MAT4X4_IDENTITY(mat)                                        \
  G_STMT_START {                                                        \
    PGM_MAT4X4_ROW_INIT (mat, 0, 1.0f, 0.0f, 0.0f, 0.0f);               \
    PGM_MAT4X4_ROW_INIT (mat, 1, 0.0f, 1.0f, 0.0f, 0.0f);               \
    PGM_MAT4X4_ROW_INIT (mat, 2, 0.0f, 0.0f, 1.0f, 0.0f);               \
    PGM_MAT4X4_ROW_INIT (mat, 3, 0.0f, 0.0f, 0.0f, 1.0f);               \
  } G_STMT_END

/**
 * PGM_MAT4X4_COPY:
 * @dest: the destination 4x4 matrix. 
 * @src: the source 4x4 matrix.
 *
 * Copies contents from @src 4x4 matrix to @dest 4x4 matrix.
 */
#define PGM_MAT4X4_COPY(dest,src)                                       \
  G_STMT_START {                                                        \
    memcpy ((void *) (dest), (void *) (src), sizeof (PgmMat4x4));       \
  } G_STMT_END

/* Linear algebra functions */

gfloat pgm_vec3_length               (const PgmVec3 *v);

void   pgm_vec3_normalize            (PgmVec3 *v);

void   pgm_vec3_scale                (PgmVec3 *v,
                                      const PgmVec3 *v1,
                                      const gfloat k);

void   pgm_vec3_sum                  (PgmVec3 *v,
                                      const PgmVec3 *v1,
                                      const PgmVec3 *v2);

void   pgm_vec3_cross_product        (PgmVec3 *v,
                                      const PgmVec3 *v1,
                                      const PgmVec3 *v2);

gfloat pgm_vec3_dot_product          (const PgmVec3 *v1,
                                      const PgmVec3 *v2);


void   pgm_vec4_mult_mat4x4_vec4     (PgmVec4 *v,
                                      const PgmMat4x4 *m1,
                                      const PgmVec4 *v1);

void   pgm_mat4x4_transpose          (PgmMat4x4 *m,
                                      const PgmMat4x4 *m1);

gboolean pgm_mat4x4_inverse          (PgmMat4x4 *m,
                                      const PgmMat4x4 *m1);

void   pgm_mat4x4_rotate_x           (PgmMat4x4 *m,
                                      gfloat theta);

void   pgm_mat4x4_rotate_y           (PgmMat4x4 *m,
                                      gfloat theta);

void   pgm_mat4x4_rotate_z           (PgmMat4x4 *m,
                                      gfloat theta);

void   pgm_mat4x4_rotate             (PgmMat4x4 *m,
                                      const PgmVec3 *axis,
                                      gfloat theta);

void   pgm_mat4x4_translate          (PgmMat4x4 *m,
                                      const PgmVec3 *t);

void   pgm_mat4x4_scale              (PgmMat4x4 *m,
                                      const PgmVec3 *s);

void   pgm_mat4x4_scale_along_axis   (PgmMat4x4 *m,
                                      const PgmVec3 *axis,
                                      gfloat k);

void   pgm_mat4x4_mult_mat4x4_mat4x4 (PgmMat4x4 *m,
                                      const PgmMat4x4 *m1,
                                      const PgmMat4x4 *m2);


gboolean pgm_intersection_line_plane (PgmVec3 *v,
                                      const PgmVec3 *l1,
                                      const PgmVec3 *l2,
                                      const PgmVec3 *p,
                                      const PgmVec3 *pu,
                                      const PgmVec3 *pv);

gboolean pgm_vec3_belongs_rectangle  (const PgmVec3 *p,
                                      const PgmVec3 *r,
                                      const PgmVec3 *ru,
                                      const PgmVec3 *rv);


G_END_DECLS

#endif /* __PGM_LINALG_H__ */
