/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment media rendering library
 *
 * Copyright © 2006, 2007 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author(s): Loïc Molinari <loic@fluendo.com>
 *            Julien Moutte <julien@fluendo.com>
 */

#ifndef __PGM_VIEWPORT_H__
#define __PGM_VIEWPORT_H__

/* pgmviewport.h and pgmviewportfactory.h include eachother */
typedef struct _PgmViewport      PgmViewport;
typedef struct _PgmViewportClass PgmViewportClass;

#include <gdk-pixbuf/gdk-pixbuf.h>
#include "pgmcanvas.h"
#include "pgmviewportfactory.h"
#include "pgmlinearalgebra.h"
#include "pgmevents.h"

G_BEGIN_DECLS

#define PGM_TYPE_VIEWPORT  (pgm_viewport_get_type ())
#define PGM_VIEWPORT(obj) \
  (G_TYPE_CHECK_INSTANCE_CAST((obj), PGM_TYPE_VIEWPORT, PgmViewport))
#define PGM_VIEWPORT_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_CAST((klass), PGM_TYPE_VIEWPORT, PgmViewportClass))
#define PGM_IS_VIEWPORT(obj) \
  (G_TYPE_CHECK_INSTANCE_TYPE((obj), PGM_TYPE_VIEWPORT))
#define PGM_IS_VIEWPORT_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_TYPE((klass), PGM_TYPE_VIEWPORT))
#define PGM_VIEWPORT_GET_CLASS(obj) \
  (G_TYPE_INSTANCE_GET_CLASS((obj), PGM_TYPE_VIEWPORT, PgmViewportClass))
#define PGM_VIEWPORT_CAST(obj) ((PgmViewport *) (obj))

/**
 * PGM_VIEWPORT_HAS_CAPS:
 * @viewport: a #PgmViewport object.
 * @mask: the #PgmViewportCapacity to check, can be a mask of capacities.
 *
 * Retrieves whether @viewport supports the @mask of #PgmViewportCapacity.
 */
#define PGM_VIEWPORT_HAS_CAPS(viewport,mask) \
  ((PGM_VIEWPORT(viewport)->caps_mask) & (mask) == (mask))

/**
 * PgmViewportCursor:
 * @PGM_VIEWPORT_LEFT_ARROW: Standard left arrow system cursor.
 * @PGM_VIEWPORT_INHERIT: Inherited cursor from the environment.
 * @PGM_VIEWPORT_NONE: Hided cursor.
 *
 * Pigment system cursor type.
 */
typedef enum {
  PGM_VIEWPORT_LEFT_ARROW = 0,
  PGM_VIEWPORT_INHERIT    = 1,
  PGM_VIEWPORT_NONE       = 2
} PgmViewportCursor;

/**
 * PgmViewportCapacity:
 * @PGM_VIEWPORT_HARDWARE_ACCELERATION: Supports accelerated rendering thanks
 * to dedicated hardware.
 *
 * Capacities supported by the viewport.
 */
typedef enum {
  PGM_VIEWPORT_HARDWARE_ACCELERATION = (1 << 0),
} PgmViewportCapacity;

/**
 * PgmViewport:
 * @factory: the factory used to instantiate the #PgmViewport.
 * @canvas: the attached canvas.
 * @title: the viewport title.
 * @cursor: the current cursor.
 * @fullscreen: the fullscreen state.
 * @visible: the visible state.
 * @icon: the current icon.
 * @width: the viewport width.
 * @height: the viewport height.
 * @width_mm: the viewport screen width in millimeters.
 * @height_mm: the viewport screen height in millimeters.
 * @projection: the 4x4 projection matrix.
 * @inv_projection: the 4x4 inverse projection matrix.
 * @projected_x: the attached canvas projected x position.
 * @projected_y: the attached canvas projected y position.
 * @projected_width: the attached canvas projected width.
 * @projected_height: the attached canvas projected height.
 * @pixel_aspect_ratio: the pixel-aspect-ratio.
 * @viewport_ratio: the viewport ratio.
 * @canvas_ratio: the attached canvas ratio with pixel-aspect-ratio correction
 * applied.
 * @caps_mask: the viewport mask of supported capacities.
 *
 * The #PgmViewport structure.
 */
struct _PgmViewport {
  GstObject parent;

  /*< public >*/ /* with LOCK */
  PgmViewportFactory *factory;
  PgmCanvas          *canvas;

  gchar             *title;
  PgmViewportCursor  cursor;
  GdkPixbuf         *icon;
  gboolean           fullscreen;
  gboolean           visible;

  gint width, height;
  gint width_mm, height_mm;

  PgmMat4x4 projection, inv_projection;
  gint projected_x, projected_y;
  gint projected_width, projected_height;

  gfloat pixel_aspect_ratio;
  gfloat viewport_ratio;
  gfloat canvas_ratio;

  gulong caps_mask;

  /*< private >*/
  /* Event dedicated lock */
  GMutex *event_lock;

  /* Event handling */
  GList      *event;
  gint        event_fd[2];
  GIOChannel *channel_out;
  GIOChannel *channel_in;
  gint        source;

  /* Signal handler id */
  gulong canvas_size_handler;

  /* Padding */
  gpointer _pgm_reserved[PGM_PADDING];
};

/**
 * PgmViewportClass:
 * @parent_class: the parent class structure.
 * @button_press_event: the "button-press-event" signal.
 * @button_release_event: the "button-release-event" signal.
 * @key_press_event: the "key-press-event" signal.
 * @key_release_event: the "key-release-event" signal.
 * @configure_event: the "configure-event" signal.
 * @delete_event: the "delete-event" signal.
 * @scroll_event: the "scroll-event" signal.
 * @motion_notify_event: the "motion-notify-event" signal.
 * @expose_event: the "expose-event" signal.
 * @update: the update virtual method.
 * @set_title: the set_title virtual method.
 * @show: the show virtual method.
 * @hide: the hide virtual method.
 * @set_cursor: the set_cursor virtual method.
 * @set_icon: the set_icon virtual method.
 * @set_size: the set_size virtual method.
 * @get_size: the get_size virtual method.
 * @set_fullscreen: the set_fullscreen virtual method.
 * @get_fullscreen: the get_fullscreen virtual method.
 * @set_screen_resolution: the set_screen_resolution virtual method.
 * @get_screen_resolution: the get_screen_resolution virtual method.
 * @set_screen_size_mm: the set_screen_size_mm virtual method.
 * @get_screen_size_mm: the get_screen_size_mm virtual method.
 * @set_canvas: the set_canvas virtual method.
 * @update_projection: the update_projection virtual method.
 * @get_pixel_formats: the get_pixel_formats virtual method.
 * @get_caps_mask: the get_caps_mask virtual method.
 *
 * Pigment drawable class.
 */
struct _PgmViewportClass {
  GstObjectClass parent_class;

  /*< public >*/

  /* signals */

  void (*button_press_event)        (PgmViewport *viewport,
                                     PgmEventButton *event);

  void (*button_release_event)      (PgmViewport *viewport,
                                     PgmEventButton *event);

  void (*key_press_event)           (PgmViewport *viewport,
                                     PgmEventKey *event);

  void (*key_release_event)         (PgmViewport *viewport,
                                     PgmEventKey *event);

  void (*configure_event)           (PgmViewport *viewport,
                                     PgmEventConfigure *event);

  void (*delete_event)              (PgmViewport *viewport,
                                     PgmEventDelete *event);

  void (*scroll_event)              (PgmViewport *viewport,
                                     PgmEventScroll *event);

  void (*motion_notify_event)       (PgmViewport *viewport,
                                     PgmEventMotion *event);

  void (*expose_event)              (PgmViewport *viewport,
                                     PgmEventExpose *event);

  /* virtual methods for subclasses */

  PgmError (*update)                (PgmViewport *viewport);

  PgmError (*set_title)             (PgmViewport *viewport,
                                     const gchar *title);

  PgmError (*show)                  (PgmViewport *viewport);

  PgmError (*hide)                  (PgmViewport *viewport);

  PgmError (*set_cursor)            (PgmViewport *viewport,
                                     PgmViewportCursor cursor);

  PgmError (*set_icon)              (PgmViewport *viewport,
                                     GdkPixbuf *icon);

  PgmError (*set_size)              (PgmViewport *viewport,
                                     gint width,
                                     gint height);
  PgmError (*get_size)              (PgmViewport *viewport,
                                     gint *width,
                                     gint *height);

  PgmError (*set_fullscreen)        (PgmViewport *viewport,
                                     gboolean fullscreen);
  PgmError (*get_fullscreen)        (PgmViewport *viewport,
                                     gboolean *fullscreen);

  PgmError (*set_screen_resolution) (PgmViewport *viewport,
                                     gint width,
                                     gint height);
  PgmError (*get_screen_resolution) (PgmViewport *viewport,
                                     gint *width,
                                     gint *height);

  PgmError (*set_screen_size_mm)    (PgmViewport *viewport,
                                     gint width,
                                     gint height);
  PgmError (*get_screen_size_mm)    (PgmViewport *viewport,
                                     gint *width,
                                     gint *height);

  PgmError (*set_canvas)            (PgmViewport *viewport,
                                     PgmCanvas *canvas);

  PgmError (*update_projection)     (PgmViewport *viewport);

  PgmError (*get_pixel_formats)     (PgmViewport *viewport,
                                     gulong *formats_mask);

  PgmError (*get_caps_mask)         (PgmViewport *viewport,
                                     gulong *caps_mask);

  /*< private >*/
  /* Default projection matrix */
  PgmMat4x4 default_projection;

  /* Padding */
  gpointer _pgm_reserved[PGM_PADDING];
};

GType        pgm_viewport_get_type              (void);

PgmError     pgm_viewport_update                (PgmViewport *viewport);

PgmError     pgm_viewport_set_title             (PgmViewport *viewport,
                                                 const gchar *title);
PgmError     pgm_viewport_get_title             (PgmViewport *viewport,
                                                 gchar **title);

PgmError     pgm_viewport_show                  (PgmViewport *viewport);

PgmError     pgm_viewport_hide                  (PgmViewport *viewport);

PgmError     pgm_viewport_is_visible            (PgmViewport *viewport,
                                                 gboolean *visible);

PgmError     pgm_viewport_set_cursor            (PgmViewport *viewport,
                                                 PgmViewportCursor cursor);
PgmError     pgm_viewport_get_cursor            (PgmViewport *viewport,
                                                 PgmViewportCursor *cursor);

PgmError     pgm_viewport_set_icon              (PgmViewport *viewport,
                                                 GdkPixbuf *icon);
PgmError     pgm_viewport_get_icon              (PgmViewport *viewport,
                                                 GdkPixbuf **icon);

PgmError     pgm_viewport_set_size              (PgmViewport *viewport,
                                                 gint width,
                                                 gint height);
PgmError     pgm_viewport_get_size              (PgmViewport *viewport,
                                                 gint *width,
                                                 gint *height);

PgmError     pgm_viewport_set_fullscreen        (PgmViewport *viewport,
                                                 gboolean fullscreen);
PgmError     pgm_viewport_get_fullscreen        (PgmViewport *viewport,
                                                 gboolean *fullscreen);

PgmError     pgm_viewport_set_screen_resolution (PgmViewport *viewport,
                                                 gint width,
                                                 gint height);
PgmError     pgm_viewport_get_screen_resolution (PgmViewport *viewport,
                                                 gint *width,
                                                 gint *height);

PgmError     pgm_viewport_set_screen_size_mm    (PgmViewport *viewport,
                                                 gint width,
                                                 gint height);
PgmError     pgm_viewport_get_screen_size_mm    (PgmViewport *viewport,
                                                 gint *width,
                                                 gint *height);

PgmError     pgm_viewport_push_event            (PgmViewport *viewport,
                                                 PgmEvent *event);

PgmError     pgm_viewport_get_canvas            (PgmViewport *viewport,
                                                 PgmCanvas **canvas);
PgmError     pgm_viewport_set_canvas            (PgmViewport *viewport,
                                                 PgmCanvas *canvas);

PgmError     pgm_viewport_update_projection     (PgmViewport *viewport);

PgmError     pgm_viewport_to_canvas             (PgmViewport *viewport,
                                                 gfloat *viewport_x,
                                                 gfloat *viewport_y,
                                                 gfloat *viewport_z,
                                                 gfloat canvas_x,
                                                 gfloat canvas_y,
                                                 gfloat canvas_z);

PgmError     pgm_viewport_from_canvas           (PgmViewport *viewport,
                                                 gfloat *canvas_x,
                                                 gfloat *canvas_y,
                                                 gfloat *canvas_z,
                                                 gfloat viewport_x,
                                                 gfloat viewport_y,
                                                 gfloat viewport_z);

PgmError     pgm_viewport_get_pixel_formats     (PgmViewport *viewport,
                                                 gulong *formats_mask);

PgmError     pgm_viewport_get_caps_mask         (PgmViewport *viewport,
                                                 gulong *caps_mask);

G_END_DECLS

#endif /* __PGM_VIEWPORT_H__ */
