/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment media rendering library
 *
 * Copyright © 2006, 2007 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author(s): Loïc Molinari <loic@fluendo.com>
 *            Julien Moutte <julien@fluendo.com>
 */

/**
 * SECTION:pgmviewportfactory
 * @short_description: A factory creating viewports.
 * @see_also: #PgmViewport.
 *
 * #PgmViewportFactory is used to create instances of #PgmViewport.
 *
 * Use the pgm_viewport_factory_new() and pgm_viewport_factory_create()
 * functions to create viewport instances or use pgm_viewport_factory_make()
 * as a convenient shortcut.
 *
 * The following code example shows you how to create an OpenGL based viewport.
 *
 * <example id="pigment-factory-creation">
 * <title>Using a viewport factory</title>
 * <programlisting language="c">
 * PgmViewportFactory *factory;
 * PgmViewport *viewport;
 * &nbsp;
 * pgm_init (&amp;argc, &amp;argv);
 * &nbsp;
 * factory = pgm_viewport_factory_new ("opengl");
 * pgm_viewport_factory_create (factory, &amp;viewport);
 * </programlisting>
 * </example>
 *
 * Last reviewed on 2007-04-12 (0.1.5)
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <string.h>
#include "pgmviewportfactory.h"

/* Maximal levels of recursion in the file system to search for plugins */
#define MAX_RECURSION_LEVELS 10

GST_DEBUG_CATEGORY (pgm_viewport_factory_debug);
#define GST_CAT_DEFAULT pgm_viewport_factory_debug

static PgmViewportFactoryClass *parent_class = NULL;

/* Private methods */

/*
 * Recursively search for a Pigment plugin name in a directory with a recursion
 * limit. If a plugin is found it returns TRUE with the module and plugin_desc
 * arguments filled, in the other case it returns FALSE.
 */
static gboolean
find_plugin (const gchar *plugin_name,
             const gchar *plugin_directory,
             gint recursion_level,
             GModule **module,
             PgmPluginDesc **plugin_desc)
{
  GModule *_module;
  PgmPluginDesc *_plugin_desc;
  GDir *dir;
  const gchar *dir_entry;
  gchar *filename;
  gint plugin_name_len;
  gboolean found;

  plugin_name_len = strlen (plugin_name);
  found = FALSE;

  /* Open the directory */
  dir = g_dir_open (plugin_directory, 0, NULL);
  if (G_UNLIKELY (!dir))
    return FALSE;

  /* Parse all the files */
  while ((dir_entry = g_dir_read_name (dir)))
    {
      filename = g_strjoin ("/", plugin_directory, dir_entry, NULL);

      GST_LOG ("looking for plugin named %d at %s", plugin_name, dir_entry);

      /* Check if it's a directory and recurse */
      if (g_file_test (filename, G_FILE_TEST_IS_DIR))
        {
          if (recursion_level > 0)
            {
              GST_LOG ("found directory, recursing");
              found = find_plugin (plugin_name, filename, recursion_level - 1,
                                   module, plugin_desc);
            }
          else
            GST_LOG ("found directory, but recursion level is too deep");
          g_free (filename);

          /* Break our parsing if a plugin has been found */
          if (found)
            break;
          else
            continue;
        }

      /* Check if it's a regular file */
      if (!g_file_test (filename, G_FILE_TEST_IS_REGULAR))
        {
          GST_LOG ("not a regular file, ignoring");
          g_free (filename);
          continue;
        }

      /* Check standard shared library extensions */
      if (!g_str_has_suffix (filename, ".so")
          && !g_str_has_suffix (filename, ".sl")
          && !g_str_has_suffix (filename, ".dll")
          && !g_str_has_suffix (filename, ".dynlib"))
        {
          GST_LOG ("extension is not recognized as module file, ignoring");
          g_free (filename);
          continue;
        }

      /* Open the module */
      GST_DEBUG ("opening module %s with local binding", dir_entry);
      _module = g_module_open (filename, G_MODULE_BIND_LOCAL);
      if (G_UNLIKELY (!_module))
        {
          GST_WARNING ("failed loading module %s: %s", dir_entry,
                       g_module_error ());
          g_free (filename);
          continue;
        }

      /* Search our "pgm_plugin_desc" entry point in the plugin */
      GST_DEBUG ("looking for Pigment plugin entry point in %s", dir_entry);
      if (!g_module_symbol (_module, "pgm_plugin_desc",
                            (gpointer) &_plugin_desc))
        {
          GST_WARNING ("this does not look like a Pigment plugin, "
                       "no entry point");
          if (G_UNLIKELY (!g_module_close (_module)))
            GST_WARNING ("failed closing module %s", filename);
          _module = NULL;
          g_free (filename);
          continue;
        }

      /* Is it the plugin we are looking for? */
      if (!g_ascii_strncasecmp (_plugin_desc->name, plugin_name,
                                strlen (_plugin_desc->name)))
        {
          /* Check create function pointer */
          if (_plugin_desc->create)
            {
              gboolean init_success = TRUE;
              GST_DEBUG ("using plugin %s", filename);

              /* Initialize the plugin */
              if (_plugin_desc->init)
                {
                  init_success = FALSE;
                  GST_DEBUG ("calling plugin init func");
                  init_success = _plugin_desc->init ();
                }
              if (init_success)
                {
                  *module = _module;
                  *plugin_desc = _plugin_desc;
                  g_free (filename);
                  found = TRUE;
                  break;
                }
            }
          else
            GST_WARNING ("plugin %s has a NULL create function pointer",
                         filename);
        }

      /* It's not the plugin we are searching for, let's close the module */
      if (G_UNLIKELY (!g_module_close (_module)))
        GST_WARNING ("failed closing module %s", filename);

      _module = NULL;
      g_free (filename);
    }

  g_dir_close (dir);

  return found;
}

/* GObject stuff */

G_DEFINE_TYPE (PgmViewportFactory, pgm_viewport_factory, GST_TYPE_OBJECT);

static void
pgm_viewport_factory_dispose (GObject *object)
{
  PgmViewportFactory *factory = PGM_VIEWPORT_FACTORY (object);

  if (G_UNLIKELY (!g_module_close (factory->module)))
    GST_WARNING ("Can't close module");

  factory->module = NULL;
  factory->plugin_desc = NULL;

  GST_CALL_PARENT (G_OBJECT_CLASS, dispose, (object));
}

static void
pgm_viewport_factory_class_init (PgmViewportFactoryClass *klass)
{
  GObjectClass *gobject_class = (GObjectClass *) klass;

  parent_class = g_type_class_peek_parent (klass);

  gobject_class->dispose = GST_DEBUG_FUNCPTR (pgm_viewport_factory_dispose);

  GST_DEBUG_CATEGORY_INIT (pgm_viewport_factory_debug, "pgm_viewport_factory",
                           0, "Pigment ViewportFactory object");
}

static void
pgm_viewport_factory_init (PgmViewportFactory *factory)
{
  factory->module = NULL;
  factory->plugin_desc = NULL;
}

/* Public methods */

/**
 * pgm_viewport_factory_new:
 * @name: the name of the factory to create.
 *
 * Creates a new #PgmViewportFactory instance of the given @name.
 *
 * MT safe.
 *
 * Returns: a new #PgmViewport instance or NULL if no factory of this
 * name has been found.
 */
PgmViewportFactory *
pgm_viewport_factory_new (const gchar *name)
{
  PgmViewportFactory *factory = NULL;
  GModule *module = NULL;
  PgmPluginDesc *plugin_desc = NULL;
  const gchar *plugin_path = NULL;
  gboolean found = FALSE;

  /* Try to get the plugin from the plugin path environment variable */
  plugin_path = g_getenv (PGM_PLUGIN_PATH_NAME);
  if (plugin_path)
    {
      gchar **list = NULL;
      gint i = 0;

      GST_DEBUG ("PGM_PLUGIN_PATH set to %s", plugin_path);
      list = g_strsplit (plugin_path, G_SEARCHPATH_SEPARATOR_S, 0);
      while (list[i] && !found)
        {
          found = find_plugin (name, plugin_path, MAX_RECURSION_LEVELS, &module,
                               &plugin_desc);
          i++;
        }
      g_strfreev (list);
    }
  else
    GST_DEBUG ("PGM_PLUGIN_PATH not set");

  /* Then from our usual plugin directory */
  if (!found)
    found = find_plugin (name, PGM_PLUGIN_DIRECTORY, MAX_RECURSION_LEVELS,
                         &module, &plugin_desc);

  /* A plugin has been found, let's create and initialize the factory */
  if (G_LIKELY (found))
    {
      factory = g_object_new (PGM_TYPE_VIEWPORT_FACTORY, NULL);
      factory->module = module;
      factory->plugin_desc = plugin_desc;
      GST_DEBUG_OBJECT (factory, "Created new viewport factory");
    }
  else
    GST_WARNING ("Can't find any plugin with name '%s'", name);

  return factory;
}

/**
 * pgm_viewport_factory_get_description:
 * @factory: a #PgmViewportFactory object.
 * @description: a pointer to a pointer to a #gchar where the description
 * string is going to be be stored. g_free() after use.
 *
 * Retrieves the description of the viewports managed by @factory in
 * @description.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_factory_get_description (PgmViewportFactory *factory,
                                      gchar **description)
{
  g_return_val_if_fail (PGM_IS_VIEWPORT_FACTORY (factory), PGM_ERROR_X);
  g_return_val_if_fail (description != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (factory);

  *description = g_strdup (factory->plugin_desc->description);

  GST_OBJECT_UNLOCK (factory);

  return PGM_ERROR_OK;
}

/**
 * pgm_viewport_factory_get_license:
 * @factory: a #PgmViewportFactory object.
 * @license: a pointer to a pointer to a #gchar where the license string is
 * going to be be stored. g_free() after use.
 *
 * Retrieves the license of the viewports managed by @factory in @license.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_factory_get_license (PgmViewportFactory *factory,
                                  gchar **license)
{
  g_return_val_if_fail (PGM_IS_VIEWPORT_FACTORY (factory), PGM_ERROR_X);
  g_return_val_if_fail (license != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (factory);

  *license = g_strdup (factory->plugin_desc->license);

  GST_OBJECT_UNLOCK (factory);

  return PGM_ERROR_OK;
}

/**
 * pgm_viewport_factory_get_origin:
 * @factory: a #PgmViewportFactory object.
 * @origin: a pointer to a pointer to a #gchar where the origin string is
 * going to be be stored. g_free() after use.
 *
 * Retrieves the origin of the viewports managed by @factory in @origin.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_factory_get_origin (PgmViewportFactory *factory,
                                 gchar **origin)
{
  g_return_val_if_fail (PGM_IS_VIEWPORT_FACTORY (factory), PGM_ERROR_X);
  g_return_val_if_fail (origin != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (factory);

  *origin = g_strdup (factory->plugin_desc->origin);

  GST_OBJECT_UNLOCK (factory);

  return PGM_ERROR_OK;
}

/**
 * pgm_viewport_factory_get_author:
 * @factory: a #PgmViewportFactory object.
 * @author: a pointer to a pointer to a #gchar where the author string is
 * going to be be stored. g_free() after use.
 *
 * Retrieves the author of the viewports managed by @factory in @author.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_factory_get_author (PgmViewportFactory *factory,
                                 gchar **author)
{
  g_return_val_if_fail (PGM_IS_VIEWPORT_FACTORY (factory), PGM_ERROR_X);
  g_return_val_if_fail (author != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (factory);

  *author = g_strdup (factory->plugin_desc->author);

  GST_OBJECT_UNLOCK (factory);

  return PGM_ERROR_OK;
}

/**
 * pgm_viewport_factory_create:
 * @factory: a #PgmViewportFactory object.
 * @viewport: a pointer to a #PgmViewport's pointer where the created viewport
 * is going to be be stored.
 *
 * Creates a new viewport of the type defined by @factory in @viewport.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_factory_create (PgmViewportFactory *factory,
                             PgmViewport **viewport)
{
  g_return_val_if_fail (PGM_IS_VIEWPORT_FACTORY (factory), PGM_ERROR_X);
  g_return_val_if_fail (viewport != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (factory);

  *viewport = factory->plugin_desc->create ();
  (*viewport)->factory = gst_object_ref (GST_OBJECT_CAST (factory));

  GST_OBJECT_UNLOCK (factory);

  GST_DEBUG_OBJECT (*viewport, "created new viewport");

  return PGM_ERROR_OK;
}

/**
 * pgm_viewport_factory_make:
 * @name: the name of the viewport to make.
 * @viewport: a pointer to a #PgmViewport's pointer where the created viewport
 * is going to be be stored.
 *
 * Creates a new viewport of the given @name in @viewport.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_factory_make (const gchar *name,
                           PgmViewport **viewport)
{
  PgmViewportFactory *factory;

  g_return_val_if_fail (viewport != NULL, PGM_ERROR_X);

  factory = pgm_viewport_factory_new (name);
  g_return_val_if_fail (factory != NULL, PGM_ERROR_X);

  pgm_viewport_factory_create (factory, viewport);
  gst_object_unref (GST_OBJECT_CAST (factory));

  return PGM_ERROR_OK;
}
